<?php
session_start();
include_once '../db-config.php';

// --- SECURITY CHECK ---
if (!isset($_SESSION['user_id']) || !isset($_SESSION['user_type']) || $_SESSION['user_type'] !== 'admin') {
    header("location: ../login.php");
    exit;
}

// --- MULTI-DELETE ACTION ---
if ($_SERVER["REQUEST_METHOD"] == "POST" && isset($_POST['action']) && $_POST['action'] === 'delete_selected') {
    if (!empty($_POST['selected_ids']) && is_array($_POST['selected_ids'])) {
        $ids_to_delete = array_map('intval', $_POST['selected_ids']);
        if (!empty($ids_to_delete)) {
            $placeholders = implode(',', array_fill(0, count($ids_to_delete), '?'));
            $types = str_repeat('i', count($ids_to_delete));
            $sql = "DELETE FROM bookings WHERE id IN ($placeholders)";
            $stmt = $conn->prepare($sql);
            $stmt->bind_param($types, ...$ids_to_delete);
            if ($stmt->execute()) {
                $_SESSION['success_message'] = count($ids_to_delete) . " booking(s) deleted.";
            } else {
                $_SESSION['error_message'] = "Error: " . $stmt->error;
            }
        }
    } else {
        $_SESSION['error_message'] = "No bookings selected.";
    }
    header("Location: manage-flight-requests.php?" . http_build_query($_GET));
    exit;
}

// --- SETUP ---
$allowed_filters = ['all', 'pending', 'confirmed', 'cancelled'];
$current_filter = $_GET['filter'] ?? 'pending';
if (!in_array($current_filter, $allowed_filters)) {
    $current_filter = 'pending';
}
function e($string) { return htmlspecialchars($string ?? '', ENT_QUOTES, 'UTF-8'); }

// --- DATA FETCHING ---
function getBookings($conn, $booking_type, $filter, $items_per_page) {
    $page_key = $booking_type . '_page';
    $current_page = max(1, (int)($_GET[$page_key] ?? 1));
    $offset = ($current_page - 1) * $items_per_page;
    $base_sql_from = "FROM bookings b LEFT JOIN users u ON b.user_id = u.id";
    $where_clauses = ["b.booking_type = ?"];
    $params = [$booking_type];
    $types = "s";
    if ($filter !== 'all') {
        $where_clauses[] = "b.status = ?";
        $params[] = $filter;
        $types .= "s";
    }
    $where_sql = "WHERE " . implode(" AND ", $where_clauses);
    $count_sql = "SELECT COUNT(*) as total $base_sql_from $where_sql";
    $stmt_count = $conn->prepare($count_sql);
    $stmt_count->bind_param($types, ...$params);
    $stmt_count->execute();
    $total_items = $stmt_count->get_result()->fetch_assoc()['total'];
    $total_pages = ceil($total_items / $items_per_page);
    $data_sql = "SELECT b.*, u.name as user_name, u.user_type $base_sql_from $where_sql ORDER BY b.created_at DESC LIMIT ?, ?";
    $stmt_data = $conn->prepare($data_sql);
    $final_params = array_merge($params, [$offset, $items_per_page]);
    $final_types = $types . "ii";
    $stmt_data->bind_param($final_types, ...$final_params);
    $stmt_data->execute();
    return ['result' => $stmt_data->get_result(), 'total_pages' => $total_pages, 'current_page' => $current_page];
}

$items_per_page = 15;
$flight_data = getBookings($conn, 'flight', $current_filter, $items_per_page);
$group_data = getBookings($conn, 'group', $current_filter, $items_per_page);

$flight_pagination_params = ['filter' => $current_filter, 'group_page' => $group_data['current_page']];
$group_pagination_params = ['filter' => $current_filter, 'flight_page' => $flight_data['current_page']];

$success_message = $_SESSION['success_message'] ?? null; unset($_SESSION['success_message']);
$error_message = $_SESSION['error_message'] ?? null; unset($_SESSION['error_message']);
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <link rel="icon" type="image/png" href="../images/logo-icon.png">
    <title>Manage Flight Requests</title>
    <link rel="stylesheet" href="admin-style.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.1/css/all.min.css">
    <style>
        .page-header { border-bottom: 2px solid #e9ecef; padding-bottom: 1rem; }
        .bulk-actions-bar { display: flex; flex-wrap: wrap; justify-content: space-between; align-items: center; margin-top: 1.5rem; margin-bottom: 1.5rem; gap: 1rem; }
        .filter-bar { background-color: #f8f9fa; border-radius: 8px; padding: 5px; display: inline-flex; flex-wrap: wrap; gap: 5px; margin-bottom: 0; }
        .filter-bar a { padding: 8px 16px; border-radius: 6px; text-decoration: none; color: #495057; font-weight: 500; transition: background-color 0.2s, color 0.2s; }
        .filter-bar a.active, .filter-bar a:hover { background-color: #007bff; color: #fff; }
        .section-header { display: flex; align-items: center; gap: 10px; font-size: 1.5rem; font-weight: 600; color: var(--primary-dark); background-color: #f8f9fa; padding: 1rem; border-radius: 8px; margin-top: 2rem; margin-bottom: 1rem; }
        
        /* Default Table Styles */
        .table-responsive { overflow-x: auto; }
        .data-table { width: 100%; }
        .data-table th, .data-table td { vertical-align: middle; }
        .data-table thead { background-color: #f1f3f5; }
        .data-table thead th { font-weight: 600; white-space: nowrap; }
        .checkbox-cell { width: 40px; text-align: center; }

        /* --- TRUE RESPONSIVE TABLE STYLES (MOBILE-FIRST) --- */
        @media (max-width: 800px) {
            .content-body { padding: 1rem; }
            .page-header .page-title { font-size: 1.5rem; }
            .bulk-actions-bar { flex-direction: column; align-items: stretch; }
            .filter-bar { justify-content: center; width: 100%; }
            .section-header { font-size: 1.25rem; padding: 0.75rem; }
            .content-card { padding: 0; background-color: transparent; box-shadow: none; border: none; }
            
            .table-responsive { overflow: visible; }
            .data-table { border: 0; }
            .data-table thead {
                border: none; clip: rect(0 0 0 0); height: 1px;
                margin: -1px; overflow: hidden; padding: 0;
                position: absolute; width: 1px;
            }
            .data-table tr {
                display: block;
                border: 1px solid #dee2e6;
                border-radius: 8px;
                margin-bottom: 1rem;
                background-color: #fff;
                box-shadow: 0 2px 4px rgba(0,0,0,0.05);
                position: relative;
                padding: 45px 15px 15px 15px; /* Top padding to make space for checkbox */
            }
            .data-table td {
                display: block;
                text-align: right;
                font-size: 14px;
                border-bottom: 1px dotted #ccc;
                padding: 10px 0;
            }
            .data-table td:last-child { border-bottom: 0; }
            .data-table td::before {
                content: attr(data-label);
                float: left;
                font-weight: bold;
                text-transform: uppercase;
                font-size: 12px;
                color: #6c757d;
            }
            .checkbox-cell {
                position: absolute;
                top: 5px;
                left: 5px;
                border: none;
                padding: 5px;
            }
            .checkbox-cell::before { display: none; }
            .actions-cell { text-align: center; border: none; padding-top: 15px; }
            .actions-cell::before { display: none; }
            .pagination-controls { text-align: center; padding: 1rem 0; }
        }
    </style>
</head>
<body>
    <div class="dashboard-wrapper">
        <?php include 'sidebar.php'; ?>
        <div class="main-content">
            <header class="main-header">
                <button class="menu-toggle" id="menu-toggle"><i class="fas fa-bars"></i></button>
                <div class="user-info"><span>Manage Flight & Group Bookings</span></div>
            </header>
            <main class="content-body">
                <form action="manage-flight-requests.php?<?= http_build_query($_GET) ?>" method="POST" id="bulk-delete-form">
                    <input type="hidden" name="action" value="delete_selected">
                    
                    <div class="page-header"><h1 class="page-title">Booking Requests</h1></div>

                    <div class="bulk-actions-bar">
                        <div class="filter-bar">
                            <a href="manage-flight-requests.php?filter=all" class="<?= $current_filter === 'all' ? 'active' : '' ?>">All</a>
                            <a href="manage-flight-requests.php?filter=pending" class="<?= $current_filter === 'pending' ? 'active' : '' ?>">Pending</a>
                            <a href="manage-flight-requests.php?filter=confirmed" class="<?= $current_filter === 'confirmed' ? 'active' : '' ?>">Confirmed</a>
                            <a href="manage-flight-requests.php?filter=cancelled" class="<?= $current_filter === 'cancelled' ? 'active' : '' ?>">Cancelled</a>
                        </div>
                        <button type="submit" class="btn btn-danger" onclick="return confirm('Are you sure you want to delete selected bookings?');">
                            <i class="fas fa-trash"></i> Delete Selected
                        </button>
                    </div>

                    <?php if ($success_message): ?><div class="notice success"><?= e($success_message); ?></div><?php endif; ?>
                    <?php if ($error_message): ?><div class="notice error"><?= e($error_message); ?></div><?php endif; ?>

                    <div class="section-header"><i class="fa-solid fa-plane"></i> Flight Bookings</div>
                    <div class="content-card">
                        <div class="table-responsive">
                            <table class="data-table flight-table">
                                <thead>
                                    <tr>
                                        <th class="checkbox-cell"><input type="checkbox" class="select-all"></th>
                                        <th>Booking Ref</th><th>Submitted By</th><th>Route</th><th>Date</th><th>Status</th><th>Actions</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php if ($flight_data['result']->num_rows > 0): while ($booking = $flight_data['result']->fetch_assoc()):
                                        $flight_details = json_decode($booking['flight_details'], true);
                                        $route = (isset($flight_details['itineraries'][0])) ? e($flight_details['itineraries'][0]['origin'] ?? '') . ' - ' . e($flight_details['itineraries'][0]['destination'] ?? '') : 'N/A';
                                        $date = (isset($flight_details['itineraries'][0]['departureDate'])) ? date('d M Y', strtotime($flight_details['itineraries'][0]['departureDate'])) : 'N/A';
                                        $status_class = 'status-' . strtolower($booking['status']);
                                        $user_name_display = e($booking['user_name'] ?? 'Unknown User');
                                        $user_type_display = e(ucfirst($booking['user_type'] ?? 'N/A'));
                                    ?>
                                        <tr>
                                            <td class="checkbox-cell" data-label=""><input type="checkbox" name="selected_ids[]" class="row-checkbox" value="<?= $booking['id'] ?>"></td>
                                            <td data-label="Booking Ref"><strong><?= e($booking['booking_ref']) ?></strong></td>
                                            <td data-label="Submitted By"><?= $user_name_display ?> (<?= $user_type_display ?>)</td>
                                            <td data-label="Route"><?= $route ?></td>
                                            <td data-label="Date"><?= $date ?></td>
                                            <td data-label="Status"><span class="status-badge <?= $status_class ?>"><?= e(ucfirst($booking['status'])) ?></span></td>
                                            <td class="actions-cell" data-label="Actions"><a href="process-booking.php?id=<?= $booking['id'] ?>" class="btn btn-sm btn-primary" title="View & Process"><i class="fas fa-eye"></i> View</a></td>
                                        </tr>
                                    <?php endwhile; else: ?>
                                        <tr><td colspan="7" class="text-center">No flight bookings found for this filter.</td></tr>
                                    <?php endif; ?>
                                </tbody>
                            </table>
                        </div>
                        <?php if ($flight_data['total_pages'] > 1): ?>
                            <div class="pagination-controls">
                                <?php if ($flight_data['current_page'] > 1): ?><a href="?<?= http_build_query(array_merge($flight_pagination_params, ['flight_page' => $flight_data['current_page'] - 1])) ?>" class="btn btn-secondary btn-sm">« Previous</a><?php endif; ?>
                                <span class="page-info">Page <?= $flight_data['current_page'] ?> of <?= $flight_data['total_pages'] ?></span>
                                <?php if ($flight_data['current_page'] < $flight_data['total_pages']): ?><a href="?<?= http_build_query(array_merge($flight_pagination_params, ['flight_page' => $flight_data['current_page'] + 1])) ?>" class="btn btn-secondary btn-sm">Next »</a><?php endif; ?>
                            </div>
                        <?php endif; ?>
                    </div>

                    <div class="section-header"><i class="fa-solid fa-users"></i> Group Bookings</div>
                    <div class="content-card">
                        <div class="table-responsive">
                            <table class="data-table group-table">
                                <thead>
                                    <tr>
                                        <th class="checkbox-cell"><input type="checkbox" class="select-all"></th>
                                        <th>Booking Ref</th><th>Submitted By</th><th>Route</th><th>Date</th><th>Status</th><th>Actions</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php if ($group_data['result']->num_rows > 0): while ($booking = $group_data['result']->fetch_assoc()):
                                        $flight_details = json_decode($booking['flight_details'], true);
                                        $route = e($flight_details['route'] ?? 'N/A');
                                        $date = !empty($flight_details['departure_date']) ? date('d M Y', strtotime($flight_details['departure_date'])) : 'N/A';
                                        $status_class = 'status-' . strtolower($booking['status']);
                                        $user_name_display = e($booking['user_name'] ?? 'Unknown User');
                                        $user_type_display = e(ucfirst($booking['user_type'] ?? 'N/A'));
                                    ?>
                                        <tr>
                                            <td class="checkbox-cell" data-label=""><input type="checkbox" name="selected_ids[]" class="row-checkbox" value="<?= $booking['id'] ?>"></td>
                                            <td data-label="Booking Ref"><strong><?= e($booking['booking_ref']) ?></strong></td>
                                            <td data-label="Submitted By"><?= $user_name_display ?> (<?= $user_type_display ?>)</td>
                                            <td data-label="Route"><?= $route ?></td>
                                            <td data-label="Date"><?= $date ?></td>
                                            <td data-label="Status"><span class="status-badge <?= $status_class ?>"><?= e(ucfirst($booking['status'])) ?></span></td>
                                            <td class="actions-cell" data-label="Actions"><a href="process-booking.php?id=<?= $booking['id'] ?>" class="btn btn-sm btn-primary" title="View & Process"><i class="fas fa-eye"></i> View</a></td>
                                        </tr>
                                    <?php endwhile; else: ?>
                                        <tr><td colspan="7" class="text-center">No group bookings found for this filter.</td></tr>
                                    <?php endif; ?>
                                </tbody>
                            </table>
                        </div>
                        <?php if ($group_data['total_pages'] > 1): ?>
                            <div class="pagination-controls">
                                <?php if ($group_data['current_page'] > 1): ?><a href="?<?= http_build_query(array_merge($group_pagination_params, ['group_page' => $group_data['current_page'] - 1])) ?>" class="btn btn-secondary btn-sm">« Previous</a><?php endif; ?>
                                <span class="page-info">Page <?= $group_data['current_page'] ?> of <?= $group_data['total_pages'] ?></span>
                                <?php if ($group_data['current_page'] < $group_data['total_pages']): ?><a href="?<?= http_build_query(array_merge($group_pagination_params, ['group_page' => $group_data['current_page'] + 1])) ?>" class="btn btn-secondary btn-sm">Next »</a><?php endif; ?>
                            </div>
                        <?php endif; ?>
                    </div>
                </form>
            </main>
        </div>
    </div>

    <script>
    document.addEventListener('DOMContentLoaded', function() {
        document.querySelectorAll('.select-all').forEach(selectAllCheckbox => {
            selectAllCheckbox.addEventListener('change', function() {
                const table = this.closest('table');
                table.querySelectorAll('.row-checkbox').forEach(checkbox => { checkbox.checked = this.checked; });
            });
        });

        document.querySelectorAll('.row-checkbox').forEach(rowCheckbox => {
            rowCheckbox.addEventListener('change', function() {
                const table = this.closest('table');
                const selectAll = table.querySelector('.select-all');
                if (!this.checked) {
                    selectAll.checked = false;
                } else {
                    const allRowCheckboxes = Array.from(table.querySelectorAll('.row-checkbox'));
                    selectAll.checked = allRowCheckboxes.every(cb => cb.checked);
                }
            });
        });

        const menuToggle = document.getElementById('menu-toggle');
        if (menuToggle) {
            menuToggle.addEventListener('click', () => document.querySelector('.dashboard-wrapper').classList.toggle('sidebar-open'));
        }
    });
    </script>
</body>
</html>