<?php
/**
 * manage-group-fares.php (v6.1 - URL/Upload Combo)
 * - The airline form now has two fields: a file upload and a text URL input.
 * - The backend logic prioritizes the file upload. If no file is uploaded, it uses the URL input.
 * - This gives the admin the flexibility to use either locally hosted logos or external ones.
 * - Also fixes logo display within the admin panel tables.
 */

// Handles security, sessions, and the database connection ($conn)
include_once '../db-config.php';

// =======================================================
// SECURITY GATEWAY
// =======================================================
if (!isset($_SESSION['user_id']) || !isset($_SESSION['user_type']) || $_SESSION['user_type'] !== 'admin') {
    header("location: login.php");
    exit;
}

// Define the directory for airline logo uploads
define('AIRLINE_UPLOAD_DIR', 'uploads/airlines/');

// Helper function to get the correct display path for logos in the admin panel
function get_admin_logo_src($logo_path) {
    if (empty($logo_path)) return '';
    if (preg_match('/^(https?:)?\/\//', $logo_path)) {
        return htmlspecialchars($logo_path); // It's a URL
    }
    return htmlspecialchars($logo_path); // It's a local path relative to the admin folder
}


// --- HANDLE ALL FORM SUBMISSIONS (Airlines and Fares) ---
if ($_SERVER["REQUEST_METHOD"] == "POST" && isset($_POST['action'])) {

    // --- Actions for the Airline Library ---
    if ($_POST['form_type'] === 'airline') {
        
        // --- START: Combined File Upload & URL Logic ---
        $logo_path_to_save = $_POST['current_logo_path'] ?? ''; // Default to existing logo on update

        // PRIORITY 1: Check if a new file has been uploaded
        if (isset($_FILES['logo_upload']) && $_FILES['logo_upload']['error'] == UPLOAD_ERR_OK) {
            if (!is_dir(AIRLINE_UPLOAD_DIR)) {
                mkdir(AIRLINE_UPLOAD_DIR, 0755, true);
            }

            $file_name = time() . '_' . basename($_FILES['logo_upload']['name']);
            $target_path = AIRLINE_UPLOAD_DIR . $file_name;

            if (move_uploaded_file($_FILES['logo_upload']['tmp_name'], $target_path)) {
                // If this is an update and there was an old logo that was a local file, delete it
                if (!empty($logo_path_to_save) && !preg_match('/^(https?:)?\/\//', $logo_path_to_save) && file_exists($logo_path_to_save)) {
                    unlink($logo_path_to_save);
                }
                $logo_path_to_save = $target_path; // Set the new local path to be saved in DB
            } else {
                $_SESSION['error_message'] = "Error uploading the logo file.";
                header("Location: manage-group-fares.php");
                exit();
            }
        } 
        // PRIORITY 2: If no file was uploaded, check if a URL was provided
        elseif (!empty(trim($_POST['logo_url']))) {
             // If this is an update and the old logo was a local file, delete it as we are switching to a URL
            if (!empty($logo_path_to_save) && !preg_match('/^(https?:)?\/\//', $logo_path_to_save) && file_exists($logo_path_to_save)) {
                unlink($logo_path_to_save);
            }
            $logo_path_to_save = trim($_POST['logo_url']);
        }
        // --- END: Combined Logic ---

        if ($_POST['action'] === 'add_airline') {
            $stmt = $conn->prepare("INSERT INTO airlines (airline_name, logo_url) VALUES (?, ?)");
            $stmt->bind_param("ss", $_POST['airline_name'], $logo_path_to_save);
            $stmt->execute();
            $_SESSION['success_message'] = "Airline added successfully!";
        }
        if ($_POST['action'] === 'update_airline' && isset($_POST['id'])) {
            $stmt = $conn->prepare("UPDATE airlines SET airline_name = ?, logo_url = ? WHERE id = ?");
            $stmt->bind_param("ssi", $_POST['airline_name'], $logo_path_to_save, $_POST['id']);
            $stmt->execute();
            $_SESSION['success_message'] = "Airline updated successfully!";
        }
        if ($_POST['action'] === 'delete_airline') {
            // First, get the logo path to delete the file if it's local
            $stmt_get = $conn->prepare("SELECT logo_url FROM airlines WHERE id = ?");
            $stmt_get->bind_param("i", $_POST['id']);
            $stmt_get->execute();
            $result = $stmt_get->get_result();
            if($row = $result->fetch_assoc()){
                // Only delete if it's NOT a URL and the file exists
                if (!empty($row['logo_url']) && !preg_match('/^(https?:)?\/\//', $row['logo_url']) && file_exists($row['logo_url'])) {
                    unlink($row['logo_url']);
                }
            }
            // Now, delete the record from DB
            $stmt = $conn->prepare("DELETE FROM airlines WHERE id = ?");
            $stmt->bind_param("i", $_POST['id']);
            $stmt->execute();
            $_SESSION['success_message'] = "Airline deleted successfully!";
        }
    }

    // --- Actions for Group Fares (no changes here) ---
    if ($_POST['form_type'] === 'group_fare') {
        $flight_details_json = json_encode(['outbound' => trim($_POST['flight_outbound']), 'inbound'  => trim($_POST['flight_inbound']), 'baggage'  => trim($_POST['flight_baggage'])]);

        if ($_POST['action'] === 'add_fare') {
            try {
                $sql = "INSERT INTO group_fares (group_ref_id, airline_id, sector, route, duration_days, departure_date, total_seats, remaining_seats, price_adult, price_adult_agent, price_child, price_child_agent, price_infant, price_infant_agent, agent_incentive, flight_details_json, is_active) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)";
                $stmt = $conn->prepare($sql);
                $remaining_seats = !empty($_POST['remaining_seats']) ? $_POST['remaining_seats'] : $_POST['total_seats'];
                
                $price_adult_agent = !empty($_POST['price_adult_agent']) ? $_POST['price_adult_agent'] : null;
                $price_child_agent = !empty($_POST['price_child_agent']) ? $_POST['price_child_agent'] : null;
                $price_infant_agent = !empty($_POST['price_infant_agent']) ? $_POST['price_infant_agent'] : null;
                $agent_incentive = !empty($_POST['agent_incentive']) ? $_POST['agent_incentive'] : 0.00;

                $stmt->bind_param(
                    "sissisiidddddddsi",
                    $_POST['group_ref_id'], $_POST['airline_id'], $_POST['sector'], $_POST['route'], $_POST['duration_days'],
                    $_POST['departure_date'], $_POST['total_seats'], $remaining_seats, $_POST['price_adult'], $price_adult_agent,
                    $_POST['price_child'], $price_child_agent, $_POST['price_infant'], $price_infant_agent, $agent_incentive,
                    $flight_details_json, $_POST['is_active']
                );
                $stmt->execute();
                $_SESSION['success_message'] = "Group Fare added successfully!";
            } catch (Exception $e) {
                $_SESSION['error_message'] = "Error: " . $e->getMessage();
            }
        }

        if ($_POST['action'] === 'update_fare' && isset($_POST['id'])) {
            try {
                $sql = "UPDATE group_fares SET group_ref_id = ?, airline_id = ?, sector = ?, route = ?, duration_days = ?, departure_date = ?, total_seats = ?, remaining_seats = ?, price_adult = ?, price_adult_agent = ?, price_child = ?, price_child_agent = ?, price_infant = ?, price_infant_agent = ?, agent_incentive = ?, flight_details_json = ?, is_active = ? WHERE id = ?";
                $stmt = $conn->prepare($sql);

                $price_adult_agent = !empty($_POST['price_adult_agent']) ? $_POST['price_adult_agent'] : null;
                $price_child_agent = !empty($_POST['price_child_agent']) ? $_POST['price_child_agent'] : null;
                $price_infant_agent = !empty($_POST['price_infant_agent']) ? $_POST['price_infant_agent'] : null;
                $agent_incentive = !empty($_POST['agent_incentive']) ? $_POST['agent_incentive'] : 0.00;

                $stmt->bind_param(
                    "sissisiidddddddsii",
                    $_POST['group_ref_id'], $_POST['airline_id'], $_POST['sector'], $_POST['route'], $_POST['duration_days'],
                    $_POST['departure_date'], $_POST['total_seats'], $_POST['remaining_seats'], $_POST['price_adult'], $price_adult_agent,
                    $_POST['price_child'], $price_child_agent, $_POST['price_infant'], $price_infant_agent, $agent_incentive,
                    $flight_details_json, $_POST['is_active'], $_POST['id']
                );
                $stmt->execute();
                $_SESSION['success_message'] = "Group Fare updated successfully!";
            } catch (Exception $e) {
                $_SESSION['error_message'] = "Error: " . $e->getMessage();
            }
        }

        if ($_POST['action'] === 'delete_fare' && isset($_POST['id'])) {
            $stmt = $conn->prepare("DELETE FROM group_fares WHERE id = ?");
            $stmt->bind_param("i", $_POST['id']);
            $stmt->execute();
            $_SESSION['success_message'] = "Group Fare deleted.";
        }
    }

    header("Location: manage-group-fares.php");
    exit();
}

// --- Fetch all data needed for the page ---
$airlines_result = $conn->query("SELECT id, airline_name, logo_url FROM airlines ORDER BY airline_name ASC");
$fare_to_edit = null;
$airline_to_edit = null;
$edit_mode_fare = false;
$edit_mode_airline = false;

// Initialize flight detail variables
$outbound_details = '';
$inbound_details = '';
$baggage_details = '';

// Check for Group Fare edit action
if (isset($_GET['action']) && $_GET['action'] === 'edit_fare' && isset($_GET['id'])) {
    $edit_mode_fare = true;
    $stmt = $conn->prepare("SELECT * FROM group_fares WHERE id = ?");
    $stmt->bind_param("i", $_GET['id']);
    $stmt->execute();
    $result = $stmt->get_result();
    $fare_to_edit = $result->fetch_assoc();

    if ($fare_to_edit && !empty($fare_to_edit['flight_details_json'])) {
        $decoded_details = json_decode($fare_to_edit['flight_details_json'], true);
        $outbound_details = $decoded_details['outbound'] ?? '';
        $inbound_details = $decoded_details['inbound'] ?? '';
        $baggage_details = $decoded_details['baggage'] ?? '';
    }
}

// Check for Airline edit action
if (isset($_GET['action']) && $_GET['action'] === 'edit_airline' && isset($_GET['id'])) {
    $edit_mode_airline = true;
    $stmt = $conn->prepare("SELECT * FROM airlines WHERE id = ?");
    $stmt->bind_param("i", $_GET['id']);
    $stmt->execute();
    $result = $stmt->get_result();
    $airline_to_edit = $result->fetch_assoc();
}

$fares_result = $conn->query("SELECT gf.*, a.airline_name, a.logo_url FROM group_fares gf LEFT JOIN airlines a ON gf.airline_id = a.id ORDER BY gf.departure_date DESC");
$current_page = basename($_SERVER['PHP_SELF']);
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Manage Group Fares & Airlines</title>
    <link rel="icon" type="image/png" href="../images/logo-icon.png">
    <link rel="stylesheet" href="admin-style.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.1/css/all.min.css">
    <style>
        .airline-logo-cell img { max-height: 25px; vertical-align: middle; margin-right: 10px; }
        .form-divider { border: 0; height: 1px; background-color: var(--border-color); margin: 2rem 0; }
        .form-subtitle { font-size: 1.1rem; margin-bottom: 1rem; color: var(--primary-color); font-weight: 600; }
        .dashboard-grid-2col { display: grid; grid-template-columns: 1fr 1fr; gap: 2rem; align-items: start; }
        .current-logo-preview { max-height: 40px; margin-bottom: 10px; display: block; border: 1px solid var(--border-color); padding: 5px; border-radius: 4px; }

        /* Responsive Styles */
        @media (max-width: 992px) {
            .dashboard-grid-2col { grid-template-columns: 1fr; }
            .form-grid { grid-template-columns: 1fr 1fr; }
        }
        @media (max-width: 768px) {
            .content-body { padding: 1rem; }
            .form-grid { grid-template-columns: 1fr; }
        }
        
        /* Table Responsive Styles */
        .table-responsive { width: 100%; overflow-x: auto; -webkit-overflow-scrolling: touch; }
        .data-table { width: 100%; min-width: 1200px; /* Force scroll on smaller screens */ }
    </style>
</head>
<body>
    <div class="dashboard-wrapper">
        <?php include 'sidebar.php'; ?>
        <div class="main-content">
            <header class="main-header">
                <button class="menu-toggle" id="menu-toggle"><i class="fas fa-bars"></i></button>
                <div class="user-info"><span>Welcome, <?php echo htmlspecialchars(ucfirst($_SESSION['user_name'] ?? 'Admin')); ?></span></div>
            </header>
            <main class="content-body" id="top">
                <?php if (isset($_SESSION['success_message'])): ?><div class="notice success"><?= $_SESSION['success_message']; unset($_SESSION['success_message']); ?></div><?php endif; ?>
                <?php if (isset($_SESSION['error_message'])): ?><div class="notice error"><?= $_SESSION['error_message']; unset($_SESSION['error_message']); ?></div><?php endif; ?>

                <div class="content-card">
                    <!-- Fare Management Form (No changes needed here) -->
                    <h2 class="form-title"><?= $edit_mode_fare ? 'Edit Group Fare' : 'Add New Group Fare' ?></h2>
                    <form action="manage-group-fares.php" method="POST" class="package-form">
                        <input type="hidden" name="form_type" value="group_fare">
                        <input type="hidden" name="action" value="<?= $edit_mode_fare ? 'update_fare' : 'add_fare' ?>">
                        <?php if ($edit_mode_fare): ?><input type="hidden" name="id" value="<?= $fare_to_edit['id'] ?>"><?php endif; ?>

                        <div class="form-grid">
                            <div class="form-group"><label for="group_ref_id">Group Ref ID *</label><input type="text" id="group_ref_id" name="group_ref_id" placeholder="e.g. SV-UMR-OCT25" value="<?= htmlspecialchars($fare_to_edit['group_ref_id'] ?? '') ?>" required></div>
                            <div class="form-group"><label for="airline_id">Airline *</label>
                                <select id="airline_id" name="airline_id" required>
                                    <option value="">-- Select an Airline --</option>
                                    <?php if ($airlines_result->num_rows > 0): $airlines_result->data_seek(0); ?>
                                        <?php while ($airline = $airlines_result->fetch_assoc()): ?>
                                            <option value="<?= $airline['id'] ?>" <?= (($fare_to_edit['airline_id'] ?? '') == $airline['id']) ? 'selected' : '' ?>><?= htmlspecialchars($airline['airline_name']) ?></option>
                                        <?php endwhile; ?>
                                    <?php endif; ?>
                                </select>
                            </div>
                            <div class="form-group"><label for="sector">Sector *</label><input type="text" id="sector" name="sector" placeholder="e.g. Umrah Group" value="<?= htmlspecialchars($fare_to_edit['sector'] ?? 'Umrah Group') ?>" required></div>
                            <div class="form-group"><label for="route">Route *</label><input type="text" id="route" name="route" placeholder="e.g. ISB-JED-ISB" value="<?= htmlspecialchars($fare_to_edit['route'] ?? '') ?>" required></div>
                            <div class="form-group"><label for="duration_days">Duration (Days) *</label><input type="number" id="duration_days" name="duration_days" placeholder="e.g. 15" value="<?= htmlspecialchars($fare_to_edit['duration_days'] ?? '') ?>" required></div>
                            <div class="form-group"><label for="departure_date">Departure Date *</label><input type="date" id="departure_date" name="departure_date" value="<?= htmlspecialchars($fare_to_edit['departure_date'] ?? '') ?>" required></div>
                            <div class="form-group"><label for="total_seats">Total Seats *</label><input type="number" id="total_seats" name="total_seats" placeholder="e.g. 25" value="<?= htmlspecialchars($fare_to_edit['total_seats'] ?? '') ?>" required></div>
                            <div class="form-group"><label for="remaining_seats">Remaining Seats</label><input type="number" id="remaining_seats" name="remaining_seats" placeholder="Leave blank to match total" value="<?= htmlspecialchars($fare_to_edit['remaining_seats'] ?? '') ?>"></div>
                            <div class="form-group"><label for="is_active">Status</label><select id="is_active" name="is_active">
                                    <option value="1" <?= (($fare_to_edit['is_active'] ?? 1) == 1) ? 'selected' : '' ?>>Active</option>
                                    <option value="0" <?= (($fare_to_edit['is_active'] ?? 1) == 0) ? 'selected' : '' ?>>Inactive</option>
                                </select></div>
                        </div>
                        <hr class="form-divider">
                        <h3 class="form-subtitle">Customer Pricing (PKR)</h3>
                        <div class="form-grid">
                            <div class="form-group"><label for="price_adult">Adult Price *</label><input type="number" step="1" id="price_adult" name="price_adult" placeholder="e.g. 250000" value="<?= htmlspecialchars($fare_to_edit['price_adult'] ?? '') ?>" required></div>
                            <div class="form-group"><label for="price_child">Child Price</label><input type="number" step="1" id="price_child" name="price_child" placeholder="e.g. 220000" value="<?= htmlspecialchars($fare_to_edit['price_child'] ?? '0') ?>"></div>
                            <div class="form-group"><label for="price_infant">Infant Price</label><input type="number" step="1" id="price_infant" name="price_infant" placeholder="e.g. 80000" value="<?= htmlspecialchars($fare_to_edit['price_infant'] ?? '0') ?>"></div>
                        </div>
                        <hr class="form-divider">
                        <h3 class="form-subtitle">Agent Pricing (PKR)</h3>
                        <div class="form-grid">
                            <div class="form-group"><label for="price_adult_agent">Agent Adult Price</label><input type="number" step="1" id="price_adult_agent" name="price_adult_agent" placeholder="Agent rate for adults" value="<?= htmlspecialchars($fare_to_edit['price_adult_agent'] ?? '') ?>"></div>
                            <div class="form-group"><label for="price_child_agent">Agent Child Price</label><input type="number" step="1" id="price_child_agent" name="price_child_agent" placeholder="Agent rate for children" value="<?= htmlspecialchars($fare_to_edit['price_child_agent'] ?? '') ?>"></div>
                            <div class="form-group"><label for="price_infant_agent">Agent Infant Price</label><input type="number" step="1" id="price_infant_agent" name="price_infant_agent" placeholder="Agent rate for infants" value="<?= htmlspecialchars($fare_to_edit['price_infant_agent'] ?? '') ?>"></div>
                             <div class="form-group"><label for="agent_incentive">Agent Incentive</label><input type="number" step="1" id="agent_incentive" name="agent_incentive" placeholder="e.g. 5000" value="<?= htmlspecialchars($fare_to_edit['agent_incentive'] ?? '0') ?>"></div>
                        </div>
                        <hr class="form-divider">
                        <h3 class="form-subtitle">Flight & Baggage Details</h3>
                        <div class="form-group"><label for="flight_outbound">Outbound (e.g. SV727 31 Jul ISB-JED 0300 0610)</label><textarea id="flight_outbound" name="flight_outbound" rows="2" placeholder="One line per segment if multiple stops"><?= htmlspecialchars($outbound_details) ?></textarea></div>
                        <div class="form-group"><label for="flight_inbound">Inbound (e.g. SV726 20 Aug JED-ISB 1810 0120)</label><textarea id="flight_inbound" name="flight_inbound" rows="2" placeholder="One line per segment if multiple stops"><?= htmlspecialchars($inbound_details) ?></textarea></div>
                        <div class="form-group"><label for="flight_baggage">Baggage Allowance</label><textarea id="flight_baggage" name="flight_baggage" rows="2" placeholder="e.g. 1 piece 23 KG Check-in, 7 KG Cabin"><?= htmlspecialchars($baggage_details) ?></textarea></div>

                        <div class="form-actions"><button type="submit" class="btn btn-primary"><?= $edit_mode_fare ? 'Update Fare' : 'Add Fare' ?></button><?php if ($edit_mode_fare): ?><a href="manage-group-fares.php" class="btn btn-secondary">Cancel Edit</a><?php endif; ?></div>
                    </form>
                </div>

                <div class="content-card">
                    <h2 class="form-title">Existing Group Fares</h2>
                    <div class="table-responsive">
                        <table class="data-table">
                             <thead>
                                <tr>
                                    <th>Ref ID</th>
                                    <th>Airline</th>
                                    <th>Route</th>
                                    <th>Dep. Date</th>
                                    <th>Seats</th>
                                    <th>Cust. Price (Adult)</th>
                                    <th>Agent Price (Adult)</th>
                                    <th>Status</th>
                                    <th>Actions</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php if ($fares_result->num_rows > 0): while ($fare = $fares_result->fetch_assoc()): ?>
                                    <tr class="<?= ($edit_mode_fare && isset($fare_to_edit['id']) && $fare['id'] === $fare_to_edit['id']) ? 'editing' : '' ?>">
                                        <td><strong><?= htmlspecialchars($fare['group_ref_id']) ?></strong></td>
                                        <td class="airline-logo-cell">
                                            <?php $logo_src = get_admin_logo_src($fare['logo_url']); if (!empty($logo_src)): ?>
                                                <img src="<?= $logo_src ?>" alt="Logo">
                                            <?php endif; ?>
                                            <?= htmlspecialchars($fare['airline_name']) ?>
                                        </td>
                                        <td><?= htmlspecialchars($fare['route']) ?></td>
                                        <td><?= date("d M Y", strtotime($fare['departure_date'])) ?></td>
                                        <td><?= htmlspecialchars($fare['remaining_seats']) ?> / <?= htmlspecialchars($fare['total_seats']) ?></td>
                                        <td><?= number_format($fare['price_adult']) ?></td>
                                        <td><?= isset($fare['price_adult_agent']) ? number_format($fare['price_adult_agent']) : 'N/A' ?></td>
                                        <td><span class="status <?= $fare['is_active'] ? 'active' : 'inactive' ?>"><?= $fare['is_active'] ? 'Active' : 'Inactive' ?></span></td>
                                        <td class="actions-cell">
                                            <a href="?action=edit_fare&id=<?= $fare['id'] ?>#top" class="btn btn-sm btn-secondary" title="Edit"><i class="fas fa-edit"></i></a>
                                            <form method="POST" action="manage-group-fares.php" onsubmit="return confirm('Delete this fare?');" style="display:inline;"><input type="hidden" name="form_type" value="group_fare"><input type="hidden" name="action" value="delete_fare"><input type="hidden" name="id" value="<?= $fare['id'] ?>"><button type="submit" class="btn btn-sm btn-danger" title="Delete"><i class="fas fa-trash"></i></button></form>
                                        </td>
                                    </tr>
                                <?php endwhile; else: ?>
                                    <tr><td colspan="9" class="empty-state">No group fares found.</td></tr>
                                <?php endif; ?>
                            </tbody>
                        </table>
                    </div>
                </div>

                <div class="dashboard-grid-2col">
                    <div class="content-card">
                        <h2 class="form-title"><?= $edit_mode_airline ? 'Edit Airline' : 'Add New Airline to Library' ?></h2>
                        <form action="manage-group-fares.php" method="POST" class="package-form" enctype="multipart/form-data">
                            <input type="hidden" name="form_type" value="airline">
                            <input type="hidden" name="action" value="<?= $edit_mode_airline ? 'update_airline' : 'add_airline' ?>">
                            <?php if ($edit_mode_airline): ?>
                                <input type="hidden" name="id" value="<?= $airline_to_edit['id'] ?>">
                                <input type="hidden" name="current_logo_path" value="<?= htmlspecialchars($airline_to_edit['logo_url'] ?? '') ?>">
                            <?php endif; ?>

                            <div class="form-group">
                                <label for="airline_name">Airline Name *</label>
                                <input type="text" id="airline_name" name="airline_name" placeholder="e.g. Saudia" value="<?= htmlspecialchars($airline_to_edit['airline_name'] ?? '') ?>" required>
                            </div>
                            
                            <!-- === MODIFIED SECTION: URL OR UPLOAD === -->
                            <div class="form-group">
                                <label for="logo_upload">Upload Logo (Overrides URL)</label>
                                <?php if ($edit_mode_airline && !empty($airline_to_edit['logo_url'])): ?>
                                    <img src="<?= get_admin_logo_src($airline_to_edit['logo_url']) ?>" alt="Current Logo" class="current-logo-preview">
                                <?php endif; ?>
                                <input type="file" id="logo_upload" name="logo_upload" accept="image/*">
                            </div>
                            
                            <div class="form-group">
                                <label for="logo_url">Or Enter Logo URL</label>
                                <?php
                                    $logo_url_value = '';
                                    if ($edit_mode_airline && !empty($airline_to_edit['logo_url'])) {
                                        // Only show the URL in the box if it's actually a URL
                                        if (preg_match('/^(https?:)?\/\//', $airline_to_edit['logo_url'])) {
                                            $logo_url_value = $airline_to_edit['logo_url'];
                                        }
                                    }
                                ?>
                                <input type="text" id="logo_url" name="logo_url" placeholder="https://example.com/logo.png" value="<?= htmlspecialchars($logo_url_value) ?>">
                            </div>
                            <!-- === END MODIFIED SECTION === -->

                            <div class="form-actions">
                                <button type="submit" class="btn btn-primary"><?= $edit_mode_airline ? 'Update Airline' : 'Add Airline' ?></button>
                                <?php if ($edit_mode_airline): ?><a href="manage-group-fares.php" class="btn btn-secondary">Cancel Edit</a><?php endif; ?>
                            </div>
                        </form>
                    </div>
                    <div class="content-card">
                        <h2 class="form-title">Airline Library</h2>
                        <div class="table-responsive">
                            <table class="data-table" style="min-width: 400px;">
                                <thead><tr><th>Airline</th><th>Actions</th></tr></thead>
                                <tbody>
                                    <?php if ($airlines_result->num_rows > 0): $airlines_result->data_seek(0); ?>
                                        <?php while ($airline = $airlines_result->fetch_assoc()): ?>
                                            <tr class="<?= ($edit_mode_airline && isset($airline_to_edit['id']) && $airline['id'] === $airline_to_edit['id']) ? 'editing' : '' ?>">
                                                <td class="airline-logo-cell">
                                                    <?php $logo_src = get_admin_logo_src($airline['logo_url']); if (!empty($logo_src)): ?>
                                                        <img src="<?= $logo_src ?>" alt="Logo">
                                                    <?php endif; ?>
                                                    <strong><?= htmlspecialchars($airline['airline_name']) ?></strong>
                                                </td>
                                                <td class="actions-cell">
                                                    <a href="?action=edit_airline&id=<?= $airline['id'] ?>#top" class="btn btn-sm btn-secondary" title="Edit"><i class="fas fa-edit"></i></a>
                                                    <form method="POST" action="manage-group-fares.php" onsubmit="return confirm('Delete this airline?');" style="display:inline;"><input type="hidden" name="form_type" value="airline"><input type="hidden" name="action" value="delete_airline"><input type="hidden" name="id" value="<?= $airline['id'] ?>"><button type="submit" class="btn btn-sm btn-danger" title="Delete"><i class="fas fa-trash"></i></button></form>
                                                </td>
                                            </tr>
                                        <?php endwhile;
                                    else: ?><tr><td colspan="2" class="empty-state">No airlines in library.</td></tr><?php endif; ?>
                                </tbody>
                            </table>
                        </div>
                    </div>
                </div>
            </main>
        </div>
    </div>
    <script>
        document.getElementById('menu-toggle').addEventListener('click', function() {
            document.querySelector('.dashboard-wrapper').classList.toggle('sidebar-open');
        });
        // This is a user preference and I will keep it.
        document.addEventListener('contextmenu', function(e) { e.preventDefault(); });
    </script>
</body>
</html>