<?php
/**
 * manage-hotels-and-visa.php (v1.5 - Fully Responsive Overhaul)
 * - Manages static visa package rates for groups and hotel room rates.
 * - OVERHAUL: Implemented a fully responsive design for the entire page, including forms, tables, and the main layout.
 * - FIX: All form grids now stack correctly on mobile devices, resolving alignment issues.
 * - FIX: The main data table is now horizontally scrollable on small screens to prevent layout breaking.
 * - FIX: Sidebar is now collapsible and toggles correctly on mobile and tablet views.
 * - PHP logic remains the same; this is a comprehensive UI/UX update.
 */
session_start();
include_once '../db-config.php';

if (!isset($_SESSION['user_id']) || !isset($_SESSION['user_type']) || $_SESSION['user_type'] !== 'admin') {
    header("location: ../login.php");
    exit;
}

$edit_mode = false;
$hotel_to_edit = null;

if (isset($_SESSION['success_message'])) { $success_message = $_SESSION['success_message']; unset($_SESSION['success_message']); }
if (isset($_SESSION['error_message'])) { $error_message = $_SESSION['error_message']; unset($_SESSION['error_message']); }

// --- Handle ALL POST Actions ---
if ($_SERVER["REQUEST_METHOD"] == "POST" && isset($_POST['action'])) {

    // --- Action: Update VISA PACKAGE Rates ---
    if ($_POST['action'] === 'update_visa_rates') {
        $stmt = $conn->prepare("UPDATE visa_package_rates SET g1_pax=?, g1_rate=?, g2_pax=?, g2_rate=?, g3_pax=?, g3_rate=?, g4_pax=?, g4_rate=?, infant_rate=? WHERE id = 1");
        $stmt->bind_param("sdsdsdsdd", $_POST['g1_pax'], $_POST['g1_rate'], $_POST['g2_pax'], $_POST['g2_rate'], $_POST['g3_pax'], $_POST['g3_rate'], $_POST['g4_pax'], $_POST['g4_rate'], $_POST['infant_rate']);
        $stmt->execute();
        $stmt->close();
        $_SESSION['success_message'] = "Visa Package rates updated successfully.";
    }

    // --- Action: Add or Update a HOTEL ---
    if ($_POST['action'] === 'add_hotel' || $_POST['action'] === 'update_hotel') {
        $hotel_name = $_POST['hotel_name'];
        $location = $_POST['location'];
        $city = $_POST['city'];
        $star_rating = (int)$_POST['star_rating'];
        
        // Customer Rates
        $sharing_rate = !empty($_POST['sharing_rate']) ? (float)$_POST['sharing_rate'] : 0.00;
        $double_rate = !empty($_POST['double_rate']) ? (float)$_POST['double_rate'] : 0.00;
        $triple_rate = !empty($_POST['triple_rate']) ? (float)$_POST['triple_rate'] : 0.00;
        $quad_rate = !empty($_POST['quad_rate']) ? (float)$_POST['quad_rate'] : 0.00;
        $room = !empty($_POST['room']) ? (float)$_POST['room'] : 0.00;

        // Agent Rates - Default to 0.00 instead of null to prevent errors
        $sharing_rate_agent = !empty($_POST['sharing_rate_agent']) ? (float)$_POST['sharing_rate_agent'] : 0.00;
        $double_rate_agent = !empty($_POST['double_rate_agent']) ? (float)$_POST['double_rate_agent'] : 0.00;
        $triple_rate_agent = !empty($_POST['triple_rate_agent']) ? (float)$_POST['triple_rate_agent'] : 0.00;
        $quad_rate_agent = !empty($_POST['quad_rate_agent']) ? (float)$_POST['quad_rate_agent'] : 0.00;
        $room_agent = !empty($_POST['room_agent']) ? (float)$_POST['room_agent'] : 0.00;
        $agent_incentive = !empty($_POST['agent_incentive']) ? (float)$_POST['agent_incentive'] : 0.00;

        if ($_POST['action'] === 'add_hotel') {
            $sql = "INSERT INTO package_hotels (hotel_name, location, city, star_rating, sharing_rate, sharing_rate_agent, double_rate, double_rate_agent, triple_rate, triple_rate_agent, quad_rate, quad_rate_agent, room, room_agent, agent_incentive) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)";
            $stmt = $conn->prepare($sql);
            $stmt->bind_param("sssiddddddddddd", $hotel_name, $location, $city, $star_rating, $sharing_rate, $sharing_rate_agent, $double_rate, $double_rate_agent, $triple_rate, $triple_rate_agent, $quad_rate, $quad_rate_agent, $room, $room_agent, $agent_incentive);
            $_SESSION['success_message'] = "Hotel '$hotel_name' created successfully.";
        } else { // 'update_hotel'
            $id = (int)$_POST['id'];
            $sql = "UPDATE package_hotels SET hotel_name=?, location=?, city=?, star_rating=?, sharing_rate=?, sharing_rate_agent=?, double_rate=?, double_rate_agent=?, triple_rate=?, triple_rate_agent=?, quad_rate=?, quad_rate_agent=?, room=?, room_agent=?, agent_incentive=? WHERE id=?";
            $stmt = $conn->prepare($sql);
            $stmt->bind_param("sssidddddddddddi", $hotel_name, $location, $city, $star_rating, $sharing_rate, $sharing_rate_agent, $double_rate, $double_rate_agent, $triple_rate, $triple_rate_agent, $quad_rate, $quad_rate_agent, $room, $room_agent, $agent_incentive, $id);
            $_SESSION['success_message'] = "Hotel '$hotel_name' updated successfully.";
        }
        $stmt->execute();
        $stmt->close();
    }

    // --- Action: Delete a HOTEL ---
    if ($_POST['action'] === 'delete_hotel' && isset($_POST['id'])) {
        $id = (int)$_POST['id'];
        $stmt = $conn->prepare("DELETE FROM package_hotels WHERE id = ?");
        $stmt->bind_param("i", $id);
        $stmt->execute();
        $stmt->close();
        $_SESSION['success_message'] = "The hotel has been deleted.";
    }

    header("Location: " . basename($_SERVER['PHP_SELF']));
    exit();
}

// --- Fetch data for displaying on the page ---
$visa_rates_result = $conn->query("SELECT * FROM visa_package_rates WHERE id = 1");
$visa_rates = $visa_rates_result ? $visa_rates_result->fetch_assoc() : null;

if (isset($_GET['action']) && $_GET['action'] === 'edit_hotel' && isset($_GET['id'])) {
    $edit_mode = true;
    $id = (int)$_GET['id'];
    $stmt = $conn->prepare("SELECT * FROM package_hotels WHERE id = ?");
    $stmt->bind_param("i", $id);
    $stmt->execute();
    $hotel_to_edit = $stmt->get_result()->fetch_assoc();
    $stmt->close();
}

$hotels_result = $conn->query("SELECT * FROM package_hotels ORDER BY city, hotel_name ASC");
$current_page = basename($_SERVER['PHP_SELF']);
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Manage Hotels & Visa Rates</title>
    <link rel="icon" type="image/png" href="../images/logo-icon.png">
    <link rel="stylesheet" href="admin-style.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.1/css/all.min.css">
    <style>
        /* === RESPONSIVE OVERHAUL === */
        
        /* Form Fieldsets and Grids */
        .package-form fieldset { 
            border: 1px solid var(--border-color); 
            padding: 1.5rem; 
            border-radius: 8px; 
            margin-bottom: 2rem; 
            background-color: #fdfdfd; 
        }
        .package-form legend { 
            font-weight: 600; 
            padding: 0 10px; 
            font-size: 1.1rem; 
            color: var(--primary-color); 
        }
        .responsive-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
            gap: 1rem;
            align-items: end;
        }

        /* Table Responsiveness */
        .table-responsive { 
            width: 100%; 
            overflow-x: auto; 
            -webkit-overflow-scrolling: touch; 
            border: 1px solid var(--border-color);
            border-radius: 8px;
            margin-top: 1rem;
        }
        .data-table { 
            width: 100%; 
            min-width: 1200px; /* Force scroll on screens smaller than this */
        }
        
        /* Main Layout & Sidebar for Mobile */
        @media (max-width: 992px) {
            /* This is likely in your admin-style.css, but included here for completeness */
            .sidebar {
                position: fixed;
                left: 0;
                top: 0;
                height: 100%;
                transform: translateX(-100%);
                transition: transform 0.3s ease-in-out;
                z-index: 1000;
                box-shadow: 0 0 15px rgba(0,0,0,0.2);
            }
            .main-content {
                width: 100%;
                margin-left: 0;
            }
            .dashboard-wrapper.sidebar-open .sidebar {
                transform: translateX(0);
            }
            .main-header .menu-toggle {
                display: block;
            }
        }
        
        /* General Mobile Adjustments */
        @media (max-width: 768px) {
            .content-body { 
                padding: 1rem; 
            }
            .package-form fieldset { 
                padding: 1rem; 
            }
            .form-actions {
                display: flex;
                flex-direction: column;
                gap: 0.5rem;
            }
            .form-actions .btn {
                width: 100%;
                text-align: center;
            }
        }
        @media (max-width: 576px) {
             .responsive-grid {
                grid-template-columns: 1fr; /* Stack all form items into a single column on small phones */
             }
        }
    </style>
</head>
<body>
    <div class="dashboard-wrapper">
        <?php include 'sidebar.php'; ?>
        <div class="main-content">
            <header class="main-header">
                <button class="menu-toggle" id="menu-toggle"><i class="fas fa-bars"></i></button>
                <div class="user-info"><span>Welcome, <?php echo htmlspecialchars(ucfirst($_SESSION['user_name'] ?? 'Admin')); ?></span></div>
            </header>
            <main class="content-body" id="top">

                <?php if (isset($success_message)): ?><div class="notice success"><?= htmlspecialchars($success_message) ?></div><?php endif; ?>
                <?php if (isset($error_message)): ?><div class="notice error"><?= htmlspecialchars($error_message) ?></div><?php endif; ?>

                <div class="content-card">
                    <h2 class="form-title">Manage Visa Package Rates</h2>
                    <form action="<?= basename($_SERVER['PHP_SELF']) ?>" method="POST" class="package-form">
                        <input type="hidden" name="action" value="update_visa_rates">
                        <div class="responsive-grid">
                            <div class="form-group"><label>Group 1 Pax</label><input type="text" name="g1_pax" value="<?= htmlspecialchars($visa_rates['g1_pax'] ?? '1') ?>"></div>
                            <div class="form-group"><label>Group 1 Rate</label><input type="number" step="0.01" name="g1_rate" value="<?= htmlspecialchars($visa_rates['g1_rate'] ?? '0.00') ?>"></div>
                            <div class="form-group"><label>Group 2 Pax</label><input type="text" name="g2_pax" value="<?= htmlspecialchars($visa_rates['g2_pax'] ?? '2') ?>"></div>
                            <div class="form-group"><label>Group 2 Rate</label><input type="number" step="0.01" name="g2_rate" value="<?= htmlspecialchars($visa_rates['g2_rate'] ?? '0.00') ?>"></div>
                            <div class="form-group"><label>Group 3 Pax</label><input type="text" name="g3_pax" value="<?= htmlspecialchars($visa_rates['g3_pax'] ?? '3') ?>"></div>
                            <div class="form-group"><label>Group 3 Rate</label><input type="number" step="0.01" name="g3_rate" value="<?= htmlspecialchars($visa_rates['g3_rate'] ?? '0.00') ?>"></div>
                            <div class="form-group"><label>Group 4 Pax</label><input type="text" name="g4_pax" value="<?= htmlspecialchars($visa_rates['g4_pax'] ?? '4') ?>"></div>
                            <div class="form-group"><label>Group 4 Rate</label><input type="number" step="0.01" name="g4_rate" value="<?= htmlspecialchars($visa_rates['g4_rate'] ?? '0.00') ?>"></div>
                            <div class="form-group"><label>Infant Rate</label><input type="number" step="0.01" name="infant_rate" value="<?= htmlspecialchars($visa_rates['infant_rate'] ?? '0.00') ?>"></div>
                        </div>
                        <div class="form-actions"><button type="submit" class="btn btn-primary"><i class="fas fa-save"></i> Update Visa Rates</button></div>
                    </form>
                </div>

                <div class="content-card">
                    <h2 class="form-title"><?= $edit_mode ? 'Edit Hotel' : 'Add New Hotel' ?></h2>
                    <form action="<?= basename($_SERVER['PHP_SELF']) ?>" method="POST" class="package-form">
                        <input type="hidden" name="action" value="<?= $edit_mode ? 'update_hotel' : 'add_hotel' ?>">
                        <?php if ($edit_mode && $hotel_to_edit): ?><input type="hidden" name="id" value="<?= $hotel_to_edit['id'] ?>"><?php endif; ?>
                        
                        <fieldset>
                            <legend>Hotel Details</legend>
                            <div class="responsive-grid">
                                <div class="form-group"><label>Hotel Name *</label><input type="text" name="hotel_name" value="<?= htmlspecialchars($hotel_to_edit['hotel_name'] ?? '') ?>" required></div>
                                <div class="form-group"><label>City *</label><input type="text" name="city" placeholder="Makkah or Madinah" value="<?= htmlspecialchars($hotel_to_edit['city'] ?? '') ?>" required></div>
                                <div class="form-group"><label>Location / Address</label><input type="text" name="location" value="<?= htmlspecialchars($hotel_to_edit['location'] ?? '') ?>"></div>
                                <div class="form-group"><label>Star Rating *</label>
                                    <select name="star_rating" required>
                                        <option value="5" <?= (($hotel_to_edit['star_rating'] ?? 5) == 5) ? 'selected' : '' ?>>5 Stars</option>
                                        <option value="4" <?= (($hotel_to_edit['star_rating'] ?? 0) == 4) ? 'selected' : '' ?>>4 Stars</option>
                                        <option value="3" <?= (($hotel_to_edit['star_rating'] ?? 0) == 3) ? 'selected' : '' ?>>3 Stars</option>
                                        <option value="2" <?= (($hotel_to_edit['star_rating'] ?? 0) == 2) ? 'selected' : '' ?>>2 Stars</option>
                                        <option value="1" <?= (($hotel_to_edit['star_rating'] ?? 0) == 1) ? 'selected' : '' ?>>1 Star</option>
                                    </select>
                                </div>
                            </div>
                        </fieldset>

                        <fieldset>
                            <legend>Customer Room Rates</legend>
                            <div class="responsive-grid">
                                <div class="form-group"><label>Sharing Rate</label><input type="number" step="0.01" name="sharing_rate" value="<?= htmlspecialchars($hotel_to_edit['sharing_rate'] ?? '0.00') ?>"></div>
                                <div class="form-group"><label>Quad Rate</label><input type="number" step="0.01" name="quad_rate" value="<?= htmlspecialchars($hotel_to_edit['quad_rate'] ?? '0.00') ?>"></div>
                                <div class="form-group"><label>Triple Rate</label><input type="number" step="0.01" name="triple_rate" value="<?= htmlspecialchars($hotel_to_edit['triple_rate'] ?? '0.00') ?>"></div>
                                <div class="form-group"><label>Double Rate</label><input type="number" step="0.01" name="double_rate" value="<?= htmlspecialchars($hotel_to_edit['double_rate'] ?? '0.00') ?>"></div>
                                <div class="form-group"><label>Room Rate</label><input type="number" step="0.01" name="room" value="<?= htmlspecialchars($hotel_to_edit['room'] ?? '0.00') ?>"></div>
                            </div>
                        </fieldset>

                        <fieldset>
                            <legend>Agent Room Rates</legend>
                             <div class="responsive-grid">
                                <div class="form-group"><label>Agent Sharing</label><input type="number" step="0.01" name="sharing_rate_agent" value="<?= htmlspecialchars($hotel_to_edit['sharing_rate_agent'] ?? '') ?>"></div>
                                <div class="form-group"><label>Agent Quad</label><input type="number" step="0.01" name="quad_rate_agent" value="<?= htmlspecialchars($hotel_to_edit['quad_rate_agent'] ?? '') ?>"></div>
                                <div class="form-group"><label>Agent Triple</label><input type="number" step="0.01" name="triple_rate_agent" value="<?= htmlspecialchars($hotel_to_edit['triple_rate_agent'] ?? '') ?>"></div>
                                <div class="form-group"><label>Agent Double</label><input type="number" step="0.01" name="double_rate_agent" value="<?= htmlspecialchars($hotel_to_edit['double_rate_agent'] ?? '') ?>"></div>
                                <div class="form-group"><label>Agent Room</label><input type="number" step="0.01" name="room_agent" value="<?= htmlspecialchars($hotel_to_edit['room_agent'] ?? '') ?>"></div>
                                <div class="form-group"><label>Agent Incentive</label><input type="number" step="0.01" name="agent_incentive" value="<?= htmlspecialchars($hotel_to_edit['agent_incentive'] ?? '0.00') ?>"></div>
                            </div>
                        </fieldset>

                        <div class="form-actions">
                            <button type="submit" class="btn btn-primary"><?= $edit_mode ? 'Update Hotel' : 'Add Hotel' ?></button>
                            <?php if ($edit_mode): ?><a href="<?= basename($_SERVER['PHP_SELF']) ?>" class="btn btn-secondary">Cancel Edit</a><?php endif; ?>
                        </div>
                    </form>
                </div>

                <div class="content-card">
                    <h2 class="form-title">Existing Hotels</h2>
                    <div class="table-responsive">
                        <table class="data-table">
                            <thead>
                                <tr>
                                    <th>Hotel</th>
                                    <th>City</th>
                                    <th>Sharing</th>
                                    <th>Quad</th>
                                    <th>Triple</th>
                                    <th>Double</th>
                                    <th>Room</th>
                                    <th>Agent Sharing</th>
                                    <th>Agent Quad</th>
                                    <th>Agent Triple</th>
                                    <th>Agent Double</th>
                                    <th>Agent Room</th>
                                    <th>Actions</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php if ($hotels_result && $hotels_result->num_rows > 0): ?>
                                    <?php while ($hotel = $hotels_result->fetch_assoc()): ?>
                                        <tr class="<?= ($edit_mode && isset($hotel_to_edit['id']) && $hotel['id'] === $hotel_to_edit['id']) ? 'editing' : '' ?>">
                                            <td><strong><?= htmlspecialchars($hotel['hotel_name']) ?></strong><br><small><?= str_repeat('⭐', $hotel['star_rating']) ?></small></td>
                                            <td><?= htmlspecialchars($hotel['city']) ?></td>
                                            <td><?= !empty($hotel['sharing_rate']) ? number_format((float)$hotel['sharing_rate']) : 'N/A' ?></td>
                                            <td><?= !empty($hotel['quad_rate']) ? number_format((float)$hotel['quad_rate']) : 'N/A' ?></td>
                                            <td><?= !empty($hotel['triple_rate']) ? number_format((float)$hotel['triple_rate']) : 'N/A' ?></td>
                                            <td><?= !empty($hotel['double_rate']) ? number_format((float)$hotel['double_rate']) : 'N/A' ?></td>
                                            <td><?= !empty($hotel['room']) ? number_format((float)$hotel['room']) : 'N/A' ?></td>
                                            <td><?= !empty($hotel['sharing_rate_agent']) ? number_format((float)$hotel['sharing_rate_agent']) : 'N/A' ?></td>
                                            <td><?= !empty($hotel['quad_rate_agent']) ? number_format((float)$hotel['quad_rate_agent']) : 'N/A' ?></td>
                                            <td><?= !empty($hotel['triple_rate_agent']) ? number_format((float)$hotel['triple_rate_agent']) : 'N/A' ?></td>
                                            <td><?= !empty($hotel['double_rate_agent']) ? number_format((float)$hotel['double_rate_agent']) : 'N/A' ?></td>
                                            <td><?= !empty($hotel['room_agent']) ? number_format((float)$hotel['room_agent']) : 'N/A' ?></td>
                                            <td class="actions-cell">
                                                <a href="?action=edit_hotel&id=<?= $hotel['id'] ?>#top" class="btn btn-sm btn-secondary" title="Edit"><i class="fas fa-edit"></i></a>
                                                <form action="<?= basename($_SERVER['PHP_SELF']) ?>" method="POST" onsubmit="return confirm('Delete this hotel?');" style="display:inline;">
                                                    <input type="hidden" name="action" value="delete_hotel"><input type="hidden" name="id" value="<?= $hotel['id'] ?>">
                                                    <button type="submit" class="btn btn-sm btn-danger" title="Delete"><i class="fas fa-trash"></i></button>
                                                </form>
                                            </td>
                                        </tr>
                                    <?php endwhile; ?>
                                <?php else: ?>
                                    <tr>
                                        <td colspan="13" class="empty-state">No hotels found.</td>
                                    </tr>
                                <?php endif; ?>
                            </tbody>
                        </table>
                    </div>
                </div>
            </main>
        </div>
    </div>
    <script>
        document.getElementById('menu-toggle').addEventListener('click', function() {
            document.querySelector('.dashboard-wrapper').classList.toggle('sidebar-open');
        });
        document.addEventListener('contextmenu', function(e) {
            e.preventDefault();
        });
    </script>
</body>
</html>