<?php
session_start();
include_once '../db-config.php';

// --- SECURITY CHECK ---
if (!isset($_SESSION['user_id']) || !isset($_SESSION['user_type']) || $_SESSION['user_type'] !== 'admin') {
    header("location: ../login.php");
    exit;
}

function e($string) { return htmlspecialchars($string ?? '', ENT_QUOTES, 'UTF-8'); }

// --- HANDLE POST ACTIONS ---
if ($_SERVER["REQUEST_METHOD"] == "POST") {
    // --- API: MARK AS READ (from JavaScript) ---
    if (isset($_SERVER['HTTP_X_REQUESTED_WITH']) && strtolower($_SERVER['HTTP_X_REQUESTED_WITH']) == 'xmlhttprequest') {
        $data = json_decode(file_get_contents('php://input'), true);
        if (isset($data['action']) && $data['action'] === 'mark_read' && isset($data['id'])) {
            $notification_id = (int)$data['id'];
            $stmt = $conn->prepare("UPDATE notifications SET is_read = 1 WHERE id = ?");
            $stmt->bind_param("i", $notification_id);
            header('Content-Type: application/json');
            echo json_encode(['success' => $stmt->execute()]);
            exit();
        }
    }

    // --- FORM ACTION: MARK ALL READ ---
    if (isset($_POST['mark_all_read'])) {
        $conn->query("UPDATE notifications SET is_read = 1 WHERE is_read = 0");
        $_SESSION['success_message'] = "All unread notifications marked as read.";
    }

    // --- FORM ACTION: DELETE SELECTED ---
    if (isset($_POST['delete_selected']) && !empty($_POST['selected_ids'])) {
        $ids_to_delete = array_map('intval', $_POST['selected_ids']);
        $placeholders = implode(',', array_fill(0, count($ids_to_delete), '?'));
        $types = str_repeat('i', count($ids_to_delete));
        $stmt = $conn->prepare("DELETE FROM notifications WHERE id IN ($placeholders)");
        $stmt->bind_param($types, ...$ids_to_delete);
        if ($stmt->execute()) {
            $_SESSION['success_message'] = $stmt->affected_rows . " notification(s) deleted.";
        }
    }
    
    header("Location: notifications.php");
    exit();
}

// --- DATA FETCHING & PAGINATION ---
$items_per_page = 20;
$current_page = max(1, (int)($_GET['page'] ?? 1));
$offset = ($current_page - 1) * $items_per_page;
$total_items = $conn->query("SELECT COUNT(*) FROM notifications")->fetch_row()[0];
$total_pages = ceil($total_items / $items_per_page);
$notifications = $conn->query("SELECT * FROM notifications ORDER BY created_at DESC LIMIT $offset, $items_per_page")->fetch_all(MYSQLI_ASSOC);

$success_message = $_SESSION['success_message'] ?? null; unset($_SESSION['success_message']);
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Notifications</title>
    <link rel="icon" type="image/png" href="../images/logo-icon.png">
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <link rel="stylesheet" href="admin-style.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.1/css/all.min.css">
    <style>
        .main-content { min-width: 0; }
        .content-body { padding: 1.5rem; }
        .notification-item {
            display: flex; gap: 1rem; align-items: flex-start;
            padding: 1rem; text-decoration: none; color: inherit;
            border-left: 4px solid transparent;
            transition: background-color 0.2s;
        }
        .notification-item:hover { background-color: #f8f9fa; }
        .notification-item.unread { border-left-color: #0d6efd; background-color: #fcfdfe; }
        .notification-item .icon { font-size: 1.5rem; color: #adb5bd; }
        .notification-item.unread .icon { color: #0d6efd; }
        .notification-content .time { font-size: 0.8rem; color: #6c757d; }
    </style>
</head>
<body>
<div class="dashboard-wrapper">
    <?php include 'sidebar.php'; ?>
    <div class="main-content">
        <header class="main-header">
            <button class="menu-toggle" id="menu-toggle"><i class="fas fa-bars"></i></button>
            <div class="user-info"><span>Notifications</span></div>
        </header>
        <main class="content-body">
            <form action="notifications.php" method="POST" id="notifications-form">
                <div class="d-flex flex-wrap gap-2 justify-content-between align-items-center mb-4">
                    <h1 class="page-title mb-0">Notifications</h1>
                    <div class="d-flex gap-2">
                        <button type="submit" name="mark_all_read" class="btn btn-secondary btn-sm">Mark All as Read</button>
                        <button type="submit" name="delete_selected" class="btn btn-danger btn-sm" onclick="return confirm('Delete selected notifications?');">Delete Selected</button>
                    </div>
                </div>

                <?php if ($success_message): ?><div class="alert alert-success"><?= e($success_message); ?></div><?php endif; ?>

                <div class="card">
                    <div class="list-group list-group-flush">
                        <?php if (count($notifications) > 0): ?>
                            <div class="list-group-item d-none d-md-flex align-items-center">
                                <input type="checkbox" class="form-check-input" id="select-all" title="Select All">
                            </div>
                            <?php foreach ($notifications as $n):
                                $is_read_class = $n['is_read'] ? '' : 'unread';
                                $has_link = !empty($n['link']);
                            ?>
                                <a href="<?= $has_link ? e($n['link']) : '#' ?>" class="list-group-item list-group-item-action notification-item <?= $is_read_class ?>" data-id="<?= $n['id'] ?>" data-has-link="<?= $has_link ? 'true' : 'false' ?>">
                                    <input type="checkbox" name="selected_ids[]" value="<?= $n['id'] ?>" class="form-check-input row-checkbox" onclick="event.stopPropagation()">
                                    <div class="icon"><i class="fas fa-bell"></i></div>
                                    <div class="flex-grow-1">
                                        <div class="d-flex justify-content-between">
                                            <strong class="mb-1"><?= e(ucwords(str_replace('_', ' ', $n['type']))) ?></strong>
                                            <small class="time"><?= date('d M, Y H:i', strtotime($n['created_at'])) ?></small>
                                        </div>
                                        <p class="mb-1 small"><?= e($n['message']) ?></p>
                                    </div>
                                </a>
                            <?php endforeach; ?>
                        <?php else: ?>
                            <div class="list-group-item text-center p-4">You have no notifications.</div>
                        <?php endif; ?>
                    </div>
                </div>
            </form>

            <?php if ($total_pages > 1): ?>
                <nav class="d-flex justify-content-center mt-4"><ul class="pagination">
                    <li class="page-item <?= $current_page <= 1 ? 'disabled' : '' ?>"><a class="page-link" href="?page=<?= $current_page - 1 ?>">« Previous</a></li>
                    <li class="page-item <?= $current_page >= $total_pages ? 'disabled' : '' ?>"><a class="page-link" href="?page=<?= $current_page + 1 ?>">Next »</a></li>
                </ul></nav>
            <?php endif; ?>
        </main>
    </div>
</div>
<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/js/bootstrap.bundle.min.js"></script>
<script>
document.addEventListener('DOMContentLoaded', function() {
    // Sidebar toggle
    const menuToggle = document.getElementById('menu-toggle');
    if (menuToggle) {
        menuToggle.addEventListener('click', () => document.querySelector('.dashboard-wrapper').classList.toggle('sidebar-open'));
    }

    // --- NOTIFICATION CLICK LOGIC ---
    document.querySelectorAll('.notification-item').forEach(item => {
        item.addEventListener('click', function(event) {
            const isUnread = this.classList.contains('unread');
            const hasLink = this.dataset.hasLink === 'true';
            
            // Only act if the notification is unread
            if (isUnread) {
                const notificationId = this.dataset.id;

                // If it's a link, prevent the default jump to allow fetch to complete first
                if (hasLink) {
                    event.preventDefault();
                }

                // Optimistically update UI
                this.classList.remove('unread');

                // Send request to server to mark as read
                fetch('notifications.php', {
                    method: 'POST',
                    headers: { 'Content-Type': 'application/json', 'X-Requested-With': 'XMLHttpRequest' },
                    body: JSON.stringify({ action: 'mark_read', id: notificationId })
                })
                .then(response => response.json())
                .then(data => {
                    if (data.success && hasLink) {
                        // If successful and it has a link, now go to the link
                        window.location.href = this.href;
                    } else if (!data.success) {
                        // If server failed, revert the UI change
                        this.classList.add('unread');
                    }
                })
                .catch(error => {
                    console.error('Error marking notification as read:', error);
                    this.classList.add('unread'); // Revert on network error
                });
            }
            // If it's already read and has a link, the click will proceed as normal.
        });
    });

    // --- SELECT ALL CHECKBOX LOGIC ---
    const selectAll = document.getElementById('select-all');
    const rowCheckboxes = document.querySelectorAll('.row-checkbox');
    if(selectAll) {
        selectAll.addEventListener('change', function() {
            rowCheckboxes.forEach(cb => {
                cb.checked = this.checked;
            });
        });
    }
});
</script>
</body>
</html>