<?php
/**
 * manage-umrah-packages.php (Automated SQL Version)
 * - Self-contained CRUD system for managing Umrah Packages.
 * - FIX: Replaced manual bind_param with a dynamic and secure query builder to prevent type/count mismatch errors.
 * - This new method is more robust and easier to maintain.
 * - UPDATE: Added Agent Pricing fields and a toggle for logo background.
 * - UPDATE: Made the entire interface responsive for mobile and tablet use.
 * - UPDATE: Added fields for Makkah/Madinah nights and distance.
 * - MAJOR UPDATE: Integrated hotel selection from custom rates, auto-price calculation in SAR, and file uploads for main image.
 * - FINAL UPDATE: Fixed PHP null value errors and implemented a complete mobile-first responsive design.
 * - FEATURE UPDATE: Implemented bulk selection and deletion of packages.
 * - INCENTIVE UPDATE: Changed Agent Incentive to SAR and made it auto-calculated based on hotel nights.
 * - IMAGE UPDATE: Added a default main image for new packages if none is uploaded.
 * - BUG FIX: Corrected an issue where hotel dropdowns and prices were not pre-filled correctly when editing a package.
 * - FEATURE ADDITION: Integrated the new 'rate_sharing' from custom hotel rates into the price calculator.
 * - **MAJOR FEATURE: Added City Category selection and management for Umrah packages.**
 */

// db-config.php should handle session_start()
include_once '../db-config.php';

// =======================================================
// SECURITY GATEWAY
// =======================================================
if (!isset($_SESSION['user_id']) || !isset($_SESSION['user_type']) || $_SESSION['user_type'] !== 'admin') {
    header("location: ../login.php");
    exit;
}

//======================================================================
//  1. SETUP & HELPER FUNCTIONS
//======================================================================
define('PAGE_TEMPLATE_PATH', '../umrah-package-template.php');
define('GENERATED_PAGES_DIR', '../');
define('MAIN_IMAGE_UPLOAD_DIR', '../images/');
define('DEFAULT_MAIN_IMAGE', 'images/default-package-image.jpg');

function sanitize_slug($slug) {
    if ($slug === null) return '';
    $slug = trim($slug);
    $slug = strtolower($slug);
    $slug = preg_replace('/[\s_]+/', '-', $slug);
    $slug = preg_replace('/[^a-z0-9\-]/', '', $slug);
    return trim($slug, '-');
}

function create_package_page($slug) {
    if (empty($slug)) return ['success' => true];
    $sanitized_slug = sanitize_slug($slug);
    if (empty($sanitized_slug)) return ['success' => true];
    $new_filepath = GENERATED_PAGES_DIR . $sanitized_slug . '.php';
    if (!file_exists(PAGE_TEMPLATE_PATH)) return ['success' => false, 'message' => "Template file 'umrah-package-template.php' not found."];
    if (file_exists($new_filepath)) return ['success' => true, 'message' => "Page '{$sanitized_slug}.php' already exists."];
    $template_content = file_get_contents(PAGE_TEMPLATE_PATH);
    if ($template_content === false) return ['success' => false, 'message' => "Could not read template file."];
    if (file_put_contents($new_filepath, $template_content) === false) return ['success' => false, 'message' => "Could not create page. Check folder permissions."];
    return ['success' => true, 'message' => "Page '{$sanitized_slug}.php' was created successfully."];
}

function delete_package_page($slug) {
    if (empty($slug)) return ['success' => true];
    $sanitized_slug = sanitize_slug($slug);
    if (empty($sanitized_slug)) return ['success' => true];
    $filepath = GENERATED_PAGES_DIR . $sanitized_slug . '.php';
    if (file_exists($filepath)) {
        if (!is_writable($filepath)) return ['success' => false, 'message' => "Cannot delete page '{$filepath}'. Check permissions."];
        if (!unlink($filepath)) return ['success' => false, 'message' => "Failed to delete page file."];
    }
    return ['success' => true];
}

$edit_mode = false;
$package_to_edit = null;
if (isset($_SESSION['success_message'])) { $success_message = $_SESSION['success_message']; unset($_SESSION['success_message']); }
if (isset($_SESSION['error_message'])) { $error_message = $_SESSION['error_message']; unset($_SESSION['error_message']); }

//======================================================================
//  2. HANDLE FORM SUBMISSIONS (CREATE, UPDATE, DELETE, CITIES)
//======================================================================
if ($_SERVER["REQUEST_METHOD"] == "POST" && isset($_POST['action'])) {

    // --- MANAGE CITIES ---
    if ($_POST['action'] === 'add_city') {
        $city_name = trim($_POST['city_name']);
        if (!empty($city_name)) {
            $stmt = $conn->prepare("INSERT INTO cities (name) VALUES (?)");
            $stmt->bind_param("s", $city_name);
            if ($stmt->execute()) {
                $_SESSION['success_message'] = "City '{$city_name}' added successfully.";
            } else {
                $_SESSION['error_message'] = "Failed to add city. It might already exist.";
            }
        }
        header("Location: manage-umrah-packages.php#cities-section");
        exit();
    }
    
    if ($_POST['action'] === 'update_city') {
        $city_name = trim($_POST['city_name']);
        $city_id = (int)$_POST['id'];
        if (!empty($city_name) && $city_id > 0) {
            $stmt = $conn->prepare("UPDATE cities SET name = ? WHERE id = ?");
            $stmt->bind_param("si", $city_name, $city_id);
            if ($stmt->execute()) {
                $_SESSION['success_message'] = "City updated successfully.";
            } else {
                $_SESSION['error_message'] = "Failed to update city.";
            }
        }
        header("Location: manage-umrah-packages.php#cities-section");
        exit();
    }

    if ($_POST['action'] === 'delete_city' && isset($_POST['id'])) {
        $stmt = $conn->prepare("DELETE FROM cities WHERE id = ?");
        $stmt->bind_param("i", $_POST['id']);
        if ($stmt->execute()) {
            $_SESSION['success_message'] = "City deleted successfully.";
        } else {
            $_SESSION['error_message'] = "Failed to delete city. Note: Packages in this city will become uncategorized.";
        }
        header("Location: manage-umrah-packages.php#cities-section");
        exit();
    }
    
    if ($_POST['action'] === 'delete_multiple_cities' && !empty($_POST['city_ids'])) {
        $city_ids = array_map('intval', $_POST['city_ids']);
        $placeholders = implode(',', array_fill(0, count($city_ids), '?'));
        $types = str_repeat('i', count($city_ids));
        
        $stmt = $conn->prepare("DELETE FROM cities WHERE id IN ($placeholders)");
        $stmt->bind_param($types, ...$city_ids);
        
        if ($stmt->execute()) {
            $_SESSION['success_message'] = count($city_ids) . " cities deleted successfully.";
        } else {
            $_SESSION['error_message'] = "Failed to delete cities.";
        }
        header("Location: manage-umrah-packages.php#cities-section");
        exit();
    }

    // Helper for handling main image upload
    function handle_main_image_upload($current_path = null) {
        if (isset($_FILES['main_image_file']) && $_FILES['main_image_file']['error'] === UPLOAD_ERR_OK) {
            if (!is_dir(MAIN_IMAGE_UPLOAD_DIR)) {
                if (!mkdir(MAIN_IMAGE_UPLOAD_DIR, 0755, true)) { throw new Exception("Failed to create main image upload directory. Check permissions."); }
            }
            $filename = time() . '_' . basename(preg_replace("/[^a-zA-Z0-9.\-\_]/", "", $_FILES['main_image_file']['name']));
            $target_path = MAIN_IMAGE_UPLOAD_DIR . $filename;
            $allowed_types = ['image/jpeg', 'image/png', 'image/gif', 'image/webp'];
            if (!in_array($_FILES['main_image_file']['type'], $allowed_types)) { throw new Exception("Invalid file type for main image. Please use JPG, PNG, GIF, or WEBP."); }
            if (move_uploaded_file($_FILES['main_image_file']['tmp_name'], $target_path)) {
                return 'images/' . $filename;
            } else {
                throw new Exception("Failed to move uploaded main image file. Check server permissions.");
            }
        }
        return $current_path;
    }
    
    // Calculate agent incentive based on nights
    $makkah_nights_val = !empty($_POST['makkah_nights']) ? (int)$_POST['makkah_nights'] : 0;
    $madinah_nights_val = !empty($_POST['madinah_nights']) ? (int)$_POST['madinah_nights'] : 0;
    $calculated_agent_incentive = ($makkah_nights_val + $madinah_nights_val) * 2;

    // Central place to define our package data structure
    $package_data = [
        'package_id' => $_POST['package_id'] ?? null,
        'package_name' => $_POST['package_name'] ?? null,
        'airline_id' => (!empty($_POST['airline_id']) && is_numeric($_POST['airline_id'])) ? (int)$_POST['airline_id'] : null,
        'city_id' => (!empty($_POST['city_id']) && is_numeric($_POST['city_id'])) ? (int)$_POST['city_id'] : null,
        'price_double' => !empty($_POST['price_double']) ? (float)$_POST['price_double'] : null,
        'price_triple' => !empty($_POST['price_triple']) ? (float)$_POST['price_triple'] : null,
        'price_quad' => !empty($_POST['price_quad']) ? (float)$_POST['price_quad'] : null,
        'price_quint' => !empty($_POST['price_quint']) ? (float)$_POST['price_quint'] : null,
        'price_double_agent' => !empty($_POST['price_double_agent']) ? (float)$_POST['price_double_agent'] : null,
        'price_triple_agent' => !empty($_POST['price_triple_agent']) ? (float)$_POST['price_triple_agent'] : null,
        'price_quad_agent' => !empty($_POST['price_quad_agent']) ? (float)$_POST['price_quad_agent'] : null,
        'price_quint_agent' => !empty($_POST['price_quint_agent']) ? (float)$_POST['price_quint_agent'] : null,
        'agent_incentive' => $calculated_agent_incentive,
        'makkah_hotel' => $_POST['makkah_hotel'] ?? null,
        'makkah_nights' => !empty($_POST['makkah_nights']) ? (int)$_POST['makkah_nights'] : null,
        'makkah_hotel_distance' => $_POST['makkah_hotel_distance'] ?? null,
        'madinah_hotel' => $_POST['madinah_hotel'] ?? null,
        'madinah_nights' => !empty($_POST['madinah_nights']) ? (int)$_POST['madinah_nights'] : null,
        'madinah_hotel_distance' => $_POST['madinah_hotel_distance'] ?? null,
        'transportation' => $_POST['transportation'] ?? 'Sharing',
        'ziyarat' => $_POST['ziyarat'] ?? 'No',
        'days' => !empty($_POST['days']) ? (int)$_POST['days'] : null,
        'page_link' => sanitize_slug($_POST['page_link'] ?? ''),
        'main_image_link' => $_POST['main_image_link'] ?? null,
        'logo_has_background' => isset($_POST['logo_has_background']) ? (int)$_POST['logo_has_background'] : 1,
        'is_active' => isset($_POST['is_active']) ? (int)$_POST['is_active'] : 1,
        'overview' => $_POST['overview'] ?? null,
        'itinerary' => $_POST['itinerary'] ?? null,
        'whats_included' => $_POST['whats_included'] ?? null,
        'whats_extra' => $_POST['whats_extra'] ?? null,
        'outbound_flight_details' => $_POST['outbound_flight_details'] ?? null,
        'inbound_flight_details' => $_POST['inbound_flight_details'] ?? null
    ];

    function handle_flyer_upload($current_path = null) {
        if (isset($_FILES['flyer_image']) && $_FILES['flyer_image']['error'] === UPLOAD_ERR_OK) {
            $upload_dir = '../uploads/flyers/';
            if (!is_dir($upload_dir)) {
                if (!mkdir($upload_dir, 0755, true)) { throw new Exception("Failed to create flyer upload directory. Check permissions."); }
            }
            $filename = time() . '_' . basename(preg_replace("/[^a-zA-Z0-9.\-\_]/", "", $_FILES['flyer_image']['name']));
            $target_path = $upload_dir . $filename;
            $allowed_types = ['image/jpeg', 'image/png', 'image/gif', 'image/webp'];
            if (!in_array($_FILES['flyer_image']['type'], $allowed_types)) { throw new Exception("Invalid file type for flyer. Please use JPG, PNG, GIF, or WEBP."); }
            if (move_uploaded_file($_FILES['flyer_image']['tmp_name'], $target_path)) {
                if ($current_path && file_exists($current_path)) { @unlink($current_path); }
                return $target_path;
            } else { throw new Exception("Failed to move uploaded flyer file. Check server permissions."); }
        }
        return $current_path;
    }

    if ($_POST['action'] === 'add') {
        try {
            $package_data['main_image_link'] = handle_main_image_upload();
            if (empty($package_data['main_image_link'])) {
                $package_data['main_image_link'] = DEFAULT_MAIN_IMAGE;
            }
            
            $package_data['flyer_image_path'] = handle_flyer_upload();
            $columns = array_keys($package_data);
            $placeholders = array_fill(0, count($columns), '?');
            $sql = "INSERT INTO umrah_packages (" . implode(', ', $columns) . ") VALUES (" . implode(', ', $placeholders) . ")";
            $stmt = $conn->prepare($sql);
            if ($stmt === false) throw new Exception("Database prepare failed: " . $conn->error);
            $types = ''; foreach ($package_data as $value) { if (is_int($value)) $types .= 'i'; elseif (is_float($value)) $types .= 'd'; else $types .= 's'; }
            $stmt->bind_param($types, ...array_values($package_data));
            if (!$stmt->execute()) throw new Exception("Database execution failed: " . $stmt->error);
            $_SESSION['success_message'] = "Package '{$package_data['package_id']}' added successfully. ";
            $page_result = create_package_page($package_data['page_link']);
            if ($page_result['success']) { if (isset($page_result['message'])) $_SESSION['success_message'] .= $page_result['message']; } else { $_SESSION['error_message'] = "DB record created, but page creation failed: " . $page_result['message']; }
        } catch (Exception $e) { $_SESSION['error_message'] = "Error: " . $e->getMessage(); }
        header("Location: manage-umrah-packages.php"); exit();
    }

    if ($_POST['action'] === 'update' && isset($_POST['original_package_id'])) {
        try {
            $original_id = $_POST['original_package_id'];
            $stmt_old = $conn->prepare("SELECT flyer_image_path, page_link, main_image_link FROM umrah_packages WHERE package_id = ?"); $stmt_old->bind_param("s", $original_id); $stmt_old->execute(); $pkg_old = $stmt_old->get_result()->fetch_assoc();
            
            $package_data['main_image_link'] = handle_main_image_upload($pkg_old['main_image_link'] ?? null);

            $old_slug = $pkg_old['page_link'] ?? ''; $current_flyer_path = $pkg_old['flyer_image_path'] ?? null; $flyer_path_to_save = $current_flyer_path;
            if (isset($_POST['remove_flyer']) && $_POST['remove_flyer'] == '1') { if ($current_flyer_path && file_exists($current_flyer_path)) @unlink($current_flyer_path); $flyer_path_to_save = null; } else { $flyer_path_to_save = handle_flyer_upload($current_flyer_path); }
            $package_data['flyer_image_path'] = $flyer_path_to_save;
            $new_slug = $package_data['page_link']; $page_op_message = ''; if ($new_slug !== $old_slug) { delete_package_page($old_slug); $page_result = create_package_page($new_slug); $page_op_message = $page_result['message'] ?? ''; }
            $set_parts = []; foreach ($package_data as $key => $value) { $set_parts[] = "$key = ?"; }
            $sql = "UPDATE umrah_packages SET " . implode(', ', $set_parts) . " WHERE package_id = ?";
            $stmt = $conn->prepare($sql); if ($stmt === false) throw new Exception("Database prepare failed: " . $conn->error);
            $types = ''; foreach ($package_data as $value) { if (is_int($value)) $types .= 'i'; elseif (is_float($value)) $types .= 'd'; else $types .= 's'; } $types .= 's';
            $values = array_values($package_data); $values[] = $original_id;
            $stmt->bind_param($types, ...$values);
            if (!$stmt->execute()) throw new Exception("Database execution failed: " . $stmt->error);
            $_SESSION['success_message'] = "Package '{$package_data['package_id']}' updated successfully. " . $page_op_message;
        } catch (Exception $e) { $_SESSION['error_message'] = "Error: " . $e->getMessage(); }
        header("Location: manage-umrah-packages.php"); exit();
    }
    
    if ($_POST['action'] === 'delete' && isset($_POST['package_id'])) {
        $package_id_to_delete = $_POST['package_id'];
        try {
            $stmt_select = $conn->prepare("SELECT flyer_image_path, page_link, main_image_link FROM umrah_packages WHERE package_id = ?"); $stmt_select->bind_param("s", $package_id_to_delete); $stmt_select->execute(); $result = $stmt_select->get_result();
            if ($row = $result->fetch_assoc()) { 
                if (!empty($row['flyer_image_path']) && file_exists($row['flyer_image_path'])) { @unlink($row['flyer_image_path']); } 
                if (!empty($row['main_image_link']) && $row['main_image_link'] !== DEFAULT_MAIN_IMAGE && file_exists('../' . $row['main_image_link'])) { @unlink('../' . $row['main_image_link']); }
                delete_package_page($row['page_link']); 
            } $stmt_select->close();
            $stmt_delete = $conn->prepare("DELETE FROM umrah_packages WHERE package_id = ?"); $stmt_delete->bind_param("s", $package_id_to_delete);
            if ($stmt_delete->execute()) { $_SESSION['success_message'] = "Package '{$package_id_to_delete}' has been deleted."; } else { throw new Exception("Failed to delete the package from the database."); } $stmt_delete->close();
        } catch (Exception $e) { $_SESSION['error_message'] = "Error deleting package: " . $e->getMessage(); }
        header("Location: manage-umrah-packages.php"); exit();
    }

    if ($_POST['action'] === 'bulk_delete' && isset($_POST['package_ids']) && is_array($_POST['package_ids'])) {
        $ids_to_delete = array_filter($_POST['package_ids']);
        if (empty($ids_to_delete)) {
            $_SESSION['error_message'] = "No packages were selected for deletion.";
            header("Location: manage-umrah-packages.php");
            exit();
        }

        try {
            $placeholders = implode(',', array_fill(0, count($ids_to_delete), '?'));
            $types = str_repeat('s', count($ids_to_delete));

            $stmt_select = $conn->prepare("SELECT flyer_image_path, page_link, main_image_link FROM umrah_packages WHERE package_id IN ($placeholders)");
            $stmt_select->bind_param($types, ...$ids_to_delete);
            $stmt_select->execute();
            $result = $stmt_select->get_result();

            while ($row = $result->fetch_assoc()) {
                if (!empty($row['flyer_image_path']) && file_exists($row['flyer_image_path'])) { @unlink($row['flyer_image_path']); }
                if (!empty($row['main_image_link']) && $row['main_image_link'] !== DEFAULT_MAIN_IMAGE && file_exists('../' . $row['main_image_link'])) { @unlink('../' . $row['main_image_link']); }
                delete_package_page($row['page_link']);
            }
            $stmt_select->close();

            $stmt_delete = $conn->prepare("DELETE FROM umrah_packages WHERE package_id IN ($placeholders)");
            $stmt_delete->bind_param($types, ...$ids_to_delete);
            if ($stmt_delete->execute()) {
                $deleted_count = $stmt_delete->affected_rows;
                $_SESSION['success_message'] = "Successfully deleted {$deleted_count} packages.";
            } else {
                throw new Exception("Failed to delete the packages from the database.");
            }
            $stmt_delete->close();
        } catch (Exception $e) {
            $_SESSION['error_message'] = "Error during bulk deletion: " . $e->getMessage();
        }
        header("Location: manage-umrah-packages.php");
        exit();
    }
}

// --- Fetch data for the page ---
if (isset($_GET['action']) && $_GET['action'] === 'edit' && isset($_GET['id'])) {
    $edit_mode = true;
    $stmt = $conn->prepare("SELECT * FROM umrah_packages WHERE package_id = ?"); $stmt->bind_param("s", $_GET['id']); $stmt->execute();
    $package_to_edit = $stmt->get_result()->fetch_assoc();
}
$airlines_result = $conn->query("SELECT id, airline_name FROM airlines ORDER BY airline_name ASC");

// Fetch all cities for the dropdowns and management table
$cities_result = $conn->query("SELECT * FROM cities ORDER BY name ASC");
$cities_for_dropdown = [];
if ($cities_result && $cities_result->num_rows > 0) {
    while($row = $cities_result->fetch_assoc()) {
        $cities_for_dropdown[] = $row;
    }
    $cities_result->data_seek(0); 
}

$packages_result = $conn->query("
    SELECT up.*, a.airline_name, a.logo_url, c.name as city_name 
    FROM umrah_packages up 
    LEFT JOIN airlines a ON up.airline_id = a.id 
    LEFT JOIN cities c ON up.city_id = c.id
    ORDER BY up.last_updated DESC");

// Fetch hotel data for dropdowns
$makkah_hotels_result = $conn->query("SELECT * FROM custom_package_hotel_rates WHERE city = 'Makkah' ORDER BY hotel_name");
$madinah_hotels_result = $conn->query("SELECT * FROM custom_package_hotel_rates WHERE city = 'Madinah' ORDER BY hotel_name");
$current_page = basename($_SERVER['PHP_SELF']);
?>

<!DOCTYPE html>

<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Manage Umrah Packages</title>
    <link rel="icon" type="image/png" href="../images/logo-icon.png">
    <link rel="stylesheet" href="admin-style.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.1/css/all.min.css">
    <!-- Select2 CSS -->
    <link href="https://cdn.jsdelivr.net/npm/select2@4.1.0-rc.0/dist/css/select2.min.css" rel="stylesheet" />
    <style>
        /* --- Base and Overhaul Styles --- */
        :root {
            --form-grid-gap: 1.25rem;
            --body-bg: #f4f7fa;
            --card-shadow: 0 4px 12px rgba(0, 0, 0, 0.08);
            --border-color: #dee2e6;
        }
        * {
            box-sizing: border-box;
        }
        .content-body {
            background-color: var(--body-bg);
        }
        .content-card {
            box-shadow: var(--card-shadow);
            border-radius: 12px;
            border: none;
            margin-bottom: 2rem;
            background-color: #fff;
        }
        .page-header {
            padding-bottom: 1rem;
            border-bottom: 1px solid var(--border-color);
            margin-bottom: 1.5rem;
        }
        .page-title {
            color: #333;
        }
        .flyer-preview, .image-preview { margin-top: 15px; border: 1px dashed #ccc; padding: 10px; border-radius: 8px; background-color: #f9f9f9; }
        .flyer-preview img, .image-preview img { max-width: 200px; max-height: 200px; border: 1px solid #ddd; border-radius: 4px; display: block; margin-bottom: 10px; }
        .flyer-preview-actions label { font-size: 0.9em; cursor: pointer; color: #d9534f; }
        
        .package-form fieldset {
            border: 1px solid #e9ecef;
            padding: 1.5rem;
            border-radius: 12px;
            margin-bottom: 2rem;
            background-color: #fff;
        }
        .package-form legend {
            font-weight: 700;
            padding: 0 10px;
            font-size: 1.2rem;
            color: var(--primary-color);
        }
        .form-hint { font-size: 0.8em; color: #777; display: block; margin-top: 4px; }
        .form-group label {
            font-weight: 600;
            margin-bottom: 0.5rem;
            display: block;
            color: #495057;
        }
        .btn {
            transition: all 0.2s ease-in-out;
        }
        .btn:hover {
            transform: translateY(-2px);
            box-shadow: 0 4px 8px rgba(0,0,0,0.1);
        }
        
        .select2-container .select2-selection--single { height: 40px; border: 1px solid var(--border-color); border-radius: 5px; }
        .select2-container--default .select2-selection--single .select2-selection__rendered { line-height: 38px; }
        .select2-container--default .select2-selection--single .select2-selection__arrow { height: 38px; }

        /* --- Grid System for Forms --- */
        .form-grid {
            display: grid;
            gap: var(--form-grid-gap);
        }
        .form-grid.grid-cols-1 { grid-template-columns: 1fr; }
        .form-grid.grid-cols-2 { grid-template-columns: repeat(auto-fit, minmax(250px, 1fr)); }
        .form-grid.grid-cols-4 { grid-template-columns: repeat(auto-fit, minmax(200px, 1fr)); }
        .form-grid.grid-cols-5 { grid-template-columns: repeat(auto-fit, minmax(150px, 1fr)); }

        .table-toolbar { margin-bottom: 1rem; }

        /* --- Mobile First Table --- */
        .table-responsive { width: 100%; }
        .data-table { width: 100%; border-collapse: collapse; }
        .data-table thead { display: none; } /* Hide headers on mobile */
        .data-table tr {
            display: block;
            margin-bottom: 1rem;
            border: 1px solid var(--border-color);
            border-radius: 8px;
            overflow: hidden;
            background: #fff;
        }
        .data-table td {
            display: flex;
            justify-content: space-between;
            align-items: center;
            text-align: right;
            font-size: 0.9em;
            border-bottom: 1px solid #eee;
            padding: 0.75rem 1rem;
            white-space: normal;
        }
        .data-table td:before {
            content: attr(data-label);
            font-weight: 600;
            color: var(--text-dark);
            text-align: left;
            padding-right: 1rem;
        }
        .data-table td:last-child { border-bottom: 0; }
        .data-table .actions-cell { justify-content: center; background-color: #f9fafb; }
        
        /* --- Tablet and Desktop Styles --- */
        @media (min-width: 851px) {
            .data-table thead { display: table-header-group; }
            .data-table tr { display: table-row; margin-bottom: 0; border: 0; border-bottom: 1px solid var(--border-color); border-radius: 0; }
            .data-table tr:last-child { border-bottom: 0; }
            .data-table td { display: table-cell; text-align: left; border-bottom: 0; }
            .data-table td:before { display: none; }
            .data-table .actions-cell { text-align: center; background-color: transparent;}
            .data-table th:first-child, .data-table td:first-child { text-align: center; }
            .data-table th.checkbox-cell, .data-table td.checkbox-cell { width: 40px; }
        }

        /* General spacing and layout adjustments for mobile */
        @media (max-width: 768px) {
            .content-body { padding: 0.75rem; }
            .content-card, .package-form fieldset { padding: 1rem; }
            .page-header { margin-bottom: 1rem; }
            .page-title { font-size: 1.5rem; }
            .form-actions { display: flex; flex-direction: column; gap: 0.5rem; }
            .form-actions .btn { width: 100%; }
        }
</style>
</head>
<body>
    <div class="dashboard-wrapper">
        <?php include 'sidebar.php'; ?>
        <div class="main-content">
            <header class="main-header">
                <button class="menu-toggle" id="menu-toggle"><i class="fas fa-bars"></i></button>
                <div class="user-info"><span>Welcome, <?php echo htmlspecialchars(ucfirst($_SESSION['user_name'] ?? 'Admin')); ?></span></div>
            </header>
            <main class="content-body" id="top">

        <div class="page-header">
            <h1 class="page-title"><?= $edit_mode ? 'Edit Umrah Package' : 'Manage Umrah Packages' ?></h1>
        </div>

        <?php if (isset($success_message)): ?><div class="notice success"><?= htmlspecialchars($success_message) ?></div><?php endif; ?>
        <?php if (isset($error_message)): ?><div class="notice error"><?= htmlspecialchars($error_message) ?></div><?php endif; ?>
        
        <div class="content-card">
            <h2 class="card-title"><?= $edit_mode ? 'Editing Package: ' . htmlspecialchars($package_to_edit['package_name']) : 'Add New Umrah Package' ?></h2>
            <form action="manage-umrah-packages.php" method="POST" class="package-form" enctype="multipart/form-data">
                <input type="hidden" name="action" value="<?= $edit_mode ? 'update' : 'add' ?>">
                <?php if ($edit_mode): ?>
                    <input type="hidden" name="original_package_id" value="<?= htmlspecialchars($package_to_edit['package_id']) ?>">
                    <input type="hidden" name="main_image_link" value="<?= htmlspecialchars($package_to_edit['main_image_link'] ?? '') ?>">
                <?php endif; ?>
                
                <fieldset><legend>Core Details</legend>
                    <div class="form-grid" style="grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));">
                        <div class="form-group"><label for="package_id">Package ID *</label><input type="text" id="package_id" name="package_id" value="<?= htmlspecialchars($package_to_edit['package_id'] ?? '') ?>" required></div>
                        <div class="form-group"><label for="package_name">Package Name *</label><input type="text" id="package_name" name="package_name" value="<?= htmlspecialchars($package_to_edit['package_name'] ?? '') ?>" required></div>
                        <div class="form-group"><label for="airline_id_select">Airline</label>
                            <select name="airline_id" id="airline_id_select" class="select2-field">
                                <option value="">-- No Airline --</option>
                                <?php if ($airlines_result && $airlines_result->num_rows > 0): $airlines_result->data_seek(0); ?>
                                    <?php while ($airline = $airlines_result->fetch_assoc()): ?>
                                        <option value="<?= $airline['id'] ?>" <?= (($package_to_edit['airline_id'] ?? '') == $airline['id']) ? 'selected' : '' ?>><?= htmlspecialchars($airline['airline_name']) ?></option>
                                    <?php endwhile; ?>
                                <?php endif; ?>
                                <option value="add_new_airline" style="font-weight: bold; color: #007bff;">+ Create New Airline</option>
                            </select>
                        </div>
                        <div class="form-group">
                            <label for="city_id">City Category</label>
                            <select id="city_id" name="city_id" class="select2-field">
                                <option value="">-- No Category --</option>
                                <?php foreach ($cities_for_dropdown as $city): ?>
                                    <option value="<?= $city['id'] ?>" <?= (($package_to_edit['city_id'] ?? null) == $city['id']) ? 'selected' : '' ?>>
                                        <?= htmlspecialchars($city['name']) ?>
                                    </option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                        <div class="form-group"><label for="days">Duration (Days) *</label><input type="number" id="days" name="days" value="<?= htmlspecialchars($package_to_edit['days'] ?? '') ?>" required min="0"></div>
                        <div class="form-group"><label for="is_active">Status</label><select id="is_active" name="is_active" class="select2-field"><option value="1" <?= (($package_to_edit['is_active'] ?? 1) == 1) ? 'selected' : '' ?>>Active</option><option value="0" <?= (($package_to_edit['is_active'] ?? 1) == 0) ? 'selected' : '' ?>>Inactive</option></select></div>
                    </div>
                </fieldset>

                <fieldset><legend>Accommodation & Services</legend>
                    <p>Select hotels to auto-fill details and calculate prices in SAR. You can manually edit any field.</p>
                     <div class="form-grid grid-cols-2" style="margin-top: 1.5rem;">
                        <div class="form-group">
                            <label for="makkah_hotel_select">Select Makkah Hotel</label>
                            <select id="makkah_hotel_select" class="select2-field price-calculator-trigger">
                                <option value="">-- None --</option>
                                <?php if ($makkah_hotels_result->num_rows > 0): $makkah_hotels_result->data_seek(0); ?>
                                    <?php while($hotel = $makkah_hotels_result->fetch_assoc()): ?>
                                        <?php $isSelected = ($edit_mode && isset($package_to_edit['makkah_hotel']) && $package_to_edit['makkah_hotel'] === $hotel['hotel_name']) ? 'selected' : ''; ?>
                                        <option value="<?= $hotel['id'] ?>"
                                            <?= $isSelected ?>
                                            data-name="<?= htmlspecialchars($hotel['hotel_name'] ?? '') ?>"
                                            data-distance="<?= htmlspecialchars($hotel['distance_info'] ?? '') ?>"
                                            data-sharing="<?= htmlspecialchars($hotel['rate_sharing'] ?? '') ?>"
                                            data-quad="<?= htmlspecialchars($hotel['rate_quad'] ?? '') ?>"
                                            data-trpl="<?= htmlspecialchars($hotel['rate_trpl'] ?? '') ?>"
                                            data-dubl="<?= htmlspecialchars($hotel['rate_dubl'] ?? '') ?>"
                                        ><?= htmlspecialchars($hotel['hotel_name'] ?? '') ?></option>
                                    <?php endwhile; ?>
                                <?php endif; ?>
                            </select>
                        </div>
                         <div class="form-group">
                            <label for="madinah_hotel_select">Select Madinah Hotel</label>
                            <select id="madinah_hotel_select" class="select2-field price-calculator-trigger">
                                <option value="">-- None --</option>
                                <?php if ($madinah_hotels_result->num_rows > 0): $madinah_hotels_result->data_seek(0); ?>
                                    <?php while($hotel = $madinah_hotels_result->fetch_assoc()): ?>
                                        <?php $isSelected = ($edit_mode && isset($package_to_edit['madinah_hotel']) && $package_to_edit['madinah_hotel'] === $hotel['hotel_name']) ? 'selected' : ''; ?>
                                        <option value="<?= $hotel['id'] ?>"
                                            <?= $isSelected ?>
                                            data-name="<?= htmlspecialchars($hotel['hotel_name'] ?? '') ?>"
                                            data-distance="<?= htmlspecialchars($hotel['distance_info'] ?? '') ?>"
                                            data-sharing="<?= htmlspecialchars($hotel['rate_sharing'] ?? '') ?>"
                                            data-quad="<?= htmlspecialchars($hotel['rate_quad'] ?? '') ?>"
                                            data-trpl="<?= htmlspecialchars($hotel['rate_trpl'] ?? '') ?>"
                                            data-dubl="<?= htmlspecialchars($hotel['rate_dubl'] ?? '') ?>"
                                        ><?= htmlspecialchars($hotel['hotel_name'] ?? '') ?></option>
                                    <?php endwhile; ?>
                                <?php endif; ?>
                            </select>
                        </div>
                     </div>
                     <div class="form-grid grid-cols-4" style="margin-top: 1rem;">
                        <div class="form-group"><label for="makkah_nights">Makkah Nights</label><input type="number" id="makkah_nights" name="makkah_nights" value="<?= htmlspecialchars($package_to_edit['makkah_nights'] ?? '0') ?>" min="0" class="price-calculator-trigger"></div>
                        <div class="form-group"><label for="makkah_hotel_distance">Makkah Hotel Distance</label><input type="text" placeholder="e.g., 500m" id="makkah_hotel_distance" name="makkah_hotel_distance" value="<?= htmlspecialchars($package_to_edit['makkah_hotel_distance'] ?? '') ?>"></div>
                        <div class="form-group"><label for="madinah_nights">Madinah Nights</label><input type="number" id="madinah_nights" name="madinah_nights" value="<?= htmlspecialchars($package_to_edit['madinah_nights'] ?? '0') ?>" min="0" class="price-calculator-trigger"></div>
                        <div class="form-group"><label for="madinah_hotel_distance">Madinah Hotel Distance</label><input type="text" placeholder="e.g., 200m" id="madinah_hotel_distance" name="madinah_hotel_distance" value="<?= htmlspecialchars($package_to_edit['madinah_hotel_distance'] ?? '') ?>"></div>
                     </div>
                     <div class="form-group" style="margin-top: 1rem;">
                        <label for="makkah_hotel">Makkah Hotel Name (Final)</label>
                        <input type="text" id="makkah_hotel" name="makkah_hotel" value="<?= htmlspecialchars($package_to_edit['makkah_hotel'] ?? '') ?>">
                        <label for="madinah_hotel" style="margin-top: 1rem;">Madinah Hotel Name (Final)</label>
                        <input type="text" id="madinah_hotel" name="madinah_hotel" value="<?= htmlspecialchars($package_to_edit['madinah_hotel'] ?? '') ?>">
                     </div>
                     <div class="form-grid grid-cols-2" style="margin-top: 1rem;">
                        <div class="form-group"><label for="transportation">Transportation</label><select id="transportation" name="transportation" class="select2-field"><option value="Sharing" <?= (($package_to_edit['transportation'] ?? 'Sharing') === 'Sharing') ? 'selected' : '' ?>>Sharing</option><option value="Private" <?= (($package_to_edit['transportation'] ?? '') === 'Private') ? 'selected' : '' ?>>Private</option></select></div>
                        <div class="form-group"><label for="ziyarat">Ziyarat Included</label><select id="ziyarat" name="ziyarat" class="select2-field"><option value="Yes" <?= (($package_to_edit['ziyarat'] ?? 'No') === 'Yes') ? 'selected' : '' ?>>Yes</option><option value="No" <?= (($package_to_edit['ziyarat'] ?? 'No') === 'No') ? 'selected' : '' ?>>No</option></select></div>
                    </div>
                </fieldset>

                <fieldset><legend>Customer Pricing (SAR)</legend>
                    <div class="form-grid grid-cols-4">
                        <div class="form-group"><label for="price_quint">Sharing/Quint</label><input type="number" id="price_quint" step="0.01" name="price_quint" value="<?= htmlspecialchars($package_to_edit['price_quint'] ?? '') ?>"></div>
                        <div class="form-group"><label for="price_quad">Quad</label><input type="number" id="price_quad" step="0.01" name="price_quad" value="<?= htmlspecialchars($package_to_edit['price_quad'] ?? '') ?>"></div>
                        <div class="form-group"><label for="price_triple">Triple</label><input type="number" id="price_triple" step="0.01" name="price_triple" value="<?= htmlspecialchars($package_to_edit['price_triple'] ?? '') ?>"></div>
                        <div class="form-group"><label for="price_double">Double</label><input type="number" id="price_double" step="0.01" name="price_double" value="<?= htmlspecialchars($package_to_edit['price_double'] ?? '') ?>"></div>
                    </div>
                </fieldset>

                <fieldset><legend>Agent Pricing (SAR)</legend>
                    <div class="form-grid grid-cols-5">
                         <div class="form-group"><label for="price_quint_agent">Agent Sharing</label><input type="number" id="price_quint_agent" step="0.01" name="price_quint_agent" value="<?= htmlspecialchars($package_to_edit['price_quint_agent'] ?? '') ?>"></div>
                        <div class="form-group"><label for="price_quad_agent">Agent Quad</label><input type="number" id="price_quad_agent" step="0.01" name="price_quad_agent" value="<?= htmlspecialchars($package_to_edit['price_quad_agent'] ?? '') ?>"></div>
                        <div class="form-group"><label for="price_triple_agent">Agent Triple</label><input type="number" id="price_triple_agent" step="0.01" name="price_triple_agent" value="<?= htmlspecialchars($package_to_edit['price_triple_agent'] ?? '') ?>"></div>
                        <div class="form-group"><label for="price_double_agent">Agent Double</label><input type="number" id="price_double_agent" step="0.01" name="price_double_agent" value="<?= htmlspecialchars($package_to_edit['price_double_agent'] ?? '') ?>"></div>
                        <div class="form-group">
                            <label for="agent_incentive">Agent Incentive (SAR)</label>
                            <input type="number" id="agent_incentive" step="0.01" name="agent_incentive" value="<?= htmlspecialchars($package_to_edit['agent_incentive'] ?? '0.00') ?>" readonly>
                            <small class="form-hint">Auto-calculated: (Total Nights * 2)</small>
                        </div>
                    </div>
                </fieldset>

                <fieldset><legend>Flight Details</legend>
                    <div class="form-grid grid-cols-2">
                        <div class="form-group"><label for="outbound_flight_details">Outbound</label><small class="form-hint">e.g., SV727 - ISB to JED - 03:00 / 06:10</small><textarea name="outbound_flight_details" id="outbound_flight_details" rows="3"><?= htmlspecialchars($package_to_edit['outbound_flight_details'] ?? '') ?></textarea></div>
                        <div class="form-group"><label for="inbound_flight_details">Inbound</label><small class="form-hint">e.g., SV726 - JED to ISB - 18:10 / 01:20</small><textarea name="inbound_flight_details" id="inbound_flight_details" rows="3"><?= htmlspecialchars($package_to_edit['inbound_flight_details'] ?? '') ?></textarea></div>
                    </div>
                </fieldset>
                
                <fieldset><legend>Website Content</legend>
                     <div class="form-grid grid-cols-2">
                        <div class="form-group"><label for="page_link">Page Link (URL Slug)</label><input type="text" id="page_link" name="page_link" placeholder="e.g. 15-days-economy-umrah" value="<?= htmlspecialchars($package_to_edit['page_link'] ?? '') ?>"><small class="form-hint">Creates a page like `your-slug.php`. Leave blank for no page.</small></div>
                         <div class="form-group">
                            <label for="main_image_file">Main Image</label>
                            <?php if ($edit_mode && !empty($package_to_edit['main_image_link'])): ?>
                            <div class="image-preview">
                                <p><strong>Current Image:</strong></p> 
                                <img src="../<?= htmlspecialchars($package_to_edit['main_image_link']) ?>" alt="Main Package Image">
                            </div>
                            <small class="form-hint">To replace, just upload a new file below.</small>
                            <?php endif; ?>
                            <input type="file" name="main_image_file" id="main_image_file" accept="image/jpeg,image/png,image/gif,image/webp">
                            <small class="form-hint">If no image is uploaded for a new package, a default will be used.</small>
                        </div>
                        <div class="form-group"><label for="logo_has_background">Company Logo Style</label>
                            <select id="logo_has_background" name="logo_has_background" class="select2-field">
                                <option value="1" <?= (($package_to_edit['logo_has_background'] ?? 1) == 1) ? 'selected' : '' ?>>With White Background (Circle)</option>
                                <option value="0" <?= (($package_to_edit['logo_has_background'] ?? 1) == 0) ? 'selected' : '' ?>>Transparent (No Background)</option>
                            </select>
                        </div>
                    </div>
                    <div class="form-group">
                        <label for="flyer_image">Package Flyer Image</label>
                        <?php if ($edit_mode && !empty($package_to_edit['flyer_image_path'])): ?>
                        <div class="flyer-preview">
                            <p><strong>Current Flyer:</strong></p> <img src="<?= htmlspecialchars($package_to_edit['flyer_image_path']) ?>" alt="Flyer">
                            <div class="flyer-preview-actions"> <input type="checkbox" name="remove_flyer" id="remove_flyer" value="1"> <label for="remove_flyer">Remove current flyer.</label> </div>
                        </div><small class="form-hint">To replace, just upload a new file below.</small>
                        <?php endif; ?>
                        <input type="file" name="flyer_image" id="flyer_image" accept="image/jpeg,image/png,image/gif,image/webp">
                    </div>
                    <div class="form-group"><label for="overview">Overview</label><textarea id="overview" name="overview" rows="4"><?= htmlspecialchars($package_to_edit['overview'] ?? '') ?></textarea></div>
                    <div class="form-group"><label for="itinerary">Itinerary</label><small class="form-hint">Enter one point per line.</small><textarea id="itinerary" name="itinerary" rows="6"><?= htmlspecialchars($package_to_edit['itinerary'] ?? '') ?></textarea></div>
                    <div class="form-group"><label for="whats_included">What's Included</label><small class="form-hint">Enter one point per line or separate with commas.</small><textarea id="whats_included" name="whats_included" rows="6"><?= htmlspecialchars($package_to_edit['whats_included'] ?? '') ?></textarea></div>
                    <div class="form-group"><label for="whats_extra">What's Extra</label><small class="form-hint">Enter one point per line.</small><textarea id="whats_extra" name="whats_extra" rows="6"><?= htmlspecialchars($package_to_edit['whats_extra'] ?? '') ?></textarea></div>
                </fieldset>

                <div class="form-actions">
                    <button type="submit" class="btn btn-primary"><?= $edit_mode ? 'Update Package' : 'Add Package' ?></button>
                    <?php if ($edit_mode): ?><a href="manage-umrah-packages.php" class="btn btn-secondary">Cancel Edit</a><?php endif; ?>
                </div>
            </form>
        </div>
        
        <div class="content-card">
            <h2 class="card-title">Existing Umrah Packages</h2>
            <form action="manage-umrah-packages.php" method="POST" id="bulk-delete-form">
                <div class="table-toolbar">
                    <button type="submit" name="action" value="bulk_delete" class="btn btn-danger"><i class="fas fa-trash"></i> Delete Selected</button>
                </div>
                <div class="table-responsive">
                    <table class="data-table">
                        <thead>
                            <tr>
                                <th class="checkbox-cell"><input type="checkbox" id="select-all"></th>
                                <th>ID</th>
                                <th>Name</th>
                                <th>Airline</th>
                                <th>City</th>
                                <th>Status</th>
                                <th>Page Link</th>
                                <th>Actions</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php if ($packages_result && $packages_result->num_rows > 0): ?>
                                <?php while ($pkg = $packages_result->fetch_assoc()): ?>
                                <tr class="<?= ($edit_mode && isset($package_to_edit['package_id']) && $pkg['package_id'] === $package_to_edit['package_id']) ? 'editing' : '' ?>">
                                    <td class="checkbox-cell" data-label="Select"><input type="checkbox" name="package_ids[]" class="package-checkbox" value="<?= htmlspecialchars($pkg['package_id']) ?>"></td>
                                    <td data-label="ID"><strong><?= htmlspecialchars($pkg['package_id']) ?></strong></td>
                                    <td data-label="Name"><?= htmlspecialchars($pkg['package_name']) ?></td>
                                    <td data-label="Airline">
                                        <?php if (!empty($pkg['logo_url'])): ?><img src="<?= htmlspecialchars($pkg['logo_url']) ?>" alt="Airline Logo" style="max-height: 25px; vertical-align: middle; margin-right: 10px;"><?php endif; ?>
                                        <?= htmlspecialchars($pkg['airline_name'] ?? 'N/A') ?>
                                    </td>
                                    <td data-label="City"><?= htmlspecialchars($pkg['city_name'] ?? 'N/A') ?></td>
                                    <td data-label="Status"><span class="status-badge <?= $pkg['is_active'] ? 'status-completed' : 'status-rejected' ?>"><?= $pkg['is_active'] ? 'Active' : 'Inactive' ?></span></td>
                                    <td data-label="Page Link">
                                        <?php if (!empty($pkg['page_link'])): $page_path = '../' . htmlspecialchars($pkg['page_link']) . '.php'; ?>
                                            <?php if (file_exists($page_path)): ?>
                                                <a href="<?= $page_path ?>" target="_blank"><?= htmlspecialchars($pkg['page_link']) ?>.php <i class="fas fa-external-link-alt fa-xs"></i></a>
                                            <?php else: ?>
                                                <span style="color: #dc3545;"><?= htmlspecialchars($pkg['page_link']) ?>.php (missing)</span>
                                            <?php endif; ?>
                                        <?php else: ?>
                                            <span style="color: #999;">No page</span>
                                        <?php endif; ?>
                                    </td>
                                    <td class="actions-cell" data-label="Actions">
                                        <a href="?action=edit&id=<?= urlencode($pkg['package_id']) ?>#top" class="btn btn-sm btn-secondary" title="Edit"><i class="fas fa-edit"></i></a>
                                        <form action="manage-umrah-packages.php" method="POST" onsubmit="return confirm('Are you sure you want to permanently delete this package and its page? This cannot be undone.');" style="display:inline;">
                                            <input type="hidden" name="action" value="delete">
                                            <input type="hidden" name="package_id" value="<?= htmlspecialchars($pkg['package_id']) ?>">
                                            <button type="submit" class="btn btn-sm btn-danger" title="Delete"><i class="fas fa-trash"></i></button>
                                        </form>
                                    </td>
                                </tr>
                                <?php endwhile; ?>
                            <?php else: ?>
                                <tr><td colspan="8" class="empty-state">No Umrah packages have been created yet.</td></tr>
                            <?php endif; ?>
                        </tbody>
                    </table>
                </div>
            </form>
        </div>

        <div class="content-card" id="cities-section">
            <form id="cities-form" action="manage-umrah-packages.php" method="POST">
                <input type="hidden" name="action" value="delete_multiple_cities">
                <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 1rem;">
                    <h2 class="card-title">Manage City Categories</h2>
                    <button type="submit" class="btn btn-danger" onclick="return confirm('Are you sure you want to delete the selected cities?');">Delete Selected</button>
                </div>
                <div class="table-responsive">
                    <table class="data-table" style="min-width: 0;">
                        <thead>
                            <tr>
                                <th style="width: 40px;"><input type="checkbox" id="selectAllCities"></th>
                                <th>City Name</th>
                                <th style="width: 150px;">Action</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php if ($cities_result && $cities_result->num_rows > 0): ?>
                                <?php while ($city = $cities_result->fetch_assoc()): ?>
                                <tr id="city-row-<?= $city['id'] ?>">
                                    <td><input type="checkbox" name="city_ids[]" value="<?= $city['id'] ?>" class="city-checkbox"></td>
                                    <td class="city-name-cell"><?= htmlspecialchars($city['name']) ?></td>
                                    <td class="actions-cell">
                                        <button type="button" class="btn btn-sm btn-secondary edit-city-btn" data-id="<?= $city['id'] ?>" data-name="<?= htmlspecialchars($city['name']) ?>"><i class="fas fa-edit"></i> Edit</button>
                                        <form action="manage-umrah-packages.php" method="POST" onsubmit="return confirm('Are you sure you want to delete this city?');" style="display:inline;">
                                            <input type="hidden" name="action" value="delete_city">
                                            <input type="hidden" name="id" value="<?= $city['id'] ?>">
                                            <button type="submit" class="btn btn-sm btn-danger" title="Delete"><i class="fas fa-trash"></i></button>
                                        </form>
                                    </td>
                                </tr>
                                <?php endwhile; ?>
                            <?php else: ?>
                                <tr><td colspan="3" class="empty-state">No cities found.</td></tr>
                            <?php endif; ?>
                        </tbody>
                    </table>
                </div>
            </form>
            <div class="form-grid" style="margin-top: 2rem;">
                <div class="form-group">
                    <label for="city_name_add"><strong>Add New City</strong></label>
                    <form action="manage-umrah-packages.php" method="POST" style="display: flex; gap: 1rem;">
                        <input type="hidden" name="action" value="add_city">
                        <input type="text" id="city_name_add" name="city_name" placeholder="e.g. Lahore" required style="flex-grow: 1;">
                        <button type="submit" class="btn btn-primary">Add City</button>
                    </form>
                </div>
            </div>
        </div>

    </main>
</div>
</div>
<!-- JQuery (Required for Select2) -->
<script src="https://code.jquery.com/jquery-3.6.0.min.js"></script>
<!-- Select2 JS -->
<script src="https://cdn.jsdelivr.net/npm/select2@4.1.0-rc.0/dist/js/select2.min.js"></script>

<script>
document.addEventListener('DOMContentLoaded', function() {
    // --- Initialize Select2 ---
    $('.select2-field').select2({
        width: '100%' // Ensure select2 takes full width of its container
    });

    // --- Sidebar Toggle ---
    document.getElementById('menu-toggle').addEventListener('click', function() {
        document.querySelector('.dashboard-wrapper').classList.toggle('sidebar-open');
    });

    const airlineSelect = document.getElementById('airline_id_select');
    if(airlineSelect) {
        $(airlineSelect).on('change', function() {
            if (this.value === 'add_new_airline') {
                window.location.href = 'manage-group-fares.php';
            }
        });
    }

    // --- Bulk Delete Functionality ---
    const selectAllCheckbox = document.getElementById('select-all');
    const packageCheckboxes = document.querySelectorAll('.package-checkbox');
    const bulkDeleteForm = document.getElementById('bulk-delete-form');

    if (selectAllCheckbox) {
        selectAllCheckbox.addEventListener('change', function() {
            packageCheckboxes.forEach(checkbox => {
                checkbox.checked = this.checked;
            });
        });
    }

    if (bulkDeleteForm) {
        bulkDeleteForm.addEventListener('submit', function(e) {
            const actionButton = document.activeElement;
            if (actionButton && actionButton.value !== 'bulk_delete') {
                return;
            }
            
            const anyChecked = [...packageCheckboxes].some(c => c.checked);
            if (!anyChecked) {
                alert('Please select at least one package to delete.');
                e.preventDefault();
                return;
            }
            
            if (!confirm('Are you sure you want to permanently delete all selected packages and their associated pages? This action cannot be undone.')) {
                e.preventDefault();
            }
        });
    }


    // --- Auto Price & Detail Calculation ---
    const makkahHotelSelect = document.getElementById('makkah_hotel_select');
    const madinahHotelSelect = document.getElementById('madinah_hotel_select');
    const makkahNightsInput = document.getElementById('makkah_nights');
    const madinahNightsInput = document.getElementById('madinah_nights');

    const makkahHotelNameInput = document.getElementById('makkah_hotel');
    const makkahDistanceInput = document.getElementById('makkah_hotel_distance');
    const madinahHotelNameInput = document.getElementById('madinah_hotel');
    const madinahDistanceInput = document.getElementById('madinah_hotel_distance');

    function updateHotelDetails() {
        const selectedMakkah = makkahHotelSelect.options[makkahHotelSelect.selectedIndex];
        if (selectedMakkah && selectedMakkah.value) {
            makkahHotelNameInput.value = selectedMakkah.dataset.name || '';
            makkahDistanceInput.value = selectedMakkah.dataset.distance || '';
        }

        const selectedMadinah = madinahHotelSelect.options[madinahHotelSelect.selectedIndex];
        if (selectedMadinah && selectedMadinah.value) {
            madinahHotelNameInput.value = selectedMadinah.dataset.name || '';
            madinahDistanceInput.value = selectedMadinah.dataset.distance || '';
        }
    }

    function calculateAndFillPrices() {
        const makkahNights = parseFloat(makkahNightsInput.value) || 0;
        const madinahNights = parseFloat(madinahNightsInput.value) || 0;

        const totalNights = makkahNights + madinahNights;
        const agentIncentive = totalNights * 2;
        document.getElementById('agent_incentive').value = agentIncentive.toFixed(2);

        const selectedMakkah = makkahHotelSelect.options[makkahHotelSelect.selectedIndex];
        const selectedMadinah = madinahHotelSelect.options[madinahHotelSelect.selectedIndex];

        const getRate = (option, rateType) => (option && option.value) ? (parseFloat(option.dataset[rateType]) || 0) : 0;
        
        const rates = {
            dubl: { makkah: getRate(selectedMakkah, 'dubl'), madinah: getRate(selectedMadinah, 'dubl') },
            trpl: { makkah: getRate(selectedMakkah, 'trpl'), madinah: getRate(selectedMadinah, 'trpl') },
            quad: { makkah: getRate(selectedMakkah, 'quad'), madinah: getRate(selectedMadinah, 'quad') },
            sharing: { makkah: getRate(selectedMakkah, 'sharing'), madinah: getRate(selectedMadinah, 'sharing') }
        };

        const calculatePrice = (rateType, isAgent) => {
            const agentDeduction = isAgent ? 2 : 0;
            const makkahRate = rates[rateType].makkah > 0 ? rates[rateType].makkah - agentDeduction : 0;
            const madinahRate = rates[rateType].madinah > 0 ? rates[rateType].madinah - agentDeduction : 0;
            const totalSar = (makkahRate * makkahNights) + (madinahRate * madinahNights);
            return totalSar > 0 ? totalSar.toFixed(2) : '';
        };

        document.getElementById('price_double').value = calculatePrice('dubl', false);
        document.getElementById('price_triple').value = calculatePrice('trpl', false);
        document.getElementById('price_quad').value = calculatePrice('quad', false);
        document.getElementById('price_quint').value = calculatePrice('sharing', false);

        document.getElementById('price_double_agent').value = calculatePrice('dubl', true);
        document.getElementById('price_triple_agent').value = calculatePrice('trpl', true);
        document.getElementById('price_quad_agent').value = calculatePrice('quad', true);
        document.getElementById('price_quint_agent').value = calculatePrice('sharing', true);
    }
    
    document.querySelectorAll('.price-calculator-trigger').forEach(el => {
        if (el.tagName === 'SELECT') {
            $(el).on('change', () => {
                updateHotelDetails();
                calculateAndFillPrices();
            });
        } else {
            el.addEventListener('input', calculateAndFillPrices);
        }
    });

    if (document.querySelector('input[name="action"]').value === 'update') {
        updateHotelDetails();
    }
    
    // --- City Management Logic ---
    document.getElementById('selectAllCities')?.addEventListener('change', function(e) {
        document.querySelectorAll('.city-checkbox').forEach(checkbox => {
            checkbox.checked = e.target.checked;
        });
    });

    document.querySelectorAll('.edit-city-btn').forEach(button => {
        button.addEventListener('click', function() {
            const id = this.dataset.id;
            const name = this.dataset.name;
            const row = document.getElementById(`city-row-${id}`);
            const nameCell = row.querySelector('.city-name-cell');
            
            nameCell.innerHTML = `
                <form action="manage-umrah-packages.php" method="POST" style="display: flex; gap: 5px;">
                    <input type="hidden" name="action" value="update_city">
                    <input type="hidden" name="id" value="${id}">
                    <input type="text" name="city_name" value="${name}" required style="width: 100%; padding: 8px;">
                    <button type="submit" class="btn btn-sm btn-primary">Save</button>
                    <button type="button" class="btn btn-sm btn-secondary cancel-edit-city">Cancel</button>
                </form>
            `;
            
            row.querySelector('.cancel-edit-city').addEventListener('click', () => {
                nameCell.innerHTML = name;
            });
        });
    });
});
</script>

</body>
</html>