<?php
session_start();
include_once '../db-config.php';

// A helper function for safely echoing output
function e($string)
{
    return htmlspecialchars($string ?? '', ENT_QUOTES, 'UTF-8');
}

// --- SECURITY CHECK: Ensure user is a logged-in admin ---
if (!isset($_SESSION['user_id']) || !isset($_SESSION['user_type']) || $_SESSION['user_type'] !== 'admin') {
    header("location: ../login.php");
    exit;
}

// --- 1. SETUP FILTERS (SYNCED WITH VENDOR-LEDGER.PHP) ---
$filter_vendor_id = (int)($_GET['vendor_id'] ?? 0);
$filter_start_date = $_GET['start_date'] ?? '';
$filter_end_date = $_GET['end_date'] ?? '';

$selected_vendor_details = null;
$transactions = [];
$opening_balance = 0;

if ($filter_vendor_id <= 0) {
    die("No Vendor ID specified. Please go back and select a vendor.");
}

// --- 2. FETCH SELECTED VENDOR DETAILS (SYNCED WITH VENDOR-LEDGER.PHP) ---
$stmt_vendor = $conn->prepare("SELECT * FROM vendors WHERE id = ?");
$stmt_vendor->bind_param("i", $filter_vendor_id);
$stmt_vendor->execute();
$result_vendor = $stmt_vendor->get_result();
if ($result_vendor) {
    $selected_vendor_details = $result_vendor->fetch_assoc();
}
$stmt_vendor->close();

if (!$selected_vendor_details) {
    die("Could not find the selected vendor.");
}


// --- 3. CALCULATE OPENING BALANCE (SYNCED & DETAILED) ---
if (!empty($filter_start_date)) {
    $opening_sql_parts = [];
    $opening_params = [];
    $opening_types = '';

    // Costs from Package Invoices (Detailed)
    $opening_sql_parts[] = "(SELECT SUM(ip.visa_price_sar_cost * i.exchange_rate) as credit, 0 as debit FROM invoice_pilgrims ip JOIN invoices i ON ip.invoice_id = i.id WHERE i.pilgrims_vendor_id = ? AND i.issue_date < ?)";
    array_push($opening_params, $filter_vendor_id, $filter_start_date);
    $opening_types .= 'is';
    $opening_sql_parts[] = "(SELECT SUM(it.total_amount_cost) as credit, 0 as debit FROM invoice_transports it JOIN invoices i ON it.invoice_id = i.id WHERE i.transport_vendor_id = ? AND i.issue_date < ?)";
    array_push($opening_params, $filter_vendor_id, $filter_start_date);
    $opening_types .= 'is';
    $opening_sql_parts[] = "(SELECT SUM(iat.total_amount_cost) as credit, 0 as debit FROM invoice_airline_tickets iat JOIN invoices i ON iat.invoice_id = i.id WHERE i.tickets_vendor_id = ? AND i.issue_date < ?)";
    array_push($opening_params, $filter_vendor_id, $filter_start_date);
    $opening_types .= 'is';
    $opening_sql_parts[] = "(SELECT SUM(ih.total_sar_cost * i.exchange_rate) as credit, 0 as debit FROM invoice_hotels ih JOIN invoices i ON ih.invoice_id = i.id WHERE ih.vendor_id = ? AND i.issue_date < ?)";
    array_push($opening_params, $filter_vendor_id, $filter_start_date);
    $opening_types .= 'is';
    $opening_sql_parts[] = "(SELECT SUM(ios.total_amount_cost * i.exchange_rate) as credit, 0 as debit FROM invoice_other_services ios JOIN invoices i ON ios.invoice_id = i.id WHERE ios.vendor_id = ? AND i.issue_date < ?)";
    array_push($opening_params, $filter_vendor_id, $filter_start_date);
    $opening_types .= 'is';
    $opening_sql_parts[] = "(SELECT SUM(ih.total_sar_cost * i.exchange_rate) as credit, 0 as debit FROM invoice_hotels ih JOIN invoices i ON ih.invoice_id = i.id WHERE i.vendor_id = ? AND ih.vendor_id IS NULL AND i.issue_date < ?)";
    array_push($opening_params, $filter_vendor_id, $filter_start_date);
    $opening_types .= 'is';
    $opening_sql_parts[] = "(SELECT SUM(ios.total_amount_cost * i.exchange_rate) as credit, 0 as debit FROM invoice_other_services ios JOIN invoices i ON ios.invoice_id = i.id WHERE i.vendor_id = ? AND ios.vendor_id IS NULL AND i.issue_date < ?)";
    array_push($opening_params, $filter_vendor_id, $filter_start_date);
    $opening_types .= 'is';

    // Costs from Standalone Ticket Invoices
    $opening_sql_parts[] = "(SELECT ti.grand_total_pkr_cost as credit, 0 as debit FROM ticket_invoices ti WHERE ti.vendor_id = ? AND ti.issue_date < ?)";
    array_push($opening_params, $filter_vendor_id, $filter_start_date);
    $opening_types .= 'is';

    // Payments
    $opening_sql_parts[] = "(SELECT p.credit_amount as credit, p.debit_amount as debit FROM payments p WHERE p.vendor_id = ? AND p.payment_date < ?)";
    array_push($opening_params, $filter_vendor_id, $filter_start_date);
    $opening_types .= 'is';

    $opening_balance_sql = "SELECT SUM(credit - debit) as opening_balance FROM (" . implode(" UNION ALL ", $opening_sql_parts) . ") AS opening_transactions";

    $stmt_opening = $conn->prepare($opening_balance_sql);
    $stmt_opening->bind_param($opening_types, ...$opening_params);
    $stmt_opening->execute();
    $opening_balance = (float)($stmt_opening->get_result()->fetch_assoc()['opening_balance'] ?? 0);
    $stmt_opening->close();
}

// --- 4. FETCH ALL TRANSACTIONS (SYNCED & CONSOLIDATED) ---
$sql_parts = [];
$params = [];
$types = '';

// --- NEW: Consolidated query for all package-related costs for the vendor ---
$vendor_costs_sql = "
    SELECT
        i.id as transaction_id,
        i.issue_date as date,
        'Booking Cost' as type,
        i.invoice_number as number,
        CASE
            WHEN i.vendor_id = ? THEN CONCAT(i.guest_name, ' x ', (SELECT COUNT(*) FROM invoice_pilgrims ip_count WHERE ip_count.invoice_id = i.id), ' Pax')
            ELSE
                TRIM(TRAILING ', ' FROM CONCAT_WS(', ',
                    CASE WHEN i.pilgrims_vendor_id = ? THEN 'Visa Cost' ELSE NULL END,
                    CASE WHEN i.transport_vendor_id = ? THEN 'Transport Cost' ELSE NULL END,
                    CASE WHEN i.tickets_vendor_id = ? THEN 'Ticket Cost (Pkg)' ELSE NULL END,
                    CASE WHEN SUM(CASE WHEN ih.vendor_id = ? THEN 1 ELSE 0 END) > 0 THEN 'Hotel Cost' ELSE NULL END,
                    CASE WHEN SUM(CASE WHEN ios.vendor_id = ? THEN 1 ELSE 0 END) > 0 THEN 'Service Cost' ELSE NULL END
                ))
        END as particulars,
        0 as debit,
        SUM(
            IFNULL(CASE WHEN i.pilgrims_vendor_id = ? THEN ip.visa_price_sar_cost * i.exchange_rate ELSE 0 END, 0) +
            IFNULL(CASE WHEN i.transport_vendor_id = ? THEN it.total_amount_cost ELSE 0 END, 0) +
            IFNULL(CASE WHEN i.tickets_vendor_id = ? THEN iat.total_amount_cost ELSE 0 END, 0) +
            IFNULL(CASE WHEN i.vendor_id = ? AND ih.vendor_id IS NULL THEN ih.total_sar_cost * i.exchange_rate ELSE 0 END, 0) +
            IFNULL(CASE WHEN i.vendor_id = ? AND ios.vendor_id IS NULL THEN ios.total_amount_cost * i.exchange_rate ELSE 0 END, 0) +
            IFNULL(CASE WHEN ih.vendor_id = ? THEN ih.total_sar_cost * i.exchange_rate ELSE 0 END, 0) +
            IFNULL(CASE WHEN ios.vendor_id = ? THEN ios.total_amount_cost * i.exchange_rate ELSE 0 END, 0)
        ) as credit,
        'package' as source
    FROM invoices i
    LEFT JOIN invoice_pilgrims ip ON i.id = ip.invoice_id
    LEFT JOIN invoice_transports it ON i.id = it.invoice_id
    LEFT JOIN invoice_airline_tickets iat ON i.id = iat.invoice_id
    LEFT JOIN invoice_hotels ih ON i.id = ih.invoice_id
    LEFT JOIN invoice_other_services ios ON i.id = ios.invoice_id
    WHERE 
        (i.pilgrims_vendor_id = ? OR i.transport_vendor_id = ? OR i.tickets_vendor_id = ? OR i.vendor_id = ? OR ih.vendor_id = ? OR ios.vendor_id = ?)
        %s
    GROUP BY i.id, i.issue_date, i.invoice_number, i.guest_name
    ";

$date_filter_sql = "";
$vendor_params = array_fill(0, 19, $filter_vendor_id);
$vendor_types = str_repeat('i', 19);
$date_params = [];
$date_types = '';

if (!empty($filter_start_date)) {
    $date_filter_sql .= " AND i.issue_date >= ?";
    $date_params[] = $filter_start_date;
    $date_types .= 's';
}
if (!empty($filter_end_date)) {
    $date_filter_sql .= " AND i.issue_date <= ?";
    $date_params[] = $filter_end_date;
    $date_types .= 's';
}

$sql_parts[] = sprintf($vendor_costs_sql, $date_filter_sql);
$params = array_merge($params, $vendor_params, $date_params);
$types .= $vendor_types . $date_types;

// --- Standalone Ticket Costs (unchanged) ---
$ticket_cost_sql = "(SELECT ti.id as transaction_id, ti.issue_date as date, 'Ticket Cost' as type, ti.invoice_number as number, CONCAT('Ticket for ', ti.guest_name) as particulars, 0 as debit, ti.grand_total_pkr_cost as credit, 'ticket' as source FROM ticket_invoices ti WHERE ti.vendor_id = ? AND ti.grand_total_pkr_cost > 0";
$local_params = [$filter_vendor_id];
$local_types = 'i';
if (!empty($filter_start_date)) {
    $ticket_cost_sql .= " AND ti.issue_date >= ?";
    $local_params[] = $filter_start_date;
    $local_types .= 's';
}
if (!empty($filter_end_date)) {
    $ticket_cost_sql .= " AND ti.issue_date <= ?";
    $local_params[] = $filter_end_date;
    $local_types .= 's';
}
$ticket_cost_sql .= ")";
$sql_parts[] = $ticket_cost_sql;
$params = array_merge($params, $local_params);
$types .= $local_types;

// --- Payments (unchanged) ---
$payment_sql = "(SELECT p.id as transaction_id, p.payment_date as date, CASE WHEN p.debit_amount > 0 AND p.payment_method IN ('Bank Transfer', 'Card') THEN 'BP' WHEN p.debit_amount > 0 AND p.payment_method = 'Cash' THEN 'CP' WHEN p.credit_amount > 0 AND p.payment_method IN ('Bank Transfer', 'Card') THEN 'BR' WHEN p.credit_amount > 0 AND p.payment_method = 'Cash' THEN 'CR' ELSE 'Payment' END as type, CONCAT(CASE WHEN p.debit_amount > 0 AND p.payment_method IN ('Bank Transfer', 'Card') THEN 'BP-' WHEN p.debit_amount > 0 AND p.payment_method = 'Cash' THEN 'CP-' WHEN p.credit_amount > 0 AND p.payment_method IN ('Bank Transfer', 'Card') THEN 'BR-' WHEN p.credit_amount > 0 AND p.payment_method = 'Cash' THEN 'CR-' ELSE 'PAY-' END, p.id) as number, p.notes as particulars, p.debit_amount as debit, p.credit_amount as credit, 'payment' as source FROM payments p WHERE p.vendor_id = ?";
$local_params = [$filter_vendor_id];
$local_types = 'i';
if (!empty($filter_start_date)) {
    $payment_sql .= " AND p.payment_date >= ?";
    $local_params[] = $filter_start_date;
    $local_types .= 's';
}
if (!empty($filter_end_date)) {
    $payment_sql .= " AND p.payment_date <= ?";
    $local_params[] = $filter_end_date;
    $local_types .= 's';
}
$payment_sql .= ")";
$sql_parts[] = $payment_sql;
$params = array_merge($params, $local_params);
$types .= $local_types;

$final_sql = "SELECT transaction_id, date, type, number, particulars, debit, credit, source FROM (" . implode(" UNION ALL ", $sql_parts) . ") AS all_transactions ORDER BY date ASC, transaction_id ASC";

$stmt_period = $conn->prepare($final_sql);
if ($stmt_period) {
    if (!empty($params)) {
        $stmt_period->bind_param($types, ...$params);
    }
    $stmt_period->execute();
    $transactions = $stmt_period->get_result()->fetch_all(MYSQLI_ASSOC);
    $stmt_period->close();
}

// --- 5. PROCESS DATA FOR DISPLAY ---
$total_debit_period = 0;
$total_credit_period = 0;
foreach ($transactions as $t) {
    $total_debit_period += (float)$t['debit'];
    $total_credit_period += (float)$t['credit'];
}
$closing_balance = $opening_balance + $total_credit_period - $total_debit_period;
?>
<!DOCTYPE html>
<html lang="en">

<head>
    <meta charset="UTF-8">
    <title>Vendor Ledger Statement for <?= e($selected_vendor_details['name']) ?></title>
    <link rel="icon" type="image/png" href="../images/logo-icon.png">
    <style>
        :root {
            --theme-color: #f0f0f0;
            --border-color: #ccc;
        }

        body {
            font-family: -apple-system, BlinkMacSystemFont, "Segoe UI", Roboto, "Helvetica Neue", Arial, sans-serif;
            background-color: #e9e9e9;
            margin: 0;
            padding: 20px;
            font-size: 10pt;
            color: #333;
        }

        .actions-bar {
            max-width: 1100px;
            margin: 0 auto 15px auto;
            display: flex;
            justify-content: flex-end;
            gap: 10px;
        }

        .btn {
            padding: 8px 15px;
            border: none;
            border-radius: 4px;
            color: white;
            font-size: 14px;
            cursor: pointer;
            text-decoration: none;
            display: inline-block;
        }

        .btn-print {
            background-color: #0d2d4c;
        }

        .print-wrapper {
            max-width: 1100px;
            margin: 0 auto;
            padding: 30px;
            border: 1px solid #ccc;
            background-color: #fff;
            box-shadow: 0 0 10px rgba(0, 0, 0, 0.1);
        }

        table {
            width: 100%;
            border-collapse: collapse;
        }

        td,
        th {
            padding: 4px;
            vertical-align: top;
        }

        .header-table td {
            border: none;
            padding: 0;
        }

        .company-logo-container {
            width: 50%;
            text-align: center;
        }

        .company-logo-container img {
            max-height: 50px;
        }

        .company-details {
            font-size: 9pt;
            line-height: 1.4;
            padding-top: 5px;
        }

        .statement-meta {
            width: 50%;
        }

        .statement-meta table {
            border: 1px solid var(--border-color);
        }

        .statement-meta td {
            padding: 5px 8px;
            font-size: 9pt;
        }

        .statement-meta td:first-child {
            font-weight: bold;
            background-color: var(--theme-color);
            width: 100px;
        }

        .vendor-details {
            border: 1px solid var(--border-color);
            margin-top: 20px;
            padding: 15px;
            background: #fafafa;
            border-radius: 5px;
        }

        .vendor-details h3 {
            margin: 0 0 10px 0;
            font-size: 12pt;
            border-bottom: 1px solid #eee;
            padding-bottom: 5px;
        }

        .vendor-details table td {
            padding: 3px 0;
        }

        .vendor-details table td:first-child {
            font-weight: bold;
            width: 120px;
        }

        .summary-container {
            display: grid;
            grid-template-columns: repeat(4, 1fr);
            gap: 1rem;
            margin: 20px 0;
        }

        .summary-item {
            text-align: center;
            padding: 1rem;
            background-color: #fff;
            border-radius: 6px;
            border: 1px solid #e0e0e0;
        }

        .summary-item .label {
            font-size: 0.9em;
            color: #6c757d;
            margin-bottom: 5px;
            text-transform: uppercase;
            letter-spacing: 0.5px;
        }

        .summary-item .value {
            font-size: 1.5em;
            font-weight: 600;
        }

        .summary-item .debit {
            color: #44bd32;
        }

        .summary-item .credit {
            color: #c23616;
        }

        .summary-item .balance {
            color: #0056b3;
        }

        .ledger-table th {
            background-color: var(--theme-color);
            border: 1px solid var(--border-color);
            padding: 8px;
            text-align: left;
            font-weight: 600;
        }

        .ledger-table td {
            border: 1px solid var(--border-color);
            padding: 7px;
            vertical-align: middle;
        }

        .ledger-table td.number {
            text-align: right;
            font-family: 'SFMono-Regular', Consolas, 'Liberation Mono', Menlo, Courier, monospace;
        }

        .ledger-table .particulars {
            white-space: normal;
            word-break: break-word;
        }

        .ledger-table .balance-row,
        .ledger-table .totals-row {
            font-weight: bold;
            background-color: #f9f9f9;
        }

        .footer {
            text-align: center;
            margin-top: 30px;
            font-size: 9pt;
            color: #777;
            border-top: 1px solid #eee;
            padding-top: 15px;
        }

        @media print {
            body {
                background-color: #fff;
                margin: 0;
                padding: 0;
                font-size: 9pt;
            }

            .actions-bar {
                display: none;
            }

            .print-wrapper {
                box-shadow: none;
                border: none;
                margin: 0;
                padding: 0;
                max-width: 100%;
            }

            * {
                color-adjust: exact !important;
                -webkit-print-color-adjust: exact !important;
                print-color-adjust: exact !important;
            }
        }
    </style>
</head>

<body>
    <div class="actions-bar">
        <a href="javascript:window.print()" class="btn btn-print">Print Statement</a>
    </div>
    <div class="print-wrapper">
        <header>
            <table class="header-table">
                <tr>
                    <td class="company-logo-container"><img src="../images/logo.png" alt="Company Logo">
                        <div class="company-details">
                            AL Quresh Near Railway Pahatak,  Infront of Al Quresh Housing Scheme Sher Shah Road Multan<br>Mob: 0092 305 23 94 810, 0092 305 23 94 810 UAN</div>
                    </td>
                    <td class="statement-meta">
                        <table>
                            <tr>
                                <td colspan="2" style="text-align:center; font-weight:bold; font-size: 14pt; background: #333; color: #fff;">Vendor Ledger</td>
                            </tr>
                            <tr>
                                <td>Statement Date:</td>
                                <td><?= date('d M, Y') ?></td>
                            </tr>
                            <tr>
                                <td>Period:</td>
                                <td><?= !empty($filter_start_date) || !empty($filter_end_date) ? e(date('d M Y', strtotime($filter_start_date))) . ' to ' . e(date('d M Y', strtotime($filter_end_date))) : 'All Time' ?></td>
                            </tr>
                        </table>
                    </td>
                </tr>
            </table>
        </header>

        <section class="vendor-details">
            <h3>Statement For:</h3>
            <table>
                <tr>
                    <td>Vendor:</td>
                    <td><?= e($selected_vendor_details['name']) ?></td>
                </tr>
                <?php if (!empty($selected_vendor_details['company_name'])): ?><tr>
                        <td>Company:</td>
                        <td><?= e($selected_vendor_details['company_name']) ?></td>
                    </tr><?php endif; ?>
                <?php if (!empty($selected_vendor_details['phone_number'])): ?><tr>
                        <td>Phone:</td>
                        <td><?= e($selected_vendor_details['phone_number']) ?></td>
                    </tr><?php endif; ?>
            </table>
        </section>

        <section class="summary-container">
            <div class="summary-item">
                <div class="label">Opening Balance</div>
                <div class="value balance"><?= number_format($opening_balance, 2) ?></div>
            </div>
            <div class="summary-item">
                <div class="label">Total Cost (Credit)</div>
                <div class="value credit"><?= number_format($total_credit_period, 2) ?></div>
            </div>
            <div class="summary-item">
                <div class="label">Total Paid (Debit)</div>
                <div class="value debit"><?= number_format($total_debit_period, 2) ?></div>
            </div>
            <div class="summary-item">
                <div class="label">Closing Balance</div>
                <div class="value balance"><?= number_format($closing_balance, 2) ?></div>
            </div>
        </section>

        <main>
            <table class="ledger-table">
                <thead>
                    <tr>
                        <th style="width: 10%;">Date</th>
                        <th style="width: 12%;">Type</th>
                        <th style="width: 12%;">Reference</th>
                        <th>Particulars</th>
                        <th class="number" style="width: 12%;">Paid (Debit)</th>
                        <th class="number" style="width: 12%;">Cost (Credit)</th>
                        <th class="number" style="width: 12%;">Balance</th>
                    </tr>
                </thead>
                <tbody>
                    <tr class="balance-row">
                        <td colspan="6"><strong>Opening Balance</strong></td>
                        <td class="number"><strong><?= number_format($opening_balance, 2) ?></strong></td>
                    </tr>
                    <?php if (empty($transactions)): ?>
                        <tr>
                            <td colspan="7" style="text-align: center; padding: 20px;">No transactions found in the selected period.</td>
                        </tr>
                        <?php else:
                        $running_balance = $opening_balance;
                        foreach ($transactions as $transaction):
                            $debit = (float)$transaction['debit'];
                            $credit = (float)$transaction['credit'];
                            $running_balance += $credit - $debit;
                        ?>
                            <tr>
                                <td><?= date('d M, y', strtotime(e($transaction['date']))) ?></td>
                                <td><?= e($transaction['type']) ?></td>
                                <td><?= e($transaction['number']) ?></td>
                                <td class="particulars"><?= e($transaction['particulars']) ?></td>
                                <td class="number"><?= $debit > 0 ? number_format($debit, 2) : '' ?></td>
                                <td class="number"><?= $credit > 0 ? number_format($credit, 2) : '' ?></td>
                                <td class="number"><?= number_format($running_balance, 2) ?></td>
                            </tr>
                    <?php endforeach;
                    endif; ?>
                    <tr class="totals-row">
                        <td colspan="4" style="text-align:right;"><strong>Period Totals</strong></td>
                        <td class="number"><strong><?= number_format($total_debit_period, 2) ?></strong></td>
                        <td class="number"><strong><?= number_format($total_credit_period, 2) ?></strong></td>
                        <td></td>
                    </tr>
                    <tr class="balance-row">
                        <td colspan="6" style="text-align:right;"><strong>Closing Balance</strong></td>
                        <td class="number"><strong><?= number_format($closing_balance, 2) ?></strong></td>
                    </tr>
                </tbody>
            </table>
        </main>
        <footer class="footer">This is a computer-generated statement and does not require a signature.</footer>
    </div>


    <script>
        // This script disables the right-click context menu on the entire page.
        document.addEventListener('contextmenu', function(e) {
            e.preventDefault();
        });
    </script>
</body>

</html>