<?php
session_start();
include_once '../db-config.php';

// =======================================================
// SECURITY GATEWAY
// =======================================================
if (!isset($_SESSION['user_id']) || !isset($_SESSION['user_type']) || $_SESSION['user_type'] !== 'admin') {
    header("location: ../login.php");
    exit;
}

function e($string) { return htmlspecialchars($string ?? '', ENT_QUOTES, 'UTF-8'); }

//======================================================================
//  HANDLE POST REQUESTS (STATUS UPDATE & BULK DELETE)
//======================================================================
if ($_SERVER["REQUEST_METHOD"] == "POST") {
    // --- API ENDPOINT FOR LIVE STATUS UPDATES ---
    if (isset($_SERVER['HTTP_X_REQUESTED_WITH']) && strtolower($_SERVER['HTTP_X_REQUESTED_WITH']) == 'xmlhttprequest') {
        header('Content-Type: application/json');
        try {
            $data = json_decode(file_get_contents('php://input'), true, 512, JSON_THROW_ON_ERROR);
            if (!isset($data['action']) || $data['action'] !== 'update_status' || !isset($data['id'], $data['status'])) {
                http_response_code(400);
                echo json_encode(['success' => false, 'message' => 'Invalid API request.']);
                exit();
            }
            $inquiry_id = (int)$data['id'];
            $new_status = $data['status'];
            $allowed_statuses = ['Pending', 'In Progress', 'Completed', 'Rejected'];
            if (!in_array($new_status, $allowed_statuses)) {
                http_response_code(400);
                echo json_encode(['success' => false, 'message' => 'Invalid status value provided.']);
                exit();
            }
            $stmt = $conn->prepare("UPDATE umrah_inquiries SET status = ? WHERE id = ?");
            $stmt->bind_param("si", $new_status, $inquiry_id);
            if ($stmt->execute()) {
                echo json_encode(['success' => true, 'message' => 'Status updated successfully.']);
            } else {
                throw new Exception("Database query execution failed: " . $stmt->error);
            }
            $stmt->close();
        } catch (Throwable $e) {
            http_response_code(500);
            error_log("API Error on view-inquiries.php: " . $e->getMessage());
            echo json_encode(['success' => false, 'message' => 'A critical server error occurred.']);
        }
        exit();
    }

    // --- NEW: BULK DELETE LOGIC ---
    if (isset($_POST['action']) && $_POST['action'] === 'bulk_delete') {
        if (empty($_POST['inquiry_ids']) || !is_array($_POST['inquiry_ids'])) {
            $_SESSION['error_message'] = "Please select inquiries to delete.";
        } else {
            $ids_to_delete = array_map('intval', $_POST['inquiry_ids']);
            $placeholders = implode(',', array_fill(0, count($ids_to_delete), '?'));
            $types = str_repeat('i', count($ids_to_delete));
            
            $sql = "DELETE FROM umrah_inquiries WHERE id IN ($placeholders)";
            $stmt = $conn->prepare($sql);
            $stmt->bind_param($types, ...$ids_to_delete);
            
            if ($stmt->execute()) {
                $deleted_count = $stmt->affected_rows;
                $_SESSION['success_message'] = "Successfully deleted {$deleted_count} inquiries.";
            } else {
                $_SESSION['error_message'] = "Error deleting inquiries: " . $stmt->error;
            }
            $stmt->close();
        }
        header("Location: " . $_SERVER['PHP_SELF'] . "?" . $_SERVER['QUERY_STRING']);
        exit();
    }
}


//======================================================================
//  HTML PAGE RENDERING LOGIC
//======================================================================
$allowed_filters = ['all', 'pending', 'in-progress', 'completed', 'rejected'];
$current_filter = $_GET['filter'] ?? 'pending';
if (!in_array($current_filter, $allowed_filters)) $current_filter = 'pending';
$items_per_page = 20;
$current_page = max(1, (int)($_GET['page'] ?? 1));
$offset = ($current_page - 1) * $items_per_page;

$base_sql_from = "FROM umrah_inquiries";
$where_clause = "";
$params = [];
$types = "";
if ($current_filter !== 'all') {
    $status_for_query = str_replace('-', ' ', ucwords($current_filter, '-'));
    $where_clause = " WHERE status = ?";
    $params[] = $status_for_query;
    $types = "s";
}

$count_sql = "SELECT COUNT(*) as total " . $base_sql_from . $where_clause;
$stmt_count = $conn->prepare($count_sql);
if (!empty($params)) $stmt_count->bind_param($types, ...$params);
$stmt_count->execute();
$total_items = $stmt_count->get_result()->fetch_assoc()['total'];
$stmt_count->close();
$total_pages = ($total_items > 0) ? ceil($total_items / $items_per_page) : 1;

$all_inquiries_count = $conn->query("SELECT COUNT(*) FROM umrah_inquiries")->fetch_row()[0];
$pending_count = $conn->query("SELECT COUNT(*) FROM umrah_inquiries WHERE status = 'Pending'")->fetch_row()[0];
$inprogress_count = $conn->query("SELECT COUNT(*) FROM umrah_inquiries WHERE status = 'In Progress'")->fetch_row()[0];
$completed_count = $conn->query("SELECT COUNT(*) FROM umrah_inquiries WHERE status = 'Completed'")->fetch_row()[0];
$rejected_count = $conn->query("SELECT COUNT(*) FROM umrah_inquiries WHERE status = 'Rejected'")->fetch_row()[0];

$data_sql = "SELECT * " . $base_sql_from . $where_clause . " ORDER BY created_at DESC LIMIT ?, ?";
$data_params = $params;
$data_params[] = $offset;
$data_params[] = $items_per_page;
$data_types = $types . "ii";
$stmt_data = $conn->prepare($data_sql);
$stmt_data->bind_param($data_types, ...$data_params);
$stmt_data->execute();
$inquiries_result = $stmt_data->get_result();
$stmt_data->close();
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Manage Umrah Inquiries</title>
    <link rel="icon" type="image/png" href="../images/logo-icon.png">
    <link rel="stylesheet" href="admin-style.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.1/css/all.min.css">
    <style>
        .filter-bar { display: flex; flex-wrap: wrap; gap: 8px; margin-bottom: 25px; }
        .filter-bar a { text-decoration: none; padding: 8px 15px; border-radius: 6px; color: var(--text-dark); font-weight: 600; font-size: 0.9rem; transition: all 0.2s ease; display: flex; align-items: center; gap: 8px; background-color: #e9ecef; }
        .filter-bar a:hover { background-color: #dee2e6; }
        .filter-bar a.active { background-color: var(--primary-color); color: #fff; }
        .count-badge { background-color: var(--secondary-color); color: var(--sidebar-bg); font-size: 0.8rem; padding: 2px 8px; border-radius: 10px; }
        .inquiry-list { border-radius: 8px; overflow: hidden; }
        .inquiry-item { padding: 15px 20px; border-bottom: 1px solid var(--border-color); background-color: #fff; transition: background-color 0.2s; }
        .inquiry-list .inquiry-item:last-child { border-bottom: none; }
        .inquiry-summary { display: flex; align-items: center; gap: 15px; cursor: pointer; }
        .inquiry-icon { font-size: 1.8rem; color: var(--primary-color); }
        .inquiry-primary-info { flex-grow: 1; }
        .inquiry-primary-info strong { font-size: 1.1rem; color: var(--text-dark); }
        .inquiry-primary-info span { font-size: 0.85rem; color: var(--text-muted); }
        .inquiry-meta { display: flex; align-items: center; gap: 15px; }
        .expand-arrow { font-size: 0.9rem; color: var(--text-muted); transition: transform 0.3s ease; }
        .inquiry-item.expanded .expand-arrow { transform: rotate(180deg); }
        .status-Pending { background-color: #ffc107; color: #212529;}
        .status-In-Progress { background-color: #0dcaf0; color: #212529;}
        .status-Completed { background-color: #198754; color: #fff; }
        .status-Rejected { background-color: #dc3545; color: #fff;}
        .inquiry-details { max-height: 0; overflow: hidden; transition: all 0.4s ease-in-out; }
        .inquiry-item.expanded .inquiry-details { max-height: 800px; padding-top: 20px; margin-top: 20px; border-top: 1px solid #e9ecef; }
        .details-grid { display: grid; grid-template-columns: repeat(auto-fit, minmax(300px, 1fr)); gap: 15px 30px; font-size: 0.95rem; margin-bottom: 15px; }
        .detail-item { color: #555; }
        .detail-item strong { color: var(--text-dark); display: block; margin-bottom: 3px; }
        .inquiry-actions { margin-top: 20px; border-top: 1px solid #e9ecef; padding-top: 20px; display: flex; align-items: center; gap: 10px; }
        .toolbar { margin-bottom: 1rem; display: flex; align-items: center; gap: 1rem; }
    </style>
</head>
<body>
    <div class="dashboard-wrapper">
        <?php include 'sidebar.php'; ?>
        <div class="main-content">
            <header class="main-header">
                <button class="menu-toggle" id="menu-toggle"><i class="fas fa-bars"></i></button>
                <div class="user-info"><span>Web Inquiries</span></div>
            </header>
            <main class="content-body">
                <h1 class="page-title">Umrah Inquiries</h1>
                
                <?php if(isset($_SESSION['success_message'])): ?>
                    <div class="notice success"><?= e($_SESSION['success_message']); unset($_SESSION['success_message']); ?></div>
                <?php endif; ?>
                <?php if(isset($_SESSION['error_message'])): ?>
                    <div class="notice error"><?= e($_SESSION['error_message']); unset($_SESSION['error_message']); ?></div>
                <?php endif; ?>

                <div class="filter-bar">
                    <a href="?filter=all" class="<?= $current_filter === 'all' ? 'active' : '' ?>">All (<?= $all_inquiries_count ?>)</a>
                    <a href="?filter=pending" class="<?= $current_filter === 'pending' ? 'active' : '' ?>">Pending <?php if ($pending_count > 0) echo "<span class='count-badge'>$pending_count</span>"; ?></a>
                    <a href="?filter=in-progress" class="<?= $current_filter === 'in-progress' ? 'active' : '' ?>">In Progress (<?= $inprogress_count ?>)</a>
                    <a href="?filter=completed" class="<?= $current_filter === 'completed' ? 'active' : '' ?>">Completed (<?= $completed_count ?>)</a>
                    <a href="?filter=rejected" class="<?= $current_filter === 'rejected' ? 'active' : '' ?>">Rejected (<?= $rejected_count ?>)</a>
                </div>

                <div class="content-card">
                    <form id="bulk-action-form" method="POST">
                        <div class="toolbar">
                            <button type="submit" name="action" value="bulk_delete" class="btn btn-danger"><i class="fas fa-trash"></i> Delete Selected</button>
                            <div class="form-check">
                                <input type="checkbox" class="form-check-input" id="select-all">
                                <label class="form-check-label" for="select-all">Select All</label>
                            </div>
                        </div>
                        <div class="inquiry-list" id="inquiry-list">
                            <?php if ($inquiries_result && $inquiries_result->num_rows > 0): while ($inquiry = $inquiries_result->fetch_assoc()): ?>
                                <div class="inquiry-item" data-id="<?php echo $inquiry['id']; ?>">
                                    <div class="inquiry-summary">
                                        <input type="checkbox" name="inquiry_ids[]" value="<?php echo $inquiry['id']; ?>" class="form-check-input inquiry-checkbox" style="margin-right: 15px;">
                                        <div class="inquiry-icon"><i class="fas fa-kaaba"></i></div>
                                        <div class="inquiry-primary-info">
                                            <strong><?php echo e($inquiry['customer_name']); ?></strong>
                                            <span>Inquiry ID: #<?php echo $inquiry['id']; ?> | Submitted: <?php echo date('d M Y, g:ia', strtotime($inquiry['created_at'])); ?></span>
                                        </div>
                                        <div class="inquiry-meta">
                                            <span class="status-badge status-<?php echo str_replace(' ', '-', e($inquiry['status'])); ?>"><?php echo e($inquiry['status']); ?></span>
                                            <div class="expand-arrow"><i class="fas fa-chevron-down"></i></div>
                                        </div>
                                    </div>
                                    <div class="inquiry-details">
                                        <div class="details-grid">
                                            <div class="detail-item"><strong>Email:</strong> <span><a href="mailto:<?= e($inquiry['customer_email']) ?>"><?= e($inquiry['customer_email']) ?></a></span></div>
                                            <div class="detail-item"><strong>Phone:</strong> <span><a href="tel:<?= e($inquiry['customer_phone']) ?>"><?= e($inquiry['customer_phone']) ?></a></span></div>
                                            <div class="detail-item"><strong>Package Name:</strong> <?php echo e($inquiry['package_name']); ?></div>
                                            <div class="detail-item"><strong>Package ID:</strong> <?php echo e($inquiry['package_id']); ?></div>
                                            <div class="detail-item"><strong>Room Type:</strong> <?php echo e($inquiry['room_type']); ?></div>
                                            <div class="detail-item"><strong>Pax:</strong> <?php echo e($inquiry['pax']); ?></div>
                                        </div>
                                        <div class="inquiry-actions">
                                            <label for="status-change-<?php echo $inquiry['id']; ?>">Update Status:</label>
                                            <select id="status-change-<?php echo $inquiry['id']; ?>" class="form-control status-change-select" style="width: auto;">
                                                <option value="Pending" <?php echo ($inquiry['status'] == 'Pending') ? 'selected' : ''; ?>>Pending</option>
                                                <option value="In Progress" <?php echo ($inquiry['status'] == 'In Progress') ? 'selected' : ''; ?>>In Progress</option>
                                                <option value="Completed" <?php echo ($inquiry['status'] == 'Completed') ? 'selected' : ''; ?>>Completed</option>
                                                <option value="Rejected" <?php echo ($inquiry['status'] == 'Rejected') ? 'selected' : ''; ?>>Rejected</option>
                                            </select>
                                        </div>
                                    </div>
                                </div>
                            <?php endwhile; else: ?>
                                <div class="empty-state"><i class="fas fa-folder-open"></i><p>No inquiries found for the "<?php echo e($current_filter); ?>" filter.</p></div>
                            <?php endif; ?>
                        </div>
                    </form>

                    <?php if ($total_pages > 1): ?>
                        <div class="pagination-controls" style="margin-top: 20px; text-align: right;">
                            <?php if ($current_page > 1): ?><a href="?filter=<?= e($current_filter) ?>&page=<?= $current_page - 1 ?>" class="btn btn-secondary btn-sm">« Previous</a><?php endif; ?>
                            <span style="padding: 0 10px;">Page <?php echo $current_page; ?> of <?php echo $total_pages; ?></span>
                            <?php if ($current_page < $total_pages): ?><a href="?filter=<?= e($current_filter) ?>&page=<?= $current_page + 1 ?>" class="btn btn-secondary btn-sm">Next »</a><?php endif; ?>
                        </div>
                    <?php endif; ?>
                </div>
            </main>
        </div>
    </div>

    <script>
        document.addEventListener('DOMContentLoaded', function() {
            const inquiryList = document.getElementById('inquiry-list');
            const bulkActionForm = document.getElementById('bulk-action-form');
            const selectAllCheckbox = document.getElementById('select-all');
            const inquiryCheckboxes = document.querySelectorAll('.inquiry-checkbox');

            if (inquiryList) {
                inquiryList.addEventListener('click', function(event) {
                    const inquiryItem = event.target.closest('.inquiry-item');
                    if (!inquiryItem) return;
                    if (!event.target.closest('select, a, input[type=checkbox]')) { // Prevent toggle on checkbox click
                        inquiryItem.classList.toggle('expanded');
                    }
                });
                inquiryList.addEventListener('change', function(event) {
                    if (!event.target.classList.contains('status-change-select')) return;
                    const selectElement = event.target;
                    const inquiryItem = selectElement.closest('.inquiry-item');
                    const inquiryId = inquiryItem.dataset.id;
                    const newStatus = selectElement.value;
                    fetch(window.location.pathname, {
                        method: 'POST',
                        headers: { 'Content-Type': 'application/json', 'X-Requested-With': 'XMLHttpRequest' },
                        body: JSON.stringify({ action: 'update_status', id: inquiryId, status: newStatus })
                    })
                    .then(response => response.json())
                    .then(data => {
                        if (data.success) {
                            const statusBadge = inquiryItem.querySelector('.status-badge');
                            statusBadge.className = 'status-badge status-' + newStatus.replace(' ', '-');
                            statusBadge.textContent = newStatus;
                        } else { throw new Error(data.message || 'The server reported a failure.'); }
                    })
                    .catch(error => {
                        console.error('Fetch Error:', error);
                        alert('Could not update status. ' + error.message);
                    });
                });
            }

            // --- Bulk Delete and Select All Logic ---
            if (selectAllCheckbox) {
                selectAllCheckbox.addEventListener('change', function() {
                    inquiryCheckboxes.forEach(checkbox => {
                        checkbox.checked = this.checked;
                    });
                });
            }

            if (bulkActionForm) {
                bulkActionForm.addEventListener('submit', function(e) {
                    const actionButton = document.activeElement;
                    if (!actionButton || actionButton.value !== 'bulk_delete') return;

                    const anyChecked = [...inquiryCheckboxes].some(c => c.checked);
                    if (!anyChecked) {
                        alert('Please select at least one inquiry to delete.');
                        e.preventDefault();
                        return;
                    }
                    
                    if (!confirm('Are you sure you want to permanently delete the selected inquiries?')) {
                        e.preventDefault();
                    }
                });
            }
        });
    </script>
</body>
</html>