<?php

/**
 * view-receipt.php (v1.1 - DYNAMIC & PRINT-OPTIMIZED)
 * - FIXED: A fatal error when viewing receipts for ticket invoices. The SQL query
 *   is now dynamic and works for both package and ticket invoices.
 * - NEW: The "Go Back" button now correctly links to the appropriate edit page
 *   (edit-invoice.php or ticket-invoice-edit.php).
 * - NEW: Added a golden border visible only during printing for a premium look.
 * - All previous professional print styles remain intact.
 */
session_start();
include_once '../db-config.php';

$payment_id = (int)($_GET['id'] ?? 0);
if ($payment_id <= 0) {
    die("Invalid Payment ID.");
}

// =================================================================
// FIX: The SQL now uses LEFT JOIN on both invoice tables and COALESCE
// to fetch the guest_name from whichever table is relevant.
// This works for both 'package' and 'ticket' invoice types.
// =================================================================
$sql = "SELECT 
            p.*, 
            COALESCE(i.guest_name, ti.guest_name) as guest_name,
            COALESCE(i.invoice_number, ti.invoice_number) as invoice_number
        FROM payments p
        LEFT JOIN invoices i ON p.invoice_id = i.id AND p.invoice_type = 'package'
        LEFT JOIN ticket_invoices ti ON p.invoice_id = ti.id AND p.invoice_type = 'ticket'
        WHERE p.id = ?";
$stmt = $conn->prepare($sql);

if ($stmt === false) {
    die("Failed to prepare the SQL statement: " . $conn->error);
}

$stmt->bind_param("i", $payment_id);
$stmt->execute();
$result = $stmt->get_result();
$receipt = $result->fetch_assoc();

if (!$receipt) {
    die("Payment receipt not found.");
}

function e($string)
{
    return htmlspecialchars($string ?? '', ENT_QUOTES, 'UTF-8');
}

// --- DYNAMIC "Go Back" Link ---
$back_link = '#';
if ($receipt['invoice_id']) {
    $back_link = ($receipt['invoice_type'] === 'ticket') ? 'ticket-invoice-edit.php?id=' . e($receipt['invoice_id']) : 'edit-invoice.php?id=' . e($receipt['invoice_id']);
}

?>
<!DOCTYPE html>
<html lang="en">

<head>
    <meta charset="UTF-8">
    <title>Payment Receipt #<?= e($receipt['id']) ?></title>
    <link rel="icon" type="image/png" href="../images/logo-icon.png">

    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.1/css/all.min.css">
    <style>
        /* --- GENERAL STYLES FOR SCREEN --- */
        body {
            background-color: #f4f7f6;
            font-family: -apple-system, BlinkMacSystemFont, "Segoe UI", Roboto, "Helvetica Neue", Arial, sans-serif;
            color: #333;
            margin: 0;
            padding: 20px;
        }

        .receipt-wrapper {
            max-width: 800px;
            margin: 20px auto;
            background: #fff;
            box-shadow: 0 4px 15px rgba(0, 0, 0, 0.1);
            border-radius: 8px;
            overflow: hidden;

        }

        .receipt-actions {
            padding: 15px 20px;
            background-color: #e9ecef;
            text-align: right;
            border-bottom: 1px solid #dee2e6;
        }

        .btn {
            padding: 10px 15px;
            border: none;
            border-radius: 5px;
            color: #fff;
            text-decoration: none;
            display: inline-flex;
            align-items: center;
            gap: 8px;
            cursor: pointer;
            font-weight: 500;
            transition: background-color 0.2s;
        }

        .btn-secondary {
            background-color: #6c757d;
        }

        .btn-secondary:hover {
            background-color: #5a6268;
        }

        .btn-primary {
            background-color: #007bff;
        }

        .btn-primary:hover {
            background-color: #0069d9;
        }

        .receipt-box {
            padding: 20px;
            border: 5px solid #D4AF37;
            /* A classic gold color */
        }

        .receipt-header {
            display: flex;
            justify-content: space-between;
            align-items: flex-start;
            padding: 20px;
            border-bottom: 2px solid #000;
        }

        .receipt-title {
            text-align: right;
        }

        .receipt-title h1 {
            margin: 0;
            font-size: 24px;
            text-transform: uppercase;
            letter-spacing: 1px;
        }

        .receipt-title p {
            margin: 5px 0 0;
            color: #555;
        }

        .receipt-details {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
            gap: 20px;
            padding: 20px;
            border-bottom: 1px solid #eee;
        }

        .detail-item span {
            display: block;
            font-size: 12px;
            color: #888;
            text-transform: uppercase;
            margin-bottom: 4px;
        }

        .detail-item strong {
            font-size: 16px;
        }

        .receipt-body {
            padding: 30px 20px;
        }

        .amount-paid {
            text-align: center;
            margin: 20px 0;
        }

        .amount-paid small {
            display: block;
            font-size: 14px;
            color: #888;
            text-transform: uppercase;
        }

        .amount-paid span {
            display: block;
            font-size: 42px;
            font-weight: bold;
            color: #000;
        }

        .payment-for {
            text-align: center;
            font-size: 16px;
            margin-top: 20px;
            color: #444;
        }

        .receipt-notes {
            margin-top: 30px;
            padding: 20px;
            border-top: 1px solid #eee;
            background-color: #fafafa;
            border-radius: 4px;
        }

        .receipt-notes h4 {
            margin-top: 0;
        }

        .receipt-footer {
            text-align: center;
            padding: 20px;
            border-top: 1px solid #eee;
            margin-top: 30px;
        }

        .receipt-footer p {
            margin: 5px 0;
        }

        .contact-info {
            font-size: 12px;
            color: #666;
        }

        /* --- PRINT STYLES --- */
        @media print {
            .no-print {
                display: none !important;
            }

            @page {
                size: A4;
                margin: 0;
                /* Remove default browser margins */
            }

            body {
                background-color: #fff !important;
                margin: 0;
                padding: 0;
                font-size: 12pt;
            }

            * {
                color: #000 !important;
                -webkit-print-color-adjust: exact;
                print-color-adjust: exact;
            }

            /* NEW: Golden border for the entire printed page */
            .print-container {
                border: 3px solid #D4AF37;
                /* A classic gold color */
                height: 100vh;
                box-sizing: border-box;
                padding: 20mm;
                /* Re-apply margin as padding inside the border */
            }

            .receipt-wrapper {
                max-width: 100%;
                width: 100%;
                margin: 0;
                padding: 0;
                box-shadow: none !important;
                border-radius: 0;
                border: none;
            }

            .receipt-box {
                padding: 0;
                border: none;
            }
        }
    </style>
</head>

<body>

    <!-- This new container is specifically for the print border -->
    <div class="print-container">
        <div class="receipt-wrapper">
            <div class="receipt-actions no-print">
                <a href="<?= e($back_link) ?>" class="btn btn-secondary"><i class="fas fa-arrow-left"></i> Go Back</a>
                <button onclick="window.print()" class="btn btn-primary"><i class="fas fa-print"></i> Print Receipt</button>
            </div>
            <div class="receipt-box">
                <header class="receipt-header">
                    <div class="company-logo">
                        <img src="../images/lofo.png" alt="Company Logo" style="max-height: 500px;">
                    </div>
                    <div class="receipt-title">
                        <h1>PAYMENT RECEIPT</h1>
                        <p><strong>Receipt #:</strong> <?= e($receipt['id']) ?></p>
                    </div>
                </header>

                <section class="receipt-details">
                    <div class="detail-item">
                        <span>Payment Date</span>
                        <strong><?= date('F j, Y', strtotime($receipt['payment_date'])) ?></strong>
                    </div>
                    <div class="detail-item">
                        <span>Payment Method</span>
                        <strong><?= e($receipt['payment_method']) ?></strong>
                    </div>
                    <div class="detail-item">
                        <span>Received From</span>
                        <strong><?= e($receipt['guest_name'] ?? 'N/A') ?></strong>
                    </div>
                </section>

                <section class="receipt-body">
                    <div class="amount-paid">
                        <small>AMOUNT PAID</small>
                        <span>PKR <?= number_format($receipt['payment_amount'], 2) ?></span>
                    </div>
                    <p class="payment-for">
                        <?php if ($receipt['invoice_id']): ?>
                            This payment is for <strong>Invoice #<?= e($receipt['invoice_number']) ?></strong>.
                        <?php else: ?>
                            This is a general payment not linked to a specific invoice.
                        <?php endif; ?>
                    </p>
                    <?php if (!empty($receipt['notes'])): ?>
                        <div class="receipt-notes">
                            <h4>Notes:</h4>
                            <p><?= nl2br(e($receipt['notes'])) ?></p>
                        </div>
                    <?php endif; ?>
                </section>

                <footer class="receipt-footer">
                    <p>Thank you for your business!</p>
                    <p class="contact-info">Questions? Contact us at <strong>+92 300 1119170</strong> or <strong>
                            travelfirst786@gmail.com</strong></p>
                </footer>
            </div>
        </div>
    </div>



    <script>
        // This script disables the right-click context menu on the entire page.
        document.addEventListener('contextmenu', function(e) {
            e.preventDefault();
        });
    </script>
</body>

</html>