<?php
session_start();
include_once 'db-config.php';

// --- SECURITY CHECK 1: Is user logged in? ---
if (!isset($_SESSION['user_id'])) {
    header("location: login.php");
    exit;
}

// --- SECURITY CHECK 2: Is the logged-in user an AGENT? ---
if (!isset($_SESSION['user_type']) || $_SESSION['user_type'] !== 'agent') {
    header("location: my-account.php"); // Redirect non-agents
    exit;
}

// A helper function for safely echoing output
function e($string)
{
    return htmlspecialchars($string ?? '', ENT_QUOTES, 'UTF-8');
}

// Get the logged-in agent's ID and name from the session
$logged_in_agent_id = (int)$_SESSION['user_id'];
$user_name = $_SESSION['user_name'];

// Initialize variables
$start_date = '';
$end_date = '';
$incentive_data = [];
$summary = ['total_bookings' => 0, 'total_incentive' => 0];
$show_results = false; // Flag to show table only after filtering

// Handle form submission for filtering
if ($_SERVER["REQUEST_METHOD"] == "GET" && isset($_GET['filter'])) {
    $start_date = !empty($_GET['start_date']) ? $_GET['start_date'] : null;
    $end_date = !empty($_GET['end_date']) ? $_GET['end_date'] : null;
    $show_results = true;

    if ($start_date && $end_date) {
        $sql = "SELECT
                    ui.id as inquiry_id,
                    ui.customer_name,
                    ui.package_name,
                    ui.created_at as inquiry_date,
                    up.agent_incentive
                FROM
                    umrah_inquiries ui
                JOIN
                    umrah_packages up ON ui.package_id = up.package_id
                WHERE
                    ui.referred_by_agent_id = ?
                    AND DATE(ui.created_at) BETWEEN ? AND ?
                    AND up.agent_incentive > 0
                ORDER BY
                    ui.created_at DESC";
        
        $stmt = $conn->prepare($sql);
        if ($stmt) {
            $stmt->bind_param("iss", $logged_in_agent_id, $start_date, $end_date);
            $stmt->execute();
            $result = $stmt->get_result();
            while ($row = $result->fetch_assoc()) {
                $incentive_data[] = $row;
                $summary['total_bookings']++;
                $summary['total_incentive'] += (float)$row['agent_incentive'];
            }
            $stmt->close();
        } else {
            $error_message = "Database query failed to prepare.";
        }
    } else {
        $error_message = "Please select both a start and end date to filter.";
    }
}
?>
<!DOCTYPE html>
<html>
<head>
    <title>My Incentives</title>
    <link rel="icon" type="image/png" href="images/logo-icon.png">
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <link rel="stylesheet" href="css/style.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.2.0/css/all.min.css">
    <link rel="stylesheet" href="css/account-style.css">
    <style>
        .filter-form-container {
            margin-bottom: 2rem;
            padding: 1.5rem;
            background-color: #f8f9fa;
            border: 1px solid #dee2e6;
            border-radius: 8px;
        }
        .filter-form {
            display: grid;
            grid-template-columns: 1fr 1fr auto;
            gap: 1rem;
            align-items: end;
        }
        .filter-form label {
            font-weight: 500;
            margin-bottom: 0.5rem;
            display: block;
        }
        .filter-form input {
            width: 100%;
            padding: 0.5rem;
            border: 1px solid #ced4da;
            border-radius: 4px;
        }
        .filter-form button {
            padding: 0.5rem 1.5rem;
            background-color: #007bff;
            color: white;
            border: none;
            border-radius: 4px;
            cursor: pointer;
            font-weight: 500;
        }
        .report-summary {
            margin-top: 2rem;
            background-color: #e9f5ff;
            border: 1px solid #bde0fe;
            border-radius: 8px;
            padding: 1.5rem;
            display: flex;
            justify-content: space-around;
            text-align: center;
        }
        .summary-item .label {
            font-size: 0.9em;
            color: #495057;
            margin-bottom: 5px;
            text-transform: uppercase;
        }
        .summary-item .value {
            font-size: 1.8em;
            font-weight: 600;
            color: #0d2d4c;
        }
        @media (max-width: 768px) {
            .filter-form {
                grid-template-columns: 1fr;
            }
            .report-summary {
                flex-direction: column;
                gap: 1rem;
            }
        }
    </style>
</head>
<body>
    <?php include 'header.php'; ?>

    <main class="account-page-wrapper">
        <div class="account-container">
            <?php include 'agent-sidebar.php'; ?>
            <div class="account-content">
                <div class="content-card">
                    <h2><i class="fa-solid fa-hand-holding-dollar"></i> My Incentives Report</h2>
                    <p class="content-description">View the incentives you've earned from referred Umrah package bookings. Select a date range to see your performance.</p>

                    <div class="filter-form-container">
                        <form action="agent-my-incentives.php" method="GET" class="filter-form">
                            <div class="form-group">
                                <label for="start_date">Start Date</label>
                                <input type="date" name="start_date" id="start_date" value="<?= e($start_date) ?>" required>
                            </div>
                            <div class="form-group">
                                <label for="end_date">End Date</label>
                                <input type="date" name="end_date" id="end_date" value="<?= e($end_date) ?>" required>
                            </div>
                            <button type="submit" name="filter" value="1"><i class="fa-solid fa-filter"></i> Filter Report</button>
                        </form>
                    </div>

                    <?php if (isset($error_message)): ?>
                        <div class="notice error"><?= e($error_message) ?></div>
                    <?php endif; ?>

                    <?php if ($show_results): ?>
                        <?php if (!empty($incentive_data)): ?>
                            <div class="table-responsive">
                                <table class="bookings-table">
                                    <thead>
                                        <tr>
                                            <th>Booking Date</th>
                                            <th>Customer Name</th>
                                            <th>Package Name</th>
                                            <th>Incentive (SAR)</th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        <?php foreach ($incentive_data as $row): ?>
                                        <tr>
                                            <td><?= e(date('d M, Y H:i', strtotime($row['inquiry_date']))) ?></td>
                                            <td><?= e($row['customer_name']) ?></td>
                                            <td><?= e($row['package_name']) ?></td>
                                            <td><strong><?= e(number_format($row['agent_incentive'], 2)) ?></strong></td>
                                        </tr>
                                        <?php endforeach; ?>
                                    </tbody>
                                </table>
                            </div>
                            
                            <div class="report-summary">
                                <div class="summary-item">
                                    <div class="label">Total Referrals</div>
                                    <div class="value"><?= e($summary['total_bookings']) ?></div>
                                </div>
                                <div class="summary-item">
                                    <div class="label">Total Incentive Earned</div>
                                    <div class="value"><?= e(number_format($summary['total_incentive'], 2)) ?> SAR</div>
                                </div>
                            </div>
                        <?php else: ?>
                            <p class="no-bookings-found">No incentive records found for the selected date range.</p>
                        <?php endif; ?>
                    <?php endif; ?>
                </div>
            </div>
        </div>
    </main>

    <?php include 'footer.php'; ?>
</body>
</html>