<?php
// Ensure session is started
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

include 'db-config.php';

// --- SECURITY CHECK 1: Is user logged in? ---
if (!isset($_SESSION['user_id'])) {
    header("location: login.php");
    exit;
}

// --- SECURITY CHECK 2: Is the logged-in user an AGENT? ---
if (!isset($_SESSION['user_type']) || $_SESSION['user_type'] !== 'agent') {
    header("location: my-account.php"); // Redirect non-agents away
    exit;
}

// Get the logged-in agent's ID and name
$user_id = $_SESSION['user_id'];
$user_name = $_SESSION['user_name'];

// Initialize variables
$company_name = '';
$company_address = '';
$company_phone_1 = ''; // NEW
$company_phone_2 = ''; // NEW
$city = '';
$branch = '';
$logo_path = '';
$success_message = '';
$error_message = '';

// --- FETCH current agent data to pre-fill the form ---
$stmt = $conn->prepare("SELECT company_name, company_address, company_phone_1, company_phone_2, city, branch, logo_path FROM users WHERE id = ?");
$stmt->bind_param("i", $user_id);
$stmt->execute();
$result = $stmt->get_result();
if ($user = $result->fetch_assoc()) {
    $company_name = $user['company_name'];
    $company_address = $user['company_address'];
    $company_phone_1 = $user['company_phone_1']; // NEW
    $company_phone_2 = $user['company_phone_2']; // NEW
    $city = $user['city'];
    $branch = $user['branch'];
    $logo_path = $user['logo_path'];
}
$stmt->close();

// --- PROCESS form submission ---
if ($_SERVER["REQUEST_METHOD"] == "POST") {
    if (isset($_POST['update_company_profile'])) {
        $new_company_name = trim($_POST['company_name']);
        $new_company_address = trim($_POST['company_address']);
        $new_company_phone_1 = trim($_POST['company_phone_1']); // NEW
        $new_company_phone_2 = trim($_POST['company_phone_2']); // NEW
        $new_city = trim($_POST['city']);
        $new_branch = trim($_POST['branch']);
        $new_logo_path = $logo_path;

        if (empty($new_company_name)) {
            $error_message = "Company Name field cannot be empty.";
        } else {
            try {
                if (isset($_FILES['company_logo']) && $_FILES['company_logo']['error'] == UPLOAD_ERR_OK) {
                    $upload_dir = __DIR__ . '/uploads/logos/';
                    if (!is_dir($upload_dir)) mkdir($upload_dir, 0755, true);

                    $file_tmp = $_FILES['company_logo']['tmp_name'];
                    $file_name = basename($_FILES['company_logo']['name']);
                    $file_size = $_FILES['company_logo']['size'];
                    $file_ext = strtolower(pathinfo($file_name, PATHINFO_EXTENSION));
                    $allowed_exts = ['jpg', 'jpeg', 'png', 'gif'];

                    if (!in_array($file_ext, $allowed_exts)) throw new Exception("Invalid file type.");
                    if ($file_size > 2 * 1024 * 1024) throw new Exception("File is too large (max 2MB).");

                    $unique_filename = 'logo-' . $user_id . '-' . uniqid() . '.' . $file_ext;
                    $destination = $upload_dir . $unique_filename;

                    if (move_uploaded_file($file_tmp, $destination)) {
                        if (!empty($logo_path) && file_exists(__DIR__ . '/' . $logo_path)) {
                           @unlink(__DIR__ . '/' . $logo_path);
                        }
                        $new_logo_path = 'uploads/logos/' . $unique_filename;
                    } else {
                        throw new Exception("Failed to save uploaded file.");
                    }
                }

                // --- UPDATE DATABASE (with new phone fields) ---
                $stmt = $conn->prepare("UPDATE users SET company_name = ?, company_address = ?, company_phone_1 = ?, company_phone_2 = ?, city = ?, branch = ?, logo_path = ? WHERE id = ?");
                $stmt->bind_param("sssssssi", $new_company_name, $new_company_address, $new_company_phone_1, $new_company_phone_2, $new_city, $new_branch, $new_logo_path, $user_id);

                if ($stmt->execute()) {
                    $company_name = $new_company_name;
                    $company_address = $new_company_address;
                    $company_phone_1 = $new_company_phone_1; // NEW
                    $company_phone_2 = $new_company_phone_2; // NEW
                    $city = $new_city;
                    $branch = $new_branch;
                    $logo_path = $new_logo_path; 
                    $_SESSION['logo_path'] = $new_logo_path;
                    $success_message = "Company profile updated successfully!";
                } else {
                    $error_message = "Error updating profile. Please try again.";
                }
                $stmt->close();
            } catch (Exception $e) {
                $error_message = $e->getMessage();
            }
        }
    }
}
?>
<!DOCTYPE html>
<html>
<head>
    <title>Edit Company Profile</title>
    <link rel="icon" type="image/png" href="images/logo-icon.png">
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <link rel="stylesheet" href="css/style.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.2.0/css/all.min.css">
    <link rel="stylesheet" href="css/account-style.css">
    <style>
        .current-logo-preview { margin-bottom: 15px; }
        .current-logo-preview label { display: block; margin-bottom: 5px; font-weight: bold; color: #555; }
        .current-logo-preview img { max-width: 150px; max-height: 100px; border: 1px solid #ddd; border-radius: 4px; padding: 5px; background: #f9f9f9; }
    </style>
</head>
<body>

    <?php include 'header.php'; ?>

    <main class="account-page-wrapper">
        <div class="account-container">
            <?php include 'agent-sidebar.php'; ?>
            <div class="account-content">
                <div class="content-card">
                    <h2><i class="fa-solid fa-building"></i> Edit Company Profile</h2>
                    <p class="content-description">Update your company's information and logo below.</p>

                    <?php if ($success_message): ?><div class="form-message success"><?php echo $success_message; ?></div><?php endif; ?>
                    <?php if ($error_message): ?><div class="form-message error"><?php echo $error_message; ?></div><?php endif; ?>

                    <form class="account-form" method="POST" action="" enctype="multipart/form-data">
                        <div class="form-group">
                            <label for="company_name">Company Name*</label>
                            <input type="text" id="company_name" name="company_name" value="<?php echo htmlspecialchars($company_name ?? ''); ?>" required>
                        </div>
                        <div class="form-group">
                            <label for="company_address">Company Address</label>
                            <textarea id="company_address" name="company_address" rows="3"><?php echo htmlspecialchars($company_address ?? ''); ?></textarea>
                        </div>
                        <!-- NEW PHONE NUMBER FIELDS -->
                        <div class="form-row">
                            <div class="form-group">
                                <label for="company_phone_1">Company Phone 1</label>
                                <input type="tel" id="company_phone_1" name="company_phone_1" value="<?php echo htmlspecialchars($company_phone_1 ?? ''); ?>">
                            </div>
                            <div class="form-group">
                                <label for="company_phone_2">Company Phone 2</label>
                                <input type="tel" id="company_phone_2" name="company_phone_2" value="<?php echo htmlspecialchars($company_phone_2 ?? ''); ?>">
                            </div>
                        </div>
                        <div class="form-row">
                            <div class="form-group">
                                <label for="city">City</label>
                                <input type="text" id="city" name="city" value="<?php echo htmlspecialchars($city ?? ''); ?>">
                            </div>
                            <div class="form-group">
                                <label for="branch">Branch</label>
                                <input type="text" id="branch" name="branch" value="<?php echo htmlspecialchars($branch ?? ''); ?>">
                            </div>
                        </div>

                        <?php if (!empty($logo_path)): ?>
                            <div class="current-logo-preview">
                                <label>Current Logo</label>
                                <img src="<?php echo htmlspecialchars($logo_path); ?>" alt="Current Company Logo">
                            </div>
                        <?php endif; ?>

                        <div class="form-group">
                            <label for="company_logo">Upload New Logo (Optional)</label>
                            <input type="file" id="company_logo" name="company_logo" accept="image/png, image/jpeg, image/gif">
                            <small class="form-text">Max file size 2MB. Leave blank to keep the current logo.</small>
                        </div>

                        <div class="form-footer">
                            <button type="submit" name="update_company_profile" class="btn-primary">Save Company Info</button>
                        </div>
                    </form>
                </div>
            </div>
        </div>
    </main>

    <?php include 'footer.php'; ?>
</body>
</html>