<?php
// session_start() is in db-config.php, so it should be included first.
include 'db-config.php';

// Security: Redirect user to login page if they are not logged in
if (!isset($_SESSION['user_id'])) {
    header("location: login.php");
    exit;
}

// Get the logged-in user's ID
$user_id = $_SESSION['user_id'];

// Initialize variables
$user_name = '';
$user_email = '';
$mobile_number = '';
$user_type = 'customer';
$success_message = '';
$error_message = '';

// --- FETCH current user data to pre-fill the form ---
$stmt = $conn->prepare("SELECT name, email, mobile_number, user_type FROM users WHERE id = ?");
$stmt->bind_param("i", $user_id);
$stmt->execute();
$result = $stmt->get_result();
if ($user = $result->fetch_assoc()) {
    $user_name = $user['name'];
    $user_email = $user['email'];
    $mobile_number = $user['mobile_number'];
    $user_type = $user['user_type'];
}
$stmt->close();

// Security: If the logged-in user is an agent, redirect them away to their proper dashboard
if ($user_type === 'agent') {
    header("location: my-account.php");
    exit;
}

// --- PROCESS form submissions ---
if ($_SERVER["REQUEST_METHOD"] == "POST") {

    // --- Handle Profile Info Update ---
    if (isset($_POST['update_profile'])) {
        $new_mobile = trim($_POST['mobile_number']);

        $stmt = $conn->prepare("UPDATE users SET mobile_number = ? WHERE id = ?");
        $stmt->bind_param("si", $new_mobile, $user_id);

        if ($stmt->execute()) {
            $mobile_number = $new_mobile; // Update variable for the current page
            $success_message = "Profile updated successfully!";
        } else {
            $error_message = "Error updating profile. Please try again.";
        }
        $stmt->close();
    }

    // --- Handle Password Update ---
    if (isset($_POST['update_password'])) {
        $current_password = $_POST['current_password'];
        $new_password = $_POST['new_password'];
        $confirm_password = $_POST['confirm_password'];

        if (empty($current_password) || empty($new_password) || empty($confirm_password)) {
            $error_message = "All password fields are required.";
        } elseif ($new_password !== $confirm_password) {
            $error_message = "New password and confirm password do not match.";
        } else {
            $stmt = $conn->prepare("SELECT password FROM users WHERE id = ?");
            $stmt->bind_param("i", $user_id);
            $stmt->execute();
            $result = $stmt->get_result();
            $user_data = $result->fetch_assoc();

            if ($user_data && password_verify($current_password, $user_data['password'])) {
                $new_hashed_password = password_hash($new_password, PASSWORD_DEFAULT);
                $update_stmt = $conn->prepare("UPDATE users SET password = ? WHERE id = ?");
                $update_stmt->bind_param("si", $new_hashed_password, $user_id);
                if ($update_stmt->execute()) {
                    $success_message = "Password updated successfully!";
                } else {
                    $error_message = "Error updating password. Please try again.";
                }
                $update_stmt->close();
            } else {
                $error_message = "Incorrect current password.";
            }
            $stmt->close();
        }
    }
}

?>
<!DOCTYPE html>
<html>

<head>
    <title>Edit Profile</title>
    <link rel="icon" type="image/png" href="images/logo-icon.png">

    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <link rel="stylesheet" href="css/style.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.2.0/css/all.min.css">
    <link rel="stylesheet" href="css/account-style.css">

    <!-- === NEW DEDICATED CSS FOR THIS PAGE === -->
    <style>
        .account-form {
            margin-top: 1.5rem;
        }

        .form-row {
            display: grid;
            grid-template-columns: 1fr 1fr;
            gap: 1.5rem;
            margin-bottom: 1.5rem;
        }

        .form-group {
            display: flex;
            flex-direction: column;
            margin-bottom: 1.5rem;
        }

        .form-group:last-child {
            margin-bottom: 0;
        }

        .form-group label {
            font-size: 0.9rem;
            font-weight: 500;
            color: #495057;
            margin-bottom: 0.5rem;
        }

        .form-group input[type="text"],
        .form-group input[type="tel"],
        .form-group input[type="email"],
        .form-group input[type="password"] {
            width: 100%;
            padding: 0.8rem 1rem;
            border: 1px solid #ced4da;
            border-radius: 6px;
            font-size: 1rem;
            font-family: inherit;
            box-sizing: border-box;
            /* Important for consistent sizing */
            transition: border-color 0.2s, box-shadow 0.2s;
        }

        .form-group input:focus {
            outline: none;
            border-color: #31a7e2;
            box-shadow: 0 0 0 3px rgba(49, 167, 226, 0.2);
        }

        .form-group input[disabled] {
            background-color: #e9ecef;
            cursor: not-allowed;
            color: #6c757d;
        }

        .form-footer {
            display: flex;
            justify-content: flex-end;
            margin-top: 1.5rem;
            padding-top: 1.5rem;
            border-top: 1px solid #f0f0f0;
        }

        .btn-primary {
            /* Assuming this class exists, otherwise adjust as needed */
            background-color: #31a7e2;
            color: #fff;
            padding: 0.8rem 2rem;
            border: none;
            border-radius: 6px;
            font-size: 1rem;
            font-weight: 500;
            cursor: pointer;
            transition: background-color 0.3s;
        }

        .btn-primary:hover {
            background-color: #238ab8;
        }

        /* Message styling */
        .form-message {
            padding: 1rem;
            margin-bottom: 1.5rem;
            border-radius: 6px;
            font-weight: 500;
        }

        .form-message.success {
            background-color: #d4edda;
            color: #155724;
            border: 1px solid #c3e6cb;
        }

        .form-message.error {
            background-color: #f8d7da;
            color: #721c24;
            border: 1px solid #f5c6cb;
        }

        /* Responsive adjustment */
        @media (max-width: 768px) {
            .form-row {
                grid-template-columns: 1fr;
                gap: 0;
            }
        }
    </style>
</head>

<body>

    <?php include 'header.php'; ?>

    <main class="account-page-wrapper">
        <div class="account-container">

            <?php include 'customer-sidebar.php'; ?>

            <div class="account-content">
                <!-- Edit Profile Card -->
                <div class="content-card">
                    <h2><i class="fa-solid fa-user-pen"></i> Edit Profile</h2>
                    <p class="content-description">Update your contact information below.</p>

                    <?php if ($success_message && isset($_POST['update_profile'])): ?><div class="form-message success"><?php echo $success_message; ?></div><?php endif; ?>
                    <?php if ($error_message && isset($_POST['update_profile'])): ?><div class="form-message error"><?php echo $error_message; ?></div><?php endif; ?>

                    <form class="account-form" method="POST" action="edit-profile.php">
                        <div class="form-row">
                            <div class="form-group">
                                <label for="name">Full Name (cannot be changed)</label>
                                <input type="text" id="name" name="name" value="<?php echo htmlspecialchars($user_name); ?>" disabled>
                            </div>
                            <div class="form-group">
                                <label for="mobile_number">Mobile Number</label>
                                <input type="tel" id="mobile_number" name="mobile_number" value="<?php echo htmlspecialchars($mobile_number ?? ''); ?>">
                            </div>
                        </div>
                        <div class="form-group">
                            <label for="email">Email Address (cannot be changed)</label>
                            <input type="email" id="email" name="email" value="<?php echo htmlspecialchars($user_email); ?>" disabled>
                        </div>
                        <div class="form-footer">
                            <button type="submit" name="update_profile" class="btn-primary">Save Changes</button>
                        </div>
                    </form>
                </div>

                <!-- Change Password Card -->
                <div class="content-card">
                    <h2><i class="fa-solid fa-lock"></i> Change Password</h2>
                    <p class="content-description">For your security, please enter your current password to make changes.</p>

                    <?php if ($success_message && isset($_POST['update_password'])): ?><div class="form-message success"><?php echo $success_message; ?></div><?php endif; ?>
                    <?php if ($error_message && isset($_POST['update_password'])): ?><div class="form-message error"><?php echo $error_message; ?></div><?php endif; ?>

                    <form class="account-form" method="POST" action="edit-profile.php">
                        <div class="form-group">
                            <label for="current_password">Current Password</label>
                            <input type="password" id="current_password" name="current_password" required>
                        </div>
                        <div class="form-row">
                            <div class="form-group">
                                <label for="new_password">New Password</label>
                                <input type="password" id="new_password" name="new_password" required>
                            </div>
                            <div class="form-group">
                                <label for="confirm_password">Confirm New Password</label>
                                <input type="password" id="confirm_password" name="confirm_password" required>
                            </div>
                        </div>
                        <div class="form-footer">
                            <button type="submit" name="update_password" class="btn-primary">Update Password</button>
                        </div>
                    </form>
                </div>
            </div>
        </div>
    </main>

    <?php include 'floating-icon.php'; ?>
    <?php include 'footer.php'; ?>

</body>

</html>