    <?php
    // Make sure your database connection and session are started.
    require_once 'db-config.php';
    if (session_status() === PHP_SESSION_NONE) {
        session_start();
    }

    // --- HELPER FUNCTIONS ---
    function format_flight_date($date_string) {
        if (empty($date_string)) return 'N/A';
        return date("D, d M, Y", strtotime($date_string));
    }

    function format_flight_time($time_string) {
        if (empty($time_string)) return 'N/A';
        return date("h:i A", strtotime($time_string));
    }

    /**
     * NEW: Parses a flight detail string into a structured array.
     * Example input: "SV727 31 Jul ISB-JED 0300 0610"
     * Example output: ['flight_no' => 'SV727', 'date' => '31 Jul', 'departure_airport' => 'ISB', ...]
     */
    function parse_flight_string($flight_string) {
        if (empty($flight_string)) {
            return []; // Return empty array if input is empty
        }
        // Regex to capture: (FlightNo) (Date) (DepAirport)-(ArrAirport) (DepTime) (ArrTime)
        $pattern = '/(\w{2}\d{3,4})\s+(.*?)\s+([A-Z]{3})-([A-Z]{3})\s+(\d{4})\s+(\d{4})/';
        
        if (preg_match($pattern, $flight_string, $matches)) {
            return [
                'flight_no'         => $matches[1],
                'date'              => $matches[2],
                'departure_airport' => $matches[3],
                'arrival_airport'   => $matches[4],
                'departure_time'    => $matches[5],
                'arrival_time'      => $matches[6],
            ];
        }
        
        return []; // Return empty array if pattern does not match
    }


    // --- 1. GET SEARCH CRITERIA FROM THE FORM (FROM URL) ---
    $selected_sector = $_GET['sector'] ?? '';
    $selected_route = $_GET['route'] ?? '';

    // --- 2. DYNAMICALLY BUILD THE SQL QUERY ---
    $sql = "SELECT gf.*, al.logo_url FROM group_fares gf 
            LEFT JOIN airlines al ON gf.airline_id = al.id
            WHERE gf.is_active = 1 AND gf.remaining_seats > 0";
    $params = [];
    $types = "";

    if (!empty($selected_sector)) {
        $sql .= " AND gf.sector = ?";
        $params[] = $selected_sector;
        $types .= "s";
    }
    if (!empty($selected_route)) {
        $sql .= " AND gf.route = ?";
        $params[] = $selected_route;
        $types .= "s";
    }

    $sql .= " ORDER BY gf.price_adult ASC";

    // --- 3. PREPARE AND EXECUTE THE QUERY SECURELY ---
    $stmt = $conn->prepare($sql);
    if (!empty($params)) {
        $stmt->bind_param($types, ...$params);
    }
    $stmt->execute();
    $result = $stmt->get_result();
    ?>

    <!DOCTYPE html>
    <html lang="en">
    <head>
        <meta charset="UTF-8">
        <meta name="viewport" content="width=device-width, initial-scale=1.0">
        <title>Group Fare Results - TravelUstaad</title>
        
        <link rel="stylesheet" href="css/style.css">
        <link rel="stylesheet" href="css/results-page.css">
        <link rel="preconnect" href="https://fonts.googleapis.com">
        <link rel="preconnect" href="https://fonts.gstatic.com" crossorigin>
        <link href="https://fonts.googleapis.com/css2?family=Poppins:wght@400;500;600;700&display=swap" rel="stylesheet">
        <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.2.0/css/all.min.css">
    </head>
    <body>

        <?php include 'header.php'; ?>

        <div class="page-wrapper">
            <main class="main-content">
                <div class="content-header">
                    <h1>Group Fare Results</h1>
                    <p>
                        Showing results for: 
                        <strong><?= !empty($selected_sector) ? htmlspecialchars($selected_sector) : 'Any Type' ?></strong> | 
                        <strong><?= !empty($selected_route) ? htmlspecialchars($selected_route) : 'Any Route' ?></strong>
                    </p>
                </div>

                <div id="results-container">
                    <?php if ($result && $result->num_rows > 0): ?>
                        <?php while ($fare = $result->fetch_assoc()): 
                            // --- ** THE CRITICAL FIX IS HERE ** ---
                            // 1. Decode the main JSON from the database
                            $flight_data = json_decode($fare['flight_details_json'], true) ?: [];

                            // 2. Parse the individual strings into structured arrays
                            $outbound = parse_flight_string($flight_data['outbound'] ?? '');
                            $inbound = parse_flight_string($flight_data['inbound'] ?? '');
                            
                            // 3. Handle multi-line baggage info
                            $baggage_lines = explode("\n", trim($flight_data['baggage'] ?? ''));
                            $outbound_baggage = trim($baggage_lines[0] ?? 'N/A');
                            // If there's no second line for inbound, use the outbound baggage info
                            $inbound_baggage = trim($baggage_lines[1] ?? $outbound_baggage);
                        ?>
                        <div class="fare-card">
                            <div class="flight-info-grid">
                                <!-- Outbound Leg -->
                                <div class="flight-leg">
                                    <div class="flight-airline">
                                        <img src="<?= htmlspecialchars($fare['logo_url'] ?? 'placeholder.png') ?>" alt="Airline Logo">
                                        <span><?= htmlspecialchars($outbound['flight_no'] ?? 'N/A') ?></span>
                                    </div>
                                    <div class="flight-departure">
                                        <strong><?= format_flight_time($outbound['departure_time'] ?? '') ?></strong>
                                        <span><?= htmlspecialchars($outbound['departure_airport'] ?? 'N/A') ?></span>
                                    </div>
                                    <div class="flight-travel-arrow">
                                        <!-- Use the main fare departure date for the outbound leg -->
                                        <span><?= format_flight_date($fare['departure_date'] ?? '') ?></span>
                                        <div class="arrow"></div>
                                    </div>
                                    <div class="flight-arrival">
                                        <strong><?= format_flight_time($outbound['arrival_time'] ?? '') ?></strong>
                                        <span><?= htmlspecialchars($outbound['arrival_airport'] ?? 'N/A') ?></span>
                                    </div>
                                    <div class="flight-baggage">
                                        <i class="fa-solid fa-suitcase-rolling"></i>
                                        <span><?= htmlspecialchars($outbound_baggage) ?></span>
                                    </div>
                                </div>

                                <!-- Inbound Leg (only show if data exists) -->
                                <?php if (!empty($inbound)): ?>
                                <div class="flight-leg">
                                    <div class="flight-airline">
                                        <img src="<?= htmlspecialchars($fare['logo_url'] ?? 'placeholder.png') ?>" alt="Airline Logo">
                                        <span><?= htmlspecialchars($inbound['flight_no'] ?? 'N/A') ?></span>
                                    </div>
                                    <div class="flight-departure">
                                        <strong><?= format_flight_time($inbound['departure_time'] ?? '') ?></strong>
                                        <span><?= htmlspecialchars($inbound['departure_airport'] ?? 'N/A') ?></span>
                                    </div>
                                    <div class="flight-travel-arrow">
                                        <!-- Use the date parsed from the inbound string -->
                                        <span><?= format_flight_date($inbound['date'] ?? '') ?></span>
                                        <div class="arrow"></div>
                                    </div>
                                    <div class="flight-arrival">
                                        <strong><?= format_flight_time($inbound['arrival_time'] ?? '') ?></strong>
                                        <span><?= htmlspecialchars($inbound['arrival_airport'] ?? 'N/A') ?></span>
                                    </div>
                                    <div class="flight-baggage">
                                        <i class="fa-solid fa-suitcase-rolling"></i>
                                        <span><?= htmlspecialchars($inbound_baggage) ?></span>
                                    </div>
                                </div>
                                <?php endif; ?>
                            </div>

                            <div class="pricing-info">
                                <div class="price-display">
                                    <span><?= htmlspecialchars($fare['price_currency'] ?? 'PKR') ?></span> 
                                    <strong><?= number_format($fare['price_adult'] ?? 0) ?></strong>
                                    <small>/person</small>
                                </div>
                                <a href="group-booking-detail.php?ref=<?= urlencode($fare['group_ref_id'] ?? '') ?>" class="btn-book">Book <i class="fa-solid fa-arrow-right"></i></a>
                                <div class="meta-tags">
                                    <span class="seats-available"><i class="fa-solid fa-chair"></i> Available: <?= htmlspecialchars($fare['remaining_seats'] ?? '0') ?></span>
                                    <span>Total Seats: <?= htmlspecialchars($fare['total_seats'] ?? '0') ?></span>
                                    <span>Days: <?= htmlspecialchars($fare['duration_days'] ?? '0') ?></span>
                                </div>
                            </div>
                        </div>
                        <?php endwhile; ?>
                    <?php else: ?>
                        <div class="no-results">
                            <h3>No Group Fares Found</h3>
                            <p>We couldn't find any results matching your search criteria.</p>
                            <p><a href="group-fares.php">Try a different search</a></p>
                        </div>
                    <?php endif; ?>
                </div>
            </main>
        </div>

        <?php include 'footer.php'; ?>

    </body>
    </html>