$(document).ready(function() {
    // --- STATE & INITIAL SETUP ---
    let state = {
        flight: { price: 0, pax: 1, arrivalAirport: '', departureAirport: '', customerName: '' },
        visa: { id: null, typeName: '', duration_days: 0, pax: 0, costSAR: 0, perPersonCostSAR: 0, required: true },
        accommodation: { stops: [], totalNights: 0, totalCostSAR: 0, required: true, rate_list_type: '' },
        transport: { costSAR: 0, required: true, type: '', details: '' },
        ziyarat: { costSAR: 0, required: false },
        settings: {},
        user_type: 'customer',
        selected_customer: null,
        service_type: ''
    };

    if (allRates.settings && allRates.settings.sar_to_pkr_rate) {
        state.settings.sar_to_pkr = parseFloat(allRates.settings.sar_to_pkr_rate);
    }
    state.user_type = allRates.user_type || 'customer';
    
    // Initialize Select2
    $('.select2-init').select2({ theme: "default", width: '100%' });
    $('#arrival-city, #departure-city, #service-type-select').select2({ theme: "default" });

    // --- AGENT-SPECIFIC LOGIC ---
    if (state.user_type === 'agent') {
        $('#agent-customer-select').on('change', function() {
            $('#start-building-btn').prop('disabled', !$(this).val());
        });
    }

    // --- INITIAL SERVICE SELECTION LOGIC ---
    $('#service-selection-screen').show();

    $('#service-type-select').on('change', function() {
        $('#proceed-service-selection').prop('disabled', !$(this).val());
    });

    $('#proceed-service-selection').on('click', function() {
        const service = $('#service-type-select').val();
        state.service_type = service;
        
        if (service === 'pkg_with_ticket') {
            window.location.href = 'index.php?service_redirect=custom_package';
        } else if (service === 'pkg_without_ticket') {
            configureForService(service);
            $('#service-selection-screen').fadeOut(() => $('#package-builder-start').fadeIn());
        } else if (service === 'only_hotels') {
            window.location.href = 'hotels.php';
        } else {
            configureForService(service);
            $('#service-selection-screen').fadeOut(() => $('#package-builder-start').fadeIn());
        }
    });

    function configureForService(serviceType) {
        state.visa.required = (serviceType === 'pkg_without_ticket');
        state.accommodation.required = (serviceType === 'pkg_without_ticket');
        state.transport.required = (serviceType === 'pkg_without_ticket' || serviceType === 'only_transport' || serviceType === 'transport_ziyarat');
        state.ziyarat.required = (serviceType === 'only_ziyarat');

        $('#visa-section').toggle(state.visa.required);
        $('#accommodation-section').toggle(state.accommodation.required);
        $('#transport-section').toggle(state.transport.required);
        $('#ziyarat-section').toggle(serviceType === 'pkg_without_ticket' || serviceType === 'only_ziyarat' || serviceType === 'transport_ziyarat');
        
        $('#summary-visa').toggle(state.visa.required);
        $('#summary-accommodation').toggle(state.accommodation.required);
        $('#summary-transport').toggle(state.transport.required);
        $('#summary-ziyarat').toggle(serviceType === 'pkg_without_ticket' || serviceType === 'only_ziyarat' || serviceType === 'transport_ziyarat');
    }

    // --- INITIAL PROMPT & CALCULATOR START ---
    $('#start-building-btn').on('click', function() {
        let customerName = '';
        if (state.user_type === 'agent') {
            const selectedCustomerOption = $('#agent-customer-select option:selected');
            if (!selectedCustomerOption.val()) return alert('Please select a customer to proceed.');
            state.selected_customer = { id: selectedCustomerOption.val(), name: selectedCustomerOption.text() };
            customerName = selectedCustomerOption.text().split(' (')[0];
        } else {
            customerName = $('#customer-name').val().trim();
            if (!customerName) return alert('Please enter a customer name.');
        }

        const price = parseFloat($('#manual-ticket-price').val()) || 0;
        const pax = parseInt($('#manual-pax').val());
        const arrival = $('#arrival-city').val();
        const departure = $('#departure-city').val();
        
        if (isNaN(pax) || pax <= 0) return alert('Please enter a valid number of passengers.');
        if (state.visa.required || state.accommodation.required || state.transport.required) {
            if (!arrival || !departure) return alert('Please select arrival and departure airports for this service.');
        }
        
        state.flight = { customerName, price, pax, arrivalAirport: arrival, departureAirport: departure };
        startCalculator();
    });

    if (ticketFromRedirect) {
        state.service_type = 'pkg_without_ticket';
        configureForService(state.service_type);
        $('#package-builder-start').show();
        $('#service-selection-screen').hide();
        const ticketPrice = $('#manual-ticket-price').val();
        if (ticketPrice > 0) {
            $('#manual-ticket-price').prop('readonly', true);
            $('#manual-pax').prop('readonly', true);
        }
    }

    function startCalculator() {
        $('#package-builder-start').fadeOut(() => {
            populateCalculator();
            $('#calculator-wrapper').fadeIn();
        });
    }

    function populateCalculator() {
        $('#customer-name-display').text(state.flight.customerName);
        $('#pax-summary-display').text(`${state.flight.pax} Person(s)`);
        $('#flight-route-display').text(`${state.flight.arrivalAirport} → ${state.flight.departureAirport}`);
        
        if(state.visa.required) renderVisaOptions();
        if(state.transport.required) renderTransportOptions();
        renderZiyaratOptions();
        if(state.accommodation.required) renderRateListCards();
        $("#accommodation-stops-container").sortable({ handle: ".drag-handle", placeholder: "ui-state-highlight", update: calculateCosts });
        updateSummary();
    }
    
    // --- EVENT LISTENERS ---
    $('#custom-package-form').on('submit', function(e) {
        e.preventDefault();
        const proceedBtn = $('#proceed-btn');
        proceedBtn.prop('disabled', true).html('<i class="fas fa-spinner fa-spin"></i> Submitting...');

        $.ajax({
            url: 'custom-package.php', type: 'POST', data: { action: 'submit_inquiry', package_data: JSON.stringify(state) }, dataType: 'json',
            success: function(response) {
                if (response.success && response.redirect) {
                    window.location.href = response.redirect;
                } else {
                    alert('Submission failed: ' + (response.message || 'Unknown error.'));
                    proceedBtn.prop('disabled', false).text('Proceed with Inquiry');
                }
            },
            error: function() {
                alert('An error occurred. Please try again.');
                proceedBtn.prop('disabled', false).text('Proceed with Inquiry');
            }
        });
    });

    $('#visa-type-select').on('change', calculateCosts);
    $('#visa-pax-select, #visa-pax-group-count').on('change input', function() {
        $('#visa-pax-group-container').toggle($('#visa-pax-select').val() === '5+');
        calculateCosts();
    });
    
    $('body').on('click', '.vehicle-selector-container', function(e) {
        e.stopPropagation();
    });

    $('body').on('click', '.transport-option', function(e) {
        if (!$(e.target).closest('.vehicle-selector-container').length) {
            if ($(this).hasClass('selected')) return;
            $('.transport-option').removeClass('selected').find('.vehicle-selector-container').slideUp();
            $(this).addClass('selected');
            if ($(this).data('type') === 'private') {
                $(this).find('.vehicle-selector-container').slideDown();
            }
            calculateCosts();
        }
    });
    $('body').on('change', '.transport-vehicle-select', calculateCosts);

    $('body').on('change', '.ziyarat-type-selector input, .ziyarat-bus-pax-count, .ziyarat-vehicle-select', function() {
        const card = $(this).closest('.ziyarat-option-card');
        const type = card.find('input[type="radio"]:checked').val();
        card.find('.vehicle-selector-container').toggle(type === 'private');
        card.find('.ziyarat-bus-pax-container').toggle(type === 'bus');
        calculateCosts();
    });
    
    $('body').on('click', '.rate-list-card', function(e) {
        if ($(e.target).hasClass('btn-preview-list')) {
            e.stopPropagation();
            const rateType = $(this).data('rate-type');
            generateAndShowRateList(rateType);
            $('#rate-list-modal').fadeIn();
            return;
        }
        $('.rate-list-card').removeClass('selected');
        $(this).addClass('selected');
        const selectedList = $(this).data('rate-type');
        state.accommodation.rate_list_type = selectedList;
        $('#accommodation-lock-overlay').hide();
        $('#accommodation-content').show();
        $('#accommodation-stops-container').empty();
        calculateCosts();
    });

    $('.modal-close').on('click', () => $('.modal').fadeOut());
    $('#add-accommodation-btn').on('click', addAccommodationStop);
    $('body').on('click', '.btn-remove-stop', function() { $(this).closest('.accommodation-stop').remove(); calculateCosts(); });
    $('body').on('input change', '.accommodation-stop input, .accommodation-stop select', calculateCosts);

    $('#download-post-btn').on('click', function() {
        $(this).prop('disabled', true).html('<i class="fas fa-spinner fa-spin"></i> Generating...');
        $('#social-post-preview-area').slideDown();
        generateSocialPostCanvas().then(() => {
            const canvas = document.getElementById('social-post-canvas');
            const link = document.createElement('a');
            const customerNameSlug = state.flight.customerName.replace(/[^a-z0-9]/gi, '_').toLowerCase();
            link.download = `umrah-package-${customerNameSlug}.png`;
            link.href = canvas.toDataURL('image/png');
            link.click();
            $(this).prop('disabled', false).html('<i class="fas fa-download"></i> Download Package Post');
        }).catch(err => {
            console.error("Canvas generation failed:", err);
            alert("Could not generate the image. Please check console for errors.");
            $(this).prop('disabled', false).html('<i class="fas fa-download"></i> Download Package Post');
        });
    });

    // --- SECTION SPECIFIC LOGIC ---
    function renderVisaOptions() {
        const typeSelect = $('#visa-type-select').empty().append('<option value="">Select Visa Type...</option>');
        allRates.visa_rates.forEach(visa => typeSelect.append(`<option value="${visa.id}">${visa.visa_type}</option>`));
        const paxSelect = $('#visa-pax-select');
        paxSelect.empty().append('<option value="0">0</option>');
        for (let i = 1; i <= 4; i++) paxSelect.append(`<option value="${i}">${i} Person${i > 1 ? 's' : ''}</option>`);
        paxSelect.append('<option value="5+">5 or more</option>');
    }

    function getVisaPaxCount() {
        const selection = $('#visa-pax-select').val();
        if (selection === '5+') {
            const paxCount = parseInt($('#visa-pax-group-count').val()) || 5;
            return Math.max(5, Math.min(47, paxCount));
        }
        return parseInt(selection) || 0;
    }

    function renderTransportOptions() {
        const container = $('#transport-options-container').empty();
        const { arrivalAirport, departureAirport } = state.flight;
        const routesToShow = [];
        if (arrivalAirport === 'JED' && departureAirport === 'JED') { routesToShow.push({ name: 'Round Trip B', details: 'JED APT → MAK → MED → JED APT' }, { name: 'Round Trip B Flipped', details: 'JED APT → MED → MAK → JED APT' }, { name: 'Round Trip C', details: 'JED APT → MAK → MED → MAK → JED APT' }); }
        else if (arrivalAirport === 'MED' && departureAirport === 'JED') { routesToShow.push({ name: 'Round Trip A Flipped', details: 'MAD APT → MED → MAK → JED APT' }); }
        else if (arrivalAirport === 'JED' && departureAirport === 'MED') { routesToShow.push({ name: 'Round Trip A', details: 'JED APT → MAK → MED → MED APT' }); }
        
        if (routesToShow.length === 0) { container.html(`<p style="text-align:center; color:#6c757d;">No standard round trips available for this route.</p>`); return; }

        routesToShow.forEach(route => {
            const originalRouteName = route.name.includes('Flipped') ? route.name.replace(' Flipped', '') : route.name;
            const busRate = allRates.transport_rates.find(r => r.sector_name === 'Round Trip C' && r.vehicle_type.toLowerCase() === 'bus');
            if (route.name === 'Round Trip C' && busRate) { container.append(`<div class="transport-option" data-route-details="${route.details}" data-type="free"><span class="badge free">INCLUDED IN VISA</span><h5>Complimentary Bus</h5><div class="route-def">${route.details}</div><div class="price-info">SAR 0.00</div></div>`); }
            const privateRates = allRates.transport_rates.filter(r => r.sector_name === originalRouteName && r.vehicle_type.toLowerCase() !== 'bus');
            if (privateRates.length > 0) {
                const cheapestRate = Math.min(...privateRates.map(r => parseFloat(r.rate_sar)));
                const vehicleOptionsHtml = [...new Set(privateRates.map(r => r.vehicle_type))].map(v => `<option value="${v}">${v}</option>`).join('');
                container.append(`<div class="transport-option" data-route-name="${originalRouteName}" data-type="private"><span class="badge private">PRIVATE</span><h5>Ya Transport (Private)</h5><div class="route-def">${route.details}</div><div class="price-info">From SAR ${cheapestRate.toFixed(2)}</div><div class="vehicle-selector-container"><label>Select Vehicle</label><select class="form-field transport-vehicle-select"><option value="">Select Vehicle</option>${vehicleOptionsHtml}</select></div></div>`);
            }
        });
        
        container.append(`<p style="font-size: 0.85rem; color: #6c757d; text-align: center; margin-top: 1.5rem; padding-top: 1rem; border-top: 1px solid #e9ecef;">Please note: Complimentary bus transport is typically included with your visa.</p>`);
    }

    function renderZiyaratOptions() {
        const container = $('#ziyarat-options-container').empty();
        const locations = [...new Set(allRates.ziyarat_rates.map(r => r.city))];
        locations.forEach(city => {
            const rates = allRates.ziyarat_rates.filter(r => r.city === city);
            const hasBus = rates.some(r => r.vehicle_type.toLowerCase() === 'bus');
            const privateVehicles = rates.filter(r => r.vehicle_type.toLowerCase() !== 'bus');
            const privateOptions = [...new Set(privateVehicles.map(v => v.vehicle_type))].map(v => `<option value="${v}">${v}</option>`).join('');
            container.append(`<div class="ziyarat-option-card" data-city="${city}"><h5>${city} Ziyarat</h5><div class="ziyarat-type-selector"><input type="radio" id="ziyarat_${city.replace(/\s+/g,'')}_none" name="ziyarat_type_${city.replace(/\s+/g,'')}" value="none" checked><label for="ziyarat_${city.replace(/\s+/g,'')}_none">None</label>${hasBus?`<input type="radio" id="ziyarat_${city.replace(/\s+/g,'')}_bus" name="ziyarat_type_${city.replace(/\s+/g,'')}" value="bus"><label for="ziyarat_${city.replace(/\s+/g,'')}_bus">Bus</label>`:''}${privateVehicles.length>0?`<input type="radio" id="ziyarat_${city.replace(/\s+/g,'')}_private" name="ziyarat_type_${city.replace(/\s+/g,'')}" value="private"><label for="ziyarat_${city.replace(/\s+/g,'')}_private">Private</label>`:''}</div><div class="ziyarat-bus-pax-container form-field"><label>Number of Passengers for Bus</label><input type="number" class="ziyarat-bus-pax-count" min="1" value="${state.flight.pax}"></div><div class="vehicle-selector-container form-field"><label>Select Private Vehicle</label><select class="ziyarat-vehicle-select"><option value="">Select Vehicle</option>${privateOptions}</select></div><div class="ziyarat-cost-display" style="display:none;"></div></div>`);
        });
    }
    
    function renderRateListCards() {
        const container = $('#rate-list-selector-container').empty();
        const s = allRates.settings;
        const listTypes = ['Rates 1', 'Rates 2'];
        listTypes.forEach(type => {
            const title = `${s.rate_list_main_title || 'Umrah Packages'} - ${type}`;
            const bgImage = s.rate_list_bg_image ? `../${s.rate_list_bg_image}` : '';
            const thumbClass = bgImage ? '' : 'no-img';
            const thumbStyle = bgImage ? `background-image: url('${bgImage}')` : '';
            const cardHtml = `<div class="rate-list-card" data-rate-type="${type}"><button type="button" class="btn-preview-list">Preview</button><div class="rate-list-card-thumb ${thumbClass}" style="${thumbStyle}">${!bgImage ? '<i class="fas fa-image"></i>' : ''}</div><div class="rate-list-card-body"><h5>${title}</h5></div></div>`;
            container.append(cardHtml);
        });
    }

    function calculateCosts() {
        if(state.visa.required) calculateVisa();
        if(state.transport.required) calculateTransport();
        if(state.accommodation.required) calculateAccommodation();
        calculateZiyarat();
        updateSummary();
    }

    function calculateVisa() {
        const visaId = $('#visa-type-select').val();
        const pax = getVisaPaxCount();
        let cost = 0, perPersonCost = 0, duration = 0, typeName = '';
        if (visaId && pax > 0) {
            const data = allRates.visa_rates.find(v => v.id == visaId);
            if (data) {
                duration = parseInt(data.duration_days);
                typeName = data.visa_type;
                let rateKey = (pax >= 5) ? 'rate_per_pax_5_plus' : `rate_per_pax_${pax}`;
                perPersonCost = parseFloat(data[rateKey] || 0);
                cost = perPersonCost * pax;
                $('#visa-cost-display').html(`Per Person: <strong>SAR ${perPersonCost.toFixed(2)}</strong> | Total for ${pax} Person(s): <strong>SAR ${cost.toFixed(2)}</strong>`);
            }
        } else { $('#visa-cost-display').text('Select visa options to see cost.'); }
        state.visa = { ...state.visa, id: visaId, typeName, duration_days: duration, pax, costSAR: cost, perPersonCostSAR: perPersonCost };
    }
    
    function calculateTransport() {
        const selected = $('.transport-option.selected');
        let cost = 0, type = '', details = '';
        if (selected.length > 0) {
            type = selected.data('type');
            details = selected.find('.route-def').text();
            if (type === 'private') {
                const vehicle = selected.find('.transport-vehicle-select').val();
                const routeName = selected.data('routeName');
                if (vehicle && routeName) {
                    const rateData = allRates.transport_rates.find(r => r.sector_name === routeName && r.vehicle_type === vehicle);
                    if (rateData) cost = parseFloat(rateData.rate_sar);
                }
            }
        }
        state.transport = { ...state.transport, costSAR: cost, type: type, details: details };
    }

    function calculateAccommodation() {
        let totalCost = 0, totalNights = 0;
        let stops = [];
        const rateListType = state.accommodation.rate_list_type;
        if (!rateListType) {
            state.accommodation = { ...state.accommodation, stops: [], totalCostSAR: 0, totalNights: 0 };
            return;
        }
        $('.accommodation-stop').each(function() {
            const $stop = $(this);
            const nights = parseInt($stop.find('.hotel-nights').val()) || 0;
            totalNights += nights;
            const hotelId = $stop.find('.hotel-select').val(), roomType = $stop.find('.room-type').val(), city = $stop.find('.hotel-city').val(), display = $stop.find('.stop-cost-display');
            let stopDetails = { id: $stop.attr('id'), city, hotelId, hotelName: '', nights, roomType, stopCostSAR: 0 };
            if (!hotelId || nights <= 0 || !roomType || !city) { display.text('Fill all details to see cost.'); stops.push(stopDetails); return; }
            const hotelData = allRates.hotel_rates[rateListType]?.[city]?.find(h => h.id == hotelId);
            if (!hotelData) { display.text('Hotel data not found.'); stops.push(stopDetails); return; }
            stopDetails.hotelName = hotelData.hotel_name;
            let ratePerPerson = hotelData[`rate_${roomType === 'triple' ? 'trpl' : roomType}`] !== null ? parseFloat(hotelData[`rate_${roomType === 'triple' ? 'trpl' : roomType}`]) : null;
            if (ratePerPerson === null) {
                const roomRate = parseFloat(hotelData.rate_room_basis);
                const paxMap = { double: 2, triple: 3, quad: 4 };
                if (!isNaN(roomRate) && paxMap[roomType]) ratePerPerson = roomRate / paxMap[roomType]; else ratePerPerson = 0;
            }
            if (state.user_type === 'agent') {
                if (city === 'Makkah') ratePerPerson = Math.max(0, ratePerPerson - 2);
                if (city === 'Madinah') ratePerPerson = Math.max(0, ratePerPerson - 3);
            }
            const stopPP = ratePerPerson * nights, stopTotal = stopPP * state.flight.pax;
            totalCost += stopTotal;
            stopDetails.stopCostSAR = stopTotal;
            display.html(`Per Person: <strong>SAR ${stopPP.toFixed(2)}</strong> | Total: <strong>SAR ${stopTotal.toFixed(2)}</strong>`);
            stops.push(stopDetails);
        });
        state.accommodation = { ...state.accommodation, stops, totalCostSAR: totalCost, totalNights };
        const visaDuration = state.visa.duration_days || 0;
        const limitReached = visaDuration > 0 && totalNights >= visaDuration;
        $('#add-accommodation-btn').prop('disabled', limitReached);
        $('#total-nights-warning').html(`<strong>Warning:</strong> Total nights (${totalNights}) exceed the ${visaDuration}-day visa limit.`).toggle(state.visa.required && visaDuration > 0 && totalNights > visaDuration);
    }

    function calculateZiyarat() {
        let totalCost = 0;
        $('.ziyarat-option-card').each(function() {
            const card = $(this), city = card.data('city'), type = card.find('input:checked').val(), display = card.find('.ziyarat-cost-display');
            let stopCost = 0;
            if (type === 'bus') {
                const rateData = allRates.ziyarat_rates.find(r => r.city === city && r.vehicle_type.toLowerCase() === 'bus');
                if (rateData) {
                    let rate = parseFloat(rateData.rate_sar);
                    if (state.user_type === 'agent') rate = Math.max(0, rate - 5);
                    const pax = parseInt(card.find('.ziyarat-bus-pax-count').val()) || 0;
                    stopCost = rate * pax;
                    display.html(`Per Person: <strong>SAR ${rate.toFixed(2)}</strong> | Total for ${pax}: <strong>SAR ${stopCost.toFixed(2)}</strong>`).show();
                }
            } else if (type === 'private') {
                const vehicle = card.find('.ziyarat-vehicle-select').val();
                if (vehicle) {
                    const rateData = allRates.ziyarat_rates.find(r => r.city === city && r.vehicle_type === vehicle);
                    if (rateData) {
                        stopCost = parseFloat(rateData.rate_sar);
                        display.html(`Total Cost: <strong>SAR ${stopCost.toFixed(2)}</strong>`).show();
                    }
                } else { display.hide(); }
            } else { display.hide(); }
            totalCost += stopCost;
        });
        state.ziyarat = { ...state.ziyarat, costSAR: totalCost };
    }
    
    function updateSummary() {
        const groundTotalSAR = (state.visa.costSAR) + (state.accommodation.totalCostSAR) + (state.transport.costSAR) + (state.ziyarat.costSAR);
        const totalPKR_noTicket = groundTotalSAR * (state.settings.sar_to_pkr || 75.00);
        const totalPKR_withTicket = totalPKR_noTicket + state.flight.price;
        const perPersonGroundPKR = state.flight.pax > 0 ? totalPKR_noTicket / state.flight.pax : 0;
        const perPersonTotalPKR = state.flight.pax > 0 ? totalPKR_withTicket / state.flight.pax : 0;
        
        $('#ticket-total').text(`PKR ${Math.round(state.flight.price).toLocaleString()}`);
        $('#visa-total').text(`SAR ${state.visa.costSAR.toFixed(2)}`);
        $('#accommodation-total').text(`SAR ${state.accommodation.totalCostSAR.toFixed(2)}`);
        $('#transport-total').text(`SAR ${state.transport.costSAR.toFixed(2)}`);
        $('#ziyarat-total').text(`SAR ${state.ziyarat.costSAR.toFixed(2)}`);
        $('#grand-total-sar').text(`SAR ${groundTotalSAR.toFixed(2)}`);
        $('#grand-total-pkr-noticket').text(`PKR ${Math.round(totalPKR_noTicket).toLocaleString()}`);
        $('#grand-total-pkr').text(`PKR ${Math.round(totalPKR_withTicket).toLocaleString()}`);
        $('#per-person-ground').text(`PKR ${Math.round(perPersonGroundPKR).toLocaleString()}`);
        $('#per-person-total').text(`PKR ${Math.round(perPersonTotalPKR).toLocaleString()}`);
        
        const hasTicket = state.flight.price > 0;
        $('#summary-ticket, #summary-total-with-ticket, #summary-pp-with-ticket').toggle(hasTicket);
        
        let isValid = true;
        if (state.visa.required && (!state.visa.id || state.visa.pax <= 0)) isValid = false;
        if (state.accommodation.required && state.accommodation.stops.length === 0) isValid = false;
        if (state.transport.required && !state.transport.type) isValid = false;

        const allStopsValid = state.accommodation.stops.every(s => s.hotelId && s.nights > 0 && s.roomType && s.city);
        if (state.accommodation.required && !allStopsValid) isValid = false;

        $('#proceed-btn, #download-post-btn').prop('disabled', !isValid);
    }

    function addAccommodationStop() {
        const id = `stop_${Date.now()}`;
        const html = `<div class="accommodation-stop" id="${id}"><div class="stop-header"><h4><i class="fas fa-arrows-alt drag-handle"></i> Hotel Stay</h4><button type="button" class="btn-remove-stop"><i class="fas fa-times-circle"></i></button></div><div class="form-row col-2"><div class="form-field"><label>City</label><select class="hotel-city"><option value="">Select</option><option value="Makkah">Makkah</option><option value="Madinah">Madinah</option></select></div><div class="form-field"><label>Hotel</label><select class="hotel-select select2-init" style="width:100%;" disabled><option value="">Select city first</option></select></div></div><div class="form-row col-2"><div class="form-field"><label>Nights</label><input type="number" class="hotel-nights" min="1" value="1"></div><div class="form-field"><label>Room Type</label><select class="room-type"><option value="sharing">Sharing</option><option value="quad">Quad</option><option value="triple">Triple</option><option value="double">Double</option></select></div></div><div class="stop-cost-display">Fill all details to see cost.</div></div>`;
        $('#accommodation-stops-container').append(html);
        $(`#${id} .select2-init`).select2({ theme: "default" });
        calculateCosts();
    }

    $('body').on('change', '.hotel-city', function() {
        const city = $(this).val();
        const rateListType = state.accommodation.rate_list_type;
        const hotelSelect = $(this).closest('.accommodation-stop').find('.hotel-select').empty().prop('disabled', true);
        if (city && rateListType && allRates.hotel_rates[rateListType]?.[city]) {
            hotelSelect.append(new Option('Select Hotel', ''));
            allRates.hotel_rates[rateListType][city].forEach(h => { hotelSelect.append(new Option(`${h.hotel_name} ${h.star_rating ? `(${h.star_rating} Star)`:''} - ${h.distance_info}`, h.id)); });
            hotelSelect.prop('disabled', false);
        } else { hotelSelect.append(new Option('Select city first', '')); }
        hotelSelect.trigger('change');
    });

    function generateAndShowRateList(rateType) {
        const s = allRates.settings;
        const hotelsForList = allRates.all_hotels_by_list[rateType] || [];
        const makkahHotels = hotelsForList.filter(h => h.city === 'Makkah');
        const madinahHotels = hotelsForList.filter(h => h.city === 'Madinah');
        const modalTitle = `${s.rate_list_main_title || 'Umrah Packages'} - ${type}`;
        $('#rate-list-modal-title').text(modalTitle);
        const tableHTML = `<div style="padding: 1rem; background: #fff;">${generateModalHotelTableHTML(`${rateType} - MAKKAH HOTELS`, makkahHotels)}${generateModalHotelTableHTML(`${rateType} - MADINAH HOTELS`, madinahHotels)}</div>`;
        $('#rate-list-modal-content').html(tableHTML);
    }

    function generateModalHotelTableHTML(title, hotels) {
        if (!hotels || hotels.length === 0) return `<p style="text-align:center; padding: 2rem;">No hotels found for this category.</p>`;
        const rows = hotels.map(h => `<tr><td>${e(h.hotel_name)}<br><small style="color:#6c757d; font-size: 0.7rem;">${e(h.notes)}</small></td><td>${e(h.star_rating)}</td><td>${e(h.distance_info)}</td><td>${h.rate_sharing?parseFloat(h.rate_sharing).toFixed(0):'N/A'}</td><td>${h.rate_quad?parseFloat(h.rate_quad).toFixed(0):'N/A'}</td><td>${h.rate_trpl?parseFloat(h.rate_trpl).toFixed(0):'N/A'}</td><td>${h.rate_dubl?parseFloat(h.rate_dubl).toFixed(0):'N/A'}</td></tr>`).join('');
        return `<h5 style="margin-top:0;margin-bottom:0.8rem; font-size: 1.1rem;">${title}</h5><table class="rl-table"><thead><tr style="background-color: #343A40; color: #fff;"><th>HOTELS</th><th>STAR</th><th>DISTANCE</th><th>SHARING</th><th>QUAD</th><th>TRIPLE</th><th>DOUBLE</th></tr></thead><tbody>${rows}</tbody></table>`;
    }

    function e(str) { return String(str||'').replace(/[&<>"']/g, s => ({'&':'&amp;','<':'&lt;','>':'&gt;','"':'&quot;',"'":'&#39;'}[s])); }

    // --- FINAL SOCIAL POST GENERATOR - TEMPLATE BASED ---
    async function generateSocialPostCanvas() {
        const wrapper = document.getElementById('social-post-canvas-wrapper');
        wrapper.innerHTML = '';
        const canvas = document.createElement('canvas');
        canvas.id = 'social-post-canvas';
        canvas.width = 1080;
        canvas.height = 1080;
        wrapper.appendChild(canvas);
        const ctx = canvas.getContext('2d');

        const loadImage = src => new Promise((resolve, reject) => {
            if (!src) return reject(new Error('Image source is missing. Check asset names in PHP.'));
            const img = new Image();
            img.onload = () => resolve(img);
            img.onerror = () => reject(new Error(`Failed to load image: ${src.substring(0, 40)}...`));
            img.src = src;
        });

        const sarToPkrRate = state.settings.sar_to_pkr || 75.00;
        const groundTotalSAR = (state.visa.costSAR) + (state.accommodation.totalCostSAR) + (state.transport.costSAR) + (state.ziyarat.costSAR);
        const groundTotalPKR = groundTotalSAR * sarToPkrRate;
        const perPersonGroundPKR = state.flight.pax > 0 ? Math.round(groundTotalPKR / state.flight.pax) : 0;

        try {
            const assets = allRates.post_assets;
            // Simplified the required assets to only the essential ones for this template
            const [
                templateImg, circ1Img, circ2Img
            ] = await Promise.all([
                loadImage(assets.post_bg_main),
                loadImage(assets.circular_1),
                loadImage(assets.circular_2)
            ]);

            ctx.clearRect(0, 0, 1080, 1080);
            
            // 1. Draw the main template image as the base
            ctx.drawImage(templateImg, 0, 0, 1080, 1080);
            
            // 2. Draw the circular images inside their designated spots
            const drawClippedImage = (img, x, y, radius) => {
                ctx.save();
                ctx.beginPath();
                ctx.arc(x, y, radius, 0, Math.PI * 2);
                ctx.closePath();
                ctx.clip();
                const aspect = img.width / img.height;
                let iw, ih, ix, iy;
                if (aspect > 1) { ih = radius * 2; iw = ih * aspect; ix = x - iw / 2; iy = y - radius; } 
                else { iw = radius * 2; ih = iw / aspect; ix = x - radius; iy = y - ih / 2; }
                ctx.drawImage(img, ix, iy, iw, ih);
                ctx.restore();
            };
            drawClippedImage(circ1Img, 678, 290, 132);
            drawClippedImage(circ2Img, 842, 458, 98);

            // 3. Calculate Nights
            const makkahNights = state.accommodation.stops
                .filter(s => s.city === 'Makkah')
                .reduce((sum, s) => sum + s.nights, 0);
            const madinahNights = state.accommodation.stops
                .filter(s => s.city === 'Madinah')
                .reduce((sum, s) => sum + s.nights, 0);
            
            // 4. Draw all dynamic text
            ctx.fillStyle = '#FFFFFF';
            ctx.textBaseline = 'middle';
            ctx.textAlign = 'center';

            // Nights Text
            ctx.font = 'bold 28px Poppins';
            ctx.fillText(`${makkahNights} NIGHTS`, 155, 580);
            ctx.fillText(`${madinahNights} NIGHTS`, 380, 580);

            // Package Includes List
            ctx.textAlign = 'left';
            ctx.font = '500 22px Poppins';
            let inclusionsY = 688; // Adjusted Y for alignment
            const inclusions = [];
            if (state.visa.costSAR > 0) inclusions.push('Visa');
            if (state.accommodation.totalCostSAR > 0) inclusions.push('Hotels');
            if (state.transport.costSAR > 0 || state.transport.type === 'free') inclusions.push('Transport');
            if (state.ziyarat.costSAR > 0) inclusions.push('Ziyarat');
            if (state.flight.price > 0) inclusions.push('Return Tickets');
            
            inclusions.slice(0, 5).forEach(item => {
                ctx.fillText(item, 120, inclusionsY);
                inclusionsY += 32;
            });
            
            // Traveling Date
            ctx.textAlign = 'center';
            ctx.font = 'bold 28px Poppins';
            ctx.fillText("As Per Your Choice", 785, 730);
            
            // Prices
            ctx.fillStyle = '#097BED'; // Blue color for prices
            ctx.font = 'bold 44px Poppins';
            ctx.fillText(`PKR ${perPersonGroundPKR.toLocaleString()}`, 310, 848);
            ctx.fillText(`PKR ${Math.round(groundTotalPKR).toLocaleString()}`, 715, 848);

            // Footer Contact Info
            ctx.fillStyle = '#000000';
            ctx.textAlign = 'left';
            ctx.font = '500 18px Poppins';
            ctx.fillText("AL QURESH NEAR RLY PHATAK, SHERSHAH RD, MULTAN", 168, 958);
            
            ctx.font = 'bold 18px Poppins';
            ctx.fillText("RAEES: +92 305 2394810 | SANWAL: +92 309 9705022", 625, 958);

        } catch (error) {
            console.error('Error during canvas generation:', error);
            ctx.fillStyle = 'red';
            ctx.font = '30px Arial';
            ctx.textAlign = 'center';
            ctx.textBaseline = 'middle';
            ctx.fillText('Image data is missing or corrupt.', canvas.width / 2, canvas.height / 2 - 20);
            ctx.font = '20px Arial';
            ctx.fillText('Check filenames in /images/post-assets/.', canvas.width / 2, canvas.height / 2 + 20);
        }
    }
});