<?php

/**
 * manage-requests.php (v3 - All-in-One with Notifications)
 * - A single-page dashboard for agents to list, view, create, edit, and delete their voucher requests.
 * - Uses URL parameters (?action=...) to switch between different views.
 * - Triggers a notification to admin on create, update, and delete actions.
 */

session_start();
include 'db-config.php';

// --- SECURITY CHECK: AGENT-ONLY PAGE ---
if (!isset($_SESSION['user_id']) || !isset($_SESSION['user_type']) || $_SESSION['user_type'] !== 'agent') {
    header("location: login.php");
    exit;
}
$agent_id = $_SESSION['user_id'];

// --- HELPER FUNCTIONS ---
function e($string)
{
    return htmlspecialchars($string ?? '', ENT_QUOTES, 'UTF-8');
}

/**
 * Creates a notification for the admin.
 * @param mysqli $conn The database connection.
 * @param string $type The type of notification (e.g., 'new_request').
 * @param string $message The notification message.
 * @param string $link The URL link for the notification.
 */
function create_admin_notification($conn, $type, $message, $link)
{
    $sql = "INSERT INTO notifications (type, message, link) VALUES (?, ?, ?)";
    $stmt = $conn->prepare($sql);
    // is_read defaults to 0, created_at defaults to CURRENT_TIMESTAMP
    $stmt->bind_param("sss", $type, $message, $link);
    $stmt->execute();
    $stmt->close();
}

// --- FETCH AGENT'S NAME FOR NOTIFICATIONS ---
$stmt_agent_name = $conn->prepare("SELECT name FROM users WHERE id = ?");
$stmt_agent_name->bind_param("i", $agent_id);
$stmt_agent_name->execute();
$agent_name = $stmt_agent_name->get_result()->fetch_assoc()['name'] ?? 'An Agent';
$stmt_agent_name->close();


// --- ACTION HANDLING (CREATE/UPDATE) ---
if ($_SERVER["REQUEST_METHOD"] == "POST") {
    $conn->begin_transaction();
    try {
        // --- HANDLE CREATE REQUEST ---
        if (isset($_POST['form_action']) && $_POST['form_action'] === 'create_request') {
            $sql_request = "INSERT INTO voucher_requests (agent_id, family_head_name, package_type, package_duration_nights, pax_summary, desired_departure_date, desired_return_date, agent_notes) VALUES (?, ?, ?, ?, ?, ?, ?, ?)";
            $stmt_request = $conn->prepare($sql_request);
            $stmt_request->bind_param("ississss", $agent_id, $_POST['family_head_name'], $_POST['package_type'], $_POST['package_duration_nights'], $_POST['pax_summary'], $_POST['desired_departure_date'], $_POST['desired_return_date'], $_POST['agent_notes']);
            $stmt_request->execute();
            $request_id = $conn->insert_id;

            if ($request_id <= 0) throw new Exception("Failed to create the main voucher request.");

            // Insert Mutamers & Accommodations (same logic as before)
            if (!empty($_POST['mutamer_name'])) {
                $stmt_mutamer = $conn->prepare("INSERT INTO request_mutamers (request_id, mutamer_name, passport_no, pax_type, bed_required) VALUES (?, ?, ?, ?, ?)");
                foreach ($_POST['mutamer_name'] as $key => $name) {
                    if (empty(trim($name))) continue;
                    $stmt_mutamer->bind_param("issss", $request_id, $name, $_POST['mutamer_passport_no'][$key], $_POST['mutamer_pax_type'][$key], $_POST['mutamer_bed_required'][$key]);
                    $stmt_mutamer->execute();
                }
            }
            if (!empty($_POST['hotel_preference'])) {
                $stmt_accom = $conn->prepare("INSERT INTO request_accommodations (request_id, city, hotel_preference, check_in_date, nights, room_type, meal_plan) VALUES (?, ?, ?, ?, ?, ?, ?)");
                foreach ($_POST['hotel_preference'] as $key => $preference) {
                    if (empty(trim($preference)) && empty(trim($_POST['accom_city'][$key]))) continue;
                    $stmt_accom->bind_param("isssiss", $request_id, $_POST['accom_city'][$key], $preference, $_POST['accom_check_in'][$key], $_POST['accom_nights'][$key], $_POST['accom_room_type'][$key], $_POST['accom_meal_plan'][$key]);
                    $stmt_accom->execute();
                }
            }

            // ** TRIGGER NOTIFICATION FOR ADMIN **
            $message = e($agent_name) . " submitted a new voucher request #" . $request_id . ".";
            $link = "admin/process-request.php?id=" . $request_id;
            create_admin_notification($conn, 'new_request', $message, $link);

            $_SESSION['success_message'] = "Voucher Request #" . $request_id . " submitted successfully!";
        }

        // --- HANDLE UPDATE REQUEST ---
        if (isset($_POST['form_action']) && $_POST['form_action'] === 'update_request') {
            $request_id = (int)$_POST['request_id'];
            $stmt_verify = $conn->prepare("SELECT id FROM voucher_requests WHERE id = ? AND agent_id = ? AND status = 'Pending'");
            $stmt_verify->bind_param("ii", $request_id, $agent_id);
            $stmt_verify->execute();
            if ($stmt_verify->get_result()->num_rows !== 1) throw new Exception("Request not found or you do not have permission to edit it.");

            $sql_update = "UPDATE voucher_requests SET family_head_name=?, package_type=?, package_duration_nights=?, pax_summary=?, desired_departure_date=?, desired_return_date=?, agent_notes=? WHERE id=?";
            $stmt_update = $conn->prepare($sql_update);
            $stmt_update->bind_param("ssissssi", $_POST['family_head_name'], $_POST['package_type'], $_POST['package_duration_nights'], $_POST['pax_summary'], $_POST['desired_departure_date'], $_POST['desired_return_date'], $_POST['agent_notes'], $request_id);
            $stmt_update->execute();

            $conn->query("DELETE FROM request_mutamers WHERE request_id = $request_id");
            $conn->query("DELETE FROM request_accommodations WHERE request_id = $request_id");

            if (!empty($_POST['mutamer_name'])) {
                $stmt_mutamer = $conn->prepare("INSERT INTO request_mutamers (request_id, mutamer_name, passport_no, pax_type, bed_required) VALUES (?, ?, ?, ?, ?)");
                foreach ($_POST['mutamer_name'] as $key => $name) {
                    if (empty(trim($name))) continue;
                    $stmt_mutamer->bind_param("issss", $request_id, $name, $_POST['mutamer_passport_no'][$key], $_POST['mutamer_pax_type'][$key], $_POST['mutamer_bed_required'][$key]);
                    $stmt_mutamer->execute();
                }
            }
            if (!empty($_POST['hotel_preference'])) {
                $stmt_accom = $conn->prepare("INSERT INTO request_accommodations (request_id, city, hotel_preference, check_in_date, nights, room_type, meal_plan) VALUES (?, ?, ?, ?, ?, ?, ?)");
                foreach ($_POST['hotel_preference'] as $key => $preference) {
                    if (empty(trim($preference)) && empty(trim($_POST['accom_city'][$key]))) continue;
                    $stmt_accom->bind_param("isssiss", $request_id, $_POST['accom_city'][$key], $preference, $_POST['accom_check_in'][$key], $_POST['accom_nights'][$key], $_POST['accom_room_type'][$key], $_POST['accom_meal_plan'][$key]);
                    $stmt_accom->execute();
                }
            }

            // ** TRIGGER NOTIFICATION FOR ADMIN **
            $message = e($agent_name) . " updated their pending voucher request #" . $request_id . ".";
            $link = "admin/process-request.php?id=" . $request_id;
            create_admin_notification($conn, 'updated_request', $message, $link);

            $_SESSION['success_message'] = "Request #" . $request_id . " has been successfully updated.";
        }

        $conn->commit();
    } catch (Exception $e) {
        $conn->rollback();
        $_SESSION['error_message'] = "An error occurred: " . $e->getMessage();
    }
    header("Location: manage-requests.php");
    exit();
}

// --- ACTION HANDLING (DELETE) ---
if (isset($_GET['action']) && $_GET['action'] === 'delete' && isset($_GET['id'])) {
    $request_id = (int)$_GET['id'];
    $conn->begin_transaction();
    try {
        $stmt_verify = $conn->prepare("SELECT id FROM voucher_requests WHERE id = ? AND agent_id = ? AND status = 'Pending'");
        $stmt_verify->bind_param("ii", $request_id, $agent_id);
        $stmt_verify->execute();
        if ($stmt_verify->get_result()->num_rows !== 1) throw new Exception("Request not found or you do not have permission to delete it.");

        $conn->query("DELETE FROM request_mutamers WHERE request_id = $request_id");
        $conn->query("DELETE FROM request_accommodations WHERE request_id = $request_id");
        $conn->query("DELETE FROM voucher_requests WHERE id = $request_id");

        // ** TRIGGER NOTIFICATION FOR ADMIN **
        $message = e($agent_name) . " deleted their pending voucher request #" . $request_id . ".";
        $link = "admin/manage-requests.php"; // Link to the main list as the specific request is gone
        create_admin_notification($conn, 'deleted_request', $message, $link);

        $conn->commit();
        $_SESSION['success_message'] = "Request #" . $request_id . " has been deleted.";
    } catch (Exception $e) {
        $conn->rollback();
        $_SESSION['error_message'] = "Error deleting request: " . $e->getMessage();
    }
    header("Location: manage-requests.php");
    exit();
}

// --- PAGE VIEW ROUTING AND DATA FETCHING (No changes needed here) ---
$action = $_GET['action'] ?? 'list';
$request_id = isset($_GET['id']) ? (int)$_GET['id'] : 0;
$page_title = "Manage Requests";
$page_icon = "fa-solid fa-folder-open";
$requests_data = [];
$request = null;
if ($action === 'list') {
    $sql_requests = "SELECT id, family_head_name, request_date, status FROM voucher_requests WHERE agent_id = ? ORDER BY request_date DESC";
    $stmt_requests = $conn->prepare($sql_requests);
    $stmt_requests->bind_param("i", $agent_id);
    $stmt_requests->execute();
    $result = $stmt_requests->get_result();
    if ($result) $requests_data = $result->fetch_all(MYSQLI_ASSOC);
    $stmt_requests->close();
} elseif (($action === 'view' || $action === 'edit') && $request_id > 0) {
    $stmt_request = $conn->prepare("SELECT * FROM voucher_requests WHERE id = ? AND agent_id = ?");
    $stmt_request->bind_param("ii", $request_id, $agent_id);
    $stmt_request->execute();
    $request = $stmt_request->get_result()->fetch_assoc();
    if ($request) {
        $request['mutamers'] = $conn->query("SELECT * FROM request_mutamers WHERE request_id = $request_id ORDER BY id ASC")->fetch_all(MYSQLI_ASSOC);
        $request['accommodations'] = $conn->query("SELECT * FROM request_accommodations WHERE request_id = $request_id ORDER BY id ASC")->fetch_all(MYSQLI_ASSOC);
        if ($action === 'edit' && strtolower($request['status']) !== 'pending') {
            $_SESSION['error_message'] = "This request can no longer be edited.";
            header("Location: manage-requests.php?action=view&id=" . $request_id);
            exit();
        }
        $page_title = ($action === 'edit') ? "Edit Request #" . $request_id : "View Request #" . $request_id;
        $page_icon = ($action === 'edit') ? "fas fa-pencil-alt" : "fas fa-eye";
    } else {
        $_SESSION['error_message'] = "Request not found.";
        header("Location: manage-requests.php");
        exit();
    }
} elseif ($action === 'create') {
    $page_title = "Request New Voucher";
    $page_icon = "fa-solid fa-paper-plane";
}
?>
<!DOCTYPE html>
<html>

<head>
    <title><?= e($page_title) ?></title>
    <link rel="icon" type="image/png" href="images/logo-icon.png">
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">

    <link rel="stylesheet" href="css/style.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.2.0/css/all.min.css">
    <link rel="stylesheet" href="css/account-style.css">
    <style>
        .content-card-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            padding-bottom: 10px;
        }

        .bookings-table th,
        .bookings-table td {
            vertical-align: middle;
            padding: 14px 10px;
            text-align: left;
        }

        .bookings-table th:nth-child(1),
        .bookings-table td:nth-child(1),
        .bookings-table th:nth-child(4),
        .bookings-table td:nth-child(4),
        .bookings-table th:nth-child(5),
        .bookings-table td:nth-child(5) {
            text-align: center;
        }

        .bookings-table .action-buttons {
            width: 140px;
        }

        .action-buttons a {
            margin: 0 5px;
        }

        .btn-delete {
            color: #e74c3c;
        }

        .btn-delete:hover {
            color: #c0392b;
        }

        .form-section,
        .view-section {
            border: 1px solid #e0e0e0;
            border-radius: 8px;
            margin-top: 1.5rem;
            padding: 1.5rem;
            background-color: #fdfdfd;
        }

        .form-section h3,
        .view-section h3 {
            margin-top: 0;
            border-bottom: 1px solid #eee;
            padding-bottom: 0.5rem;
            margin-bottom: 1rem;
            font-size: 1.2em;
        }

        .dynamic-table {
            width: 100%;
            border-collapse: collapse;
        }

        .dynamic-table th,
        .dynamic-table td {
            border: 1px solid #ddd;
            padding: 8px;
            text-align: center;
            vertical-align: middle;
        }

        .dynamic-table th {
            background-color: #f7f7f7;
            font-size: 0.9em;
        }

        .dynamic-table input,
        .dynamic-table select,
        .dynamic-table textarea {
            width: 100%;
            box-sizing: border-box;
            padding: 6px;
            font-size: 0.9em;
            border-radius: 4px;
            border: 1px solid #ccc;
        }

        .dynamic-table textarea {
            min-height: 40px;
        }

        .remove-row-btn {
            background-color: #e74c3c;
            color: white;
            border: none;
            padding: 5px 10px;
            border-radius: 3px;
            cursor: pointer;
        }

        .add-row-btn {
            margin-top: 10px;
        }

        .view-section dl {
            display: grid;
            grid-template-columns: max-content 1fr;
            gap: 10px 20px;
        }

        .view-section dt {
            font-weight: bold;
            color: #555;
        }

        .view-section dd {
            margin: 0;
        }
    </style>
</head>

<body>
    <?php include 'header.php'; ?>
    <main class="account-page-wrapper">
        <div class="account-container">
            <?php include 'agent-sidebar.php'; ?>
            <div class="account-content">
                <div class="content-card">
                    <div class="content-card-header">
                        <h2><i class="<?= e($page_icon) ?>"></i> <?= e($page_title) ?></h2>
                        <?php if ($action === 'list'): ?>
                            <a href="?action=create" class="btn btn-primary"><i class="fa-solid fa-plus"></i> Request New Voucher</a>
                        <?php else: ?>
                            <a href="manage-requests.php" class="btn btn-secondary"><i class="fa-solid fa-arrow-left"></i> Back to List</a>
                        <?php endif; ?>
                    </div>

                    <?php if (isset($_SESSION['success_message'])): ?>
                        <div class="notice success" style="margin-bottom: 20px;"><?= e($_SESSION['success_message']);
                                                                                    unset($_SESSION['success_message']); ?></div>
                    <?php endif; ?>
                    <?php if (isset($_SESSION['error_message'])): ?>
                        <div class="notice error" style="margin-bottom: 20px;"><?= e($_SESSION['error_message']);
                                                                                unset($_SESSION['error_message']); ?></div>
                    <?php endif; ?>

                    <?php // --- HTML VIEW ROUTER (No changes needed here) --- 
                    ?>

                    <?php if ($action === 'list'): ?>
                        <p class="content-description">Track the status of all your submitted voucher requests. You can only edit or delete requests that are still 'Pending'.</p>
                        <div class="table-responsive">
                            <table class="bookings-table">
                                <thead>
                                    <tr>
                                        <th>Request #</th>
                                        <th>Family Head</th>
                                        <th>Request Date</th>
                                        <th>Status</th>
                                        <th>Actions</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php if (!empty($requests_data)): ?>
                                        <?php foreach ($requests_data as $req): ?>
                                            <tr class="request-row" data-request-id="<?= e($req['id']) ?>" style="cursor: pointer;">
                                                <td><strong>#<?= e($req['id']) ?></strong></td>
                                                <td><?= e($req['family_head_name']) ?></td>
                                                <td><?= date('M j, Y, g:i a', strtotime($req['request_date'])) ?></td>
                                                <td><span class="status-badge status-<?= strtolower(e($req['status'])) ?>"><?= e($req['status']) ?></span></td>
                                                <td class="action-buttons">
                                                    <a href="?action=view&id=<?= $req['id'] ?>" class="btn-sm btn-view" title="View Request"><i class="fas fa-eye"></i></a>
                                                    <?php if (strtolower($req['status']) === 'pending'): ?>
                                                        <a href="?action=edit&id=<?= $req['id'] ?>" class="btn-sm btn-edit" title="Edit Request"><i class="fas fa-pencil-alt"></i></a>
                                                        <a href="?action=delete&id=<?= $req['id'] ?>" class="btn-sm btn-delete" title="Delete Request" onclick="return confirm('Are you sure you want to permanently delete this request?');"><i class="fas fa-trash-alt"></i></a>
                                                    <?php endif; ?>
                                                </td>
                                            </tr>
                                        <?php endforeach; ?>
                                    <?php else: ?>
                                        <tr>
                                            <td colspan="5" class="no-bookings-found">You have not submitted any voucher requests yet.</td>
                                        </tr>
                                    <?php endif; ?>
                                </tbody>
                            </table>
                        </div>

                    <?php elseif ($action === 'create' || ($action === 'edit' && $request)): ?>
                        <form action="manage-requests.php" method="POST">
                            <input type="hidden" name="form_action" value="<?= ($action === 'create') ? 'create_request' : 'update_request' ?>">
                            <?php if ($action === 'edit'): ?><input type="hidden" name="request_id" value="<?= e($request['id']) ?>"><?php endif; ?>
                            <section class="form-section">
                                <h3><i class="fas fa-info-circle"></i> Basic Information</h3>
                                <div class="form-row">
                                    <div class="form-group"><label>Family Head Name*</label><input type="text" name="family_head_name" value="<?= e($request['family_head_name'] ?? '') ?>" required></div>
                                    <div class="form-group"><label>Package Type</label><input type="text" name="package_type" value="<?= e($request['package_type'] ?? '') ?>" placeholder="e.g., 5 Star, Economy"></div>
                                </div>
                                <div class="form-row">
                                    <div class="form-group"><label>Package Duration (Nights)</label><input type="number" name="package_duration_nights" value="<?= e($request['package_duration_nights'] ?? '') ?>" placeholder="e.g., 21"></div>
                                    <div class="form-group"><label>Pax Summary</label><input type="text" name="pax_summary" value="<?= e($request['pax_summary'] ?? '') ?>" placeholder="e.g., (A-4 : C-2: I-1)"></div>
                                </div>
                            </section>
                            <section class="form-section">
                                <h3><i class="fas fa-calendar-alt"></i> Desired Travel Dates</h3>
                                <div class="form-row">
                                    <div class="form-group"><label>Desired Departure Date</label><input type="date" name="desired_departure_date" value="<?= e($request['desired_departure_date'] ?? '') ?>"></div>
                                    <div class="form-group"><label>Desired Return Date</label><input type="date" name="desired_return_date" value="<?= e($request['desired_return_date'] ?? '') ?>"></div>
                                </div>
                            </section>
                            <section class="form-section">
                                <h3><i class="fas fa-hotel"></i> Accommodation Preferences</h3>
                                <div class="table-responsive">
                                    <table class="dynamic-table">
                                        <thead>
                                            <tr>
                                                <th>City*</th>
                                                <th>Hotel Preference*</th>
                                                <th>Check-In</th>
                                                <th>Nights</th>
                                                <th>Room Type</th>
                                                <th>Meal Plan</th>
                                                <th>Action</th>
                                            </tr>
                                        </thead>
                                        <tbody id="accom-tbody"><?php if ($action === 'edit' && !empty($request['accommodations'])) {
                                                                    foreach ($request['accommodations'] as $accom) {
                                                                        echo '<tr><td><input type="text" name="accom_city[]" value="' . e($accom['city']) . '" required></td><td><textarea name="hotel_preference[]" required>' . e($accom['hotel_preference']) . '</textarea></td><td><input type="date" name="accom_check_in[]" value="' . e($accom['check_in_date']) . '"></td><td><input type="number" name="accom_nights[]" value="' . e($accom['nights']) . '"></td><td><input type="text" name="accom_room_type[]" value="' . e($accom['room_type']) . '"></td><td><input type="text" name="accom_meal_plan[]" value="' . e($accom['meal_plan']) . '"></td><td><button type="button" class="remove-row-btn" onclick="removeRow(this)">X</button></td></tr>';
                                                                    }
                                                                } ?></tbody>
                                    </table>
                                </div><button type="button" class="btn btn-secondary add-row-btn" onclick="addAccommodationRow()">+ Add Accommodation</button>
                            </section>
                            <section class="form-section">
                                <h3><i class="fas fa-users"></i> Traveler Details</h3>
                                <div class="table-responsive">
                                    <table class="dynamic-table">
                                        <thead>
                                            <tr>
                                                <th>Mutamer Name*</th>
                                                <th>Passport No</th>
                                                <th>Pax Type</th>
                                                <th>Bed Required</th>
                                                <th>Action</th>
                                            </tr>
                                        </thead>
                                        <tbody id="mutamers-tbody"><?php if ($action === 'edit' && !empty($request['mutamers'])) {
                                                                        foreach ($request['mutamers'] as $mutamer) {
                                                                            echo '<tr><td><input type="text" name="mutamer_name[]" value="' . e($mutamer['mutamer_name']) . '" required></td><td><input type="text" name="mutamer_passport_no[]" value="' . e($mutamer['passport_no']) . '"></td><td><select name="mutamer_pax_type[]"><option ' . ($mutamer['pax_type'] == 'Adult' ? 'selected' : '') . '>Adult</option><option ' . ($mutamer['pax_type'] == 'Child' ? 'selected' : '') . '>Child</option><option ' . ($mutamer['pax_type'] == 'Infant' ? 'selected' : '') . '>Infant</option></select></td><td><select name="mutamer_bed_required[]"><option ' . ($mutamer['bed_required'] == 'Yes' ? 'selected' : '') . '>Yes</option><option ' . ($mutamer['bed_required'] == 'No' ? 'selected' : '') . '>No</option></select></td><td><button type="button" class="remove-row-btn" onclick="removeRow(this)">X</button></td></tr>';
                                                                        }
                                                                    } ?></tbody>
                                    </table>
                                </div><button type="button" class="btn btn-secondary add-row-btn" onclick="addMutamerRow()">+ Add Traveler</button>
                            </section>
                            <section class="form-section">
                                <h3><i class="fas fa-sticky-note"></i> Special Notes for Admin</h3>
                                <div class="form-group"><label>Include any special requests or instructions here.</label><textarea name="agent_notes" rows="4" placeholder="e.g., 'Client requires a wheelchair accessible room'"><?= e($request['agent_notes'] ?? '') ?></textarea></div>
                            </section>
                            <div class="form-actions"><button type="submit" class="btn btn-primary"><i class="fas fa-paper-plane"></i> <?= ($action === 'create') ? 'Submit Request' : 'Update Request' ?></button></div>
                        </form>

                    <?php elseif ($action === 'view' && $request): ?>
                        <div class="view-section">
                            <h3><i class="fas fa-info-circle"></i> Basic Information</h3>
                            <dl>
                                <dt>Family Head:</dt>
                                <dd><?= e($request['family_head_name']) ?></dd>
                                <dt>Package Type:</dt>
                                <dd><?= e($request['package_type'] ?: 'N/A') ?></dd>
                                <dt>Duration:</dt>
                                <dd><?= e($request['package_duration_nights'] ? $request['package_duration_nights'] . ' Nights' : 'N/A') ?></dd>
                                <dt>Pax Summary:</dt>
                                <dd><?= e($request['pax_summary'] ?: 'N/A') ?></dd>
                            </dl>
                        </div>
                        <div class="view-section">
                            <h3><i class="fas fa-calendar-alt"></i> Desired Travel Dates</h3>
                            <dl>
                                <dt>Departure:</dt>
                                <dd><?= e($request['desired_departure_date'] ? date('M j, Y', strtotime($request['desired_departure_date'])) : 'N/A') ?></dd>
                                <dt>Return:</dt>
                                <dd><?= e($request['desired_return_date'] ? date('M j, Y', strtotime($request['desired_return_date'])) : 'N/A') ?></dd>
                            </dl>
                        </div>
                        <div class="view-section">
                            <h3><i class="fas fa-hotel"></i> Accommodation Preferences</h3><?php if (!empty($request['accommodations'])): ?><div class="table-responsive">
                                    <table class="dynamic-table">
                                        <thead>
                                            <tr>
                                                <th>City</th>
                                                <th>Preference</th>
                                                <th>Check-In</th>
                                                <th>Nights</th>
                                                <th>Room Type</th>
                                                <th>Meal Plan</th>
                                            </tr>
                                        </thead>
                                        <tbody><?php foreach ($request['accommodations'] as $accom): echo '<tr><td>' . e($accom['city']) . '</td><td>' . nl2br(e($accom['hotel_preference'])) . '</td><td>' . e($accom['check_in_date'] ? date('M j, Y', strtotime($accom['check_in_date'])) : 'N/A') . '</td><td>' . e($accom['nights'] ?: 'N/A') . '</td><td>' . e($accom['room_type'] ?: 'N/A') . '</td><td>' . e($accom['meal_plan'] ?: 'N/A') . '</td></tr>';
                                                                                                endforeach; ?></tbody>
                                    </table>
                                </div><?php else: ?><p>No accommodation preferences were specified.</p><?php endif; ?>
                        </div>
                        <div class="view-section">
                            <h3><i class="fas fa-users"></i> Traveler Details</h3><?php if (!empty($request['mutamers'])): ?><div class="table-responsive">
                                    <table class="dynamic-table">
                                        <thead>
                                            <tr>
                                                <th>Name</th>
                                                <th>Passport #</th>
                                                <th>Pax Type</th>
                                                <th>Bed Required</th>
                                            </tr>
                                        </thead>
                                        <tbody><?php foreach ($request['mutamers'] as $mutamer): echo '<tr><td>' . e($mutamer['mutamer_name']) . '</td><td>' . e($mutamer['passport_no'] ?: 'N/A') . '</td><td>' . e($mutamer['pax_type']) . '</td><td>' . e($mutamer['bed_required']) . '</td></tr>';
                                                                                        endforeach; ?></tbody>
                                    </table>
                                </div><?php else: ?><p>No travelers were listed.</p><?php endif; ?>
                        </div>
                        <div class="view-section">
                            <h3><i class="fas fa-sticky-note"></i> Agent Notes</h3>
                            <p><?= nl2br(e($request['agent_notes'] ?: 'No special notes provided.')) ?></p>
                        </div>
                        <div class="view-section">
                            <h3><i class="fas fa-user-shield"></i> Admin Notes</h3>
                            <p><?= nl2br(e($request['admin_notes'] ?: 'No notes from admin yet.')) ?></p>
                        </div>

                    <?php endif; ?>
                </div>
            </div>
        </div>
    </main>
    <?php include 'footer.php'; ?>
    <script>
        function removeRow(btn) {
            btn.closest('tr').remove();
        }

        function addMutamerRow() {
            const tbody = document.getElementById('mutamers-tbody');
            if (!tbody) return;
            const row = tbody.insertRow();
            row.innerHTML = `<td><input type="text" name="mutamer_name[]" required></td><td><input type="text" name="mutamer_passport_no[]"></td><td><select name="mutamer_pax_type[]"><option selected>Adult</option><option>Child</option><option>Infant</option></select></td><td><select name="mutamer_bed_required[]"><option selected>Yes</option><option>No</option></select></td><td><button type="button" class="remove-row-btn" onclick="removeRow(this)">X</button></td>`;
        }

        function addAccommodationRow() {
            const tbody = document.getElementById('accom-tbody');
            if (!tbody) return;
            const row = tbody.insertRow();
            row.innerHTML = `<td><input type="text" name="accom_city[]" required></td><td><textarea name="hotel_preference[]" required placeholder="e.g., Movenpick Hajar or similar"></textarea></td><td><input type="date" name="accom_check_in[]"></td><td><input type="number" name="accom_nights[]"></td><td><input type="text" name="accom_room_type[]" placeholder="e.g., Quad"></td><td><input type="text" name="accom_meal_plan[]" placeholder="e.g., RO, BB"></td><td><button type="button" class="remove-row-btn" onclick="removeRow(this)">X</button></td>`;
        }

        document.addEventListener('DOMContentLoaded', function() {
            const requestRows = document.querySelectorAll('.request-row');
            requestRows.forEach(row => {
                row.addEventListener('click', function(event) {
                    if (event.target.closest('a')) return;
                    const requestId = this.dataset.requestId;
                    if (requestId) window.location.href = `?action=view&id=${requestId}`;
                });
            });

            // If on create form, add initial rows
            if (document.querySelector('form input[name="form_action"][value="create_request"]')) {
                addMutamerRow();
                addAccommodationRow();
            }
        });
    </script>
</body>

</html>