<?php
session_start();
include 'db-config.php';

// --- SECURITY CHECK 1: Is user logged in? ---
if (!isset($_SESSION['user_id'])) {
    header("location: login.php");
    exit;
}

// --- SECURITY CHECK 2: Is the logged-in user a CUSTOMER? ---
if (isset($_SESSION['user_type']) && $_SESSION['user_type'] === 'agent') {
    header("location: agent-dashboard.php");
    exit;
}

$user_id = $_SESSION['user_id'];

// --- Get the logged-in user's full details ---
$stmt_user = $conn->prepare("SELECT name, email FROM users WHERE id = ?");
$stmt_user->bind_param("i", $user_id);
$stmt_user->execute();
$current_user = $stmt_user->get_result()->fetch_assoc();
$stmt_user->close();

$user_name = $current_user['name'] ?? 'Customer';
$user_email = $current_user['email'] ?? 'N/A';

// --- 1. FETCH ALL UMRAH/PACKAGE INVOICE DATA ---
$invoices_data = [];
$sql_invoices = "SELECT *, 'package' as invoice_type FROM invoices WHERE user_id = ?";
$stmt_invoices = $conn->prepare($sql_invoices);
$stmt_invoices->bind_param("i", $user_id);
$stmt_invoices->execute();
$invoices_result = $stmt_invoices->get_result();

while ($invoice = $invoices_result->fetch_assoc()) {
    $current_invoice_id = $invoice['id'];
    $invoice['pilgrims'] = $conn->query("SELECT * FROM invoice_pilgrims WHERE invoice_id = $current_invoice_id")->fetch_all(MYSQLI_ASSOC);
    $invoice['hotels'] = $conn->query("SELECT * FROM invoice_hotels WHERE invoice_id = $current_invoice_id")->fetch_all(MYSQLI_ASSOC);
    $invoice['transports'] = $conn->query("SELECT * FROM invoice_transports WHERE invoice_id = $current_invoice_id")->fetch_all(MYSQLI_ASSOC);
    $invoice['other_services'] = $conn->query("SELECT * FROM invoice_other_services WHERE invoice_id = $current_invoice_id")->fetch_all(MYSQLI_ASSOC);
    $invoice['airline_tickets'] = $conn->query("SELECT * FROM invoice_airline_tickets WHERE invoice_id = $current_invoice_id")->fetch_all(MYSQLI_ASSOC);
    $payments = $conn->query("SELECT payment_amount FROM payments WHERE invoice_id = $current_invoice_id AND invoice_type = 'package'")->fetch_all(MYSQLI_ASSOC);
    $amount_paid = array_sum(array_column($payments, 'payment_amount'));
    $invoice['amount_paid'] = $amount_paid;
    $invoice['amount_due'] = $invoice['grand_total_pkr'] - $amount_paid;
    $invoices_data[] = $invoice;
}
$stmt_invoices->close();

// --- 2. FETCH ALL FLIGHT TICKET INVOICE DATA ---
$sql_ticket_invoices = "SELECT *, 'ticket' as invoice_type FROM ticket_invoices WHERE user_id = ?";
$stmt_ticket_invoices = $conn->prepare($sql_ticket_invoices);
$stmt_ticket_invoices->bind_param("i", $user_id);
$stmt_ticket_invoices->execute();
$ticket_invoices_result = $stmt_ticket_invoices->get_result();

while ($ticket_invoice = $ticket_invoices_result->fetch_assoc()) {
    $current_ticket_invoice_id = $ticket_invoice['id'];
    $ticket_invoice['passengers'] = $conn->query("SELECT * FROM ticket_invoice_passengers WHERE ticket_invoice_id = $current_ticket_invoice_id")->fetch_all(MYSQLI_ASSOC);
    $ticket_invoice['flights'] = $conn->query("SELECT * FROM ticket_invoice_flights WHERE ticket_invoice_id = $current_ticket_invoice_id")->fetch_all(MYSQLI_ASSOC);
    $payments = $conn->query("SELECT payment_amount FROM payments WHERE invoice_id = $current_ticket_invoice_id AND invoice_type = 'ticket'")->fetch_all(MYSQLI_ASSOC);
    $amount_paid = array_sum(array_column($payments, 'payment_amount'));
    $ticket_invoice['amount_paid'] = $amount_paid;
    $ticket_invoice['amount_due'] = $ticket_invoice['grand_total_pkr'] - $amount_paid;
    $invoices_data[] = $ticket_invoice; // Add to the same array
}
$stmt_ticket_invoices->close();

// --- 3. SORT ALL INVOICES BY DATE ---
usort($invoices_data, function ($a, $b) {
    return strtotime($b['issue_date']) - strtotime($a['issue_date']);
});

?>
<!DOCTYPE html>
<html>

<head>
    <title>My Invoices</title>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <link rel="stylesheet" href="css/style.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.2.0/css/all.min.css">
    <link rel="stylesheet" href="css/account-style.css">

    <style>
        .modal-body {
            font-family: Arial, sans-serif;
            font-size: 9pt;
            color: #000;
        }

        .modal-body .invoice-wrapper {
            position: relative;
            max-width: 100%;
            box-shadow: none;
            border: none;
            padding: 20px;
        }

        .modal-body table {
            width: 100%;
            border-collapse: collapse;
        }

        .modal-body td,
        .modal-body th {
            padding: 4px;
            vertical-align: middle;
        }

        .modal-body .header-table td {
            border: none;
            padding: 0;
            vertical-align: top;
        }

        .modal-body .company-logo-container {
            text-align: center;
        }

        .modal-body .company-logo-container img {
            max-height: 50px;
        }

        .modal-body .company-details {
            font-size: 9pt;
            line-height: 1.4;
            padding-top: 5px;
        }

        .modal-body .meta-table td {
            background-color: #f7b731;
            border: 1px solid #000;
            padding: 5px 8px;
            font-weight: bold;
        }

        .modal-body .meta-table td:first-child {
            width: 100px;
        }

        .modal-body .meta-table td:last-child {
            background-color: #fff;
            text-align: center;
        }

        .modal-body .section-title {
            background-color: black;
            color: white;
            font-weight: bold;
            text-align: center;
            border: 1px solid #000;
            padding: 5px;
            margin-top: 10px;
            font-size: 10pt;
        }

        .modal-body .detail-table {
            border: 1px solid #000;
            margin-bottom: 5px;
        }

        .modal-body .detail-table th {
            background-color: #f7b731;
            border: 1px solid #000;
            font-weight: bold;
            padding: 5px;
        }

        .modal-body .detail-table td {
            border: 1px solid #000;
            padding: 4px;
            text-align: center;
        }

        .modal-body .detail-table .text-left {
            text-align: left;
            padding-left: 5px;
        }

        .modal-body .total-row td {
            border: 1px solid #000;
            font-weight: bold;
        }

        .modal-body .total-row .total-label {
            text-align: right;
            padding-right: 15px;
        }

        .modal-body .total-row .total-value {
            background-color: #f7b731;
            text-align: center;
            font-size: 10pt;
        }

        .modal-body .footer-container {
            padding-top: 15px;
            display: flex;
            justify-content: space-between;
            align-items: flex-start;
            gap: 20px;
        }

        .modal-body .footer-notes {
            flex: 1 1 55%;
            font-size: 8pt;
            line-height: 1.5;
        }

        .modal-body .footer-notes h4 {
            margin-top: 0;
            margin-bottom: 5px;
            font-size: 9pt;
        }

        .modal-body .summary-totals {
            flex: 0 0 43%;
        }

        .modal-body .summary-totals table td {
            border: 1px solid #000;
            padding: 6px 10px;
            font-size: 10pt;
        }

        .modal-body .summary-totals table td:first-child {
            font-weight: bold;
            width: 65%;
        }

        .modal-body .summary-totals table td:last-child {
            text-align: right;
            font-weight: normal;
        }

        .modal-body .summary-totals .grand-total td {
            background-color: #f7b731;
            font-weight: bold;
        }

        .modal-body .summary-totals .discount-row td {
            font-weight: bold;
            color: #27ae60;
        }

        .modal-body .summary-totals .payment-received-row td {
            font-weight: bold;
            color: #d35400;
        }

        .modal-body .summary-totals .remaining-amount-row td {
            background-color: #c0392b;
            color: white;
            font-weight: bold;
        }

        .modal-body .fare-breakdown-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
            gap: 15px;
            margin-top: 5px;
            padding: 10px;
            border: 1px solid #000;
            background-color: rgba(0, 0, 0, 0.02);
        }

        .modal-body .fare-box {
            border: 1px solid #b0b0b0;
            background-color: #fdfdfd;
            border-radius: 4px;
            overflow: hidden;
        }

        .modal-body .fare-box-header {
            background-color: #f7b731;
            padding: 8px;
            font-weight: bold;
            text-align: center;
            font-size: 11pt;
            color: #000;
        }

        .modal-body .fare-box-content {
            padding: 10px;
            display: flex;
            justify-content: space-around;
            gap: 10px;
        }

        .modal-body .fare-detail {
            text-align: center;
            flex: 1;
            padding: 8px 5px;
            background-color: #f0f0f0;
            border: 1px solid #ddd;
            border-radius: 3px;
        }

        .modal-body .fare-detail .label {
            display: block;
            font-size: 8pt;
            color: #555;
            margin-bottom: 4px;
            text-transform: uppercase;
        }

        .modal-body .fare-detail .value {
            display: block;
            font-weight: bold;
            font-size: 12pt;
            color: #000;
        }

        .whatsapp-number {
            display: block;
            background-color: #e6f8f0;
            border: 1px solid #a3e9c5;
            padding: 1rem;
            margin: 1rem 0;
            text-align: center;
            font-size: 1.2rem;
            font-weight: bold;
            border-radius: 8px;
            color: #155724;
        }

        .whatsapp-number .fa-whatsapp {
            color: #25D366;
            margin-right: 0.5rem;
        }


        .payment-details-section {
            margin-top: 30px;
        }

        .payment-details-section h4 {
            font-size: 1.1em;
            margin-bottom: 15px;
            padding-bottom: 10px;
            border-bottom: 1px solid #eee;
        }

        .bank-accounts-container {
            display: flex;
            gap: 20px;
            margin-bottom: 20px;
            flex-wrap: wrap;
        }

        .bank-account-box {
            flex: 1;
            min-width: 300px;
            border: 1px solid #ddd;
            padding: 15px;
            border-radius: 5px;
            background-color: #fdfdfd;
            text-align: center;
        }

        .bank-logo {
            max-height: 100px;
            margin-bottom: 15px;
        }

        .bank-details-table {
            width: 100%;
            text-align: left;
        }

        .bank-details-table td {
            border: none;
            padding: 5px 0;
            font-size: 0.9em;
            vertical-align: top;
        }

        .bank-details-table td:first-child {
            font-weight: bold;
            width: 100px;
            color: #555;
        }

        .receipt-instruction {
            margin-top: 15px;
            text-align: center;
            font-size: 1em;
            font-weight: bold;
            background-color: #fffde7;
            padding: 10px;
            border: 1px dashed #fbc02d;
            border-radius: 4px;
        }

        .content-card h2 {
            margin-bottom: 5px;
        }

        .content-card .content-description {
            margin-top: 0;
        }

        .content-card+.content-card {
            margin-top: 30px;
        }
        
        /* === RESPONSIVE TABLE CSS === */
        @media screen and (max-width: 768px) {
            .bookings-table thead {
                display: none;
            }

            .bookings-table,
            .bookings-table tbody,
            .bookings-table tr,
            .bookings-table td {
                display: block;
                width: 100%;
            }

            .bookings-table tr {
                margin-bottom: 15px;
                border: 1px solid var(--border-color);
                border-radius: 8px;
                overflow: hidden;
                cursor: pointer;
            }

            .bookings-table td {
                padding-left: 50%;
                text-align: right;
                position: relative;
                border: none;
                border-bottom: 1px solid var(--border-color-light);
            }

            .bookings-table td:last-child {
                border-bottom: 0;
            }

            .bookings-table td::before {
                content: attr(data-label);
                position: absolute;
                left: 15px;
                width: calc(50% - 30px);
                padding-right: 10px;
                white-space: nowrap;
                text-align: left;
                font-weight: bold;
                color: var(--text-dark);
            }
        }
    </style>
</head>

<body>

    <?php include 'header.php'; ?>

    <main class="account-page-wrapper">
        <div class="account-container">
            <?php include 'customer-sidebar.php'; ?>
            <div class="account-content">
                <div class="content-card">
                    <h2><i class="fa-solid fa-file-invoice"></i> My Invoices</h2>
                    <p class="content-description">Here is a list of your invoices. Click on a row to see the full details.</p>
                    <div class="table-responsive">
                        <table class="bookings-table">
                            <thead>
                                <tr>
                                    <th>Invoice #</th>
                                    <th>Type</th>
                                    <th>Issue Date</th>
                                    <th>Status</th>
                                    <th>Total</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php if (!empty($invoices_data)): ?>
                                    <?php foreach ($invoices_data as $invoice): ?>
                                        <tr class="invoice-row" data-invoice-details='<?= htmlspecialchars(json_encode($invoice), ENT_QUOTES, 'UTF-8'); ?>' data-type="<?= $invoice['invoice_type'] ?>">
                                            <td data-label="Invoice #"><strong>#<?= htmlspecialchars($invoice['invoice_number'] ?: $invoice['id']) ?></strong></td>
                                            <td data-label="Type"><?= ucfirst($invoice['invoice_type']) ?></td>
                                            <td data-label="Issue Date"><?= date('M j, Y', strtotime($invoice['issue_date'])) ?></td>
                                            <td data-label="Status"><span class="status-badge status-<?= strtolower(str_replace(' ', '-', $invoice['status'])) ?>"><?= htmlspecialchars(ucfirst($invoice['status'])) ?></span></td>
                                            <td data-label="Total">PKR <?= number_format($invoice['grand_total_pkr'], 2) ?></td>
                                        </tr>
                                    <?php endforeach; ?>
                                <?php else: ?>
                                    <tr>
                                        <td colspan="5" class="no-bookings-found">You have no invoices.</td>
                                    </tr>
                                <?php endif; ?>
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>
        </div>
    </main>

    <?php include 'floating-icon.php'; ?>
    <?php include 'footer.php'; ?>

    <!-- MODAL HTML STRUCTURE -->
    <div class="modal-overlay" id="invoice-modal-overlay">
        <div class="modal-content" id="invoice-modal-content">
            <button class="modal-close-btn" id="modal-close-btn"><i class="fa-solid fa-times"></i></button>
            <div class="modal-body" id="modal-body"></div>
        </div>
    </div>

    <script>
        // This is the same modern, working JavaScript from agent-invoices.php
        document.addEventListener('DOMContentLoaded', function() {
            const modalOverlay = document.getElementById('invoice-modal-overlay');
            const modalBody = document.getElementById('modal-body');
            const closeModalBtn = document.getElementById('modal-close-btn');
            const invoiceRows = document.querySelectorAll('.invoice-row');

            const nf = (num) => (parseFloat(num) || 0).toLocaleString('en-US', {
                minimumFractionDigits: 0,
                maximumFractionDigits: 0
            });
            const nf_decimal = (num) => {
                const formatted = (parseFloat(num) || 0).toLocaleString('en-US', {
                    minimumFractionDigits: 2,
                    maximumFractionDigits: 2
                });
                return formatted.endsWith('.00') ? formatted.slice(0, -3) : formatted;
            };
            const formatDate = (dateStr) => {
                if (!dateStr || dateStr === '0000-00-00') return 'N/A';
                const months = ["Jan", "Feb", "Mar", "Apr", "May", "Jun", "Jul", "Aug", "Sep", "Oct", "Nov", "Dec"];
                const d = new Date(dateStr);
                return `${('0' + d.getDate()).slice(-2)}-${months[d.getMonth()]}-${d.getFullYear()}`;
            };
            const formatDateTime = (dateStr) => {
                if (!dateStr || dateStr === '0000-00-00 00:00:00') return 'N/A';
                const d = new Date(dateStr);
                const datePart = `${('0' + d.getDate()).slice(-2)}-${['Jan','Feb','Mar','Apr','May','Jun','Jul','Aug','Sep','Oct','Nov','Dec'][d.getMonth()]}-${d.getFullYear().toString().slice(-2)}`;
                const timePart = `${('0' + d.getHours()).slice(-2)}:${('0' + d.getMinutes()).slice(-2)}`;
                return `${datePart} ${timePart}`;
            }

            function openModal() {
                modalOverlay.classList.add('is-visible');
                document.body.style.overflow = 'hidden';
            }

            function closeModal() {
                modalOverlay.classList.remove('is-visible');
                document.body.style.overflow = '';
                modalBody.innerHTML = '';
            }

            invoiceRows.forEach(row => {
                row.addEventListener('click', function(event) {
                    try {
                        const invoiceData = JSON.parse(this.dataset.invoiceDetails);
                        const invoiceType = this.dataset.type;

                        if (invoiceType === 'ticket') {
                            populateTicketModal(invoiceData);
                        } else {
                            populateUmrahModal(invoiceData);
                        }
                        openModal();
                    } catch (e) {
                        console.error("Failed to parse invoice data:", e);
                        alert("An error occurred while loading invoice details.");
                    }
                });
            });

            if (closeModalBtn) closeModalBtn.addEventListener('click', closeModal);
            if (modalOverlay) modalOverlay.addEventListener('click', function(event) {
                if (event.target === modalOverlay) closeModal();
            });

            function getPaymentSummaryHtml(invoice) {
                if (invoice.amount_paid > 0) {
                    return `
                <tr class="payment-received-row">
                    <td>Payment Received</td>
                    <td>- ${nf_decimal(invoice.amount_paid)}</td>
                </tr>
                <tr class="remaining-amount-row">
                    <td>Remaining Amount</td>
                    <td>${nf_decimal(invoice.amount_due)}</td>
                </tr>
            `;
                }
                return '';
            }

            function populateTicketModal(invoice) {
                const invoice_display_number = invoice.invoice_number || invoice.id;

                let passengersHtml = '';
                if (invoice.passengers && invoice.passengers.length > 0) {
                    passengersHtml = `<div class="section-title">Passenger Information</div><table class="detail-table"><thead><tr><th>Sr.</th><th>Full Name</th><th>Type</th><th>Passport No.</th><th>PNR</th><th>Ticket No.</th></tr></thead><tbody>
            ${invoice.passengers.map((p, i) => `<tr><td>${i+1}</td><td class="text-left">${p.full_name}</td><td>${p.passenger_type}</td><td>${p.passport_no}</td><td>${p.pnr}</td><td>${p.ticket_number}</td></tr>`).join('')}
            </tbody></table>`;
                }

                let flightsHtml = '';
                if (invoice.flights && invoice.flights.length > 0) {
                    flightsHtml = `<div class="section-title">Flight Itinerary</div><table class="detail-table"><thead><tr><th>Airline</th><th>Flight No.</th><th>Sector</th><th>Departure</th><th>Arrival</th></tr></thead><tbody>
            ${invoice.flights.map(f => `<tr><td class="text-left">${f.airline}</td><td>${f.flight_no}</td><td>${f.sector}</td><td>${formatDateTime(f.departure_datetime)}</td><td>${formatDateTime(f.arrival_datetime)}</td></tr>`).join('')}
            </tbody></table>`;
                }

                let fareBreakdownHtml = '';
                if (invoice.adult_qty > 0 || invoice.child_qty > 0 || invoice.infant_qty > 0) {
                    fareBreakdownHtml += '<div class="section-title">Fare Breakdown</div><div class="fare-breakdown-grid">';
                    if (invoice.adult_qty > 0) {
                        fareBreakdownHtml += `<div class="fare-box"><div class="fare-box-header">Adults</div><div class="fare-box-content"><div class="fare-detail"><span class="label">Quantity</span><span class="value">${nf(invoice.adult_qty)}</span></div><div class="fare-detail"><span class="label">Rate (PKR)</span><span class="value">${nf_decimal(invoice.adult_rate)}</span></div></div></div>`;
                    }
                    if (invoice.child_qty > 0) {
                        fareBreakdownHtml += `<div class="fare-box"><div class="fare-box-header">Children</div><div class="fare-box-content"><div class="fare-detail"><span class="label">Quantity</span><span class="value">${nf(invoice.child_qty)}</span></div><div class="fare-detail"><span class="label">Rate (PKR)</span><span class="value">${nf_decimal(invoice.child_rate)}</span></div></div></div>`;
                    }
                    if (invoice.infant_qty > 0) {
                        fareBreakdownHtml += `<div class="fare-box"><div class="fare-box-header">Infants</div><div class="fare-box-content"><div class="fare-detail"><span class="label">Quantity</span><span class="value">${nf(invoice.infant_qty)}</span></div><div class="fare-detail"><span class="label">Rate (PKR)</span><span class="value">${nf_decimal(invoice.infant_rate)}</span></div></div></div>`;
                    }
                    fareBreakdownHtml += '</div>';
                }

                const summaryHtml = `
            <table>
                <tr><td>Total Fare (PKR)</td><td>${nf_decimal(invoice.total_fare_pkr)}</td></tr>
                <tr><td>Service Fee (PKR)</td><td>${nf_decimal(invoice.service_fee_pkr)}</td></tr>
                ${invoice.discount_pkr > 0 ? `<tr class="discount-row"><td>Discount (PKR)</td><td>- ${nf_decimal(invoice.discount_pkr)}</td></tr>` : ''}
                <tr class="grand-total"><td>Grand Total (PKR)</td><td>${nf_decimal(invoice.grand_total_pkr)}</td></tr>
                ${getPaymentSummaryHtml(invoice)}
            </table>`;

                modalBody.innerHTML = `
        <div class="invoice-wrapper">
            <div class="invoice-content">
                <header><table class="header-table"><tr>
                    <td style="width: 33%;"></td>
                    <td style="width: 34%;" class="company-logo-container">
                        <img src="images/logo.png" alt="Logo">
                        <div class="company-details">AL Quresh Near Railway Pahatak,  Infront of Al Quresh Housing Scheme Sher Shah Road Multan<br>Mob: 0092 305 23 94 810, 0092 305 23 94 810 UAN</div>
                    </td>
                    <td style="width: 33%;"><table class="meta-table">
                        <tr><td>Invoice No:</td><td>${invoice_display_number}</td></tr>
                        <tr><td>Guest Name:</td><td>${invoice.guest_name}</td></tr>
                        <tr><td>Dated:</td><td>${formatDate(invoice.issue_date)}</td></tr>
                    </table></td>
                </tr></table></header>
                <main>${passengersHtml}${flightsHtml}${fareBreakdownHtml}</main>
                <footer class="footer-container">
                    <div class="footer-notes">${invoice.notes ? `<h4>Terms & Conditions:</h4><p>${invoice.notes.replace(/\n/g, '<br>')}</p>` : ''}</div>
                    <div class="summary-totals">${summaryHtml}</div>
                </footer>
            </div>
        </div>`;
            }

            function populateUmrahModal(invoice) {
                const invoice_display_number = invoice.invoice_number || invoice.id;
                const total_sar = invoice.pilgrims.reduce((sum, p) => sum + parseFloat(p.visa_price_sar), 0) + invoice.hotels.reduce((sum, h) => sum + parseFloat(h.total_sar), 0) + invoice.transports.reduce((sum, t) => sum + parseFloat(t.total_amount), 0) + invoice.other_services.reduce((sum, s) => sum + parseFloat(s.total_amount), 0);
                let pilgrimsHtml = '',
                    hotelsHtml = '',
                    transportsHtml = '',
                    servicesHtml = '';

                if (invoice.pilgrims && invoice.pilgrims.length > 0) {
                    pilgrimsHtml = `<div class="section-title">Pilgrims Detail</div><table class="detail-table"><thead><tr><th>PASSPORT NO</th><th>PASSANGER DETAILS</th><th>Visa</th></tr></thead><tbody>
            ${invoice.pilgrims.map(p => `<tr><td>${p.passport_no}</td><td class="text-left">${p.passenger_details}</td><td>${nf(p.visa_price_sar)}</td></tr>`).join('')}
            </tbody></table>`;
                }
                if (invoice.hotels && invoice.hotels.length > 0) {
                    hotelsHtml = `<div class="section-title">Accommodation</div><table class="detail-table"><thead><tr><th>City</th><th>Hotel Name</th><th>Checkin</th><th>Nights</th><th>Checkout</th><th>Room</th><th>Meal</th><th>Net Amount</th></tr></thead><tbody>
            ${invoice.hotels.map(h => `<tr><td>${h.city}</td><td class="text-left">${h.hotel_name}</td><td>${formatDate(h.check_in)}</td><td>${h.nights}</td><td>${formatDate(h.check_out)}</td><td>${h.room_type}</td><td>${h.meal_plan}</td><td>${nf(h.total_sar)}</td></tr>`).join('')}
            </tbody></table>`;
                }
                if (invoice.transports && invoice.transports.length > 0) {
                    transportsHtml = `<div class="section-title">Transportation</div><table class="detail-table"><thead><tr><th>Vehical Type</th><th>Route</th><th>Net Amount</th></tr></thead><tbody>
            ${invoice.transports.map(t => `<tr><td class="text-left">${t.vehicle_type}</td><td class="text-left">${t.route}</td><td>${nf(t.total_amount)}</td></tr>`).join('')}
            </tbody></table>`;
                }
                if (invoice.other_services && invoice.other_services.length > 0) {
                    servicesHtml = `<div class="section-title">Other Services</div><table class="detail-table"><thead><tr><th>Service Name</th><th>Net Amount</th></tr></thead><tbody>
            ${invoice.other_services.map(s => `<tr><td class="text-left">${s.service_name}</td><td>${nf(s.total_amount)}</td></tr>`).join('')}
            </tbody></table>`;
                }

                modalBody.innerHTML = `
        <div class="invoice-wrapper">
            <div class="invoice-content">
                 <header><table class="header-table"><tr>
                    <td style="width: 33%;"></td>
                    <td style="width: 34%;" class="company-logo-container">
                        <img src="images/logo.png" alt="Logo">
                       <div class="company-details">AL Quresh Near Railway Pahatak,  Infront of Al Quresh Housing Scheme Sher Shah Road Multan<br>Mob: 0092 305 23 94 810, 0092 305 23 94 810 UAN</div>
                    </td>
                    <td style="width: 33%;"><table class="meta-table">
                        <tr><td>Invoice No:</td><td>${invoice_display_number}</td></tr>
                        <tr><td>Guest Name:</td><td>${invoice.guest_name}</td></tr>
                        <tr><td>Dated:</td><td>${formatDate(invoice.issue_date)}</td></tr>
                    </table></td>
                </tr></table></header>
                <main>${pilgrimsHtml}${hotelsHtml}${transportsHtml}${servicesHtml}</main>
                <footer class="footer-container">
                    <div class="footer-notes">${invoice.notes ? `<h4>Terms & Conditions:</h4><p>${invoice.notes.replace(/\n/g, '<br>')}</p>` : ''}</div>
                    <div class="summary-totals">
                        <table>
                            <tr><td>Total Amount (SAR)</td><td>${nf(total_sar)}</td></tr>
                            <tr><td>Exchange Rate:</td><td>${nf_decimal(invoice.exchange_rate)}</td></tr>
                            <tr><td>Total Amount (Pak Rs)</td><td>${nf(invoice.total_pkr_without_ticket)}</td></tr>
                            <tr class="grand-total"><td>Grand Total</td><td>${nf(invoice.grand_total_pkr)}</td></tr>
                            ${getPaymentSummaryHtml(invoice)}
                        </table>
                    </div>
                </footer>
            </div>
        </div>`;
            }
        });
    </script>

</body>

</html>