<?php
// Start session before any output
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

// Establishes the database connection ($conn)
require_once 'db-config.php';

// --- HELPER FUNCTION ---
// Safe htmlspecialchars function to prevent errors with null values
function safe_htmlspecialchars($string) {
    return htmlspecialchars($string ?? '', ENT_QUOTES, 'UTF-8');
}

// ===================================================================
//  1. PREPARE AGENT/COMPANY DATA FOR LOGOS AND FOOTER
//  This data will be used by the included 'all-visa-services.php' file
// ===================================================================
$is_agent = (isset($_SESSION['user_type']) && $_SESSION['user_type'] === 'agent');
$footer_logo = 'images/logo.png'; // Default company logo
$logo_style_class = 'white-bg-logo'; // Default style
$footer_address = 'Your Company Address, City, Country'; // Default address
$footer_phone_1 = '+92 123 4567890'; // Default phone
$footer_phone_2 = ''; // Default second phone (optional)

if ($is_agent && isset($_SESSION['user_id'])) {
    // Fetch the logged-in agent's specific details
    $agent_id = $_SESSION['user_id'];
    $stmt = $conn->prepare("SELECT logo_path, logo_background_style, company_address, company_phone_1, company_phone_2 FROM users WHERE id = ?");
    $stmt->bind_param("i", $agent_id);
    $stmt->execute();
    $result = $stmt->get_result();
    if ($agent_data = $result->fetch_assoc()) {
        if (!empty($agent_data['logo_path'])) {
            $footer_logo = 'admin/' . $agent_data['logo_path']; // Prepend folder path
        }
        if (!empty($agent_data['logo_background_style']) && $agent_data['logo_background_style'] === 'transparent') {
            $logo_style_class = 'transparent-logo';
        }
        if (!empty($agent_data['company_address'])) {
            $footer_address = $agent_data['company_address'];
        }
        if (!empty($agent_data['company_phone_1'])) {
            $footer_phone_1 = $agent_data['company_phone_1'];
        }
        if (!empty($agent_data['company_phone_2'])) {
            $footer_phone_2 = $agent_data['company_phone_2'];
        }
    }
    $stmt->close();
}

// ===================================================================
//  2. FETCH CITIES AND FILTERED VISA SERVICES FOR THE PAGE
//  The $visa_services_result variable will be used by the included file
// ===================================================================
$cities_result = $conn->query("SELECT name FROM cities ORDER BY name ASC");
$selected_city = isset($_GET['city']) ? trim($_GET['city']) : null;

// Base SQL query
$sql = "SELECT * FROM visas WHERE is_active = 1";
$params = [];
$types = '';

// Add city filter if selected
if ($selected_city && $selected_city !== 'all') {
    $sql .= " AND city = ?";
    $params[] = $selected_city;
    $types .= 's';
}

$sql .= " ORDER BY id DESC";

// Prepare and execute the query
$stmt = $conn->prepare($sql);
if ($stmt) {
    if (!empty($params)) {
        $stmt->bind_param($types, ...$params);
    }
    $stmt->execute();
    $visa_services_result = $stmt->get_result();
} else {
    // Handle query preparation error
    $visa_services_result = null;
    error_log("Failed to prepare visa services query: " . $conn->error);
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Visa Services - RF Travel & Tours</title>
    <link rel="icon" type="image/png" href="images/logo-icon.png">
    <link rel="stylesheet" href="css/style.css">
    <link rel="preconnect" href="https://fonts.googleapis.com">
    <link rel="preconnect" href="https://fonts.gstatic.com" crossorigin>
    <link href="https://fonts.googleapis.com/css2?family=Poppins:wght@400;500;600;700;800&display=swap" rel="stylesheet">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.1/css/all.min.css">
    
    <style>
        /* This style block only contains styles for the Hero section and page layout */
        /* All card-specific styles are now in the `all-visa-services.php` file */

        /* --- Section Styling --- */
        .visa-listings-section { padding: 60px 20px; background-color: #f8f9fa; }
        .listings-container { max-width: 1200px; margin: 0 auto; }
        .listings-header { text-align: center; margin-bottom: 30px; }
        .listings-title { font-size: 2.8rem; font-weight: 700; color: #2c3e50; margin-bottom: 10px; }
        .listings-subtitle { font-size: 1.1rem; color: #7f8c8d; }

        /* --- City Filter Navigation --- */
        .city-filter-nav { display: flex; flex-wrap: wrap; justify-content: center; gap: 10px; margin-bottom: 40px; }
        .filter-btn { padding: 10px 20px; background-color: #fff; color: #34495e; border: 1px solid #ddd; border-radius: 30px; text-decoration: none; font-weight: 500; transition: all 0.3s ease; }
        .filter-btn:hover { background-color: #f0f0f0; border-color: #ccc; }
        .filter-btn.active { background-color: #6c757d; color: #fff; border-color: #6c757d; }

        .hero-section {
    background-image: linear-gradient(rgba(0, 0, 0, 0.5), rgba(0, 0, 0, 0.5)), url('./images/bg.jpg');
    background-size: cover;
    background-position: center;
    background-repeat: no-repeat;
    position: relative;
    color: white;
    padding-top: 150px;
    padding-bottom: 100px;
    text-align: center;
}

/* ===== MOBILE RESPONSIVE ===== */
@media (max-width: 768px) {
    .hero-section {
        padding-top: 100px;
        padding-bottom: 80px;
        background-position: top center;
    }
}

@media (max-width: 480px) {
    .hero-section {
        padding-top: 80px;
        padding-bottom: 60px;
        background-image: linear-gradient(rgba(0, 0, 0, 0.6), rgba(0, 0, 0, 0.6)), url('./images/bg.jpg');
        background-position: center top;
        text-align: center;
    }

    .hero-section h1, 
    .hero-section p {
        font-size: 0.9rem;
        line-height: 1.4;
        padding: 0 1rem;
    }
}


        /* This rule removes the width limit from the search container to match the visa section below */
        .search-container {
            max-width: 1700px; /* Increased to match the new visa section width */
            padding-left: 40px;
            padding-right: 40px;
            box-sizing: border-box;
            margin: 0 auto;
            margin-top: -150px; /* Ensures it stays centered */
        }

        @media (max-width: 768px) {
            .search-container {
                padding-left: 20px;
                padding-right: 20px;
            }
        }
    </style>
</head>

<body>

    <?php include 'header.php'; ?>

    <main>
        <section class="hero-section">
            <div class="hero-content">
                <h1>Discover the World <br> and Its Wonders</h1>
                <p>Embark on journeys that inspire the soul and create unforgettable memories beyond imagination.</p>
            </div>
            <div class="search-container">
                <div class="tabs-wrapper">
                    <div class="service-tabs">
                        <a href="index.php" class="tab"><i class="fa-solid fa-plane-up"></i> Ticket</a>
                        <a href="group-fares.php" class="tab"><i class="fa-solid fa-users"></i> Groups</a>
                        <a href="umrah-packages.php" class="tab"><i class="fa-solid fa-kaaba"></i> Umrah</a>
                        <a href="hotels.php" class="tab"><i class="fa-solid fa-hotel"></i> Hotels</a>
                        <a href="visa-services.php" class="tab active"><i class="fa-solid fa-passport"></i> Visas</a>
                        <a href="custom-package.php" class="tab"><i class="fa-solid fa-kaaba"></i> Custom Package</a>
                    </div>
                </div>
                <div class="search-form-wrapper">
                    <div class="umrah-welcome-text">
                        <h2>Simplify your travel with <strong>Amere Taiba Travel & Tours</strong></h2>
                        <p> Experience seamless, reliable, and hassle-free visa processing for top destinations across the globe.</p>
                        <p class="guarantee-line">Your Gateway to Global Destinations 🌍</p>
                    </div>
                </div>
            </div>
        </section>
        
        <!-- This is where the card display component will be included -->
        <?php include 'all-visa-services.php'; ?>

        <?php include 'floating-icon.php'; ?>
    </main>
    
    <?php include 'footer.php'; ?>

    <!-- Make sure html2canvas script is available -->
    <script src="js/html2canvas.min.js"></script>
    <script>
        document.addEventListener('DOMContentLoaded', function() {
            // Sidebar toggle logic
            const menuToggle = document.querySelector('.menu-toggle');
            const mobileSidebar = document.querySelector('.mobile-sidebar');
            const sidebarOverlay = document.querySelector('.sidebar-overlay');
            if (menuToggle && mobileSidebar && sidebarOverlay) {
                menuToggle.addEventListener('click', () => {
                    mobileSidebar.classList.add('active');
                    sidebarOverlay.classList.add('active');
                });
                const closeSidebar = () => {
                    mobileSidebar.classList.remove('active');
                    sidebarOverlay.classList.remove('active');
                };
                document.querySelector('.sidebar-close-btn').addEventListener('click', closeSidebar);
                sidebarOverlay.addEventListener('click', closeSidebar);
            }

            // The download logic has been moved to the `all-visa-services.php` file
            // along with the buttons that trigger it. This keeps the code organized.
        });
    </script>
</body>
</html>