<?php
// ======================================================================
//  0. START SESSION
// ======================================================================
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

// ======================================================================
//  1. INITIALIZE & SETUP
// ======================================================================
require_once 'db-config.php';

// --- CONFIGURATION ---
$admin_email = "rftravelsandtours@gmail.com";
$whatsapp_number = "923052394810";

// ======================================================================
//  2. NEW: HANDLE AGENT REFERRAL LINK
// ======================================================================
// If an agent's referral code is in the URL, store it in the session for later.
if (isset($_GET['ref']) && !empty($_GET['ref'])) {
    // We store the agent's ID from the URL to be used when the booking form is submitted.
    $_SESSION['referring_agent_id'] = (int)$_GET['ref'];
}

// ======================================================================
//  3. HANDLE BOOKING FORM SUBMISSION (WITH COMMISSION TRACKING)
// ======================================================================
if ($_SERVER["REQUEST_METHOD"] == "POST" && isset($_POST['request_booking'])) {
    $customer_name = trim($_POST['customer_name'] ?? '');
    $email = trim($_POST['email'] ?? '');
    $phone_number = trim($_POST['phone'] ?? '');
    $room_type = $_POST['room_type'] ?? 'Sharing';
    $pax = (int)($_POST['party_size'] ?? 1);
    $package_id_from_page = $_POST['package_id'] ?? 'Unknown';
    $package_name_from_page = $_POST['package_name'] ?? 'Unknown';

    if (empty($customer_name) || empty($email) || empty($phone_number) || !filter_var($email, FILTER_VALIDATE_EMAIL) || $package_id_from_page === 'Unknown') {
        $_SESSION['form_message'] = "Please fill in all required fields with valid information.";
        $_SESSION['form_msg_type'] = "error";
        header("Location: " . basename(__FILE__) . "?id=" . urlencode($package_id_from_page) . "#booking-form-anchor");
        exit();
    }

    $user_id_to_save = isset($_SESSION['user_id']) ? (int)$_SESSION['user_id'] : null;
    // NEW: Check if a referral agent ID was stored in the session
    $referred_by_agent_id = isset($_SESSION['referring_agent_id']) ? (int)$_SESSION['referring_agent_id'] : null;
    $status_to_save = 'Pending';

    $conn->begin_transaction();
    try {
        // --- Action 1: Insert into umrah_inquiries (with the new referral ID column) ---
        $sql1 = "INSERT INTO umrah_inquiries (package_id, package_name, customer_name, customer_phone, customer_email, room_type, pax, user_id, referred_by_agent_id, status) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?)";
        $stmt1 = $conn->prepare($sql1);
        // The type string "ssssssiiis" now matches the 10 placeholders
        $stmt1->bind_param(
            "ssssssiiis",
            $package_id_from_page,
            $package_name_from_page,
            $customer_name,
            $phone_number,
            $email,
            $room_type,
            $pax,
            $user_id_to_save,
            $referred_by_agent_id, // This is the new value being saved
            $status_to_save
        );
        $stmt1->execute();
        $inquiry_id = $conn->insert_id; // Get the ID of the inquiry we just created
        $stmt1->close();

        // --- Action 2: Insert into notifications (no change here) ---
        $notification_type = "New Umrah Inquiry";
        $notification_message = "Inquiry for " . htmlspecialchars($package_name_from_page) . " from " . htmlspecialchars($customer_name) . ".";
        if ($referred_by_agent_id) {
            $notification_message .= " (Referred by Agent ID: " . $referred_by_agent_id . ")";
        }
        $notification_link = "admin/view-inquiries.php";
        $sql2 = "INSERT INTO notifications (type, message, link) VALUES (?, ?, ?)";
        $stmt2 = $conn->prepare($sql2);
        $stmt2->bind_param("sss", $notification_type, $notification_message, $notification_link);
        $stmt2->execute();
        $stmt2->close();
        
        $conn->commit();
        
        // After a successful booking via referral link, we can clear the session variable.
        if (isset($_SESSION['referring_agent_id'])) {
            unset($_SESSION['referring_agent_id']);
        }

        $_SESSION['submission_success'] = true;
        $_SESSION['success_context_name'] = $customer_name;
        header("Location: thank-you.php");
        exit();

    } catch (mysqli_sql_exception $exception) {
        $conn->rollback();
        error_log("Booking Form Database Error: " . $exception->getMessage());
        $_SESSION['form_message'] = "A server error occurred. Our team has been notified. Please try again later.";
        $_SESSION['form_msg_type'] = "error";
        header("Location: " . basename(__FILE__) . "?id=" . urlencode($package_id_from_page) . "#booking-form-anchor");
        exit();
    }
}

// ======================================================================
//  4. FETCH PACKAGE DATA FOR DISPLAY
// ======================================================================
if (!isset($_GET['id']) || empty(trim($_GET['id']))) {
    http_response_code(400);
    die("Error: No package ID was specified.");
}
$package_code_to_display = trim($_GET['id']);
$package = null;
try {
    $sql = "SELECT up.*, a.airline_name, a.logo_url FROM umrah_packages up LEFT JOIN airlines a ON up.airline_id = a.id WHERE up.package_id = ?";
    $stmt = $conn->prepare($sql);
    if ($stmt === false) throw new Exception("DB prepare failed: " . $conn->error);
    $stmt->bind_param("s", $package_code_to_display);
    $stmt->execute();
    $result = $stmt->get_result();
    if ($result->num_rows > 0) {
        $package = $result->fetch_assoc();
    }
    $stmt->close();
} catch (Exception $e) {
    error_log("Package Fetch Error: " . $e->getMessage());
    die("A database error occurred while fetching package details.");
}
if ($package === null) {
    http_response_code(404);
    die("Error: Package with ID '" . htmlspecialchars($package_code_to_display) . "' could not be found.");
}

$booking_message = '';
if (isset($_SESSION['form_message'])) {
    $msg_type = $_SESSION['form_msg_type'] === 'error' ? 'error' : 'success';
    $booking_message = '<div class="notice ' . $msg_type . '">' . htmlspecialchars($_SESSION['form_message']) . '</div>';
    unset($_SESSION['form_message'], $_SESSION['form_msg_type']);
}

function echo_list_items($text) {
    if (empty(trim($text))) return;
    $items = explode("\n", trim($text));
    foreach ($items as $item) {
        $trimmed_item = trim($item);
        if (!empty($trimmed_item)) {
            echo "<li>" . htmlspecialchars($trimmed_item) . "</li>\n";
        }
    }
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?= htmlspecialchars($package['package_name']) ?> - RF Travel & Tours</title>
    <link rel="icon" type="image/png" href="images/logo-icon.png">
    <link rel="stylesheet" href="css/style.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.2.0/css/all.min.css">
    <style>
        .notice { padding: 15px 20px; margin-bottom: 20px; border-radius: 8px; border: 1px solid transparent; }
        .notice.success { color: #155724; background-color: #d4edda; border-color: #c3e6cb; }
        .notice.error { color: #721c24; background-color: #f8d7da; border-color: #f5c6cb; }
        .input-with-icon { position: relative; }
        .input-with-icon .fa-solid { position: absolute; left: 15px; top: 50%; transform: translateY(-50%); color: #aaa; pointer-events: none; }
        .input-with-icon .booking-input { padding-left: 45px !important; }

        /* Styles for the new Agent Sharing Panel */
        .agent-share-panel {
            background-color: #fff8e1; /* A light yellow to stand out */
            border: 1px solid #ffecb3;
            border-left: 5px solid #ffc107; /* A gold accent color */
            padding: 20px;
            margin: 25px 0;
            border-radius: 8px;
        }
        .agent-share-panel h3 {
            margin-top: 0;
            font-size: 1.2rem;
            color: #333;
        }
        .agent-share-panel p {
            margin: 5px 0 15px 0;
            font-size: 0.9rem;
            color: #555;
        }
        .share-link-wrapper {
            display: flex;
            gap: 10px;
        }
        .share-link-input {
            flex-grow: 1;
            padding: 10px;
            font-size: 0.9rem;
            border: 1px solid #ccc;
            border-radius: 5px;
            background-color: #fff;
            color: #333;
        }
        .copy-link-btn {
            padding: 10px 20px;
            font-weight: 600;
            background-color: var(--primary-gold, #cfa968);
            color: white;
            border: none;
            border-radius: 5px;
            cursor: pointer;
            transition: background-color 0.2s;
        }
        .copy-link-btn:hover {
            background-color: #b89354;
        }
        .copy-link-btn i {
            margin-right: 8px;
        }
    </style>
</head>
<body>
    <?php include 'header.php'; ?>
    <main class="tour-detail-page">
        <div class="container">
            <div class="tour-layout-grid">
                <div class="tour-main-content">
                    <?= $booking_message ?>
                    <div class="tour-header">
                        <h1><?= htmlspecialchars($package['package_name']) ?></h1>
                        <div class="tour-meta-info"> <span>Package Code: <?= htmlspecialchars($package['package_id']) ?></span> </div>
                    </div>

                    <!-- ======================================================== -->
                    <!-- ===== NEW: AGENT-ONLY SHARING PANEL STARTS HERE ===== -->
                    <!-- ======================================================== -->
                    <?php if (isset($_SESSION['user_type']) && $_SESSION['user_type'] === 'agent'): ?>
                        <section class="agent-share-panel">
                            <h3><i class="fa-solid fa-share-nodes"></i> Your Agent Sharing Link</h3>
                            <p>Share this unique link with your customers. Any bookings made through it will be credited to you for commission.</p>
                            <div class="share-link-wrapper">
                                <?php
                                // Get the full current URL
                                $protocol = isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on' ? "https" : "http";
                                $full_url = "$protocol://$_SERVER[HTTP_HOST]$_SERVER[REQUEST_URI]";
                                // Remove any existing 'ref' parameter to avoid duplicates
                                $base_url = preg_replace('/([&?]ref=[0-9]+)/', '', $full_url);
                                // Add the agent's unique referral parameter
                                $separator = strpos($base_url, '?') === false ? '?' : '&';
                                $referral_link = $base_url . $separator . 'ref=' . urlencode($_SESSION['user_id']);
                                ?>
                                <input type="text" value="<?= htmlspecialchars($referral_link) ?>" id="referralLinkInput" class="share-link-input" readonly>
                                <button class="copy-link-btn" id="copyLinkBtn"><i class="fa-solid fa-copy"></i> Copy</button>
                            </div>
                        </section>
                    <?php endif; ?>
                    <!-- ===== AGENT-ONLY SHARING PANEL ENDS HERE ===== -->

                    <div class="image-gallery">
                         <div class="gallery-main-image"> <img src="<?= htmlspecialchars($package['main_image_link']) ?>" alt="Main image for <?= htmlspecialchars($package['package_name']) ?>"> </div>
                    </div>
                    
                    <section class="tour-section">
                        <h2>Overview of Your <?= htmlspecialchars($package['days']) ?>-Day Umrah Package:</h2>
                        <p><?= nl2br(htmlspecialchars($package['overview'])) ?></p>
                    </section>
                    
                    <div class="booking-card">
                        <div class="price-grid-header"><h3>Package Pricing (per person)</h3></div>
                        <div class="umrah-pricing-grid">
                            <?php if (!empty($package['price_quint'])) : ?><div class="price-cell"><label>Sharing</label><strong>PKR <?= number_format($package['price_quint']) ?>/-</strong></div><?php endif; ?>
                            <?php if (!empty($package['price_quad'])) : ?><div class="price-cell"><label>Quad</label><strong>PKR <?= number_format($package['price_quad']) ?>/-</strong></div><?php endif; ?>
                            <?php if (!empty($package['price_triple'])) : ?><div class="price-cell"><label>Triple</label><strong>PKR <?= number_format($package['price_triple']) ?>/-</strong></div><?php endif; ?>
                            <?php if (!empty($package['price_double'])) : ?><div class="price-cell"><label>Double</label><strong>PKR <?= number_format($package['price_double']) ?>/-</strong></div><?php endif; ?>
                        </div>
                    </div>
                    
                    <?php if (!empty($package['itinerary'])): ?>
                    <section class="tour-section">
                        <h2>Your <?= htmlspecialchars($package['days']) ?>-Day Umrah Itinerary</h2>
                        <ul><?php echo_list_items($package['itinerary']); ?></ul>
                    </section>
                    <?php endif; ?>

                    <section class="tour-section">
                        <?php if (!empty($package['whats_included'])): ?>
                            <h2>What's included</h2>
                            <ul><?php echo_list_items($package['whats_included']); ?></ul>
                        <?php endif; ?>

                        <?php if (!empty($package['whats_extra'])): ?>
                            <h2>What's extra</h2>
                            <ul><?php echo_list_items($package['whats_extra']); ?></ul>
                        <?php endif; ?>
                    </section>
                </div>
                
                <aside class="desktop-form tour-booking-sidebar">
                    <div class="booking-card" id="booking-form-anchor">
                        <div class="booking-price">Book Now</div>
                        <form action="" method="POST" class="booking-form">
                            <input type="hidden" name="request_booking" value="1">
                            <input type="hidden" name="package_id" value="<?= htmlspecialchars($package['package_id']) ?>">
                            <input type="hidden" name="package_name" value="<?= htmlspecialchars($package['package_name']) ?>">
                            <div class="form-group">
                                <label for="customer_name">Full Name:</label>
                                <div class="input-with-icon">
                                    <i class="fa-solid fa-user"></i>
                                    <input type="text" id="customer_name" name="customer_name" class="booking-input" placeholder="Enter your full name" required>
                                </div>
                            </div>
                            <div class="form-group">
                                <label for="phone">Phone Number:</label>
                                <div class="input-with-icon">
                                    <i class="fa-solid fa-phone"></i>
                                    <input type="tel" id="phone" name="phone" class="booking-input" placeholder="Enter your phone number" required>
                                </div>
                            </div>
                            <div class="form-group">
                                <label for="email">Email Address:</label>
                                <div class="input-with-icon">
                                    <i class="fa-solid fa-envelope"></i>
                                    <input type="email" id="email" name="email" class="booking-input" placeholder="Enter your email address" required>
                                </div>
                            </div>
                            <div class="form-group">
                                <label for="room-type">Room Type:</label>
                                <select id="room-type" name="room_type" class="booking-input">
                                    <?php if (!empty($package['price_quint'])) : ?><option value="Sharing">Sharing</option><?php endif; ?>
                                    <?php if (!empty($package['price_quad'])) : ?><option value="Quad">Quad</option><?php endif; ?>
                                    <?php if (!empty($package['price_triple'])) : ?><option value="Triple">Triple</option><?php endif; ?>
                                    <?php if (!empty($package['price_double'])) : ?><option value="Double">Double</option><?php endif; ?>
                                </select>
                            </div>
                            <div class="form-group">
                                <label>Total Pax:</label>
                                <div class="party-size-selector"><button type="button" class="party-btn minus">-</button><input type="text" id="party-size-input" name="party_size" value="1" readonly><button type="button" class="party-btn plus">+</button></div>
                            </div>
                            <button type="submit" class="btn-booking">Request Booking</button>
                        </form>
                    </div>
                </aside>
            </div>
        </div>
    </main>
    <?php include 'footer.php'; ?>
    <script>
    document.addEventListener('DOMContentLoaded', function() {
        // Party size selector logic
        document.querySelectorAll('.party-size-selector').forEach(selector => {
            const minusBtn = selector.querySelector('.party-btn.minus');
            const plusBtn = selector.querySelector('.party-btn.plus');
            const partyInput = selector.querySelector('input[type="text"]');
            minusBtn.addEventListener('click', () => {
                let value = parseInt(partyInput.value, 10);
                if (value > 1) partyInput.value = value - 1;
            });
            plusBtn.addEventListener('click', () => {
                let value = parseInt(partyInput.value, 10);
                partyInput.value = value + 1;
            });
        });

        // Agent copy link button logic
        const copyBtn = document.getElementById('copyLinkBtn');
        const linkInput = document.getElementById('referralLinkInput');
        if (copyBtn && linkInput) {
            copyBtn.addEventListener('click', function() {
                linkInput.select();
                document.execCommand('copy');
                copyBtn.innerHTML = '<i class="fa-solid fa-check"></i> Copied!';
                setTimeout(() => {
                    copyBtn.innerHTML = '<i class="fa-solid fa-copy"></i> Copy';
                }, 2000);
            });
        }
    });
    </script>
</body>
</html>