    <?php
    session_start();
    include_once '../db-config.php';

    function e($string) { return htmlspecialchars($string ?? '', ENT_QUOTES, 'UTF-8'); }

    // --- SECURITY CHECK ---
    if (!isset($_SESSION['user_id']) || !isset($_SESSION['user_type']) || $_SESSION['user_type'] !== 'admin') {
        header("location: ../login.php");
        exit;
    }

    // --- 1. GET PAYMENT ID AND FETCH DATA ---
    $payment_id = (int)($_GET['id'] ?? 0);
    if ($payment_id <= 0) {
        $_SESSION['error_message'] = "Invalid Payment ID.";
        header("Location: manage-payments.php");
        exit;
    }

    $stmt_get_payment = $conn->prepare("
        SELECT p.*, COALESCE(i.user_id, ti.user_id) as linked_invoice_user_id
        FROM payments p
        LEFT JOIN invoices i ON p.invoice_id = i.id AND p.invoice_type = 'package'
        LEFT JOIN ticket_invoices ti ON p.invoice_id = ti.id AND p.invoice_type = 'ticket'
        WHERE p.id = ?
    ");
    $stmt_get_payment->bind_param("i", $payment_id);
    $stmt_get_payment->execute();
    $payment = $stmt_get_payment->get_result()->fetch_assoc();
    if (!$payment) {
        $_SESSION['error_message'] = "Payment not found.";
        header("Location: manage-payments.php");
        exit;
    }

    // --- 2. FETCH DATA FOR FORM DROPDOWNS ---
    $users_list = $conn->query("SELECT id, name, user_type FROM users WHERE user_type IN ('customer', 'agent') ORDER BY name ASC")->fetch_all(MYSQLI_ASSOC);
    $vendors_list = $conn->query("SELECT id, name FROM vendors ORDER BY name ASC")->fetch_all(MYSQLI_ASSOC);
    $all_invoices_sql = "
        (SELECT id, 'package' as type, invoice_number, guest_name, user_id FROM invoices WHERE invoice_number IS NOT NULL AND invoice_number != '')
        UNION ALL
        (SELECT id, 'ticket' as type, invoice_number, guest_name, user_id FROM ticket_invoices WHERE invoice_number IS NOT NULL AND invoice_number != '')
        ORDER BY invoice_number DESC
    ";
    $all_invoices = $conn->query($all_invoices_sql)->fetch_all(MYSQLI_ASSOC);

    // --- 3. SET PRE-SELECTION VALUES FROM FETCHED DATA ---
    $preselected_invoice_value = ($payment['invoice_id'] && $payment['invoice_type']) ? $payment['invoice_id'] . '|' . $payment['invoice_type'] : 'none';
    $preselected_user_id_for_filter = $payment['user_id'] ?? $payment['linked_invoice_user_id'] ?? 'all';

    // --- 4. HANDLE FORM SUBMISSION FOR UPDATE ---
    if ($_SERVER["REQUEST_METHOD"] == "POST" && isset($_POST['update_payment'])) {
        $conn->begin_transaction();
        try {
            $payment_type = $_POST['payment_type'];
            $user_id = (int)($_POST['user_id'] ?? 0);
            $vendor_id = (int)($_POST['vendor_id'] ?? 0);
            $payment_amount = (float)($_POST['payment_amount'] ?? 0);
            $payment_date = $_POST['payment_date'];
            $payment_method = $_POST['payment_method'];
            $notes = trim($_POST['notes'] ?? '');
            $invoice_info = $_POST['invoice_id'] ?? 'none';
            $invoice_reference = trim($_POST['invoice_reference'] ?? '');

            if ($payment_amount <= 0) throw new Exception("Payment amount must be a positive number.");

            $receipt_filename = $payment['receipt_image_path'];
            if (isset($_POST['remove_receipt']) && $receipt_filename) {
                $file_to_remove = '../uploads/receipts/' . $receipt_filename;
                if (file_exists($file_to_remove)) unlink($file_to_remove);
                $receipt_filename = null;
            }
            if (isset($_FILES['receipt_image']) && $_FILES['receipt_image']['error'] == UPLOAD_ERR_OK) {
                if ($receipt_filename) {
                    $file_to_remove = '../uploads/receipts/' . $receipt_filename;
                    if (file_exists($file_to_remove)) unlink($file_to_remove);
                }
                $upload_dir = '../uploads/receipts/';
                if (!is_dir($upload_dir)) mkdir($upload_dir, 0755, true);
                $file_ext = strtolower(pathinfo($_FILES['receipt_image']['name'], PATHINFO_EXTENSION));
                $receipt_filename = 'receipt-' . time() . '-' . uniqid() . '.' . $file_ext;
                if (!move_uploaded_file($_FILES['receipt_image']['tmp_name'], $upload_dir . $receipt_filename)) {
                    throw new Exception("Failed to save the newly uploaded file.");
                }
            }

            $invoice_id_to_save = null;
            $invoice_type_to_save = null;
            $final_user_id_to_save = ($user_id > 0) ? $user_id : null;
            if ($invoice_info !== 'none' && !empty($invoice_info)) {
                list($invoice_id_to_save, $invoice_type_to_save) = explode('|', $invoice_info);
                $stmt_inv_user = ($invoice_type_to_save === 'package') 
                    ? $conn->prepare("SELECT user_id FROM invoices WHERE id = ?")
                    : $conn->prepare("SELECT user_id FROM ticket_invoices WHERE id = ?");
                if ($stmt_inv_user) {
                    $stmt_inv_user->bind_param("i", $invoice_id_to_save);
                    $stmt_inv_user->execute();
                    $result = $stmt_inv_user->get_result()->fetch_assoc();
                    if ($result && !empty($result['user_id'])) {
                        $final_user_id_to_save = (int)$result['user_id'];
                    }
                    $stmt_inv_user->close();
                }
            }

            $vendor_id_to_save = $vendor_id > 0 ? $vendor_id : null;
            $credit_amount = ($payment_type === 'received') ? $payment_amount : 0;
            $debit_amount = ($payment_type === 'made') ? $payment_amount : 0;

            $sql = "UPDATE payments SET invoice_id=?, invoice_type=?, user_id=?, vendor_id=?, payment_date=?, payment_amount=?, credit_amount=?, debit_amount=?, payment_method=?, invoice_reference=?, notes=?, receipt_image_path=? WHERE id=?";
            $stmt = $conn->prepare($sql);
            $stmt->bind_param("isiisddsssssi", $invoice_id_to_save, $invoice_type_to_save, $final_user_id_to_save, $vendor_id_to_save, $payment_date, $payment_amount, $credit_amount, $debit_amount, $payment_method, $invoice_reference, $notes, $receipt_filename, $payment_id);
            
            if ($stmt->execute()) {
                $conn->commit();
                $_SESSION['success_message'] = "Payment #" . str_pad($payment_id, 3, '0', STR_PAD_LEFT) . " updated successfully!";
                header("Location: manage-payments.php");
                exit();
            } else {
                throw new Exception("Database update error: " . $stmt->error);
            }
        } catch (Exception $e) {
            $conn->rollback();
            $_SESSION['error_message'] = "Error updating payment: " . $e->getMessage();
            header("Location: edit-payment.php?id=" . $payment_id);
            exit();
        }
    }
    $success_message = $_SESSION['success_message'] ?? null;
    $error_message = $_SESSION['error_message'] ?? null;
    unset($_SESSION['success_message'], $_SESSION['error_message']);
    ?>
    <!DOCTYPE html>
    <html lang="en">
    <head>
        <meta charset="UTF-8">
        <meta name="viewport" content="width=device-width, initial-scale=1.0">
        <title>Edit Payment #<?= e(str_pad($payment_id, 3, '0', STR_PAD_LEFT)) ?></title>
        <link rel="icon" type="image/png" href="../images/logo-icon.png">
        <link rel="stylesheet" href="admin-style.css">
        <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.1/css/all.min.css">
    </head>
    <body>
        <div class="dashboard-wrapper">
            <?php include 'sidebar.php'; ?>
            <div class="main-content">
                <header class="main-header">
                    <button class="menu-toggle" id="menu-toggle"><i class="fas fa-bars"></i></button>
                    <div class="user-info"><span>Edit Payment</span></div>
                </header>
                <main class="content-body">
                    <div class="content-card" style="max-width: 800px; margin: auto;">
                        <div class="card-header"><h2>Edit Payment #<?= e(str_pad($payment_id, 3, '0', STR_PAD_LEFT)) ?></h2></div>
                        <div class="card-body">
                            <?php if ($success_message): ?><div class="notice success"><?= e($success_message); ?></div><?php endif; ?>
                            <?php if ($error_message): ?><div class="notice error"><?= e($error_message); ?></div><?php endif; ?>

                            <form action="edit-payment.php?id=<?= e($payment_id) ?>" method="POST" class="styled-form" enctype="multipart/form-data">
                                <div class="form-group">
                                    <label for="payment_type">Payment Type</label>
                                    <select name="payment_type" id="payment_type" class="form-control" required>
                                        <option value="received" <?= ($payment['credit_amount'] > 0) ? 'selected' : '' ?>>Payment Received (Credit)</option>
                                        <option value="made" <?= ($payment['debit_amount'] > 0) ? 'selected' : '' ?>>Payment Made / Expense (Debit)</option>
                                    </select>
                                </div>

                                <fieldset>
                                    <legend>Link Payment To (Optional)</legend>
                                    <div class="form-row">
                                        <div class="form-group">
                                            <label for="user_id">Link to User</label>
                                            <select name="user_id" id="user_id" class="form-control">
                                                <option value="0">-- No User --</option>
                                                <?php foreach ($users_list as $user): ?><option value="<?= e($user['id']) ?>" <?= ($payment['user_id'] == $user['id']) ? 'selected' : '' ?>><?= e($user['name']) ?> - [<?= e(strtoupper(substr($user['user_type'], 0, 1))) ?>]</option><?php endforeach; ?>
                                            </select>
                                        </div>
                                        <div class="form-group">
                                            <label for="vendor_id">Link to Vendor</label>
                                            <select name="vendor_id" id="vendor_id" class="form-control">
                                                <option value="0">-- No Vendor --</option>
                                                <?php foreach ($vendors_list as $vendor): ?><option value="<?= e($vendor['id']) ?>" <?= ($payment['vendor_id'] == $vendor['id']) ? 'selected' : '' ?>><?= e($vendor['name']) ?></option><?php endforeach; ?>
                                            </select>
                                        </div>
                                    </div>
                                    <hr style="margin: 20px 0;">
                                    <div class="form-group">
                                        <label for="user_filter">Filter Invoices by User</label>
                                        <select id="user_filter" class="form-control" onchange="filterInvoices()">
                                            <option value="all" <?= ($preselected_user_id_for_filter === 'all' || $preselected_user_id_for_filter === null) ? 'selected' : '' ?>>-- Show All Invoices --</option>
                                            <option value="none" <?= ($preselected_user_id_for_filter === 'none') ? 'selected' : '' ?>>-- Show Invoices without a User --</option>
                                            <?php foreach ($users_list as $user): ?><option value="<?= e($user['id']) ?>" <?= ($preselected_user_id_for_filter == $user['id']) ? 'selected' : '' ?>><?= e($user['name']) ?> - [<?= e(strtoupper(substr($user['user_type'], 0, 1))) ?>]</option><?php endforeach; ?>
                                        </select>
                                    </div>
                                    <div class="form-group">
                                        <label for="invoice_id_select">Link to Specific Invoice</label>
                                        <select name="invoice_id" id="invoice_id_select" class="form-control"></select>
                                    </div>
                                </fieldset>

                                <div class="form-row">
                                    <div class="form-group"><label for="payment_amount">Amount (PKR)*</label><input type="number" name="payment_amount" id="payment_amount" class="form-control" step="0.01" value="<?= e($payment['payment_amount']) ?>" required></div>
                                    <div class="form-group"><label for="payment_date">Payment Date*</label><input type="date" name="payment_date" id="payment_date" class="form-control" value="<?= e($payment['payment_date']) ?>" required></div>
                                </div>
                                
                                <div class="form-row">
                                    <div class="form-group">
                                        <label for="payment_method">Payment Method</label>
                                        <select name="payment_method" id="payment_method" class="form-control">
                                            <option <?= $payment['payment_method'] == 'Bank Transfer' ? 'selected' : '' ?>>Bank Transfer</option>
                                            <option <?= $payment['payment_method'] == 'Cash' ? 'selected' : '' ?>>Cash</option>
                                            <option <?= $payment['payment_method'] == 'Card' ? 'selected' : '' ?>>Card</option>
                                            <option <?= $payment['payment_method'] == 'Other' ? 'selected' : '' ?>>Other</option>
                                        </select>
                                    </div>
                                    <div class="form-group">
                                        <label for="invoice_reference">Invoice Reference (Optional)</label>
                                        <input type="text" name="invoice_reference" id="invoice_reference" class="form-control" placeholder="e.g., Supplier Inv #" value="<?= e($payment['invoice_reference'] ?? '') ?>">
                                    </div>
                                </div>

                                <div class="form-group">
                                    <label for="receipt_image">Upload New Receipt / Proof</label>
                                    <?php if ($payment['receipt_image_path']): ?>
                                        <div class="notice info" style="margin-bottom: 10px;">
                                            Current: <a href="../uploads/receipts/<?= e($payment['receipt_image_path']) ?>" target="_blank">View File</a>
                                            <span style="margin-left: 15px;"><input type="checkbox" name="remove_receipt" id="remove_receipt"><label for="remove_receipt"> Remove</label></span>
                                        </div>
                                    <?php endif; ?>
                                    <input type="file" name="receipt_image" id="receipt_image" class="form-control" accept="image/*,.pdf">
                                    <small style="display: block; margin-top: 5px; color: var(--text-muted);">Only upload a file here if you want to replace the current one.</small>
                                </div>

                                <div class="form-group">
                                    <label for="notes">Notes / Description</label>
                                    <textarea name="notes" id="notes" class="form-control" rows="3" placeholder="e.g., 'Advance from client'"><?= e($payment['notes']) ?></textarea>
                                </div>

                                <div class="form-actions">
                                    <button type="submit" name="update_payment" class="btn btn-primary"><i class="fas fa-save"></i> Save Changes</button>
                                    <a href="manage-payments.php" class="btn btn-secondary">Cancel</a>
                                </div>
                            </form>
                        </div>
                    </div>
                </main>
            </div>
        </div>
        <script>
            const allInvoices = <?= json_encode($all_invoices); ?>;
            const preselectedInvoiceValue = <?= json_encode($preselected_invoice_value); ?>;
            function filterInvoices() {
                const userFilter = document.getElementById('user_filter').value;
                const invoiceSelect = document.getElementById('invoice_id_select');
                invoiceSelect.innerHTML = '<option value="none">-- Do not link to an invoice --</option>';
                allInvoices.forEach(inv => {
                    let userMatch = (userFilter === 'all') || 
                                    (userFilter === 'none' && (inv.user_id === null || inv.user_id == 0)) || 
                                    (inv.user_id == userFilter);
                    if (userMatch) {
                        const option = document.createElement('option');
                        const optionValue = `${inv.id}|${inv.type}`;
                        option.value = optionValue;
                        option.textContent = `Inv #${inv.invoice_number} - ${inv.guest_name} (${inv.type.charAt(0).toUpperCase() + inv.type.slice(1)})`;
                        if (optionValue === preselectedInvoiceValue) { option.selected = true; }
                        invoiceSelect.appendChild(option);
                    }
                });
            }
            document.addEventListener('DOMContentLoaded', filterInvoices);
        </script>
    </body>
    </html>