<?php
session_start();
include_once '../db-config.php';

// --- SECURITY CHECK ---
if (!isset($_SESSION['user_id']) || !isset($_SESSION['user_type']) || $_SESSION['user_type'] !== 'admin') {
    header("location: ../login.php");
    exit;
}

function e($string)
{
    return htmlspecialchars($string ?? '', ENT_QUOTES, 'UTF-8');
}

$booking_id = (int)($_GET['id'] ?? 0);
if ($booking_id <= 0) {
    header("Location: manage-quick-bookings.php");
    exit;
}

// --- HANDLE FORM SUBMISSION FOR UPDATE ---
if ($_SERVER["REQUEST_METHOD"] == "POST" && isset($_POST['update_quick_booking'])) {
    $conn->begin_transaction();
    try {
        // 1. UPDATE MAIN QUICK BOOKING RECORD
        $user_id = (int)($_POST['user_id'] ?? 0);
        $user_id_to_save = ($user_id > 0) ? $user_id : null;
        $customer_name = trim($_POST['customer_name']);
        $package_type = $_POST['package_type'];
        $room_type = $_POST['room_type'];
        $status = $_POST['status'];

        $stmt_update_booking = $conn->prepare("UPDATE quick_bookings SET user_id=?, customer_name=?, package_type=?, room_type=?, status=? WHERE id=?");
        $stmt_update_booking->bind_param("issssi", $user_id_to_save, $customer_name, $package_type, $room_type, $status, $booking_id);
        $stmt_update_booking->execute();

        // 2. REPLACE PASSENGERS IN QUICK BOOKING
        $stmt_delete_passengers = $conn->prepare("DELETE FROM quick_booking_passengers WHERE booking_id = ?");
        $stmt_delete_passengers->bind_param("i", $booking_id);
        $stmt_delete_passengers->execute();

        $family_head_index = -1; // Default if no passengers
        if (!empty($_POST['passenger_name']) && is_array($_POST['passenger_name'])) {
            $sql_passenger = "INSERT INTO quick_booking_passengers (booking_id, is_family_head, full_name, passport_no, visa_no, dob, gender, passport_issue_date, passport_expiry_date, pax_type, bed_required, group_no, pnr_no) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)";
            $stmt_passenger = $conn->prepare($sql_passenger);
            $family_head_index = $_POST['family_head'] ?? -1;
            
            // Re-index passenger_name array to ensure keys are sequential from 0
            $passenger_names = array_values($_POST['passenger_name']);
            
            foreach ($passenger_names as $key => $name) {
                if (empty(trim($name))) continue;
                
                $original_key = array_keys($_POST['passenger_name'])[$key];

                $is_head = ($original_key == $family_head_index) ? 1 : 0;
                $dob = !empty($_POST['dob'][$original_key]) ? $_POST['dob'][$original_key] : null;
                $issue_date = !empty($_POST['passport_issue_date'][$original_key]) ? $_POST['passport_issue_date'][$original_key] : null;
                $expiry_date = !empty($_POST['passport_expiry_date'][$original_key]) ? $_POST['passport_expiry_date'][$original_key] : null;
                
                $stmt_passenger->bind_param("iisssssssssss", $booking_id, $is_head, $name, $_POST['passport_no'][$original_key], $_POST['visa_no'][$original_key], $dob, $_POST['gender'][$original_key], $issue_date, $expiry_date, $_POST['pax_type'][$original_key], $_POST['bed_required'][$original_key], $_POST['group_no'][$original_key], $_POST['pnr_no'][$original_key]);
                $stmt_passenger->execute();
            }
        }

        // CASCADING UPDATE LOGIC FOR VOUCHER & INVOICE
        // ... (rest of the logic remains the same)
        $stmt_check_links = $conn->prepare("SELECT id as voucher_id, invoice_id FROM vouchers WHERE manual_no = ?");
        $stmt_check_links->bind_param("s", $booking_id);
        $stmt_check_links->execute();
        $linked_records = $stmt_check_links->get_result()->fetch_assoc();
        $voucher_id_to_update = $linked_records['voucher_id'] ?? null;
        $invoice_id_to_update = $linked_records['invoice_id'] ?? null;
        
        $family_head_name = $customer_name;
        if (isset($family_head_index) && $family_head_index != -1 && isset($_POST['passenger_name'][$family_head_index])) {
            $family_head_name = $_POST['passenger_name'][$family_head_index];
        }

        if ($voucher_id_to_update) {
            $passenger_count = isset($_POST['passenger_name']) ? count(array_filter($_POST['passenger_name'])) : 0;
            $pax_summary = $passenger_count > 0 ? '(A-' . $passenger_count . ':C-0:I-0) Beds(' . $passenger_count . ')' : '';
            $stmt_update_voucher = $conn->prepare("UPDATE vouchers SET user_id=?, family_head_name=?, package_type=?, pax_summary=? WHERE id=?");
            $stmt_update_voucher->bind_param("isssi", $user_id_to_save, $family_head_name, $package_type, $pax_summary, $voucher_id_to_update);
            $stmt_update_voucher->execute();
            $stmt_delete_mutamers = $conn->prepare("DELETE FROM voucher_mutamers WHERE voucher_id = ?");
            $stmt_delete_mutamers->bind_param("i", $voucher_id_to_update);
            $stmt_delete_mutamers->execute();
            if (!empty($_POST['passenger_name'])) {
                $stmt_mutamer = $conn->prepare("INSERT INTO voucher_mutamers (voucher_id, mutamer_name, passport_no, pax_type, bed_required, group_no, visa_no, pnr_no) VALUES (?, ?, ?, ?, ?, ?, ?, ?)");
                foreach ($_POST['passenger_name'] as $key => $name) {
                    if (empty(trim($name))) continue;
                    $stmt_mutamer->bind_param("isssssss", $voucher_id_to_update, $name, $_POST['passport_no'][$key], $_POST['pax_type'][$key], $_POST['bed_required'][$key], $_POST['group_no'][$key], $_POST['visa_no'][$key], $_POST['pnr_no'][$key]);
                    $stmt_mutamer->execute();
                }
            }
        }

        if ($invoice_id_to_update) {
            $stmt_update_invoice = $conn->prepare("UPDATE invoices SET user_id=?, guest_name=? WHERE id=?");
            $stmt_update_invoice->bind_param("isi", $user_id_to_save, $family_head_name, $invoice_id_to_update);
            $stmt_update_invoice->execute();
            $stmt_delete_pilgrims = $conn->prepare("DELETE FROM invoice_pilgrims WHERE invoice_id = ?");
            $stmt_delete_pilgrims->bind_param("i", $invoice_id_to_update);
            $stmt_delete_pilgrims->execute();
            if (!empty($_POST['passenger_name'])) {
                $stmt_pilgrim = $conn->prepare("INSERT INTO invoice_pilgrims (invoice_id, passport_no, passenger_details, dob, dox, pax, bed) VALUES (?, ?, ?, ?, ?, ?, ?)");
                foreach ($_POST['passenger_name'] as $key => $name) {
                    if (empty(trim($name))) continue;
                    $dob = !empty($_POST['dob'][$key]) ? $_POST['dob'][$key] : null;
                    $dox = !empty($_POST['passport_expiry_date'][$key]) ? $_POST['passport_expiry_date'][$key] : null;
                    $stmt_pilgrim->bind_param("issssss", $invoice_id_to_update, $_POST['passport_no'][$key], $name, $dob, $dox, $_POST['pax_type'][$key], $_POST['bed_required'][$key]);
                    $stmt_pilgrim->execute();
                }
            }
        }
        
        $conn->commit();
        $_SESSION['success_message'] = "Quick Booking #" . $booking_id . " and all linked records updated successfully!";
        header("Location: manage-quick-bookings.php");
        exit();
    } catch (Exception $e) {
        $conn->rollback();
        $_SESSION['error_message'] = "Error Updating Booking: " . $e->getMessage();
        header("Location: edit-quick-booking.php?id=" . $booking_id);
        exit();
    }
}


// --- FETCH DATA FOR DISPLAY ---
$stmt_booking = $conn->prepare("SELECT * FROM quick_bookings WHERE id = ?");
$stmt_booking->bind_param("i", $booking_id);
$stmt_booking->execute();
$booking = $stmt_booking->get_result()->fetch_assoc();
if (!$booking) {
    header("Location: manage-quick-bookings.php");
    exit;
}
$stmt_passengers = $conn->prepare("SELECT * FROM quick_booking_passengers WHERE booking_id = ? ORDER BY id ASC");
$stmt_passengers->bind_param("i", $booking_id);
$stmt_passengers->execute();
$passengers = $stmt_passengers->get_result()->fetch_all(MYSQLI_ASSOC);
$users_list = $conn->query("SELECT id, name, user_type FROM users WHERE user_type IN ('customer', 'agent') ORDER BY name ASC")->fetch_all(MYSQLI_ASSOC);

$success_message = $_SESSION['success_message'] ?? null;
$error_message = $_SESSION['error_message'] ?? null;
unset($_SESSION['success_message'], $_SESSION['error_message']);
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <link rel="icon" type="image/png" href="../images/logo-icon.png">
    <title>Edit Quick Booking #<?= e($booking_id) ?></title>
    <link rel="stylesheet" href="admin-style.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.1/css/all.min.css">
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdn.jsdelivr.net/npm/select2@4.1.0-rc.0/dist/css/select2.min.css" rel="stylesheet" />
    <style>
        .select2-container--default .select2-selection--single {
            height: calc(1.5em + .75rem + 2px); padding: .375rem .75rem; border: 1px solid #ced4da;
        }
        .select2-container--default .select2-selection--single .select2-selection__rendered { line-height: 1.5; padding-left: 0;}
        .select2-container--default .select2-selection--single .select2-selection__arrow { height: calc(1.5em + .75rem); }
        .passenger-card {
            background-color: #f8f9fa;
            border: 1px solid #dee2e6;
            border-radius: .375rem;
            margin-bottom: 1rem;
            position: relative;
        }
        .passenger-card .remove-row-btn {
            position: absolute;
            top: 10px;
            right: 10px;
        }
    </style>
</head>
<body>
    <div class="dashboard-wrapper">
        <?php include 'sidebar.php'; ?>
        <div class="main-content">
            <header class="main-header">
                <div class="user-info"><span>Edit Quick Booking #<?= e($booking_id) ?></span></div>
            </header>
            <main class="content-body">
                <div class="content-card">
                    <div class="alert alert-info">Updating this booking will also update any linked Voucher and Invoice records. Deleting a passenger here and saving will also remove them from linked records.</div>
                    <?php if ($success_message): ?><div class="alert alert-success"><?= e($success_message); ?></div><?php endif; ?>
                    <?php if ($error_message): ?><div class="alert alert-danger"><?= e($error_message); ?></div><?php endif; ?>

                    <form action="edit-quick-booking.php?id=<?= e($booking_id) ?>" method="POST">
                        <fieldset class="p-3 mb-4 border rounded-3">
                            <legend class="fw-bold h5"><i class="fas fa-book"></i> Booking Details</legend>
                            <div class="row g-3">
                                <div class="col-md-6"><label class="form-label">Select User (Optional)</label>
                                    <select id="user_selector" name="user_id" class="form-select">
                                        <option value="0" data-name="">-- Manual Customer Entry --</option>
                                        <?php foreach ($users_list as $user): ?>
                                            <option value="<?= e($user['id']) ?>" data-name="<?= e($user['name']) ?>" <?= ($booking['user_id'] == $user['id']) ? 'selected' : '' ?>>
                                                <?= e($user['name']) ?> - [<?= e(strtoupper(substr($user['user_type'], 0, 1))) ?>]
                                            </option>
                                        <?php endforeach; ?>
                                    </select>
                                </div>
                                <div class="col-md-6"><label class="form-label">Customer Name*</label><input type="text" id="customer_name" name="customer_name" class="form-control" value="<?= e($booking['customer_name']) ?>" required></div>
                                <div class="col-md-4"><label class="form-label">Package Type</label>
                                    <select name="package_type" class="form-select" required>
                                        <option value="Standard" <?= $booking['package_type'] == 'Standard' ? 'selected' : '' ?>>Standard</option>
                                        <option value="Star" <?= $booking['package_type'] == 'Star' ? 'selected' : '' ?>>Star</option>
                                    </select>
                                </div>
                                <div class="col-md-4"><label class="form-label">Room Type</label>
                                    <select name="room_type" class="form-select" required>
                                        <option value="Sharing" <?= $booking['room_type'] == 'Sharing' ? 'selected' : '' ?>>Sharing</option>
                                        <option value="Quint" <?= $booking['room_type'] == 'Quint' ? 'selected' : '' ?>>Quint</option>
                                        <option value="Quad" <?= $booking['room_type'] == 'Quad' ? 'selected' : '' ?>>Quad</option>
                                        <option value="Triple" <?= $booking['room_type'] == 'Triple' ? 'selected' : '' ?>>Triple</option>
                                        <option value="Double" <?= $booking['room_type'] == 'Double' ? 'selected' : '' ?>>Double</option>
                                    </select>
                                </div>
                                <div class="col-md-4"><label class="form-label">Status</label>
                                    <select name="status" class="form-select" required>
                                        <option value="Pending" <?= $booking['status'] == 'Pending' ? 'selected' : '' ?>>Pending</option>
                                        <option value="Confirmed" <?= $booking['status'] == 'Confirmed' ? 'selected' : '' ?>>Confirmed</option>
                                        <option value="Cancelled" <?= $booking['status'] == 'Cancelled' ? 'selected' : '' ?>>Cancelled</option>
                                    </select>
                                </div>
                            </div>
                        </fieldset>

                        <section>
                            <h3 class="h5 fw-bold mb-3"><i class="fas fa-users"></i> Passenger Details</h3>
                            <div id="passengers-container">
                                <?php foreach ($passengers as $index => $p): ?>
                                    <div class="passenger-card" id="passenger-card-<?= $index ?>">
                                        <button type="button" class="btn btn-danger btn-sm remove-row-btn" onclick="removeRow(<?= $index ?>)"><i class="fas fa-trash"></i></button>
                                        <div class="p-3">
                                            <div class="form-check mb-3">
                                                <input class="form-check-input" type="radio" name="family_head" id="family_head_<?= $index ?>" value="<?= $index ?>" <?= $p['is_family_head'] ? 'checked' : '' ?>>
                                                <label class="form-check-label fw-bold" for="family_head_<?= $index ?>">This Passenger is the Family Head</label>
                                            </div>
                                            <div class="row g-3">
                                                <div class="col-lg-4 col-md-6"><label class="form-label">Full Name*</label><input type="text" name="passenger_name[<?= $index ?>]" class="form-control" value="<?= e($p['full_name']) ?>" required></div>
                                                <div class="col-lg-4 col-md-6"><label class="form-label">Passport No.</label><input type="text" name="passport_no[<?= $index ?>]" class="form-control" value="<?= e($p['passport_no']) ?>"></div>
                                                <div class="col-lg-4 col-md-6"><label class="form-label">Visa No.</label><input type="text" name="visa_no[<?= $index ?>]" class="form-control" value="<?= e($p['visa_no']) ?>"></div>
                                                <div class="col-lg-3 col-md-6"><label class="form-label">Pax Type</label><select name="pax_type[<?= $index ?>]" class="form-select"><option value="Adult" <?= ($p['pax_type'] ?? 'Adult') == 'Adult' ? 'selected' : '' ?>>Adult</option><option value="Child" <?= ($p['pax_type'] ?? '') == 'Child' ? 'selected' : '' ?>>Child</option><option value="Infant" <?= ($p['pax_type'] ?? '') == 'Infant' ? 'selected' : '' ?>>Infant</option></select></div>
                                                <div class="col-lg-3 col-md-6"><label class="form-label">Bed Required</label><select name="bed_required[<?= $index ?>]" class="form-select"><option value="Yes" <?= ($p['bed_required'] ?? 'Yes') == 'Yes' ? 'selected' : '' ?>>Yes</option><option value="No" <?= ($p['bed_required'] ?? '') == 'No' ? 'selected' : '' ?>>No</option></select></div>
                                                <div class="col-lg-3 col-md-6"><label class="form-label">Group #</label><input type="text" name="group_no[<?= $index ?>]" class="form-control" value="<?= e($p['group_no']) ?>"></div>
                                                <div class="col-lg-3 col-md-6"><label class="form-label">PNR #</label><input type="text" name="pnr_no[<?= $index ?>]" class="form-control" value="<?= e($p['pnr_no']) ?>"></div>
                                                <div class="col-lg-3 col-md-6"><label class="form-label">Date of Birth</label><input type="date" name="dob[<?= $index ?>]" class="form-control" value="<?= e($p['dob']) ?>"></div>
                                                <div class="col-lg-3 col-md-6"><label class="form-label">Gender</label><select name="gender[<?= $index ?>]" class="form-select"><option value="Male" <?= $p['gender'] == 'Male' ? 'selected' : '' ?>>Male</option><option value="Female" <?= $p['gender'] == 'Female' ? 'selected' : '' ?>>Female</option><option value="Other" <?= $p['gender'] == 'Other' ? 'selected' : '' ?>>Other</option></select></div>
                                                <div class="col-lg-3 col-md-6"><label class="form-label">Passport Issue</label><input type="date" name="passport_issue_date[<?= $index ?>]" class="form-control" value="<?= e($p['passport_issue_date']) ?>"></div>
                                                <div class="col-lg-3 col-md-6"><label class="form-label">Passport Expiry</label><input type="date" name="passport_expiry_date[<?= $index ?>]" class="form-control" value="<?= e($p['passport_expiry_date']) ?>"></div>
                                            </div>
                                        </div>
                                    </div>
                                <?php endforeach; ?>
                            </div>
                            <button type="button" class="btn btn-secondary mt-2" onclick="addPassengerRow()"><i class="fas fa-plus"></i> Add Passenger</button>
                        </section>

                        <div class="mt-4 pt-3 border-top">
                            <button type="submit" name="update_quick_booking" class="btn btn-primary">Save Changes</button>
                            <a href="manage-quick-bookings.php" class="btn btn-secondary">Cancel</a>
                        </div>
                    </form>
                </div>
            </main>
        </div>
    </div>
    <script src="https://code.jquery.com/jquery-3.6.0.min.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/select2@4.1.0-rc.0/dist/js/select2.min.js"></script>
    <script>
        let passengerIndex = <?= count($passengers) ?>;

        document.addEventListener('DOMContentLoaded', function() {
            $('#user_selector').select2({ width: '100%' });
            const userSelector = $('#user_selector');
            const customerNameInput = $('#customer_name');
            userSelector.on('change', function() {
                const selectedOption = $(this).find('option:selected');
                customerNameInput.val(selectedOption.data('name') || '');
            });
            userSelector.trigger('change');
        });

        function addPassengerRow() {
            const container = document.getElementById('passengers-container');
            const isFirstRow = container.children.length === 0;
            const checkedAttribute = isFirstRow ? 'checked' : '';
            const newIndex = passengerIndex++;

            const card = document.createElement('div');
            card.className = 'passenger-card';
            card.id = `passenger-card-${newIndex}`;

            card.innerHTML = `
                <button type="button" class="btn btn-danger btn-sm remove-row-btn" onclick="removeRow(${newIndex})"><i class="fas fa-trash"></i></button>
                <div class="p-3">
                    <div class="form-check mb-3">
                        <input class="form-check-input" type="radio" name="family_head" id="family_head_${newIndex}" value="${newIndex}" ${checkedAttribute}>
                        <label class="form-check-label fw-bold" for="family_head_${newIndex}">This Passenger is the Family Head</label>
                    </div>
                    <div class="row g-3">
                        <div class="col-lg-4 col-md-6"><label class="form-label">Full Name*</label><input type="text" name="passenger_name[${newIndex}]" class="form-control" required></div>
                        <div class="col-lg-4 col-md-6"><label class="form-label">Passport No.</label><input type="text" name="passport_no[${newIndex}]" class="form-control"></div>
                        <div class="col-lg-4 col-md-6"><label class="form-label">Visa No.</label><input type="text" name="visa_no[${newIndex}]" class="form-control"></div>
                        <div class="col-lg-3 col-md-6"><label class="form-label">Pax Type</label><select name="pax_type[${newIndex}]" class="form-select"><option value="Adult">Adult</option><option value="Child">Child</option><option value="Infant">Infant</option></select></div>
                        <div class="col-lg-3 col-md-6"><label class="form-label">Bed Required</label><select name="bed_required[${newIndex}]" class="form-select"><option value="Yes">Yes</option><option value="No">No</option></select></div>
                        <div class="col-lg-3 col-md-6"><label class="form-label">Group #</label><input type="text" name="group_no[${newIndex}]" class="form-control"></div>
                        <div class="col-lg-3 col-md-6"><label class="form-label">PNR #</label><input type="text" name="pnr_no[${newIndex}]" class="form-control"></div>
                        <div class="col-lg-3 col-md-6"><label class="form-label">Date of Birth</label><input type="date" name="dob[${newIndex}]" class="form-control"></div>
                        <div class="col-lg-3 col-md-6"><label class="form-label">Gender</label><select name="gender[${newIndex}]" class="form-select"><option value="Male">Male</option><option value="Female">Female</option><option value="Other">Other</option></select></div>
                        <div class="col-lg-3 col-md-6"><label class="form-label">Passport Issue</label><input type="date" name="passport_issue_date[${newIndex}]" class="form-control"></div>
                        <div class="col-lg-3 col-md-6"><label class="form-label">Passport Expiry</label><input type="date" name="passport_expiry_date[${newIndex}]" class="form-control"></div>
                    </div>
                </div>
            `;
            container.appendChild(card);
        }

        function removeRow(index) {
            const cardToRemove = document.getElementById(`passenger-card-${index}`);
            if (!cardToRemove) return;
            
            const radio = cardToRemove.querySelector('input[type="radio"]');
            
            cardToRemove.remove();

            if (radio && radio.checked) {
                const firstRadio = document.querySelector('#passengers-container input[type="radio"]');
                if (firstRadio) {
                    firstRadio.checked = true;
                }
            }
        }
        
        document.addEventListener('contextmenu', function(e) { e.preventDefault(); });
    </script>
</body>
</html>