<?php
session_start();
include_once '../db-config.php';

// --- SECURITY CHECK ---
if (!isset($_SESSION['user_id']) || !isset($_SESSION['user_type']) || $_SESSION['user_type'] !== 'admin') {
    header("location: ../login.php");
    exit;
}

// --- STEP 1: VALIDATE AND FETCH VOUCHER DATA ---
if (!isset($_GET['id']) || !is_numeric($_GET['id'])) {
    die("Invalid Voucher ID.");
}
$voucher_id = (int)$_GET['id'];

// Fetch main voucher record
$stmt = $conn->prepare("SELECT * FROM vouchers WHERE id = ?");
$stmt->bind_param("i", $voucher_id);
$stmt->execute();
$result = $stmt->get_result();
$voucher = $result->fetch_assoc();

if (!$voucher) {
    die("Voucher not found.");
}

// Fetch all related data for form population
$users_list = $conn->query("SELECT id, name, user_type FROM users WHERE user_type IN ('customer', 'agent') ORDER BY name ASC")->fetch_all(MYSQLI_ASSOC);
$vendors_list = $conn->query("SELECT id, name FROM vendors ORDER BY name ASC")->fetch_all(MYSQLI_ASSOC);
$accommodations = $conn->query("SELECT * FROM voucher_accommodations WHERE voucher_id = $voucher_id ORDER BY check_in_date ASC")->fetch_all(MYSQLI_ASSOC);
$mutamers = $conn->query("SELECT * FROM voucher_mutamers WHERE voucher_id = $voucher_id ORDER BY id ASC")->fetch_all(MYSQLI_ASSOC);
$flights_data = $conn->query("SELECT * FROM voucher_flights WHERE voucher_id = $voucher_id")->fetch_all(MYSQLI_ASSOC);

$departure_flight = null;
$arrival_flight = null;
foreach ($flights_data as $flight) {
    if ($flight['direction'] == 'Pakistan To KSA') $departure_flight = $flight;
    else if ($flight['direction'] == 'KSA To Pakistan') $arrival_flight = $flight;
}

// Fetch hotels for dynamic dropdowns
$makkah_hotels = [];
$madinah_hotels = [];
$hotel_result = $conn->query("SELECT hotel_name, city FROM rate_sheets ORDER BY hotel_name ASC");
if ($hotel_result) {
    while ($row = $hotel_result->fetch_assoc()) {
        if (strtolower($row['city']) === 'makkah') $makkah_hotels[] = $row['hotel_name'];
        elseif (strtolower($row['city']) === 'madinah') $madinah_hotels[] = $row['hotel_name'];
    }
}

function e($string)
{
    return htmlspecialchars($string ?? '', ENT_QUOTES, 'UTF-8');
}

// --- STEP 2: HANDLE FORM SUBMISSION FOR UPDATE ---
if ($_SERVER["REQUEST_METHOD"] == "POST") {
    $conn->begin_transaction();
    try {
        $user_id = (int)($_POST['user_id'] ?? 0);
        $user_id_to_save = ($user_id > 0) ? $user_id : null;
        $vendor_id = (int)($_POST['vendor_id'] ?? 0);
        $vendor_id_to_save = ($vendor_id > 0) ? $vendor_id : null;

        $status = $_POST['status'];
        $voucher_date = $_POST['voucher_date'];
        $family_head_name = $_POST['family_head_name'];
        $booking_ref_no = $_POST['booking_ref_no'];
        $manual_no = $_POST['manual_no'];
        $package_type = $_POST['package_type'];
        $package_duration_nights = $_POST['package_duration_nights'];
        $pax_summary = $_POST['pax_summary'];
        $shirka_name = $_POST['shirka_name'];
        $transporter_name = $_POST['transporter_name'];
        $transport_type = $_POST['transport_type'];
        $transport_description = $_POST['transport_description'];
        $transport_brn = $_POST['transport_brn'];
        $transport_helpline_1 = $_POST['transport_helpline_1'];
        $transport_helpline_2 = $_POST['transport_helpline_2'];
        $hotel_checkin_time = $_POST['hotel_checkin_time'];
        $hotel_checkout_time = $_POST['hotel_checkout_time'];
        $notes_urdu = $_POST['notes_urdu'];

        $sql_update_voucher = "UPDATE vouchers SET user_id=?, vendor_id=?, status=?, voucher_date=?, family_head_name=?, booking_ref_no=?, manual_no=?, package_type=?, package_duration_nights=?, pax_summary=?, shirka_name=?, transporter_name=?, transport_type=?, transport_description=?, transport_brn=?, transport_helpline_1=?, transport_helpline_2=?, hotel_checkin_time=?, hotel_checkout_time=?, notes_urdu=? WHERE id=?";
        $stmt_update = $conn->prepare($sql_update_voucher);
        $stmt_update->bind_param(
            "iisssssissssssssssssi",
            $user_id_to_save, $vendor_id_to_save, $status, $voucher_date, $family_head_name,
            $booking_ref_no, $manual_no, $package_type, $package_duration_nights, $pax_summary,
            $shirka_name, $transporter_name, $transport_type, $transport_description,
            $transport_brn, $transport_helpline_1, $transport_helpline_2, $hotel_checkin_time,
            $hotel_checkout_time, $notes_urdu, $voucher_id
        );
        $stmt_update->execute();

        $conn->query("DELETE FROM voucher_mutamers WHERE voucher_id = $voucher_id");
        $conn->query("DELETE FROM voucher_accommodations WHERE voucher_id = $voucher_id");
        $conn->query("DELETE FROM voucher_flights WHERE voucher_id = $voucher_id");

        if (!empty($_POST['mutamer_name'])) {
            $stmt_mutamer = $conn->prepare("INSERT INTO voucher_mutamers (voucher_id, mutamer_name, passport_no, pax_type, bed_required, group_no, visa_no, pnr_no) VALUES (?, ?, ?, ?, ?, ?, ?, ?)");
            foreach ($_POST['mutamer_name'] as $key => $name) {
                if (empty(trim($name))) continue;
                $stmt_mutamer->bind_param("isssssss", $voucher_id, $name, $_POST['mutamer_passport_no'][$key], $_POST['mutamer_pax_type'][$key], $_POST['mutamer_bed_required'][$key], $_POST['mutamer_group_no'][$key], $_POST['mutamer_visa_no'][$key], $_POST['mutamer_pnr_no'][$key]);
                $stmt_mutamer->execute();
            }
        }
        if (!empty($_POST['hotel_name'])) {
            $stmt_accom = $conn->prepare("INSERT INTO voucher_accommodations (voucher_id, city, hotel_name, room_type, check_in_date, nights, check_out_date, meal_plan, confirmation_no, person_name, person_call_number, person_whatsapp_number) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)");
            foreach ($_POST['hotel_name'] as $key => $name) {
                if (empty(trim($name))) continue;
                $checkInDateStr = !empty($_POST['hotel_check_in'][$key]) ? date('Y-m-d', strtotime(str_replace('/', '-', $_POST['hotel_check_in'][$key]))) : null;
                $checkOutDateStr = !empty($_POST['hotel_check_out'][$key]) ? date('Y-m-d', strtotime(str_replace('/', '-', $_POST['hotel_check_out'][$key]))) : null;
                $nights = (int)($_POST['hotel_nights'][$key] ?? 0);
                $stmt_accom->bind_param("issssissssss", $voucher_id, $_POST['hotel_city'][$key], $name, $_POST['hotel_room_type'][$key], $checkInDateStr, $nights, $checkOutDateStr, $_POST['hotel_meal_plan'][$key], $_POST['hotel_confirmation_no'][$key], $_POST['hotel_person_name'][$key], $_POST['hotel_person_call'][$key], $_POST['hotel_person_whatsapp'][$key]);
                $stmt_accom->execute();
            }
        }
        if (!empty(trim($_POST['flight_no_dep']))) {
            $stmt_flight = $conn->prepare("INSERT INTO voucher_flights (voucher_id, direction, flight_no, sector, departure_datetime, arrival_datetime) VALUES (?, ?, ?, ?, ?, ?)");
            $direction_dep = 'Pakistan To KSA';
            $stmt_flight->bind_param("isssss", $voucher_id, $direction_dep, $_POST['flight_no_dep'], $_POST['sector_dep'], $_POST['departure_datetime_dep'], $_POST['arrival_datetime_dep']);
            $stmt_flight->execute();
        }
        if (!empty(trim($_POST['flight_no_arr']))) {
            $stmt_flight = $conn->prepare("INSERT INTO voucher_flights (voucher_id, direction, flight_no, sector, departure_datetime, arrival_datetime) VALUES (?, ?, ?, ?, ?, ?)");
            $direction_arr = 'KSA To Pakistan';
            $stmt_flight->bind_param("isssss", $voucher_id, $direction_arr, $_POST['flight_no_arr'], $_POST['sector_arr'], $_POST['departure_datetime_arr'], $_POST['arrival_datetime_arr']);
            $stmt_flight->execute();
        }

        $conn->commit();
        $_SESSION['success_message'] = "Voucher #" . $voucher_id . " updated successfully!";
        header("Location: view-voucher.php?id=" . $voucher_id);
        exit();
    } catch (Exception $e) {
        $conn->rollback();
        $_SESSION['error_message'] = "Error Updating Voucher: " . $e->getMessage();
        header("Location: edit-voucher.php?id=" . $voucher_id);
        exit();
    }
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Edit Voucher #<?= e($voucher['id']) ?></title>
    <link rel="icon" type="image/png" href="../images/logo-icon.png">
    <link rel="stylesheet" href="admin-style.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.1/css/all.min.css">
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdn.jsdelivr.net/npm/select2@4.1.0-rc.0/dist/css/select2.min.css" rel="stylesheet" />
    <link href="https://fonts.googleapis.com/css2?family=Noto+Nastaliq+Urdu:wght@400;700&display=swap" rel="stylesheet">
    <style>
        textarea[name="notes_urdu"] { direction: rtl; font-family: 'Noto Nastaliq Urdu', sans-serif; }
    </style>
</head>
<body>
    <div class="dashboard-wrapper">
        <?php include 'sidebar.php'; ?>
        <div class="main-content">
            <header class="main-header">
                <button class="menu-toggle" id="menu-toggle"><i class="fas fa-bars"></i></button>
                <div class="user-info"><span>Edit Voucher #<?= e($voucher['id']) ?></span></div>
            </header>
            <main class="content-body">
                <div class="content-card">
                    <?php if (isset($_SESSION['error_message'])): ?><div class="alert alert-danger"><?= e($_SESSION['error_message']); unset($_SESSION['error_message']); ?></div><?php endif; ?>

                    <form action="edit-voucher.php?id=<?= e($voucher['id']) ?>" method="POST" enctype="multipart/form-data">

                        <fieldset class="p-3 mb-4 border rounded-3 bg-light">
                            <legend class="fw-bold h5"><i class="fas fa-file-invoice"></i> Voucher Details</legend>
                            <div class="row g-3 mb-3">
                                <div class="col-md-4 col-lg-2"><label class="form-label">Manual No</label><input type="text" class="form-control" name="manual_no" value="<?= e($voucher['manual_no']) ?>"></div>
                                <div class="col-md-4 col-lg-2"><label class="form-label">Booking Ref #</label><input type="text" class="form-control" name="booking_ref_no" value="<?= e($voucher['booking_ref_no']) ?>"></div>
                                <div class="col-md-4 col-lg-2"><label class="form-label">Voucher Date*</label><input type="date" class="form-control" name="voucher_date" required value="<?= e($voucher['voucher_date']) ?>"></div>
                                <div class="col-md-6 col-lg-3"><label class="form-label">Family Head Name*</label><input type="text" class="form-control" name="family_head_name" value="<?= e($voucher['family_head_name']) ?>" required></div>
                                <div class="col-md-6 col-lg-3"><label class="form-label">Package Type</label><input type="text" class="form-control" name="package_type" value="<?= e($voucher['package_type']) ?>"></div>
                                <div class="col-md-6 col-lg-2"><label class="form-label">Duration (Nights)</label><input type="number" class="form-control" name="package_duration_nights" value="<?= e($voucher['package_duration_nights']) ?>"></div>
                                <div class="col-md-6 col-lg-4"><label class="form-label">Pax Summary</label><input type="text" class="form-control" name="pax_summary" value="<?= e($voucher['pax_summary']) ?>"></div>
                                <div class="col-md-6 col-lg-4"><label class="form-label">Shirka Name</label><input type="text" class="form-control" name="shirka_name" value="<?= e($voucher['shirka_name']) ?>"></div>
                            </div>
                        </fieldset>

                        <section class="p-3 mb-4 border rounded-3">
                            <h3 class="h5 fw-bold"><i class="fas fa-users"></i> Mutamers</h3>
                            <div class="table-responsive">
                                <table class="table table-bordered align-middle">
                                    <thead class="table-light"><tr><th>Mutamer Name</th><th>Passport No</th><th>Pax</th><th>Bed</th><th>Group #</th><th>Visa #</th><th>PNR #</th><th>Action</th></tr></thead>
                                    <tbody id="mutamers-tbody">
                                        <?php foreach ($mutamers as $mutamer): ?>
                                            <tr>
                                                <td><input type="text" class="form-control" name="mutamer_name[]" value="<?= e($mutamer['mutamer_name']) ?>"></td>
                                                <td><input type="text" class="form-control" name="mutamer_passport_no[]" value="<?= e($mutamer['passport_no']) ?>"></td>
                                                <td><select class="form-select" name="mutamer_pax_type[]"><option value="Adult" <?= ($mutamer['pax_type'] == 'Adult') ? 'selected' : '' ?>>Adult</option><option value="Child" <?= ($mutamer['pax_type'] == 'Child') ? 'selected' : '' ?>>Child</option><option value="Infant" <?= ($mutamer['pax_type'] == 'Infant') ? 'selected' : '' ?>>Infant</option></select></td>
                                                <td><select class="form-select" name="mutamer_bed_required[]"><option value="Yes" <?= ($mutamer['bed_required'] == 'Yes') ? 'selected' : '' ?>>Yes</option><option value="No" <?= ($mutamer['bed_required'] == 'No') ? 'selected' : '' ?>>No</option></select></td>
                                                <td><input type="text" class="form-control" name="mutamer_group_no[]" value="<?= e($mutamer['group_no']) ?>"></td>
                                                <td><input type="text" class="form-control" name="mutamer_visa_no[]" value="<?= e($mutamer['visa_no']) ?>"></td>
                                                <td><input type="text" class="form-control" name="mutamer_pnr_no[]" value="<?= e($mutamer['pnr_no']) ?>"></td>
                                                <td><button type="button" class="btn btn-sm btn-danger" onclick="removeRow(this)">X</button></td>
                                            </tr>
                                        <?php endforeach; ?>
                                    </tbody>
                                </table>
                            </div>
                            <button type="button" class="btn btn-secondary mt-2" onclick="addMutamerRow()">+ Add Mutamer</button>
                        </section>

                        <section class="p-3 mb-4 border rounded-3">
                            <h3 class="h5 fw-bold"><i class="fas fa-hotel"></i> Accommodation</h3>
                            <div class="table-responsive">
                                <table class="table table-bordered align-middle">
                                    <thead class="table-light"><tr><th>City</th><th>Hotel</th><th>Check-In</th><th>Nights</th><th>Check-Out</th><th>Room Type</th><th>Meal</th><th>Conf #</th><th>Action</th></tr></thead>
                                    <tbody id="hotel-tbody">
                                        <?php foreach ($accommodations as $accom): ?>
                                            <tr>
                                                <td><select class="form-select" name="hotel_city[]" onchange="updateHotelDropdown(this)"><option value="Makkah" <?= strtolower($accom['city']) == 'makkah' ? 'selected' : '' ?>>Makkah</option><option value="Madinah" <?= strtolower($accom['city']) == 'madinah' ? 'selected' : '' ?>>Madinah</option></select></td>
                                                <td><select name="hotel_name[]" class="form-select" style="width: 100%;" data-selected-hotel="<?= e($accom['hotel_name']) ?>"></select></td>
                                                <td><input type="text" class="form-control" name="hotel_check_in[]" value="<?= e($accom['check_in_date'] ? date('d/m/Y', strtotime($accom['check_in_date'])) : '') ?>" oninput="calculateCheckoutDate(this)" placeholder="dd/mm/yyyy"></td>
                                                <td><input type="number" class="form-control" name="hotel_nights[]" value="<?= e($accom['nights']) ?>" oninput="calculateCheckoutDate(this)" placeholder="e.g. 5"></td>
                                                <td><input type="text" class="form-control" name="hotel_check_out[]" value="<?= e($accom['check_out_date'] ? date('d/m/Y', strtotime($accom['check_out_date'])) : '') ?>" oninput="calculateNights(this)" placeholder="dd/mm/yyyy"></td>
                                                <td><select class="form-select" name="hotel_room_type[]">
                                                        <option value="Sharing" <?= $accom['room_type'] == 'Sharing' ? 'selected' : '' ?>>Sharing</option>
                                                        <option value="Quint" <?= $accom['room_type'] == 'Quint' ? 'selected' : '' ?>>Quint</option>
                                                        <option value="Quad" <?= $accom['room_type'] == 'Quad' ? 'selected' : '' ?>>Quad</option>
                                                        <option value="Triple" <?= $accom['room_type'] == 'Triple' ? 'selected' : '' ?>>Triple</option>
                                                        <option value="Double" <?= $accom['room_type'] == 'Double' ? 'selected' : '' ?>>Double</option>
                                                    </select></td>
                                                <td><input type="text" class="form-control" name="hotel_meal_plan[]" value="<?= e($accom['meal_plan']) ?>"></td>
                                                <td><input type="text" class="form-control" name="hotel_confirmation_no[]" value="<?= e($accom['confirmation_no']) ?>"></td>
                                                <td><button type="button" class="btn btn-sm btn-danger" onclick="removeRow(this)">X</button></td>
                                            </tr>
                                        <?php endforeach; ?>
                                    </tbody>
                                </table>
                            </div>
                            <button type="button" class="btn btn-secondary mt-2" onclick="addHotelRow()">+ Add Accommodation</button>
                            <hr>
                            <h4 class="h6 fw-bold">Accommodation Contact Details (Per Hotel)</h4>
                            <div class="table-responsive">
                                <table class="table table-bordered align-middle">
                                    <thead class="table-light"><tr><th style="width: 25%;">Hotel Name (for reference)</th><th>Person Name</th><th>Person call number</th><th>Person WhatsApp Number</th></tr></thead>
                                    <tbody id="hotel-contacts-tbody">
                                        <?php foreach ($accommodations as $accom): ?>
                                            <tr>
                                                <td class="hotel-name-ref"><?= e($accom['hotel_name']) ?></td>
                                                <td><input type="text" class="form-control" name="hotel_person_name[]" value="<?= e($accom['person_name'] ?? '') ?>"></td>
                                                <td><input type="text" class="form-control" name="hotel_person_call[]" value="<?= e($accom['person_call_number'] ?? '') ?>"></td>
                                                <td><input type="text" class="form-control" name="hotel_person_whatsapp[]" value="<?= e($accom['person_whatsapp_number'] ?? '') ?>"></td>
                                            </tr>
                                        <?php endforeach; ?>
                                    </tbody>
                                </table>
                            </div>
                        </section>

                        <fieldset class="p-3 mb-4 border rounded-3 bg-light">
                            <legend class="fw-bold h5"><i class="fas fa-bus"></i> Transport / Services</legend>
                            <div class="row g-3">
                                <div class="col-md-4 col-lg-2"><label class="form-label">Transporter</label>
                                    <select name="transporter_name" class="form-select">
                                        <option value="Company Transport" <?= $voucher['transporter_name'] == 'Company Transport' ? 'selected' : '' ?>>Company Transport</option>
                                        <option value="Private" <?= $voucher['transporter_name'] == 'Private' ? 'selected' : '' ?>>Private</option>
                                    </select>
                                </div>
                                <div class="col-md-4 col-lg-2"><label class="form-label">Type</label>
                                    <select name="transport_type" class="form-select">
                                        <option value="Economy By Bus" <?= $voucher['transport_type'] == 'Economy By Bus' ? 'selected' : '' ?>>Economy By Bus</option>
                                        <option value="Luxury By Bus" <?= $voucher['transport_type'] == 'Luxury By Bus' ? 'selected' : '' ?>>Luxury By Bus</option>
                                        <option value="Sedan Car" <?= $voucher['transport_type'] == 'Sedan Car' ? 'selected' : '' ?>>Sedan Car</option>
                                        <option value="SUV / GMC" <?= $voucher['transport_type'] == 'SUV / GMC' ? 'selected' : '' ?>>SUV / GMC</option>
                                    </select>
                                </div>
                                <div class="col-md-4 col-lg-2"><label class="form-label">BRN</label><input type="text" class="form-control" name="transport_brn" value="<?= e($voucher['transport_brn']) ?>"></div>
                                <div class="col-md-4 col-lg-2"><label class="form-label">Description</label><input type="text" class="form-control" name="transport_description" value="<?= e($voucher['transport_description']) ?>"></div>
                                <div class="col-md-4 col-lg-2"><label class="form-label">Helpline 1</label><input type="text" class="form-control" name="transport_helpline_1" value="<?= e($voucher['transport_helpline_1']) ?>"></div>
                                <div class="col-md-4 col-lg-2"><label class="form-label">Helpline 2</label><input type="text" class="form-control" name="transport_helpline_2" value="<?= e($voucher['transport_helpline_2']) ?>"></div>
                            </div>
                        </fieldset>
                        
                        <fieldset class="p-3 mb-4 border rounded-3 bg-light">
                            <legend class="fw-bold h5"><i class="fas fa-plane-departure"></i> Flight Details</legend>
                             <div class="row g-3">
                                <div class="col-lg-6 border-end-lg pe-lg-4 mb-3 mb-lg-0">
                                    <h6>Departure Flight (Pakistan to KSA)</h6>
                                    <div class="row g-2">
                                        <div class="col-md-6"><label class="form-label">Flight No</label><input type="text" name="flight_no_dep" class="form-control" value="<?= e($departure_flight['flight_no'] ?? '') ?>"></div>
                                        <div class="col-md-6"><label class="form-label">Sector</label><input type="text" name="sector_dep" class="form-control" value="<?= e($departure_flight['sector'] ?? '') ?>"></div>
                                        <div class="col-md-6"><label class="form-label">Departure</label><input type="datetime-local" name="departure_datetime_dep" class="form-control" value="<?= !empty($departure_flight['departure_datetime']) ? date('Y-m-d\TH:i', strtotime($departure_flight['departure_datetime'])) : '' ?>"></div>
                                        <div class="col-md-6"><label class="form-label">Arrival</label><input type="datetime-local" name="arrival_datetime_dep" class="form-control" value="<?= !empty($departure_flight['arrival_datetime']) ? date('Y-m-d\TH:i', strtotime($departure_flight['arrival_datetime'])) : '' ?>"></div>
                                    </div>
                                </div>
                                <div class="col-lg-6 ps-lg-4">
                                    <h6>Return Flight (KSA to Pakistan)</h6>
                                    <div class="row g-2">
                                        <div class="col-md-6"><label class="form-label">Flight No</label><input type="text" name="flight_no_arr" class="form-control" value="<?= e($arrival_flight['flight_no'] ?? '') ?>"></div>
                                        <div class="col-md-6"><label class="form-label">Sector</label><input type="text" name="sector_arr" class="form-control" value="<?= e($arrival_flight['sector'] ?? '') ?>"></div>
                                        <div class="col-md-6"><label class="form-label">Departure</label><input type="datetime-local" name="departure_datetime_arr" class="form-control" value="<?= !empty($arrival_flight['departure_datetime']) ? date('Y-m-d\TH:i', strtotime($arrival_flight['departure_datetime'])) : '' ?>"></div>
                                        <div class="col-md-6"><label class="form-label">Arrival</label><input type="datetime-local" name="arrival_datetime_arr" class="form-control" value="<?= !empty($arrival_flight['arrival_datetime']) ? date('Y-m-d\TH:i', strtotime($arrival_flight['arrival_datetime'])) : '' ?>"></div>
                                    </div>
                                </div>
                            </div>
                        </fieldset>

                        <section class="p-3 mb-4 border rounded-3">
                            <h3 class="h5 fw-bold"><i class="fas fa-sticky-note"></i> Notes & Hotel Timings</h3>
                            <div class="row g-3 mb-3">
                                <div class="col-md-6"><label class="form-label">Hotel Check In Time</label><input type="text" name="hotel_checkin_time" class="form-control" value="<?= e($voucher['hotel_checkin_time']) ?>"></div>
                                <div class="col-md-6"><label class="form-label">Hotel Check Out Time</label><input type="text" name="hotel_checkout_time" class="form-control" value="<?= e($voucher['hotel_checkout_time']) ?>"></div>
                            </div>
                            <div class="form-group"><label class="form-label">Notes / Terms & Conditions (Urdu)</label><textarea name="notes_urdu" class="form-control" rows="6"><?= e($voucher['notes_urdu']) ?></textarea></div>
                        </section>

                        <div class="row g-3 mb-4 p-3 border rounded-3 bg-light">
                            <div class="col-md-4"><label class="form-label d-block fw-bold">Status*</label><div class="form-check form-check-inline"><input class="form-check-input" type="radio" name="status" id="statusTentative" value="Tentative" <?= ($voucher['status'] == 'Tentative') ? 'checked' : '' ?>><label class="form-check-label" for="statusTentative">Tentative</label></div><div class="form-check form-check-inline"><input class="form-check-input" type="radio" name="status" id="statusConfirmed" value="Confirmed" <?= ($voucher['status'] == 'Confirmed') ? 'checked' : '' ?>><label class="form-check-label" for="statusConfirmed">Confirmed</label></div></div>
                            <div class="col-md-4"><label class="form-label fw-bold">Assign to User</label><select class="form-select" id="user_id" name="user_id"><option value="0">-- No User Assignment --</option><?php foreach ($users_list as $user): ?><option value="<?= $user['id'] ?>" <?= ($voucher['user_id'] == $user['id']) ? 'selected' : '' ?>><?= e($user['name']) ?> - [<?= e(ucfirst($user['user_type'])) ?>]</option><?php endforeach; ?></select></div>
                            <div class="col-md-4"><label class="form-label fw-bold">Assign to Vendor</label><select class="form-select" id="vendor_id" name="vendor_id"><option value="0">-- No Vendor Assignment --</option><?php foreach ($vendors_list as $vendor): ?><option value="<?= $vendor['id'] ?>" <?= ($voucher['vendor_id'] == $vendor['id']) ? 'selected' : '' ?>><?= e($vendor['name']) ?></option><?php endforeach; ?></select></div>
                        </div>

                        <div class="form-actions"><button type="submit" class="btn btn-primary">Update Voucher</button></div>
                    </form>
                </div>
            </main>
        </div>
    </div>
    <script src="https://code.jquery.com/jquery-3.6.0.min.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/select2@4.1.0-rc.0/dist/js/select2.min.js"></script>
    <script>
        const makkahHotels = <?= json_encode(array_unique($makkah_hotels)) ?>;
        const madinahHotels = <?= json_encode(array_unique($madinah_hotels)) ?>;

        const removeRow = (btn) => {
            const row = btn.closest('tr');
            const tbody = row.parentNode;
            if (tbody.id === 'hotel-tbody') {
                const rowIndex = Array.from(tbody.children).indexOf(row);
                const contactsTbody = document.getElementById('hotel-contacts-tbody');
                if (contactsTbody && contactsTbody.rows[rowIndex]) {
                    contactsTbody.deleteRow(rowIndex);
                }
            }
            row.remove();
        };

        function addMutamerRow() { document.getElementById('mutamers-tbody').insertAdjacentHTML('beforeend', `<tr><td><input type="text" name="mutamer_name[]" class="form-control"></td><td><input type="text" name="mutamer_passport_no[]" class="form-control"></td><td><select name="mutamer_pax_type[]" class="form-select"><option>Adult</option><option>Child</option><option>Infant</option></select></td><td><select name="mutamer_bed_required[]" class="form-select"><option>Yes</option><option>No</option></select></td><td><input type="text" name="mutamer_group_no[]" class="form-control"></td><td><input type="text" name="mutamer_visa_no[]" class="form-control"></td><td><input type="text" name="mutamer_pnr_no[]" class="form-control"></td><td><button type="button" class="btn btn-sm btn-danger" onclick="removeRow(this)">X</button></td></tr>`); }

        function populateHotelOptions(selectElement, hotels, selectedHotel = '') {
            $(selectElement).empty().append(new Option('Select Hotel', '', false, false));
            hotels.forEach(hotel => {
                const isSelected = (hotel === selectedHotel);
                $(selectElement).append(new Option(hotel, hotel, isSelected, isSelected));
            });
            $(selectElement).trigger('change');
        }

        function updateHotelDropdown(citySelect) {
            const row = citySelect.closest('tr');
            const hotelSelect = row.querySelector('select[name="hotel_name[]"]');
            const previouslySelectedHotel = hotelSelect.dataset.selectedHotel || '';
            if ($(hotelSelect).data('select2')) $(hotelSelect).select2('destroy');
            let hotelList = (citySelect.value.toLowerCase() === 'makkah') ? makkahHotels : madinahHotels;
            populateHotelOptions(hotelSelect, hotelList, previouslySelectedHotel);
            $(hotelSelect).select2({ placeholder: 'Search and select a hotel', allowClear: true }).on('change', function() { updateContactRef(this); });
        }

        function updateContactRef(hotelSelect) {
            const row = hotelSelect.closest('tr');
            const rowIndex = Array.from(row.parentNode.children).indexOf(row);
            const contactsRow = document.getElementById('hotel-contacts-tbody').rows[rowIndex];
            if (contactsRow) contactsRow.querySelector('.hotel-name-ref').textContent = hotelSelect.value || 'Hotel Not Selected';
        }

        function addHotelRow() {
            const hotelTbody = document.getElementById('hotel-tbody');
            const newHotelRowHTML = `
            <tr>
                <td><select name="hotel_city[]" class="form-select" onchange="updateHotelDropdown(this)"><option selected>Makkah</option><option>Madinah</option></select></td>
                <td><select name="hotel_name[]" class="form-select" style="width: 100%;" data-selected-hotel=""></select></td>
                <td><input type="text" name="hotel_check_in[]" class="form-control" oninput="calculateCheckoutDate(this)" placeholder="dd/mm/yyyy"></td>
                <td><input type="number" name="hotel_nights[]" class="form-control" oninput="calculateCheckoutDate(this)" placeholder="e.g. 5"></td>
                <td><input type="text" name="hotel_check_out[]" class="form-control" oninput="calculateNights(this)" placeholder="dd/mm/yyyy"></td>
                <td><select class="form-select" name="hotel_room_type[]"><option value="Sharing">Sharing</option><option value="Quint">Quint</option><option value="Quad" selected>Quad</option><option value="Triple">Triple</option><option value="Double">Double</option></select></td>
                <td><input type="text" class="form-control" name="hotel_meal_plan[]" value="R/O"></td>
                <td><input type="text" class="form-control" name="hotel_confirmation_no[]"></td>
                <td><button type="button" class="btn btn-sm btn-danger" onclick="removeRow(this)">X</button></td>
            </tr>`;
            hotelTbody.insertAdjacentHTML('beforeend', newHotelRowHTML);
            updateHotelDropdown(hotelTbody.lastElementChild.querySelector('select[name="hotel_city[]"]'));
            document.getElementById('hotel-contacts-tbody').insertAdjacentHTML('beforeend', `<tr><td class="hotel-name-ref">Hotel Not Selected</td><td><input type="text" name="hotel_person_name[]" class="form-control"></td><td><input type="text" name="hotel_person_call[]" class="form-control"></td><td><input type="text" name="hotel_person_whatsapp[]" class="form-control"></td></tr>`);
        }

        function formatDate(date) { let d = new Date(date), day = '' + d.getDate(), month = '' + (d.getMonth() + 1), year = d.getFullYear(); if (day.length < 2) day = '0' + day; if (month.length < 2) month = '0' + month; return [day, month, year].join('/'); }
        function parseDate(str) { if (!/^\d{1,2}\/\d{1,2}\/\d{4}$/.test(str)) return null; var parts = str.split('/'); return new Date(parts[2], parts[1] - 1, parts[0]); }

        function calculateCheckoutDate(inputElement) {
            const row = inputElement.closest('tr');
            const checkInInput = row.querySelector('input[name="hotel_check_in[]"]');
            const nightsInput = row.querySelector('input[name="hotel_nights[]"]');
            const checkOutInput = row.querySelector('input[name="hotel_check_out[]"]');
            const checkInDate = parseDate(checkInInput.value);
            const nights = parseInt(nightsInput.value, 10);
            if (checkInDate && !isNaN(nights) && nights >= 0) {
                let date = new Date(checkInDate);
                date.setDate(date.getDate() + nights);
                checkOutInput.value = formatDate(date);
            } else {
                checkOutInput.value = '';
            }
        }
        function calculateNights(inputElement) {
            const row = inputElement.closest('tr');
            const checkInInput = row.querySelector('input[name="hotel_check_in[]"]');
            const nightsInput = row.querySelector('input[name="hotel_nights[]"]');
            const checkOutInput = row.querySelector('input[name="hotel_check_out[]"]');
            const checkInDate = parseDate(checkInInput.value);
            const checkOutDate = parseDate(checkOutInput.value);
            if (checkInDate && checkOutDate) { const diff = checkOutDate - checkInDate; nightsInput.value = (diff >= 0) ? Math.ceil(diff / 864e5) : 0; }
        }

        document.addEventListener('DOMContentLoaded', () => {
             // Sidebar Toggle functionality
            const menuToggle = document.getElementById('menu-toggle');
            if (menuToggle) {
                menuToggle.addEventListener('click', () => {
                    document.querySelector('.dashboard-wrapper').classList.toggle('sidebar-open');
                });
            }
            
            document.querySelectorAll('#hotel-tbody tr').forEach(row => { updateHotelDropdown(row.querySelector('select[name="hotel_city[]"]')); });
            if (document.getElementById('mutamers-tbody') && document.querySelectorAll('#mutamers-tbody tr').length === 0) addMutamerRow();
        });
    </script>
</body>
</html>