<?php
/**
 * hotel-rate-sheets.php
 * - Manages the main "Rate Sheets" (e.g., "AL Marwa Rayhaan - Makkah").
 * - Now includes vendor association.
 * - UPDATE: Made the entire interface responsive for mobile and tablet use.
 */
include_once '../db-config.php';
// session_start() is often in db-config.php, but let's be sure.
if (session_status() == PHP_SESSION_NONE) {
    session_start();
}
if (!isset($_SESSION['user_id']) || !isset($_SESSION['user_type']) || $_SESSION['user_type'] !== 'admin') {
    header("location: login.php");
    exit;
}

$edit_mode = false;
$sheet_to_edit = null;

if (isset($_SESSION['success_message'])) { $success_message = $_SESSION['success_message']; unset($_SESSION['success_message']); }
if (isset($_SESSION['error_message'])) { $error_message = $_SESSION['error_message']; unset($_SESSION['error_message']); }

// --- Handle CUD for the RATE SHEETS ---
if ($_SERVER["REQUEST_METHOD"] == "POST" && isset($_POST['action'])) {
    if ($_POST['action'] === 'add' || $_POST['action'] === 'update') {
        $hotel_name = $_POST['hotel_name'];
        $city = $_POST['city'];
        $stars = $_POST['stars'];
        $vendor_id = !empty($_POST['vendor_id']) ? (int)$_POST['vendor_id'] : NULL;

        if ($_POST['action'] === 'add') {
            $stmt = $conn->prepare("INSERT INTO rate_sheets (hotel_name, city, stars, vendor_id) VALUES (?, ?, ?, ?)");
            $stmt->bind_param("ssii", $hotel_name, $city, $stars, $vendor_id);
            $_SESSION['success_message'] = "Rate Sheet for '$hotel_name' created successfully.";
        } else {
            $id = $_POST['id'];
            $stmt = $conn->prepare("UPDATE rate_sheets SET hotel_name = ?, city = ?, stars = ?, vendor_id = ? WHERE id = ?");
            $stmt->bind_param("ssiii", $hotel_name, $city, $stars, $vendor_id, $id);
            $_SESSION['success_message'] = "Rate Sheet for '$hotel_name' updated successfully.";
        }
        $stmt->execute();
        $stmt->close();
    }
    if ($_POST['action'] === 'delete' && isset($_POST['id'])) {
        $stmt = $conn->prepare("DELETE FROM rate_sheets WHERE id = ?");
        $stmt->bind_param("i", $_POST['id']);
        $stmt->execute();
        $stmt->close();
        $_SESSION['success_message'] = "The Rate Sheet and all its rate entries have been deleted.";
    }
    header("Location: hotel-rate-sheets.php");
    exit();
}

if (isset($_GET['action']) && $_GET['action'] === 'edit' && isset($_GET['id'])) {
    $edit_mode = true;
    $stmt = $conn->prepare("SELECT * FROM rate_sheets WHERE id = ?");
    $stmt->bind_param("i", $_GET['id']);
    $stmt->execute();
    $sheet_to_edit = $stmt->get_result()->fetch_assoc();
    $stmt->close();
}

// Fetch vendors who provide 'hotels' service for the dropdown
$hotel_vendors_result = $conn->query("SELECT id, name FROM vendors WHERE services LIKE '%hotels%' ORDER BY name ASC");
$hotel_vendors = [];
if ($hotel_vendors_result) {
    while ($row = $hotel_vendors_result->fetch_assoc()) {
        $hotel_vendors[] = $row;
    }
}

// Fetch all rate sheets and JOIN with vendors to get the vendor name
$sheets_result = $conn->query("
    SELECT rs.*, v.name AS vendor_name 
    FROM rate_sheets rs
    LEFT JOIN vendors v ON rs.vendor_id = v.id
    ORDER BY rs.city, rs.hotel_name ASC
");

$current_page = basename($_SERVER['PHP_SELF']);
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Manage Hotel Rate Sheets</title>
    <link rel="icon" type="image/png" href="../images/logo-icon.png">
    <link rel="stylesheet" href="admin-style.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.1/css/all.min.css">
    <style>
        /* Responsive Styles */
        @media (max-width: 992px) {
            .form-grid {
                grid-template-columns: 1fr 1fr;
            }
        }
        @media (max-width: 768px) {
            .form-grid {
                grid-template-columns: 1fr;
            }
            .content-body {
                padding: 1rem;
            }
        }
        
        /* Table Responsive Styles */
        .table-responsive {
            width: 100%;
            overflow-x: auto;
            -webkit-overflow-scrolling: touch;
        }
        .data-table {
            width: 100%;
            min-width: 800px; /* Force scroll on smaller screens */
        }
        .data-table .actions-cell {
            white-space: nowrap; /* Prevent action buttons from wrapping */
        }
    </style>
</head>
<body>
    <div class="dashboard-wrapper">
        <?php include 'sidebar.php'; ?>
        <div class="main-content">
            <header class="main-header">
                <button class="menu-toggle" id="menu-toggle"><i class="fas fa-bars"></i></button>
                <div class="user-info"><span>Welcome, <?php echo htmlspecialchars(ucfirst($_SESSION['user_name'] ?? 'Admin')); ?></span></div>
            </header>
            <main class="content-body" id="top">
                <div class="content-card">
                    <h2 class="form-title"><?= $edit_mode ? 'Edit Rate Sheet' : 'Add New Rate Sheet' ?></h2>
                    <?php if (isset($success_message)): ?><div class="notice success"><?= htmlspecialchars($success_message) ?></div><?php endif; ?>
                    <?php if (isset($error_message)): ?><div class="notice error"><?= htmlspecialchars($error_message) ?></div><?php endif; ?>

                    <form action="hotel-rate-sheets.php" method="POST" class="styled-form">
                        <input type="hidden" name="action" value="<?= $edit_mode ? 'update' : 'add' ?>">
                        <?php if ($edit_mode): ?><input type="hidden" name="id" value="<?= $sheet_to_edit['id'] ?>"><?php endif; ?>

                        <div class="form-grid">
                            <div class="form-group"><label for="hotel_name">Hotel Name *</label><input id="hotel_name" type="text" name="hotel_name" value="<?= htmlspecialchars($sheet_to_edit['hotel_name'] ?? '') ?>" required></div>
                            <div class="form-group"><label for="city">City *</label><input id="city" type="text" name="city" placeholder="e.g. Makkah" value="<?= htmlspecialchars($sheet_to_edit['city'] ?? '') ?>" required></div>
                            <div class="form-group"><label for="stars">Star Rating *</label>
                                <select id="stars" name="stars" required>
                                    <option value="5" <?= (($sheet_to_edit['stars'] ?? 5) == 5) ? 'selected' : '' ?>>5 Stars</option>
                                    <option value="4" <?= (($sheet_to_edit['stars'] ?? 0) == 4) ? 'selected' : '' ?>>4 Stars</option>
                                    <option value="3" <?= (($sheet_to_edit['stars'] ?? 0) == 3) ? 'selected' : '' ?>>3 Stars</option>
                                    <option value="2" <?= (($sheet_to_edit['stars'] ?? 0) == 2) ? 'selected' : '' ?>>2 Stars</option>
                                    <option value="1" <?= (($sheet_to_edit['stars'] ?? 0) == 1) ? 'selected' : '' ?>>1 Star</option>
                                </select>
                            </div>
                            <div class="form-group"><label for="vendor_id">Associated Vendor</label>
                                <select id="vendor_id" name="vendor_id">
                                    <option value="">-- No Vendor --</option>
                                    <?php foreach ($hotel_vendors as $vendor): ?>
                                        <option value="<?= $vendor['id'] ?>" <?= (($sheet_to_edit['vendor_id'] ?? null) == $vendor['id']) ? 'selected' : '' ?>>
                                            <?= htmlspecialchars($vendor['name']) ?>
                                        </option>
                                    <?php endforeach; ?>
                                </select>
                            </div>
                        </div>

                        <div class="form-actions">
                            <button type="submit" class="btn btn-primary"><?= $edit_mode ? 'Update Sheet' : 'Create Sheet' ?></button>
                            <?php if ($edit_mode): ?><a href="hotel-rate-sheets.php" class="btn btn-secondary">Cancel Edit</a><?php endif; ?>
                        </div>
                    </form>
                </div>

                <div class="content-card">
                    <h2 class="form-title">Existing Rate Sheets</h2>
                    <div class="table-responsive">
                        <table class="data-table">
                            <thead>
                                <tr>
                                    <th>Hotel Name</th>
                                    <th>City</th>
                                    <th>Vendor</th>
                                    <th>Actions</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php if ($sheets_result && $sheets_result->num_rows > 0): ?>
                                    <?php while ($sheet = $sheets_result->fetch_assoc()): ?>
                                        <tr class="<?= ($edit_mode && isset($sheet_to_edit['id']) && $sheet['id'] === $sheet_to_edit['id']) ? 'editing' : '' ?>">
                                            <td><strong><?= htmlspecialchars($sheet['hotel_name']) ?></strong><br><small><?= str_repeat('⭐', $sheet['stars']) ?></small></td>
                                            <td><?= htmlspecialchars($sheet['city']) ?></td>
                                            <td><?= htmlspecialchars($sheet['vendor_name'] ?? 'N/A') ?></td>
                                            <td class="actions-cell">
                                                <a href="manage-hotel-rates.php?sheet_id=<?= $sheet['id'] ?>" class="btn btn-sm btn-primary" title="Manage Rates"><i class="fas fa-dollar-sign"></i></a>
                                                <a href="?action=edit&id=<?= $sheet['id'] ?>#top" class="btn btn-sm btn-secondary" title="Edit Sheet"><i class="fas fa-edit"></i></a>
                                                <form action="hotel-rate-sheets.php" method="POST" onsubmit="return confirm('DELETE this entire sheet and ALL its rates?');" style="display:inline;">
                                                    <input type="hidden" name="action" value="delete"><input type="hidden" name="id" value="<?= $sheet['id'] ?>">
                                                    <button type="submit" class="btn btn-sm btn-danger" title="Delete Sheet"><i class="fas fa-trash"></i></button>
                                                </form>
                                            </td>
                                        </tr>
                                    <?php endwhile; ?>
                                <?php else: ?>
                                    <tr>
                                        <td colspan="4" class="empty-state">No rate sheets found. Use the form above to create one.</td>
                                    </tr>
                                <?php endif; ?>
                            </tbody>
                        </table>
                    </div>
                </div>
            </main>
        </div>
    </div>
    <script>
        document.getElementById('menu-toggle').addEventListener('click', function() {
            document.querySelector('.dashboard-wrapper').classList.toggle('sidebar-open');
        });
        document.addEventListener('contextmenu', function(e) {
            e.preventDefault();
        });
    </script>
</body>
</html>