<?php

/**
 * manage-hotel-rates.php
 * - Manages the individual rate entries for a SINGLE rate sheet.
 */
include_once '../db-config.php';
if (!isset($_SESSION['user_id']) || !isset($_SESSION['user_type']) || $_SESSION['user_type'] !== 'admin') {
    header("location: login.php");
    exit;
}

if (!isset($_GET['sheet_id']) || !filter_var($_GET['sheet_id'], FILTER_VALIDATE_INT)) {
    $_SESSION['error_message'] = "No rate sheet was selected.";
    header("Location: hotel-rate-sheets.php");
    exit;
}
$sheet_id = (int)$_GET['sheet_id'];

$stmt = $conn->prepare("SELECT hotel_name, city, stars FROM rate_sheets WHERE id = ?");
$stmt->bind_param("i", $sheet_id);
$stmt->execute();
$sheet = $stmt->get_result()->fetch_assoc();
if (!$sheet) {
    $_SESSION['error_message'] = "Rate sheet not found.";
    header("Location: hotel-rate-sheets.php");
    exit;
}

$edit_mode = false;
$rate_to_edit = null;
if (isset($_SESSION['success_message'])) {
    $success_message = $_SESSION['success_message'];
    unset($_SESSION['success_message']);
}
if (isset($_SESSION['error_message'])) {
    $error_message = $_SESSION['error_message'];
    unset($_SESSION['error_message']);
}

if ($_SERVER["REQUEST_METHOD"] == "POST" && isset($_POST['action'])) {
    $redirect_url = "manage-hotel-rates.php?sheet_id=" . $sheet_id;
    try {
        if ($_POST['action'] === 'add' || $_POST['action'] === 'update') {
            $rate_4n = !empty($_POST['rate_4_nights']) ? $_POST['rate_4_nights'] : NULL;
            if ($_POST['action'] === 'add') {
                $sql = "INSERT INTO rate_entries (rate_sheet_id, period_from, period_till, room_type, rate_weekday, rate_weekend, rate_4_nights, supplement_ex_bed, supplement_meal_plan, meal_rate_lunch, meal_rate_dinner) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)";
                $stmt = $conn->prepare($sql);
                $stmt->bind_param("isssdddsdsd", $sheet_id, $_POST['period_from'], $_POST['period_till'], $_POST['room_type'], $_POST['rate_weekday'], $_POST['rate_weekend'], $rate_4n, $_POST['supplement_ex_bed'], $_POST['supplement_meal_plan'], $_POST['meal_rate_lunch'], $_POST['meal_rate_dinner']);
                $_SESSION['success_message'] = "New rate entry added successfully!";
            } else {
                $sql = "UPDATE rate_entries SET period_from = ?, period_till = ?, room_type = ?, rate_weekday = ?, rate_weekend = ?, rate_4_nights = ?, supplement_ex_bed = ?, supplement_meal_plan = ?, meal_rate_lunch = ?, meal_rate_dinner = ? WHERE id = ? AND rate_sheet_id = ?";
                $stmt = $conn->prepare($sql);
                $stmt->bind_param("sssdddsdsdii", $_POST['period_from'], $_POST['period_till'], $_POST['room_type'], $_POST['rate_weekday'], $_POST['rate_weekend'], $rate_4n, $_POST['supplement_ex_bed'], $_POST['supplement_meal_plan'], $_POST['meal_rate_lunch'], $_POST['meal_rate_dinner'], $_POST['id'], $sheet_id);
                $_SESSION['success_message'] = "Rate entry updated successfully!";
            }
        }
        if ($_POST['action'] === 'delete') {
            $stmt = $conn->prepare("DELETE FROM rate_entries WHERE id = ? AND rate_sheet_id = ?");
            $stmt->bind_param("ii", $_POST['id'], $sheet_id);
            $_SESSION['success_message'] = "Rate entry deleted.";
        }
        $stmt->execute();
    } catch (Exception $e) {
        $_SESSION['error_message'] = "An error occurred: " . $e->getMessage();
    }
    header("Location: " . $redirect_url);
    exit();
}

if (isset($_GET['action']) && $_GET['action'] === 'edit' && isset($_GET['id'])) {
    $edit_mode = true;
    $stmt = $conn->prepare("SELECT * FROM rate_entries WHERE id = ? AND rate_sheet_id = ?");
    $stmt->bind_param("ii", $_GET['id'], $sheet_id);
    $stmt->execute();
    $rate_to_edit = $stmt->get_result()->fetch_assoc();
}

$rates_result = $conn->query("SELECT * FROM rate_entries WHERE rate_sheet_id = $sheet_id ORDER BY period_from DESC");
$current_page = basename($_SERVER['PHP_SELF']);
?>
<!DOCTYPE html>
<html lang="en">

<head>
    <meta charset="UTF-8">
    <title>Manage Rates for <?= htmlspecialchars($sheet['hotel_name']) ?></title>
    <link rel="icon" type="image/png" href="../images/logo-icon.png">

    <link rel="stylesheet" href="admin-style.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.1/css/all.min.css">
</head>

<body>
    <div class="dashboard-wrapper">
        <?php include 'sidebar.php'; ?>
        <div class="main-content">
            <header class="main-header"><button class="menu-toggle" id="menu-toggle"><i class="fas fa-bars"></i></button>
                <div class="user-info"><span>Welcome, Admin</span></div>
            </header>
            <main class="content-body">
                <a href="hotel-rate-sheets.php" class="btn btn-secondary" style="margin-bottom: 1.5rem;"><i class="fas fa-arrow-left"></i> Back to All Rate Sheets</a>
                <div class="content-card">
                    <h2 class="form-title">
                        <?= $edit_mode ? 'Edit Rate for' : 'Add New Rate for' ?>: <strong><?= htmlspecialchars($sheet['hotel_name']) ?></strong><br>
                        <small style="font-weight: normal; color: #6c757d;"><?= htmlspecialchars($sheet['city']) ?> - <?= str_repeat('⭐', $sheet['stars']) ?></small>
                    </h2>
                    <?php if (isset($success_message)): ?><div class="notice success"><?= $success_message ?></div><?php endif; ?>
                    <form action="manage-hotel-rates.php?sheet_id=<?= $sheet_id ?>" method="POST">
                        <input type="hidden" name="action" value="<?= $edit_mode ? 'update' : 'add' ?>">
                        <?php if ($edit_mode): ?><input type="hidden" name="id" value="<?= $rate_to_edit['id'] ?>"><?php endif; ?>
                        <div class="form-grid">
                            <div class="form-group"><label>Period From *</label><input type="date" name="period_from" value="<?= htmlspecialchars($rate_to_edit['period_from'] ?? '') ?>" required></div>
                            <div class="form-group"><label>Period Till *</label><input type="date" name="period_till" value="<?= htmlspecialchars($rate_to_edit['period_till'] ?? '') ?>" required></div>
                            <div class="form-group"><label>Room Type *</label><input type="text" name="room_type" placeholder="e.g. Double" value="<?= htmlspecialchars($rate_to_edit['room_type'] ?? '') ?>" required></div>
                        </div>
                        <div class="form-grid">
                            <div class="form-group"><label>Weekday Rate (W.D) *</label><input type="number" step="0.01" name="rate_weekday" value="<?= htmlspecialchars($rate_to_edit['rate_weekday'] ?? '') ?>" required></div>
                            <div class="form-group"><label>Weekend Rate (W.E) *</label><input type="number" step="0.01" name="rate_weekend" value="<?= htmlspecialchars($rate_to_edit['rate_weekend'] ?? '') ?>" required></div>
                            <div class="form-group"><label>4 Nights Rate</label><input type="number" step="0.01" name="rate_4_nights" value="<?= htmlspecialchars($rate_to_edit['rate_4_nights'] ?? '') ?>"></div>
                        </div>
                        <div class="form-grid">
                            <div class="form-group"><label>Extra Bed</label><input type="number" step="0.01" name="supplement_ex_bed" value="<?= htmlspecialchars($rate_to_edit['supplement_ex_bed'] ?? '') ?>"></div>
                            <div class="form-group"><label>Meal Plan</label><input type="text" name="supplement_meal_plan" placeholder="BB" value="<?= htmlspecialchars($rate_to_edit['supplement_meal_plan'] ?? '') ?>"></div>
                            <div class="form-group"><label>Lunch Rate</label><input type="number" step="0.01" name="meal_rate_lunch" value="<?= htmlspecialchars($rate_to_edit['meal_rate_lunch'] ?? '') ?>"></div>
                            <div class="form-group"><label>Dinner Rate</label><input type="number" step="0.01" name="meal_rate_dinner" value="<?= htmlspecialchars($rate_to_edit['meal_rate_dinner'] ?? '') ?>"></div>
                        </div>
                        <div class="form-actions">
                            <button type="submit" class="btn btn-primary"><?= $edit_mode ? 'Update Rate Entry' : 'Add Rate Entry' ?></button>
                            <?php if ($edit_mode): ?><a href="manage-hotel-rates.php?sheet_id=<?= $sheet_id ?>" class="btn btn-secondary">Cancel Edit</a><?php endif; ?>
                        </div>
                    </form>
                </div>
                <div class="content-card">
                    <h2 class="form-title">Existing Rate Entries</h2>
                    <div class="table-responsive">
                        <table class="data-table">
                            <thead>
                                <tr>
                                    <th>Period</th>
                                    <th>Room</th>
                                    <th>Weekday</th>
                                    <th>Weekend</th>
                                    <th>Actions</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php if ($rates_result && $rates_result->num_rows > 0): ?>
                                    <?php while ($rate = $rates_result->fetch_assoc()):
                                        // --- NEW: Prepare text for the clipboard ---
                                        $clipboard_text = "Hotel: " . htmlspecialchars($sheet['hotel_name']) . ", " . htmlspecialchars($sheet['city']) . "\n";
                                        $clipboard_text .= "Period: " . date('d/m/Y', strtotime($rate['period_from'])) . " - " . date('d/m/Y', strtotime($rate['period_till'])) . "\n";
                                        $clipboard_text .= "Room: " . htmlspecialchars($rate['room_type']) . "\n";
                                        $clipboard_text .= "Weekday Rate: " . number_format($rate['rate_weekday'], 2) . "\n";
                                        $clipboard_text .= "Weekend Rate: " . number_format($rate['rate_weekend'], 2) . "\n";
                                        if (!empty($rate['supplement_meal_plan'])) $clipboard_text .= "Meal Plan: " . htmlspecialchars($rate['supplement_meal_plan']) . "\n";
                                    ?>
                                        <tr>
                                            <td><?= date('d M, Y', strtotime($rate['period_from'])) ?><br>to <?= date('d M, Y', strtotime($rate['period_till'])) ?></td>
                                            <td><?= htmlspecialchars($rate['room_type']) ?></td>
                                            <td><?= number_format($rate['rate_weekday'], 2) ?></td>
                                            <td><?= number_format($rate['rate_weekend'], 2) ?></td>
                                            <td class="actions-cell">
                                                <button class="btn btn-sm btn-copy-rate" data-clipboard-text="<?= htmlspecialchars($clipboard_text) ?>"><i class="fas fa-clipboard"></i></button>
                                                <a href="?sheet_id=<?= $sheet_id ?>&action=edit&id=<?= $rate['id'] ?>" class="btn btn-sm btn-secondary"><i class="fas fa-edit"></i></a>
                                                <form action="manage-hotel-rates.php?sheet_id=<?= $sheet_id ?>" method="POST" onsubmit="return confirm('Delete this rate entry?');" style="display:inline;">
                                                    <input type="hidden" name="action" value="delete"><input type="hidden" name="id" value="<?= $rate['id'] ?>">
                                                    <button type="submit" class="btn btn-sm btn-danger"><i class="fas fa-trash"></i></button>
                                                </form>
                                            </td>
                                        </tr>
                                    <?php endwhile; ?>
                                <?php else: ?>
                                    <tr>
                                        <td colspan="5" class="empty-state">No rates found for this sheet. Use the form above to add one.</td>
                                    </tr>
                                <?php endif; ?>
                            </tbody>
                        </table>
                    </div>
                </div>
            </main>
        </div>
    </div>
    <script>
        // --- JavaScript for Copy to Clipboard Button ---
        document.addEventListener('DOMContentLoaded', function() {
            document.querySelectorAll('.btn-copy-rate').forEach(button => {
                button.addEventListener('click', function() {
                    const textToCopy = this.getAttribute('data-clipboard-text');
                    navigator.clipboard.writeText(textToCopy).then(() => {
                        const originalIcon = this.innerHTML;
                        this.innerHTML = '<i class="fas fa-check"></i>';
                        this.classList.add('copied');
                        setTimeout(() => {
                            this.innerHTML = originalIcon;
                            this.classList.remove('copied');
                        }, 2000);
                    }).catch(err => console.error('Failed to copy text: ', err));
                });
            });
        });
    </script>


    <script>
        // This script disables the right-click context menu on the entire page.
        document.addEventListener('contextmenu', function(e) {
            e.preventDefault();
        });
    </script>
</body>

</html>