<?php
/**
 * manage-transport.php
 * - Manages transport rates for various vehicle types and sectors.
 * - UPDATE: Made the entire interface responsive for mobile and tablet use.
 */
include_once '../db-config.php';
if (session_status() == PHP_SESSION_NONE) {
    session_start();
}

if (!isset($_SESSION['user_id']) || !isset($_SESSION['user_type']) || $_SESSION['user_type'] !== 'admin') {
    header("location: login.php");
    exit;
}

$edit_mode = false;
$rate_to_edit = null;

if (isset($_SESSION['success_message'])) { $success_message = $_SESSION['success_message']; unset($_SESSION['success_message']); }
if (isset($_SESSION['error_message'])) { $error_message = $_SESSION['error_message']; unset($_SESSION['error_message']); }

// --- Handle CUD for Transport Rates ---
if ($_SERVER["REQUEST_METHOD"] == "POST" && isset($_POST['action'])) {

    // --- ADD or UPDATE a Rate ---
    if ($_POST['action'] === 'add' || $_POST['action'] === 'update') {
        $sector_name = $_POST['sector_name'];
        $sedan_rate = !empty($_POST['sedan_rate']) ? (float)$_POST['sedan_rate'] : 0.00;
        $starex_rate = !empty($_POST['starex_rate']) ? (float)$_POST['starex_rate'] : 0.00;
        $staria_rate = !empty($_POST['staria_rate']) ? (float)$_POST['staria_rate'] : 0.00;
        $gmc_rate = !empty($_POST['gmc_rate']) ? (float)$_POST['gmc_rate'] : 0.00;
        $hiace_rate = !empty($_POST['hiace_rate']) ? (float)$_POST['hiace_rate'] : 0.00;
        $coaster_rate = !empty($_POST['coaster_rate']) ? (float)$_POST['coaster_rate'] : 0.00;
        $bus_rate = !empty($_POST['bus_rate']) ? (float)$_POST['bus_rate'] : 0.00;

        if ($_POST['action'] === 'add') {
            $stmt = $conn->prepare("INSERT INTO transport_rates (sector_name, sedan_rate, starex_rate, staria_rate, gmc_rate, hiace_rate, coaster_rate, bus_rate) VALUES (?, ?, ?, ?, ?, ?, ?, ?)");
            $stmt->bind_param("sddddddd", $sector_name, $sedan_rate, $starex_rate, $staria_rate, $gmc_rate, $hiace_rate, $coaster_rate, $bus_rate);
            $_SESSION['success_message'] = "Transport rate for '$sector_name' created successfully.";
        } else { // 'update'
            $id = (int)$_POST['id'];
            $stmt = $conn->prepare("UPDATE transport_rates SET sector_name=?, sedan_rate=?, starex_rate=?, staria_rate=?, gmc_rate=?, hiace_rate=?, coaster_rate=?, bus_rate=? WHERE id=?");
            $stmt->bind_param("sdddddddi", $sector_name, $sedan_rate, $starex_rate, $staria_rate, $gmc_rate, $hiace_rate, $coaster_rate, $bus_rate, $id);
            $_SESSION['success_message'] = "Transport rate for '$sector_name' updated successfully.";
        }
        $stmt->execute();
        $stmt->close();
    }

    // --- DELETE a Rate ---
    if ($_POST['action'] === 'delete' && isset($_POST['id'])) {
        $id = (int)$_POST['id'];
        $stmt = $conn->prepare("DELETE FROM transport_rates WHERE id = ?");
        $stmt->bind_param("i", $id);
        $stmt->execute();
        $stmt->close();
        $_SESSION['success_message'] = "The transport rate has been deleted.";
    }

    header("Location: manage-transport.php");
    exit();
}

// --- Fetch data for the form if in edit mode ---
if (isset($_GET['action']) && $_GET['action'] === 'edit' && isset($_GET['id'])) {
    $edit_mode = true;
    $id = (int)$_GET['id'];
    $stmt = $conn->prepare("SELECT * FROM transport_rates WHERE id = ?");
    $stmt->bind_param("i", $id);
    $stmt->execute();
    $rate_to_edit = $stmt->get_result()->fetch_assoc();
    $stmt->close();
}

// --- Fetch all rates for the list ---
$rates_result = $conn->query("SELECT * FROM transport_rates ORDER BY sector_name ASC");
$current_page = basename($_SERVER['PHP_SELF']);
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Manage Transport Rates</title>
    <link rel="icon" type="image/png" href="../images/logo-icon.png">
    <link rel="stylesheet" href="admin-style.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.1/css/all.min.css">
    <style>
        /* Responsive Styles */
        @media (max-width: 992px) {
            .form-grid {
                grid-template-columns: repeat(3, 1fr); /* 3 columns on tablets */
            }
        }
        @media (max-width: 768px) {
            .form-grid {
                grid-template-columns: repeat(2, 1fr); /* 2 columns on mobile */
            }
            .content-body {
                padding: 1rem;
            }
        }
        @media (max-width: 576px) {
            .form-grid {
                grid-template-columns: 1fr; /* 1 column on small mobile */
            }
        }
        
        /* Table Responsive Styles */
        .table-responsive {
            width: 100%;
            overflow-x: auto;
            -webkit-overflow-scrolling: touch;
        }
        .data-table {
            width: 100%;
            min-width: 1000px; /* Force scroll on smaller screens */
        }
    </style>
</head>
<body>
    <div class="dashboard-wrapper">
        <?php include 'sidebar.php'; ?>
        <div class="main-content">
            <header class="main-header">
                <button class="menu-toggle" id="menu-toggle"><i class="fas fa-bars"></i></button>
                <div class="user-info"><span>Welcome, <?php echo htmlspecialchars(ucfirst($_SESSION['user_name'] ?? 'Admin')); ?></span></div>
            </header>
            <main class="content-body" id="top">
                <?php if (isset($success_message)): ?><div class="notice success"><?= htmlspecialchars($success_message) ?></div><?php endif; ?>
                <?php if (isset($error_message)): ?><div class="notice error"><?= htmlspecialchars($error_message) ?></div><?php endif; ?>

                <div class="content-card">
                    <h2 class="form-title"><?= $edit_mode ? 'Edit Transport Rate' : 'Add New Transport Rate' ?></h2>
                    <form action="manage-transport.php" method="POST" class="styled-form">
                        <input type="hidden" name="action" value="<?= $edit_mode ? 'update' : 'add' ?>">
                        <?php if ($edit_mode && $rate_to_edit): ?><input type="hidden" name="id" value="<?= $rate_to_edit['id'] ?>"><?php endif; ?>

                        <div class="form-group">
                            <label for="sector_name">Sector Name *</label>
                            <input type="text" id="sector_name" name="sector_name" placeholder="e.g., Jeddah Airport to Makkah Hotel" value="<?= htmlspecialchars($rate_to_edit['sector_name'] ?? '') ?>" required>
                        </div>

                        <div class="form-grid">
                            <div class="form-group"><label for="sedan_rate">Sedan Rate</label><input id="sedan_rate" type="number" step="0.01" name="sedan_rate" value="<?= htmlspecialchars($rate_to_edit['sedan_rate'] ?? '0.00') ?>"></div>
                            <div class="form-group"><label for="starex_rate">Starex Rate</label><input id="starex_rate" type="number" step="0.01" name="starex_rate" value="<?= htmlspecialchars($rate_to_edit['starex_rate'] ?? '0.00') ?>"></div>
                            <div class="form-group"><label for="staria_rate">Staria Rate</label><input id="staria_rate" type="number" step="0.01" name="staria_rate" value="<?= htmlspecialchars($rate_to_edit['staria_rate'] ?? '0.00') ?>"></div>
                            <div class="form-group"><label for="gmc_rate">GMC Rate</label><input id="gmc_rate" type="number" step="0.01" name="gmc_rate" value="<?= htmlspecialchars($rate_to_edit['gmc_rate'] ?? '0.00') ?>"></div>
                            <div class="form-group"><label for="hiace_rate">Hiace Rate</label><input id="hiace_rate" type="number" step="0.01" name="hiace_rate" value="<?= htmlspecialchars($rate_to_edit['hiace_rate'] ?? '0.00') ?>"></div>
                            <div class="form-group"><label for="coaster_rate">Coaster Rate</label><input id="coaster_rate" type="number" step="0.01" name="coaster_rate" value="<?= htmlspecialchars($rate_to_edit['coaster_rate'] ?? '0.00') ?>"></div>
                            <div class="form-group"><label for="bus_rate">Bus Rate</label><input id="bus_rate" type="number" step="0.01" name="bus_rate" value="<?= htmlspecialchars($rate_to_edit['bus_rate'] ?? '0.00') ?>"></div>
                        </div>

                        <div class="form-actions">
                            <button type="submit" class="btn btn-primary"><?= $edit_mode ? 'Update Rate' : 'Add Rate' ?></button>
                            <?php if ($edit_mode): ?><a href="manage-transport.php" class="btn btn-secondary">Cancel Edit</a><?php endif; ?>
                        </div>
                    </form>
                </div>

                <div class="content-card">
                    <h2 class="form-title">Existing Transport Rates (SAR)</h2>
                    <div class="table-responsive">
                        <table class="data-table">
                            <thead>
                                <tr>
                                    <th>Sector Name</th>
                                    <th>Sedan</th>
                                    <th>Starex</th>
                                    <th>Staria</th>
                                    <th>GMC</th>
                                    <th>Hiace</th>
                                    <th>Coaster</th>
                                    <th>Bus</th>
                                    <th>Actions</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php if ($rates_result && $rates_result->num_rows > 0): ?>
                                    <?php while ($rate = $rates_result->fetch_assoc()): ?>
                                        <tr class="<?= ($edit_mode && isset($rate_to_edit['id']) && $rate['id'] === $rate_to_edit['id']) ? 'editing' : '' ?>">
                                            <td><strong><?= htmlspecialchars($rate['sector_name']) ?></strong></td>
                                            <td><?= number_format($rate['sedan_rate']) ?></td>
                                            <td><?= number_format($rate['starex_rate']) ?></td>
                                            <td><?= number_format($rate['staria_rate']) ?></td>
                                            <td><?= number_format($rate['gmc_rate']) ?></td>
                                            <td><?= number_format($rate['hiace_rate']) ?></td>
                                            <td><?= number_format($rate['coaster_rate']) ?></td>
                                            <td><?= number_format($rate['bus_rate']) ?></td>
                                            <td class="actions-cell">
                                                <a href="?action=edit&id=<?= $rate['id'] ?>#top" class="btn btn-sm btn-secondary" title="Edit"><i class="fas fa-edit"></i></a>
                                                <form action="manage-transport.php" method="POST" onsubmit="return confirm('Are you sure you want to delete this rate?');" style="display:inline;">
                                                    <input type="hidden" name="action" value="delete"><input type="hidden" name="id" value="<?= $rate['id'] ?>">
                                                    <button type="submit" class="btn btn-sm btn-danger" title="Delete"><i class="fas fa-trash"></i></button>
                                                </form>
                                            </td>
                                        </tr>
                                    <?php endwhile; ?>
                                <?php else: ?>
                                    <tr>
                                        <td colspan="9" class="empty-state">No transport rates found. Use the form above to add one.</td>
                                    </tr>
                                <?php endif; ?>
                            </tbody>
                        </table>
                    </div>
                </div>
            </main>
        </div>
    </div>
    <script>
        document.getElementById('menu-toggle').addEventListener('click', function() {
            document.querySelector('.dashboard-wrapper').classList.toggle('sidebar-open');
        });
        document.addEventListener('contextmenu', function(e) {
            e.preventDefault();
        });
    </script>
</body>
</html>