<?php
session_start();
include_once '../db-config.php';

// --- SECURITY CHECK ---
if (!isset($_SESSION['user_id']) || !isset($_SESSION['user_type']) || $_SESSION['user_type'] !== 'admin') {
    header("location: ../login.php");
    exit;
}

function e($string) { return htmlspecialchars($string ?? '', ENT_QUOTES, 'UTF-8'); }

$error_message = '';
$success_message = '';

// --- HANDLE POST REQUESTS ---
if ($_SERVER["REQUEST_METHOD"] == "POST") {

    // --- ACTION: CREATE A NEW USER ---
    if (isset($_POST['create_user'])) {
        $name = trim($_POST['name'] ?? '');
        $email = trim($_POST['email'] ?? '');
        $password = $_POST['password'] ?? '';
        $user_type = $_POST['user_type'] ?? 'customer';
        $mobile_number = trim($_POST['mobile_number'] ?? null);
        $company_name = ($user_type === 'agent') ? trim($_POST['company_name'] ?? null) : null;
        $company_address = ($user_type === 'agent') ? trim($_POST['company_address'] ?? null) : null;

        if (empty($name) || empty($email) || empty($password) || strlen($password) < 8) {
            $error_message = "Name, email, and a password of at least 8 characters are required.";
        } else {
            $hashed_password = password_hash($password, PASSWORD_DEFAULT);
            $stmt = $conn->prepare("INSERT INTO users (name, email, password, user_type, mobile_number, company_name, company_address) VALUES (?, ?, ?, ?, ?, ?, ?)");
            $stmt->bind_param("sssssss", $name, $email, $hashed_password, $user_type, $mobile_number, $company_name, $company_address);
            if ($stmt->execute()) {
                $_SESSION['success_message'] = ucfirst($user_type) . " '" . e($name) . "' was created successfully!";
            } else {
                $_SESSION['error_message'] = "Error: That email may already be in use.";
            }
            $stmt->close();
        }
        header("Location: manage-users.php?tab=" . ($user_type === 'admin' ? 'admins' : $user_type . 's'));
        exit();
    }

    // --- ACTION: UPDATE EXISTING USER DETAILS ---
    if (isset($_POST['update_user'])) {
        $user_id = (int)($_POST['user_id'] ?? 0);
        $name = trim($_POST['name'] ?? '');
        $email = trim($_POST['email'] ?? '');
        $user_type = $_POST['user_type'] ?? 'customer';
        $mobile_number = trim($_POST['mobile_number'] ?? null);
        $company_name = ($user_type === 'agent') ? trim($_POST['company_name'] ?? null) : null;
        $company_address = ($user_type === 'agent') ? trim($_POST['company_address'] ?? null) : null;

        if ($user_id > 0) {
            $stmt = $conn->prepare("UPDATE users SET name = ?, email = ?, user_type = ?, mobile_number = ?, company_name = ?, company_address = ? WHERE id = ?");
            $stmt->bind_param("ssssssi", $name, $email, $user_type, $mobile_number, $company_name, $company_address, $user_id);
            if ($stmt->execute()) {
                $_SESSION['success_message'] = "User details updated successfully!";
            } else {
                $_SESSION['error_message'] = "Error updating user details. The email may be in use.";
            }
            $stmt->close();
        }
        header("Location: manage-users.php?tab=" . ($user_type === 'admin' ? 'admins' : $user_type . 's'));
        exit();
    }

    // --- ACTION: UPDATE A USER'S PASSWORD ---
    if (isset($_POST['update_password'])) {
        $user_id = (int)($_POST['user_id'] ?? 0);
        $new_password = $_POST['new_password'] ?? '';
        if (strlen($new_password) < 8) {
            $_SESSION['error_message'] = "New password must be at least 8 characters.";
        } else {
            $hashed_password = password_hash($new_password, PASSWORD_DEFAULT);
            $stmt = $conn->prepare("UPDATE users SET password = ? WHERE id = ?");
            $stmt->bind_param("si", $hashed_password, $user_id);
            if ($stmt->execute()) {
                $_SESSION['success_message'] = "Password for user ID #$user_id updated successfully!";
            } else {
                $_SESSION['error_message'] = "Error updating password.";
            }
            $stmt->close();
        }
        header("Location: manage-users.php?edit_id=" . $user_id);
        exit();
    }

    // --- NEW: HANDLE MULTI-DELETE ACTION ---
    if (isset($_POST['multi_delete_action'])) {
        $ids_to_delete = $_POST['user_ids'] ?? [];
        if (empty($ids_to_delete)) {
            $_SESSION['error_message'] = "No users were selected for deletion.";
        } else {
            $ids_to_delete = array_map('intval', $ids_to_delete);
            $self_delete_key = array_search((int)$_SESSION['user_id'], $ids_to_delete);
            if ($self_delete_key !== false) {
                unset($ids_to_delete[$self_delete_key]);
                $_SESSION['error_message'] = "You cannot delete your own account. Other selected users were processed if any.";
            }

            if (!empty($ids_to_delete)) {
                $placeholders = implode(',', array_fill(0, count($ids_to_delete), '?'));
                $types = str_repeat('i', count($ids_to_delete));
                
                $stmt_delete = $conn->prepare("DELETE FROM users WHERE id IN ($placeholders)");
                $stmt_delete->bind_param($types, ...$ids_to_delete);
                
                if ($stmt_delete->execute()) {
                    $count = $stmt_delete->affected_rows;
                    $_SESSION['success_message'] = "$count user(s) deleted successfully.";
                } else {
                    $_SESSION['error_message'] = "Error deleting users. This can happen if a user has related records (like invoices or bookings). Please ensure records are unlinked before deleting. DB Error: " . $stmt_delete->error;
                }
            }
        }
        header("Location: " . $_SERVER['HTTP_REFERER']);
        exit();
    }
}

// --- ACTION: DELETE A SINGLE USER (from GET request) ---
if (isset($_GET['delete_user'])) {
    $user_id_to_delete = (int)$_GET['delete_user'];
    if ($user_id_to_delete === (int)$_SESSION['user_id']) {
        $_SESSION['error_message'] = "You cannot delete your own account.";
    } else {
        $stmt = $conn->prepare("DELETE FROM users WHERE id = ?");
        $stmt->bind_param("i", $user_id_to_delete);
        if ($stmt->execute()) {
            $_SESSION['success_message'] = "User has been deleted.";
        } else {
            $_SESSION['error_message'] = "Error deleting user. This can happen if the user has related records (like invoices or bookings). DB Error: " . $stmt->error;
        }
        $stmt->close();
    }
    header("Location: manage-users.php");
    exit();
}

// --- PREPARE DATA FOR DISPLAY ---
$is_editing = false;
$user_to_edit = null;
if (isset($_GET['edit_id'])) {
    $edit_id = (int)$_GET['edit_id'];
    $stmt = $conn->prepare("SELECT * FROM users WHERE id = ?");
    $stmt->bind_param("i", $edit_id);
    $stmt->execute();
    $result = $stmt->get_result();
    if ($result->num_rows === 1) {
        $is_editing = true;
        $user_to_edit = $result->fetch_assoc();
    }
    $stmt->close();
}

// --- TABS, SEARCH & PAGINATION ---
$allowed_tabs = ['admins', 'agents', 'customers'];
$current_tab = $_GET['tab'] ?? 'agents';
if (!in_array($current_tab, $allowed_tabs)) $current_tab = 'agents';
$search_term = $_GET['search'] ?? '';
$items_per_page = 15;
$current_page = max(1, (int)($_GET['page'] ?? 1));
$offset = ($current_page - 1) * $items_per_page;
$user_type_for_query = rtrim($current_tab, 's');

$base_sql = "FROM users WHERE user_type = ?";
$params = [$user_type_for_query];
$types = "s";
if (!empty($search_term)) {
    $base_sql .= " AND (name LIKE ? OR email LIKE ? OR company_name LIKE ?)";
    $like_term = "%" . $search_term . "%";
    array_push($params, $like_term, $like_term, $like_term);
    $types .= "sss";
}

$count_sql = "SELECT COUNT(*) as total " . $base_sql;
$stmt_count = $conn->prepare($count_sql);
$stmt_count->bind_param($types, ...$params);
$stmt_count->execute();
$total_items = $stmt_count->get_result()->fetch_assoc()['total'];
$total_pages = ceil($total_items / $items_per_page);

$data_sql = "SELECT * " . $base_sql . " ORDER BY created_at DESC LIMIT ?, ?";
$params[] = $offset;
$params[] = $items_per_page;
$types .= "ii";
$stmt_data = $conn->prepare($data_sql);
$stmt_data->bind_param($types, ...$params);
$stmt_data->execute();
$users_result = $stmt_data->get_result();

$success_message = $_SESSION['success_message'] ?? null;
$error_message = $_SESSION['error_message'] ?? null;
unset($_SESSION['success_message'], $_SESSION['error_message']);
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Manage Users</title>
    <link rel="icon" type="image/png" href="../images/logo-icon.png">
    <link rel="stylesheet" href="admin-style.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.1/css/all.min.css">
    <style>
        .filter-bar { display: flex; flex-wrap: wrap; gap: 8px; background-color: var(--card-bg); padding: 10px; border-radius: 8px; margin-bottom: 25px; }
        .filter-bar a { text-decoration: none; padding: 8px 15px; border-radius: 6px; color: var(--text-dark); font-weight: 600; font-size: 0.9rem; transition: all 0.2s ease; display: flex; align-items: center; gap: 8px; }
        .filter-bar a:hover { background-color: #eef1f4; }
        .filter-bar a.active { background-color: var(--primary-color); color: #fff; }
        .search-bar { margin-bottom: 1.5rem; max-width: 500px; }
        .search-bar form { display: flex; }
        .search-bar input[type="text"] { flex-grow: 1; border-top-right-radius: 0; border-bottom-right-radius: 0; }
        .search-bar button { border-top-left-radius: 0; border-bottom-left-radius: 0; }
        .agent-fields { display: none; }
        .card-header { display: flex; justify-content: space-between; align-items: center; flex-wrap: wrap; }
    </style>
</head>
<body>
    <div class="dashboard-wrapper">
        <?php include 'sidebar.php'; ?>
        <div class="main-content">
            <header class="main-header">
                <button class="menu-toggle" id="menu-toggle"><i class="fas fa-bars"></i></button>
                <div class="user-info"><span>User Management</span></div>
            </header>
            <main class="content-body">
                <h1 class="page-title">Manage Users</h1>
                <?php if ($success_message): ?><div class="notice success"><?= e($success_message); ?></div><?php endif; ?>
                <?php if ($error_message): ?><div class="notice error"><?= e($error_message); ?></div><?php endif; ?>

                <div class="content-card">
                    <div class="card-header"><h2><?php echo $is_editing ? 'Edit User: ' . e($user_to_edit['name']) : 'Create New User'; ?></h2></div>
                    <div class="card-body">
                        <form action="manage-users.php" method="POST" class="styled-form" id="user-form">
                            <?php if ($is_editing): ?><input type="hidden" name="user_id" value="<?= e($user_to_edit['id']); ?>"><?php endif; ?>
                            <div class="form-row">
                                <div class="form-group"><label>Full Name</label><input type="text" name="name" value="<?= e($user_to_edit['name'] ?? '') ?>" required></div>
                                <div class="form-group"><label>Email Address</label><input type="email" name="email" value="<?= e($user_to_edit['email'] ?? '') ?>" required></div>
                            </div>
                            <div class="form-row">
                                <div class="form-group"><label>Mobile Number</label><input type="tel" name="mobile_number" value="<?= e($user_to_edit['mobile_number'] ?? '') ?>"></div>
                                <div class="form-group"><label>User Type</label>
                                    <select name="user_type" id="user_type_select" class="form-control" required>
                                        <option value="customer" <?= ($user_to_edit['user_type'] ?? '') == 'customer' ? 'selected' : '' ?>>Customer</option>
                                        <option value="agent" <?= ($user_to_edit['user_type'] ?? '') == 'agent' ? 'selected' : '' ?>>Agent</option>
                                        <option value="admin" <?= ($user_to_edit['user_type'] ?? '') == 'admin' ? 'selected' : '' ?>>Admin</option>
                                    </select>
                                </div>
                            </div>
                            <div class="form-row agent-fields">
                                <div class="form-group"><label>Company Name</label><input type="text" name="company_name" value="<?= e($user_to_edit['company_name'] ?? '') ?>"></div>
                                <div class="form-group"><label>Company Address</label><input type="text" name="company_address" value="<?= e($user_to_edit['company_address'] ?? '') ?>"></div>
                            </div>
                            <?php if (!$is_editing): ?>
                                <div class="form-group"><label>Password</label><input type="password" name="password" placeholder="Min. 8 characters" required></div>
                            <?php endif; ?>
                            <div class="form-actions">
                                <?php if ($is_editing): ?>
                                    <button type="submit" name="update_user" class="btn btn-success">Save Changes</button>
                                    <a href="manage-users.php?tab=<?= e($user_to_edit['user_type']) ?>s" class="btn btn-secondary">Cancel</a>
                                <?php else: ?>
                                    <button type="submit" name="create_user" class="btn btn-primary">Create User</button>
                                <?php endif; ?>
                            </div>
                        </form>
                        <?php if ($is_editing): ?>
                            <hr style="margin: 2rem 0;">
                            <h4>Update Password for "<?= e($user_to_edit['name']); ?>"</h4>
                            <form action="manage-users.php" method="POST" class="styled-form">
                                <input type="hidden" name="user_id" value="<?= e($user_to_edit['id']); ?>">
                                <div class="form-group"><label>New Password</label><input type="password" name="new_password" placeholder="Min. 8 characters" required></div>
                                <div class="form-actions"><button type="submit" name="update_password" class="btn btn-primary">Update Password</button></div>
                            </form>
                        <?php endif; ?>
                    </div>
                </div>

                <div class="content-card">
                    <form id="multi-delete-form" action="manage-users.php" method="POST" onsubmit="return confirm('Are you sure you want to delete all selected users? This action cannot be undone.');">
                        <input type="hidden" name="multi_delete_action" value="1">
                        <div class="card-header">
                            <div class="filter-bar">
                                <a href="?tab=admins" class="<?= $current_tab === 'admins' ? 'active' : '' ?>">Admins</a>
                                <a href="?tab=agents" class="<?= $current_tab === 'agents' ? 'active' : '' ?>">Agents</a>
                                <a href="?tab=customers" class="<?= $current_tab === 'customers' ? 'active' : '' ?>">Customers</a>
                            </div>
                            <?php if ($current_tab !== 'admins'): ?>
                                <button type="submit" class="btn btn-danger"><i class="fas fa-trash"></i> Delete Selected</button>
                            <?php endif; ?>
                        </div>
                        <div class="card-body">
                            <?php if ($current_tab !== 'admins'): ?>
                                <div class="search-bar">
                                    <form action="manage-users.php" method="GET">
                                        <input type="hidden" name="tab" value="<?= e($current_tab) ?>">
                                        <input type="text" name="search" class="form-control" placeholder="Search by name, email, company..." value="<?= e($search_term) ?>">
                                        <button type="submit" class="btn btn-primary"><i class="fas fa-search"></i></button>
                                    </form>
                                </div>
                            <?php endif; ?>
                            <div class="table-responsive">
                                <table>
                                    <thead><tr>
                                        <?php if ($current_tab !== 'admins'): ?><th style="width: 20px;"><input type="checkbox" id="select-all-checkbox"></th><?php endif; ?>
                                        <th>ID</th><th>Name</th><th>Email</th><th>Mobile</th><th>Company</th><th>Type</th><th>Actions</th>
                                    </tr></thead>
                                    <tbody>
                                        <?php if ($users_result->num_rows > 0): while ($user = $users_result->fetch_assoc()): ?>
                                            <tr>
                                                <?php if ($current_tab !== 'admins'): ?><td><input type="checkbox" name="user_ids[]" class="user-checkbox" value="<?= e($user['id']) ?>"></td><?php endif; ?>
                                                <td><?= $user['id']; ?></td>
                                                <td><?= e($user['name']); ?></td>
                                                <td><?= e($user['email']); ?></td>
                                                <td><?= e($user['mobile_number'] ?? '-'); ?></td>
                                                <td><?= e($user['company_name'] ?? '-'); ?></td>
                                                <td><span class="status-badge status-<?= e($user['user_type']) ?>"><?= e(ucfirst($user['user_type'])); ?></span></td>
                                                <td class="actions-cell">
                                                    <a href="manage-users.php?tab=<?= e($current_tab) ?>&edit_id=<?= $user['id']; ?>" class="btn btn-sm btn-info">Edit</a>
                                                    <a href="manage-users.php?delete_user=<?= $user['id']; ?>" class="btn btn-sm btn-danger" onclick="return confirm('Are you sure you want to delete this user?');">Delete</a>
                                                </td>
                                            </tr>
                                        <?php endwhile; else: ?>
                                            <tr><td colspan="<?= $current_tab !== 'admins' ? '8' : '7' ?>" class="empty-state">No users found.</td></tr>
                                        <?php endif; ?>
                                    </tbody>
                                </table>
                            </div>
                        </div>
                    </form>
                    <?php if ($total_pages > 1): ?>
                        <div class="pagination-controls" style="margin: 20px; text-align: right;">
                            <?php if ($current_page > 1): ?><a href="?tab=<?= e($current_tab) ?>&search=<?= e($search_term) ?>&page=<?= $current_page - 1 ?>" class="btn btn-sm btn-secondary">« Previous</a><?php endif; ?>
                            <span style="padding: 0 10px;">Page <?= $current_page ?> of <?= $total_pages ?></span>
                            <?php if ($current_page < $total_pages): ?><a href="?tab=<?= e($current_tab) ?>&search=<?= e($search_term) ?>&page=<?= $current_page + 1 ?>" class="btn btn-sm btn-secondary">Next »</a><?php endif; ?>
                        </div>
                    <?php endif; ?>
                </div>
            </main>
        </div>
    </div>
    <script>
        document.addEventListener('DOMContentLoaded', function() {
            const userTypeSelect = document.getElementById('user_type_select');
            const agentFields = document.querySelector('.agent-fields');
            function toggleAgentFields() {
                if (userTypeSelect && agentFields) {
                    agentFields.style.display = (userTypeSelect.value === 'agent') ? 'grid' : 'none';
                }
            }
            if (userTypeSelect) {
                toggleAgentFields();
                userTypeSelect.addEventListener('change', toggleAgentFields);
            }

            const selectAllCheckbox = document.getElementById('select-all-checkbox');
            if(selectAllCheckbox) {
                selectAllCheckbox.addEventListener('change', function() {
                    document.querySelectorAll('.user-checkbox').forEach(checkbox => {
                        checkbox.checked = this.checked;
                    });
                });
            }
        });
    </script>
</body>
</html>