<?php
session_start();
include_once '../db-config.php';

function e($string) { return htmlspecialchars($string ?? '', ENT_QUOTES, 'UTF-8'); }

// --- SECURITY CHECK ---
if (!isset($_SESSION['user_id']) || !isset($_SESSION['user_type']) || $_SESSION['user_type'] !== 'admin') {
    header("location: ../login.php");
    exit;
}

// --- FUNCTION TO CHECK VENDOR DEPENDENCIES ---
function vendor_has_dependencies($conn, $vendor_id) {
    $tables = ['invoices', 'ticket_invoices', 'vouchers', 'payments'];
    foreach ($tables as $table) {
        $stmt = $conn->prepare("SELECT id FROM `$table` WHERE vendor_id = ? LIMIT 1");
        $stmt->bind_param("i", $vendor_id);
        $stmt->execute();
        if ($stmt->get_result()->num_rows > 0) return true;
        
        if($table === 'invoices') {
            $specific_cols = ['pilgrims_vendor_id', 'transport_vendor_id', 'tickets_vendor_id'];
            foreach($specific_cols as $col) {
                $stmt_spec = $conn->prepare("SELECT id FROM invoices WHERE $col = ? LIMIT 1");
                $stmt_spec->bind_param("i", $vendor_id);
                $stmt_spec->execute();
                if ($stmt_spec->get_result()->num_rows > 0) return true;
            }
        }
    }
    return false;
}

// --- HANDLE ACTIONS (CREATE, UPDATE, DELETE) ---
if ($_SERVER["REQUEST_METHOD"] == "POST") {
    if (isset($_POST['create_vendor']) || isset($_POST['update_vendor'])) {
        $name = trim($_POST['name']); $company_name = trim($_POST['company_name']);
        $phone = trim($_POST['phone_number']); $email = trim($_POST['email']);
        $services = isset($_POST['services']) ? implode(',', $_POST['services']) : '';

        if (isset($_POST['create_vendor'])) {
            $stmt = $conn->prepare("INSERT INTO vendors (name, company_name, services, phone_number, email) VALUES (?, ?, ?, ?, ?)");
            $stmt->bind_param("sssss", $name, $company_name, $services, $phone, $email);
            if($stmt->execute()) $_SESSION['success_message'] = "Vendor '" . e($name) . "' created successfully!";
            else $_SESSION['error_message'] = "Database error: " . $stmt->error;
        } else {
            $vendor_id = (int)$_POST['vendor_id'];
            $stmt = $conn->prepare("UPDATE vendors SET name=?, company_name=?, services=?, phone_number=?, email=? WHERE id=?");
            $stmt->bind_param("sssssi", $name, $company_name, $services, $phone, $email, $vendor_id);
            if($stmt->execute()) $_SESSION['success_message'] = "Vendor '" . e($name) . "' updated successfully!";
            else $_SESSION['error_message'] = "Database error: " . $stmt->error;
        }
    } elseif (isset($_POST['delete_vendor'])) {
        $vendor_id = (int)$_POST['vendor_id'];
        if (vendor_has_dependencies($conn, $vendor_id)) {
            $_SESSION['error_message'] = "Cannot delete vendor. It is linked to existing records.";
        } else {
            $stmt = $conn->prepare("DELETE FROM vendors WHERE id = ?");
            $stmt->bind_param("i", $vendor_id);
            if ($stmt->execute()) $_SESSION['success_message'] = "Vendor deleted successfully.";
            else $_SESSION['error_message'] = "Error deleting vendor.";
        }
    } elseif (isset($_POST['delete_selected'])) {
        if (!empty($_POST['selected_ids']) && is_array($_POST['selected_ids'])) {
            $ids_to_delete = array_map('intval', $_POST['selected_ids']);
            $deletable_ids = [];
            $undeletable_names = [];

            foreach ($ids_to_delete as $vid) {
                if (vendor_has_dependencies($conn, $vid)) {
                    $stmt_name = $conn->prepare("SELECT name FROM vendors WHERE id = ?");
                    $stmt_name->bind_param("i", $vid);
                    $stmt_name->execute();
                    $vendor_name = $stmt_name->get_result()->fetch_assoc()['name'];
                    $undeletable_names[] = $vendor_name;
                } else {
                    $deletable_ids[] = $vid;
                }
            }

            if (!empty($deletable_ids)) {
                $placeholders = implode(',', array_fill(0, count($deletable_ids), '?'));
                $types = str_repeat('i', count($deletable_ids));
                $sql = "DELETE FROM vendors WHERE id IN ($placeholders)";
                $stmt = $conn->prepare($sql);
                $stmt->bind_param($types, ...$deletable_ids);
                $stmt->execute();
                $_SESSION['success_message'] = count($deletable_ids) . " vendor(s) deleted successfully.";
            }

            if (!empty($undeletable_names)) {
                $error_str = "Could not delete the following vendors because they are linked to existing records: " . implode(', ', $undeletable_names) . ".";
                if(isset($_SESSION['success_message'])) {
                     $_SESSION['success_message'] .= " " . $error_str;
                } else {
                    $_SESSION['error_message'] = $error_str;
                }
            }

        } else {
            $_SESSION['error_message'] = "No vendors were selected.";
        }
    }
    header("Location: manage-vendors.php");
    exit();
}

// --- FETCH DATA FOR DISPLAY ---
$is_editing = (isset($_GET['edit_id']) && is_numeric($_GET['edit_id']));
$vendor_to_edit = null;
if ($is_editing) {
    $edit_id = (int)$_GET['edit_id'];
    $stmt = $conn->prepare("SELECT * FROM vendors WHERE id = ?");
    $stmt->bind_param("i", $edit_id);
    $stmt->execute();
    $vendor_to_edit = $stmt->get_result()->fetch_assoc();
}

$vendors = $conn->query("SELECT * FROM vendors ORDER BY name ASC")->fetch_all(MYSQLI_ASSOC);
$success_message = $_SESSION['success_message'] ?? null; unset($_SESSION['success_message']);
$error_message = $_SESSION['error_message'] ?? null; unset($_SESSION['error_message']);
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Manage Vendors</title>
    <link rel="icon" type="image/png" href="../images/logo-icon.png">
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <link rel="stylesheet" href="admin-style.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.1/css/all.min.css">
</head>
<body>
<div class="dashboard-wrapper">
    <?php include 'sidebar.php'; ?>
    <div class="main-content">
        <header class="main-header">
            <button class="menu-toggle" id="menu-toggle"><i class="fas fa-bars"></i></button>
            <div class="user-info"><span>Manage Vendors</span></div>
        </header>
        <main class="content-body">
            <h1 class="page-title">Vendor Management</h1>

            <?php if ($success_message): ?><div class="alert alert-success"><?= e($success_message); ?></div><?php endif; ?>
            <?php if ($error_message): ?><div class="alert alert-danger"><?= e($error_message); ?></div><?php endif; ?>

            <div class="card mb-4">
                <div class="card-header"><h2 class="h5 mb-0"><?= $is_editing ? 'Edit Vendor: ' . e($vendor_to_edit['name']) : 'Add New Vendor'; ?></h2></div>
                <div class="card-body">
                    <form action="manage-vendors.php" method="POST">
                        <?php if ($is_editing): ?><input type="hidden" name="vendor_id" value="<?= e($vendor_to_edit['id']); ?>"><?php endif; ?>
                        <div class="row g-3">
                            <div class="col-md-6"><label class="form-label">Vendor Name*</label><input type="text" name="name" class="form-control" value="<?= e($vendor_to_edit['name'] ?? '') ?>" required></div>
                            <div class="col-md-6"><label class="form-label">Company Name</label><input type="text" name="company_name" class="form-control" value="<?= e($vendor_to_edit['company_name'] ?? '') ?>"></div>
                            <div class="col-md-6"><label class="form-label">Phone</label><input type="tel" name="phone_number" class="form-control" value="<?= e($vendor_to_edit['phone_number'] ?? '') ?>"></div>
                            <div class="col-md-6"><label class="form-label">Email</label><input type="email" name="email" class="form-control" value="<?= e($vendor_to_edit['email'] ?? '') ?>"></div>
                            <div class="col-12"><label class="form-label">Services Provided</label>
                                <div class="d-flex flex-wrap gap-3">
                                    <?php
                                    $available_services = ['Tickets', 'Hotels', 'Transport', 'Visa'];
                                    $selected_services = $is_editing ? explode(',', $vendor_to_edit['services']) : [];
                                    foreach ($available_services as $service):
                                        $service_val = strtolower($service);
                                        $checked = in_array($service_val, $selected_services) ? 'checked' : '';
                                    ?>
                                    <div class="form-check">
                                        <input class="form-check-input" type="checkbox" name="services[]" value="<?= $service_val ?>" id="service_<?= $service_val ?>" <?= $checked ?>>
                                        <label class="form-check-label" for="service_<?= $service_val ?>"><?= $service ?></label>
                                    </div>
                                    <?php endforeach; ?>
                                </div>
                            </div>
                            <div class="col-12 text-end">
                                <?php if ($is_editing): ?>
                                    <a href="manage-vendors.php" class="btn btn-secondary">Cancel</a>
                                    <button type="submit" name="update_vendor" class="btn btn-success">Save Changes</button>
                                <?php else: ?>
                                    <button type="submit" name="create_vendor" class="btn btn-primary">Add Vendor</button>
                                <?php endif; ?>
                            </div>
                        </div>
                    </form>
                </div>
            </div>

            <div class="card">
                <div class="card-header d-flex justify-content-between align-items-center flex-wrap">
                    <h2 class="h5 mb-0">Existing Vendors</h2>
                    <button type="submit" form="bulk-delete-form" name="delete_selected" id="delete-selected-btn" class="btn btn-danger btn-sm" disabled onclick="return confirm('Are you sure? This action cannot be undone.');">
                        <i class="fas fa-trash"></i> Delete Selected
                    </button>
                </div>
                
                <form action="manage-vendors.php" method="POST" id="bulk-delete-form">
                    <div class="table-responsive">
                        <table class="table table-hover align-middle mb-0">
                            <thead class="table-light">
                                <tr>
                                    <th style="width: 40px;"><input type="checkbox" id="select-all-checkbox"></th>
                                    <th>Name</th>
                                    <th>Services</th>
                                    <th>Contact</th>
                                    <th>Actions</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php if (!empty($vendors)): foreach ($vendors as $vendor): ?>
                                <tr>
                                    <td><input type="checkbox" class="row-checkbox" name="selected_ids[]" value="<?= $vendor['id'] ?>"></td>
                                    <td><strong><?= e($vendor['name']) ?></strong><br><small class="text-muted"><?= e($vendor['company_name']) ?></small></td>
                                    <td><?php $svcs = explode(',', $vendor['services']); foreach($svcs as $s) { if(!empty($s)) echo '<span class="badge bg-secondary me-1">' . e(ucfirst($s)) . '</span>'; } ?></td>
                                    <td><?= e($vendor['email']) ?><br><small><?= e($vendor['phone_number']) ?></small></td>
                                    <td>
                                        <a href="manage-vendors.php?edit_id=<?= $vendor['id'] ?>" class="btn btn-sm btn-outline-primary" title="Edit"><i class="fas fa-edit"></i></a>
                                        <form action="manage-vendors.php" method="POST" class="d-inline" onsubmit="return confirm('Are you sure? This cannot be undone.');">
                                            <input type="hidden" name="vendor_id" value="<?= $vendor['id'] ?>">
                                            <button type="submit" name="delete_vendor" class="btn btn-sm btn-outline-danger" title="Delete"><i class="fas fa-trash"></i></button>
                                        </form>
                                    </td>
                                </tr>
                                <?php endforeach; else: ?>
                                <tr><td colspan="5" class="text-center p-4">No vendors have been added yet.</td></tr>
                                <?php endif; ?>
                            </tbody>
                        </table>
                    </div>
                </form>
            </div>
        </main>
    </div>
</div>
<script>
    document.addEventListener('DOMContentLoaded', function() {
        const menuToggle = document.getElementById('menu-toggle');
        if (menuToggle) {
            menuToggle.addEventListener('click', () => document.querySelector('.dashboard-wrapper').classList.toggle('sidebar-open'));
        }

        const selectAllCheckbox = document.getElementById('select-all-checkbox');
        const rowCheckboxes = document.querySelectorAll('.row-checkbox');
        const deleteBtn = document.getElementById('delete-selected-btn');

        function toggleDeleteButton() {
            const anyChecked = Array.from(rowCheckboxes).some(cb => cb.checked);
            deleteBtn.disabled = !anyChecked;
        }

        if (selectAllCheckbox) {
            selectAllCheckbox.addEventListener('change', function() {
                rowCheckboxes.forEach(checkbox => {
                    checkbox.checked = this.checked;
                });
                toggleDeleteButton();
            });
        }

        rowCheckboxes.forEach(checkbox => {
            checkbox.addEventListener('change', function() {
                if (!this.checked) {
                    selectAllCheckbox.checked = false;
                } else {
                    const allChecked = Array.from(rowCheckboxes).every(cb => cb.checked);
                    selectAllCheckbox.checked = allChecked;
                }
                toggleDeleteButton();
            });
        });
    });
</script>
</body>
</html>