<?php
session_start();
include_once '../db-config.php';

// --- SECURITY CHECK: Ensure user is a logged-in admin ---
if (!isset($_SESSION['user_id']) || !isset($_SESSION['user_type']) || $_SESSION['user_type'] !== 'admin') {
    header("location: ../login.php");
    exit;
}

function e($string)
{
    return htmlspecialchars($string ?? '', ENT_QUOTES, 'UTF-8');
}

$booking_id = (int)($_GET['id'] ?? 0);
if ($booking_id <= 0) {
    $_SESSION['error_message'] = "Invalid Booking ID provided.";
    header("Location: manage-flight-requests.php");
    exit;
}

// --- HANDLE POST REQUESTS (STATUS AND PASSENGER UPDATES) ---
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // --- HANDLE STATUS UPDATE ---
    if (isset($_POST['update_status'])) {
        $new_status = $_POST['status'];
        $booking_id_to_update = (int)$_POST['booking_id'];

        if (in_array($new_status, ['pending', 'confirmed', 'cancelled'])) {
            $sql_update = "UPDATE bookings SET status = ? WHERE id = ?";
            $stmt_update = $conn->prepare($sql_update);
            $stmt_update->bind_param("si", $new_status, $booking_id_to_update);
            if ($stmt_update->execute()) {
                $_SESSION['success_message'] = "Booking #" . e($booking_id_to_update) . " status updated to '" . e(ucfirst($new_status)) . "'.";
            } else {
                $_SESSION['error_message'] = "Error: Failed to update booking status.";
            }
            $stmt_update->close();
        } else {
            $_SESSION['error_message'] = "Error: Invalid status selected.";
        }
        header("Location: process-booking.php?id=" . $booking_id); // Redirect back to this page
        exit;
    }

    // --- HANDLE PASSENGER PNR/TICKET NUMBER UPDATES (Applies to Group Bookings) ---
    if (isset($_POST['update_passengers'])) {
        $passenger_ids = $_POST['passenger_id'] ?? [];
        $pnrs = $_POST['pnr'] ?? [];
        $ticket_numbers = $_POST['ticket_number'] ?? [];

        if (!empty($passenger_ids)) {
            $conn->begin_transaction();
            try {
                $sql_update_pax = "UPDATE ticket_invoice_passengers SET pnr = ?, ticket_number = ? WHERE id = ?";
                $stmt_update_pax = $conn->prepare($sql_update_pax);

                foreach ($passenger_ids as $key => $pax_id) {
                    $pnr = $pnrs[$key] ?? '';
                    $ticket_number = $ticket_numbers[$key] ?? '';
                    $stmt_update_pax->bind_param("ssi", $pnr, $ticket_number, $pax_id);
                    $stmt_update_pax->execute();
                }
                $conn->commit();
                $_SESSION['success_message'] = "Passenger PNRs and Ticket Numbers have been updated successfully.";
            } catch (Exception $e) {
                $conn->rollback();
                $_SESSION['error_message'] = "Error updating passenger details: " . $e->getMessage();
            }
        }
        header("Location: process-booking.php?id=" . $booking_id);
        exit;
    }

    // --- HANDLE PASSENGER PNR/TICKET NUMBER UPDATES (Applies to Standard Flights) ---
    if (isset($_POST['update_flight_passengers'])) {
        $stmt_get = $conn->prepare("SELECT passenger_details FROM bookings WHERE id = ?");
        $stmt_get->bind_param("i", $booking_id);
        $stmt_get->execute();
        $current_json_str = $stmt_get->get_result()->fetch_assoc()['passenger_details'];
        $stmt_get->close();

        $passenger_data = json_decode($current_json_str, true);

        foreach ($_POST['pax_key'] as $index => $key) {
            list($type, $pax_index) = explode('_', $key);
            if (isset($passenger_data[$type][$pax_index])) {
                $passenger_data[$type][$pax_index]['pnr'] = $_POST['pnr'][$index] ?? '';
                $passenger_data[$type][$pax_index]['ticket_number'] = $_POST['ticket_number'][$index] ?? '';
            }
        }

        $updated_json = json_encode($passenger_data);

        $stmt_update = $conn->prepare("UPDATE bookings SET passenger_details = ? WHERE id = ?");
        $stmt_update->bind_param("si", $updated_json, $booking_id);
        if ($stmt_update->execute()) {
            $_SESSION['success_message'] = "Flight passenger PNRs and Ticket Numbers have been saved.";
        } else {
            $_SESSION['error_message'] = "Error updating flight passenger details.";
        }
        $stmt_update->close();

        header("Location: process-booking.php?id=" . $booking_id);
        exit;
    }
}

// --- FETCH ALL REQUIRED DATA for displaying the page ---
$sql_booking = "SELECT b.*, u.name as user_name, u.email as user_email, u.mobile_number as user_mobile, u.company_name, u.user_type,
                ti.id as ticket_invoice_id, ti.invoice_number as ticket_invoice_number
                FROM bookings b 
                LEFT JOIN users u ON b.user_id = u.id
                LEFT JOIN ticket_invoices ti ON b.id = ti.booking_id
                WHERE b.id = ?";
$stmt_booking = $conn->prepare($sql_booking);
$stmt_booking->bind_param("i", $booking_id);
$stmt_booking->execute();
$booking = $stmt_booking->get_result()->fetch_assoc();
$stmt_booking->close();

if (!$booking) {
    $_SESSION['error_message'] = "Booking with ID " . e($booking_id) . " was not found.";
    header("Location: manage-flight-requests.php");
    exit;
}

// --- PROCESS FETCHED DATA ---
$flight_details = json_decode($booking['flight_details'], true) ?: [];
$passenger_details_from_json = json_decode($booking['passenger_details'], true) ?: [];

$passengers_from_invoice = [];
if ($booking['booking_type'] === 'group' && !empty($booking['ticket_invoice_id'])) {
    $stmt_pass = $conn->prepare("SELECT * FROM ticket_invoice_passengers WHERE ticket_invoice_id = ? ORDER BY id ASC");
    $stmt_pass->bind_param("i", $booking['ticket_invoice_id']);
    $stmt_pass->execute();
    $passengers_from_invoice = $stmt_pass->get_result()->fetch_all(MYSQLI_ASSOC);
}

$success_message = $_SESSION['success_message'] ?? null;
$error_message = $_SESSION['error_message'] ?? null;
unset($_SESSION['success_message'], $_SESSION['error_message']);

?>
<!DOCTYPE html>
<html lang="en">

<head>
    <meta charset="UTF-8">
    <title>Process Booking #<?= e($booking['booking_ref']) ?></title>
    <link rel="icon" type="image/png" href="../images/logo-icon.png">
    <link rel="stylesheet" href="admin-style.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.1/css/all.min.css">
    <link rel="stylesheet" href="../css/account-style.css">
    <style>
        .page-layout {
            max-width: 1200px;
            margin: 0 auto;
        }

        .info-card {
            background-color: #fff;
            border: 1px solid #e9ecef;
            border-radius: 8px;
            margin-bottom: 1.5rem;
        }

        .info-card-header {
            padding: 1rem 1.5rem;
            border-bottom: 1px solid #f0f0f0;
            font-size: 1.2rem;
            font-weight: 600;
            display: flex;
            align-items: center;
            gap: 0.75rem;
        }

        .info-card-body {
            padding: 1.5rem;
        }

        .info-card-body table.detail-view td {
            padding: 0.5rem 0;
        }

        .info-card-body table.detail-view td:first-child {
            font-weight: bold;
            color: #555;
            width: 120px;
        }

        .status-form select {
            width: 100%;
            padding: 0.5rem;
            font-size: 1rem;
            border: 1px solid #ccc;
            border-radius: 4px;
        }

        .status-form button {
            width: 100%;
            margin-top: 1rem;
        }

        .baggage-info-card {
            background-color: #f8f9fa;
            border: 1px solid #e9ecef;
            border-radius: 8px;
            padding: 1rem;
            margin-top: 1rem;
            font-size: 0.9rem;
        }

        .passenger-table {
            width: 100%;
            border-collapse: collapse;
            margin-top: 1rem;
            font-size: 0.9rem;
        }

        .passenger-table th,
        .passenger-table td {
            border: 1px solid #e9ecef;
            padding: 0.75rem;
            text-align: left;
            vertical-align: middle;
        }

        .passenger-table thead th {
            background-color: #f8f9fa;
            font-weight: 600;
            white-space: nowrap;
        }

        .passenger-table tbody tr:nth-child(even) {
            background-color: #fdfdfd;
        }

        .passenger-table input {
            width: 100%;
            padding: 6px;
            border: 1px solid #ccc;
            border-radius: 4px;
            box-sizing: border-box;
        }

        .actions-card .action-group {
            margin-bottom: 1rem;
        }

        .actions-card .action-group:last-child {
            margin-bottom: 0;
        }

        .actions-card .btn {
            display: block;
            width: 100%;
            text-align: center;
            box-sizing: border-box;
            color: white !important;
        }

        .actions-grid {
            display: grid;
            grid-template-columns: 1fr 1fr;
            gap: 1rem;
        }

        @media(max-width: 768px) {
            .actions-grid {
                grid-template-columns: 1fr;
            }
        }
    </style>
</head>

<body>
    <div class="dashboard-wrapper">
        <?php include 'sidebar.php'; ?>
        <div class="main-content">
            <header class="main-header">
                <div class="user-info"><span>Process Booking</span></div>
            </header>
            <main class="content-body">
                <div class="page-header">
                    <h1 class="page-title">Booking #<?= e($booking['booking_ref']) ?></h1>
                    <a href="manage-flight-requests.php?filter=<?= e($booking['status']) ?>" class="btn btn-secondary"><i class="fas fa-arrow-left"></i> Back to List</a>
                </div>

                <?php if ($success_message): ?><div class="notice success" style="margin: 0 auto 1.5rem auto; max-width: 1200px;"><?= e($success_message) ?></div><?php endif; ?>
                <?php if ($error_message): ?><div class="notice error" style="margin: 0 auto 1.5rem auto; max-width: 1200px;"><?= e($error_message) ?></div><?php endif; ?>

                <div class="page-layout">
                    <div>
                        <div class="info-card">
                            <div class="info-card-header"><i class="fa-solid fa-user-check"></i> Submitted By</div>
                            <div class="info-card-body">
                                <table class="detail-view">
                                    <tr>
                                        <td>Name:</td>
                                        <td><?= e($booking['user_name']) ?></td>
                                    </tr>
                                    <tr>
                                        <td>Type:</td>
                                        <td><?= e(ucfirst($booking['user_type'])) ?></td>
                                    </tr>
                                    <tr>
                                        <td>Company:</td>
                                        <td><?= e($booking['company_name'] ?: 'N/A') ?></td>
                                    </tr>
                                    <tr>
                                        <td>Email:</td>
                                        <td><?= e($booking['user_email']) ?></td>
                                    </tr>
                                    <tr>
                                        <td>Mobile:</td>
                                        <td><?= e($booking['user_mobile']) ?></td>
                                    </tr>
                                </table>
                            </div>
                        </div>

                        <div class="info-card">
                            <div class="info-card-header"><i class="fa-solid fa-plane-up"></i> Flight Itinerary</div>
                            <div class="info-card-body">
                                <?php if ($booking['booking_type'] === 'group' && !empty($flight_details)): ?>
                                    <table class="detail-view">
                                        <tr>
                                            <td>Route:</td>
                                            <td><?= e($flight_details['route'] ?? 'N/A') ?></td>
                                        </tr>
                                        <tr>
                                            <td>Departure:</td>
                                            <td><?= (!empty($flight_details['departure_date']) ? date('d M Y', strtotime($flight_details['departure_date'])) : 'N/A') ?></td>
                                        </tr>
                                        <tr>
                                            <td>Airline:</td>
                                            <td><?= e($flight_details['airline_name'] ?? 'N/A') ?></td>
                                        </tr>
                                    </table>
                                    <?php if (isset($flight_details['flight_details_json']['baggage'])): ?>
                                        <div class="baggage-info-card"><i class="fa-solid fa-suitcase"></i> <strong>Baggage Info:</strong><br><?= nl2br(e($flight_details['flight_details_json']['baggage'])) ?></div>
                                    <?php endif; ?>
                                <?php elseif ($booking['booking_type'] === 'flight' && isset($flight_details['itineraries'])): ?>
                                    <?php foreach ($flight_details['itineraries'] as $index => $itinerary): ?>
                                        <div class="journey-card" style="margin-bottom: 1rem;">
                                            <div class="journey-title"><i class="fa-solid fa-plane-<?= $index === 0 ? 'departure' : 'arrival' ?>"></i> <?= $index === 0 ? 'Outbound' : 'Return/Next Leg' ?></div>
                                            <?php foreach ($itinerary['segments'] as $seg):
                                                // *** FIX IS HERE: Changed st_starts_with to str_starts_with ***
                                                $logo_url = str_starts_with($seg['airlineLogo'], '//') ? 'https:' . $seg['airlineLogo'] : $seg['airlineLogo'];
                                            ?>
                                                <div class="segment-block">
                                                    <div class="segment-header"><img src="<?= e($logo_url) ?>" alt="<?= e($seg['airlineName']) ?>">
                                                        <div><strong><?= e($seg['airlineName']) ?></strong><span><?= e($seg['flightNumber']) ?></span></div>
                                                    </div>
                                                    <div class="segment-path">
                                                        <div class="path-point"><strong><?= e($seg['departure']['time']) ?></strong><span><?= e($seg['departure']['iata']) ?></span></div>
                                                        <div class="path-line"><span class="duration"><?= e($seg['duration']) ?></span></div>
                                                        <div class="path-point"><strong><?= e($seg['arrival']['time']) ?></strong><span><?= e($seg['arrival']['iata']) ?></span></div>
                                                    </div>
                                                </div>
                                            <?php endforeach; ?>
                                        </div>
                                    <?php endforeach; ?>
                                <?php else: ?>
                                    <p>No flight details available for this booking.</p>
                                <?php endif; ?>
                            </div>
                        </div>

                        <div class="info-card">
                            <div class="info-card-header"><i class="fa-solid fa-users"></i> Passenger Details</div>
                            <div class="info-card-body">
                                <form action="process-booking.php?id=<?= e($booking['id']) ?>" method="POST">
                                    <div class="table-responsive">
                                        <table class="passenger-table">
                                            <thead>
                                                <tr>
                                                    <th>Name</th>
                                                    <th>Type</th>
                                                    <th>Passport #</th>
                                                    <th>DOB</th>
                                                    <th>Issue Date</th>
                                                    <th>Expiry Date</th>
                                                    <th>PNR</th>
                                                    <th>Ticket Number</th>
                                                </tr>
                                            </thead>
                                            <tbody>
                                                <?php if ($booking['booking_type'] === 'group' && !empty($passengers_from_invoice)): ?>
                                                    <?php foreach ($passengers_from_invoice as $pax): ?>
                                                        <tr>
                                                            <input type="hidden" name="passenger_id[]" value="<?= e($pax['id']) ?>">
                                                            <td><?= e($pax['full_name']) ?></td>
                                                            <td><?= e($pax['passenger_type']) ?></td>
                                                            <td><?= e($pax['passport_no']) ?></td>
                                                            <td><?= !empty($pax['dob']) ? date('d M Y', strtotime($pax['dob'])) : 'N/A' ?></td>
                                                            <td><?= !empty($pax['passport_issue_date']) ? date('d M Y', strtotime($pax['passport_issue_date'])) : 'N/A' ?></td>
                                                            <td><?= !empty($pax['passport_expiry_date']) ? date('d M Y', strtotime($pax['passport_expiry_date'])) : 'N/A' ?></td>
                                                            <td><input type="text" name="pnr[]" value="<?= e($pax['pnr']) ?>"></td>
                                                            <td><input type="text" name="ticket_number[]" value="<?= e($pax['ticket_number']) ?>"></td>
                                                        </tr>
                                                    <?php endforeach; ?>
                                                <?php elseif ($booking['booking_type'] === 'flight' && !empty($passenger_details_from_json)): ?>
                                                    <?php foreach ($passenger_details_from_json as $type => $passengers):
                                                        if (empty($passengers) || !is_array($passengers)) continue;
                                                        foreach ($passengers as $pax_num => $p):
                                                            $name = e(($p['title'] ?? '') . ' ' . ($p['firstName'] ?? '') . ' ' . ($p['lastName'] ?? ''));
                                                    ?>
                                                            <tr>
                                                                <input type="hidden" name="pax_key[]" value="<?= e($type . '_' . $pax_num) ?>">
                                                                <td><?= trim($name) ?></td>
                                                                <td><?= e(ucfirst(rtrim($type, 's'))) ?></td>
                                                                <td><?= e($p['passport'] ?? 'N/A') ?></td>
                                                                <td><?= !empty($p['dob']) ? date('d M Y', strtotime($p['dob'])) : 'N/A' ?></td>
                                                                <td><?= !empty($p['doi']) ? date('d M Y', strtotime($p['doi'])) : 'N/A' ?></td>
                                                                <td><?= !empty($p['doe']) ? date('d M Y', strtotime($p['doe'])) : 'N/A' ?></td>
                                                                <td><input type="text" name="pnr[]" value="<?= e($p['pnr'] ?? '') ?>"></td>
                                                                <td><input type="text" name="ticket_number[]" value="<?= e($p['ticket_number'] ?? '') ?>"></td>
                                                            </tr>
                                                    <?php endforeach;
                                                    endforeach; ?>
                                                <?php else: ?>
                                                    <tr>
                                                        <td colspan="8" style="text-align:center;">No passenger details available.</td>
                                                    </tr>
                                                <?php endif; ?>
                                            </tbody>
                                        </table>
                                    </div>
                                    <div style="text-align: right; margin-top: 1.5rem;">
                                        <?php if ($booking['booking_type'] === 'group' && !empty($passengers_from_invoice)): ?>
                                            <button type="submit" name="update_passengers" class="btn btn-primary"><i class="fas fa-save"></i> Save Group Passenger Details</button>
                                        <?php elseif ($booking['booking_type'] === 'flight' && !empty($passenger_details_from_json)): ?>
                                            <button type="submit" name="update_flight_passengers" class="btn btn-primary"><i class="fas fa-save"></i> Save Flight Passenger Details</button>
                                        <?php endif; ?>
                                    </div>
                                </form>
                            </div>
                        </div>

                        <div class="info-card actions-card">
                            <div class="info-card-header"><i class="fa-solid fa-tasks"></i> Actions & Status</div>
                            <div class="info-card-body">
                                <div class="actions-grid">
                                    <div class="action-group">
                                        <label style="font-weight: bold; display:block; margin-bottom: 0.5rem;">Invoice & Voucher</label>
                                        <?php if (!empty($booking['ticket_invoice_id'])): ?>
                                            <a href="ticket-invoice-view.php?id=<?= e($booking['ticket_invoice_id']) ?>" target="_blank" class="btn btn-primary"><i class="fas fa-eye"></i> View Ticket Invoice (#<?= e($booking['ticket_invoice_number']) ?>)</a>
                                        <?php else: ?>
                                            <a href="create-ticket-invoice.php?booking_id=<?= e($booking['id']) ?>" class="btn btn-primary"><i class="fas fa-plus-circle"></i> Create Ticket Invoice</a>
                                        <?php endif; ?>
                                        <a href="../ticket-voucher.php?booking_id=<?= e($booking['id']) ?>" target="_blank" rel="noopener noreferrer" class="btn btn-secondary" style="margin-top: 0.5rem;"><i class="fas fa-ticket-alt"></i> View/Print Voucher</a>
                                    </div>
                                    <div class="action-group">
                                        <form action="process-booking.php?id=<?= e($booking['id']) ?>" method="POST" class="status-form">
                                            <input type="hidden" name="booking_id" value="<?= e($booking['id']) ?>">
                                            <div class="form-group">
                                                <label for="status" style="font-weight: bold; display:block; margin-bottom: 0.5rem;">Booking Status</label>
                                                <p><span class="status-badge status-<?= strtolower(e($booking['status'])) ?>"><?= e(ucfirst($booking['status'])) ?></span></p>
                                                <select name="status" id="status" class="form-control" style="margin-top:0.5rem;">
                                                    <option value="pending" <?= $booking['status'] == 'pending' ? 'selected' : '' ?>>Pending</option>
                                                    <option value="confirmed" <?= $booking['status'] == 'confirmed' ? 'selected' : '' ?>>Confirmed</option>
                                                    <option value="cancelled" <?= $booking['status'] == 'cancelled' ? 'selected' : '' ?>>Cancelled</option>
                                                </select>
                                            </div>
                                            <button type="submit" name="update_status" class="btn btn-success">Update Status</button>
                                        </form>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </main>
        </div>
    </div>
</body>

</html>