<?php
session_start();
include_once '../db-config.php';

// --- SECURITY CHECK ---
if (!isset($_SESSION['user_id']) || !isset($_SESSION['user_type']) || $_SESSION['user_type'] !== 'admin') {
    header("location: ../login.php");
    exit;
}

function e($string)
{
    return htmlspecialchars($string ?? '', ENT_QUOTES, 'UTF-8');
}

// --- LOGIC TO PRE-FILL FORM FROM A QUICK BOOKING ---
$quick_booking_data = null;
$quick_booking_passengers = [];
$adult_count = 0;
$child_count = 0;
$infant_count = 0;

$quick_booking_id = (int)($_GET['quick_booking_id'] ?? 0);

if ($quick_booking_id > 0) {
    $stmt_qb = $conn->prepare("SELECT * FROM quick_bookings WHERE id = ?");
    $stmt_qb->bind_param("i", $quick_booking_id);
    $stmt_qb->execute();
    $quick_booking_data = $stmt_qb->get_result()->fetch_assoc();

    $stmt_pass = $conn->prepare("SELECT * FROM quick_booking_passengers WHERE booking_id = ? ORDER BY id ASC");
    $stmt_pass->bind_param("i", $quick_booking_id);
    $stmt_pass->execute();
    $quick_booking_passengers = $stmt_pass->get_result()->fetch_all(MYSQLI_ASSOC);

    foreach ($quick_booking_passengers as $p) {
        switch (strtolower($p['pax_type'])) {
            case 'adult':
                $adult_count++;
                break;
            case 'child':
                $child_count++;
                break;
            case 'infant':
                $infant_count++;
                break;
        }
    }
}

// Fetch users for the dropdown
$users_list = $conn->query("SELECT id, name, user_type FROM users WHERE user_type IN ('customer', 'agent') ORDER BY name ASC")->fetch_all(MYSQLI_ASSOC);
// Fetch vendors for the dropdown
$vendors_list = $conn->query("SELECT id, name FROM vendors ORDER BY name ASC")->fetch_all(MYSQLI_ASSOC);


// Handle form submission
if ($_SERVER["REQUEST_METHOD"] == "POST") {
    $conn->begin_transaction();
    try {
        $submitted_quick_booking_id = (int)($_POST['quick_booking_id'] ?? 0);

        // --- Invoice Data ---
        $user_id = (int)($_POST['user_id'] ?? 0);
        $vendor_id = (int)($_POST['vendor_id'] ?? 0);
        $guest_name = trim($_POST['guest_name']);
        $invoice_number_input = trim($_POST['invoice_number']);
        $issue_date = $_POST['issue_date'];
        $status = $_POST['status'];
        $notes = trim($_POST['notes'] ?? '');

        // --- Fare Breakdown & Financial Data ---
        $adult_qty = (int)($_POST['adult_qty'] ?? 0);
        $adult_rate = (float)($_POST['adult_rate'] ?? 0);
        $adult_cost = (float)($_POST['adult_cost'] ?? 0);
        $child_qty = (int)($_POST['child_qty'] ?? 0);
        $child_rate = (float)($_POST['child_rate'] ?? 0);
        $child_cost = (float)($_POST['child_cost'] ?? 0);
        $infant_qty = (int)($_POST['infant_qty'] ?? 0);
        $infant_rate = (float)($_POST['infant_rate'] ?? 0);
        $infant_cost = (float)($_POST['infant_cost'] ?? 0);
        $service_fee = (float)($_POST['service_fee_pkr'] ?? 0);
        $vendor_commission = (float)($_POST['vendor_commission_pkr'] ?? 0);
        $discount = (float)($_POST['discount_pkr'] ?? 0);

        if (empty($guest_name) || empty($issue_date)) throw new Exception("Guest Name and Issue Date are required.");

        // --- Server-side calculation ---
        $total_fare = ($adult_qty * $adult_rate) + ($child_qty * $child_rate) + ($infant_qty * $infant_rate);
        $grand_total = ($total_fare + $service_fee) - $discount;
        $total_fare_cost = ($adult_qty * $adult_cost) + ($child_qty * $child_cost) + ($infant_qty * $infant_cost);
        $grand_total_cost = $total_fare_cost; // Vendor commission is now handled differently, if at all

        // --- INSERT into main invoice table with quick_booking_id ---
        $sql_invoice = "INSERT INTO ticket_invoices (quick_booking_id, user_id, vendor_id, guest_name, invoice_number, issue_date, status, adult_qty, adult_rate, adult_cost, child_qty, child_rate, child_cost, infant_qty, infant_rate, infant_cost, total_fare_pkr, total_fare_pkr_cost, service_fee_pkr, vendor_commission_pkr, discount_pkr, grand_total_pkr, grand_total_pkr_cost, notes) VALUES (?, ?, ?, ?, NULL, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)";
        $stmt_invoice = $conn->prepare($sql_invoice);
        $quick_booking_id_to_save = $submitted_quick_booking_id > 0 ? $submitted_quick_booking_id : null;
        $user_id_to_save = $user_id > 0 ? $user_id : null;
        $vendor_id_to_save = $vendor_id > 0 ? $vendor_id : null;
        $stmt_invoice->bind_param("iiisssiddisddidddddddds", $quick_booking_id_to_save, $user_id_to_save, $vendor_id_to_save, $guest_name, $issue_date, $status, $adult_qty, $adult_rate, $adult_cost, $child_qty, $child_rate, $child_cost, $infant_qty, $infant_rate, $infant_cost, $total_fare, $total_fare_cost, $service_fee, $vendor_commission, $discount, $grand_total, $grand_total_cost, $notes);
        $stmt_invoice->execute();
        $invoice_id = $conn->insert_id;
        if (!$invoice_id) throw new Exception("Failed to create main invoice record.");

        // --- Generate and update the invoice number ---
        $final_invoice_number = $invoice_number_input;
        if (empty($final_invoice_number)) {
            $final_invoice_number = "TKT-" . str_pad($invoice_id, 4, '0', STR_PAD_LEFT);
        }
        $update_sql = "UPDATE ticket_invoices SET invoice_number = ? WHERE id = ?";
        $stmt_update = $conn->prepare($update_sql);
        $stmt_update->bind_param("si", $final_invoice_number, $invoice_id);
        $stmt_update->execute();

        // --- 2. INSERT passengers ---
        if (!empty($_POST['passenger_name'])) {
            $sql_pass = "INSERT INTO ticket_invoice_passengers (ticket_invoice_id, full_name, passenger_type, passport_no, pnr, ticket_number, dob, passport_issue_date, passport_expiry_date) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)";
            $stmt_pass = $conn->prepare($sql_pass);
            foreach ($_POST['passenger_name'] as $key => $name) {
                if (!empty(trim($name))) {
                    $type = $_POST['passenger_type'][$key];
                    $passport_no = $_POST['passenger_passport'][$key];
                    $pnr = $_POST['passenger_pnr'][$key] ?? '';
                    $ticket_no = $_POST['passenger_ticket_no'][$key] ?? '';
                    $dob = !empty($_POST['passenger_dob'][$key]) ? $_POST['passenger_dob'][$key] : null;
                    $pass_issue_date = !empty($_POST['passenger_issue_date'][$key]) ? $_POST['passenger_issue_date'][$key] : null;
                    $pass_expiry_date = !empty($_POST['passenger_expiry_date'][$key]) ? $_POST['passenger_expiry_date'][$key] : null;

                    $stmt_pass->bind_param("issssssss", $invoice_id, $name, $type, $passport_no, $pnr, $ticket_no, $dob, $pass_issue_date, $pass_expiry_date);
                    $stmt_pass->execute();
                }
            }
        }

        // --- 3. INSERT flights ---
        if (!empty($_POST['flight_airline'])) {
            $sql_flight = "INSERT INTO ticket_invoice_flights (ticket_invoice_id, airline, flight_no, sector, departure_datetime, arrival_datetime) VALUES (?, ?, ?, ?, ?, ?)";
            $stmt_flight = $conn->prepare($sql_flight);
            foreach ($_POST['flight_airline'] as $key => $airline) {
                if (!empty(trim($airline))) {
                    $flight_no = $_POST['flight_no'][$key];
                    $sector = $_POST['flight_sector'][$key];
                    $dep_dt = !empty($_POST['flight_departure'][$key]) ? $_POST['flight_departure'][$key] : null;
                    $arr_dt = !empty($_POST['flight_arrival'][$key]) ? $_POST['flight_arrival'][$key] : null;

                    $stmt_flight->bind_param("isssss", $invoice_id, $airline, $flight_no, $sector, $dep_dt, $arr_dt);
                    $stmt_flight->execute();
                }
            }
        }

        $conn->commit();
        $_SESSION['success_message'] = "Ticket Invoice #{$final_invoice_number} created successfully!";
        header("Location: ticket-invoices-list.php");
        exit();
    } catch (Exception $e) {
        $conn->rollback();
        $_SESSION['error_message'] = "Error creating invoice: " . $e->getMessage();
        $redirect_url = "ticket-invoice-create.php";
        if ($submitted_quick_booking_id > 0) {
            $redirect_url .= "?quick_booking_id=" . $submitted_quick_booking_id;
        }
        header("Location: " . $redirect_url);
        exit();
    }
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Create New Ticket Invoice</title>
    <link rel="icon" type="image/png" href="../images/logo-icon.png">
    <link rel="stylesheet" href="admin-style.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.1/css/all.min.css">
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdn.jsdelivr.net/npm/select2@4.1.0-rc.0/dist/css/select2.min.css" rel="stylesheet" />
    <style>
        .select2-container--default .select2-selection--single { height: calc(1.5em + .75rem + 2px); padding: .375rem .75rem; border: 1px solid #ced4da; }
        .select2-container--default .select2-selection--single .select2-selection__rendered { line-height: 1.5; padding-left: 0;}
        .select2-container--default .select2-selection--single .select2-selection__arrow { height: calc(1.5em + .75rem); }
        .dynamic-table th, .dynamic-table td { vertical-align: middle; white-space: nowrap; }
        .summary-grid { display: grid; grid-template-columns: 1fr auto; gap: 5px 20px; align-items: center; }
        .summary-label { font-weight: bold; text-align: right; }
        .grand-total { font-size: 1.1em; font-weight: bold; }
        .profit-label { color: var(--success-color); }
        .cost-label { color: var(--danger-color); }
        
        .main-content { min-width: 0; }
        .table-responsive > .table { min-width: 1400px; }
        .dynamic-table th { white-space: nowrap; }
    </style>
</head>

<body>
    <div class="dashboard-wrapper">
        <?php include 'sidebar.php'; ?>
        <div class="main-content">
            <header class="main-header">
                <button class="menu-toggle" id="menu-toggle"><i class="fas fa-bars"></i></button>
                <div class="user-info"><span>Create New Airline Ticket Invoice</span></div>
            </header>
            <main class="content-body">
                <div class="content-card">
                    <div class="card-body">
                        <?php if (isset($_SESSION['error_message'])): ?>
                        <div class="alert alert-danger"><?= e($_SESSION['error_message']); unset($_SESSION['error_message']); ?></div>
                        <?php endif; ?>

                        <form action="ticket-invoice-create.php" method="POST" id="invoice-main-form">
                            <input type="hidden" name="quick_booking_id" value="<?= e($quick_booking_id) ?>">

                             <fieldset class="mb-4"><legend class="h5 fw-bold"><i class="fas fa-file-invoice"></i> Main Details</legend>
                                <div class="row g-3">
                                    <div class="col-xl-3 col-lg-6 col-md-6 col-12"><label for="guest_name" class="form-label">Lead Passenger*</label><input type="text" class="form-control" id="guest_name" name="guest_name" required value="<?= e($quick_booking_data['customer_name'] ?? '') ?>"></div>
                                    <div class="col-xl-3 col-lg-6 col-md-6 col-12"><label for="invoice_number" class="form-label">Invoice No</label><input type="text" class="form-control" id="invoice_number" name="invoice_number" placeholder="Auto-generates"></div>
                                    <div class="col-xl-3 col-lg-6 col-md-6 col-12"><label for="issue_date" class="form-label">Dated*</label><input type="date" class="form-control" id="issue_date" name="issue_date" required value="<?= date('Y-m-d') ?>"></div>
                                    <div class="col-xl-3 col-lg-6 col-md-6 col-12"><label for="status" class="form-label">Status</label><select name="status" id="status" class="form-select"><option value="Tentative" selected>Tentative</option><option value="Approved">Approved</option><option value="Cancelled">Cancelled</option></select></div>
                                </div>
                            </fieldset>

                            <section class="mb-4"><h3 class="h5 fw-bold"><i class="fas fa-users"></i> Passenger Information</h3>
                                <div class="table-responsive"><table class="table table-bordered dynamic-table">
                                    <thead class="table-light"><tr><th>Full Name*</th><th>Type</th><th>Passport No.</th><th>Date of Birth</th><th>Passport Issue</th><th>Passport Expiry</th><th>PNR</th><th>Ticket No.</th><th class="text-center">Action</th></tr></thead>
                                    <tbody id="passengers-tbody">
                                        <?php foreach ($quick_booking_passengers as $p): ?>
                                        <tr>
                                            <td><input type="text" class="form-control" name="passenger_name[]" required value="<?= e($p['full_name']) ?>"></td>
                                            <td><select class="form-select" name="passenger_type[]">
                                                <option value="Adult" <?= strtolower($p['pax_type']) == 'adult' ? 'selected' : '' ?>>Adult</option>
                                                <option value="Child" <?= strtolower($p['pax_type']) == 'child' ? 'selected' : '' ?>>Child</option>
                                                <option value="Infant" <?= strtolower($p['pax_type']) == 'infant' ? 'selected' : '' ?>>Infant</option>
                                            </select></td>
                                            <td><input type="text" class="form-control" name="passenger_passport[]" value="<?= e($p['passport_no']) ?>"></td>
                                            <td><input type="date" class="form-control" name="passenger_dob[]" value="<?= e($p['dob']) ?>"></td>
                                            <td><input type="date" class="form-control" name="passenger_issue_date[]" value="<?= e($p['passport_issue_date']) ?>"></td>
                                            <td><input type="date" class="form-control" name="passenger_expiry_date[]" value="<?= e($p['passport_expiry_date']) ?>"></td>
                                            <td><input type="text" class="form-control" name="passenger_pnr[]"></td>
                                            <td><input type="text" class="form-control" name="passenger_ticket_no[]"></td>
                                            <td class="text-center"><button type="button" class="btn btn-sm btn-outline-danger" onclick="removeRow(this)"><i class="fas fa-trash"></i></button></td>
                                        </tr>
                                        <?php endforeach; ?>
                                    </tbody>
                                </table></div><button type="button" class="btn btn-secondary mt-2 btn-sm" onclick="addPassengerRow()"><i class="fas fa-plus"></i> Add Passenger</button>
                            </section>

                            <section class="mb-4"><h3 class="h5 fw-bold"><i class="fas fa-plane-departure"></i> Flight Itinerary</h3>
                                <div class="table-responsive"><table class="table table-bordered dynamic-table">
                                    <thead class="table-light"><tr><th>Airline*</th><th>Flight No.</th><th>Sector</th><th>Departure</th><th>Arrival</th><th class="text-center">Action</th></tr></thead>
                                    <tbody id="flights-tbody"></tbody>
                                </table></div><button type="button" class="btn btn-secondary mt-2 btn-sm" onclick="addFlightRow()"><i class="fas fa-plus"></i> Add Flight Leg</button>
                            </section>
                            
                            <fieldset class="mb-4"><legend class="h5 fw-bold"><i class="fas fa-cogs"></i> Finalization & Financials</legend>
                                <div class="row g-4">
                                    <div class="col-lg-7">
                                        <div class="row g-3 mb-3">
                                            <div class="col-md-6"><label for="user_selector" class="form-label">Assign to User</label><select id="user_selector" name="user_id" class="form-select">
                                                <option value="0" data-name="" selected>-- Direct Customer --</option>
                                                <?php foreach ($users_list as $user): ?>
                                                <option value="<?= $user['id'] ?>" data-name="<?= e($user['name']) ?>" <?= ($quick_booking_data && $quick_booking_data['user_id'] == $user['id']) ? 'selected' : '' ?>><?= e($user['name']) ?> - [<?= ($user['user_type'] === 'agent') ? 'A' : 'C' ?>]</option>
                                                <?php endforeach; ?></select>
                                            </div>
                                            <div class="col-md-6"><label for="vendor_selector" class="form-label">Assign to Vendor</label><select id="vendor_selector" name="vendor_id" class="form-select">
                                                <option value="0" selected>-- No Vendor --</option>
                                                <?php foreach ($vendors_list as $vendor): ?><option value="<?= $vendor['id'] ?>"><?= e($vendor['name']) ?></option><?php endforeach; ?></select>
                                            </div>
                                        </div>
                                        <div class="mb-3"><label for="notes" class="form-label">Terms & Conditions / Notes</label><textarea id="notes" name="notes" class="form-control" rows="12">All fares are subject to change without prior notice...</textarea></div>
                                    </div>
                                    <div class="col-lg-5">
                                        <div class="bg-light p-3 border rounded">
                                            <h4 class="h6 text-center mb-3">Invoice Summary</h4>
                                            <div class="mb-4">
                                                <h6 class="mb-3">Fare Breakdown</h6>
                                                <div class="row g-2 mb-2"><div class="col-3 fw-bold small">Type</div><div class="col-3 fw-bold small">Qty</div><div class="col-3 fw-bold small">Sale Rate</div><div class="col-3 fw-bold small">Cost</div></div>
                                                <div class="row g-2 mb-2 align-items-center"><div class="col-3"><label for="adult_qty" class="form-label mb-0">Adults</label></div><div class="col-3"><input type="number" class="form-control form-control-sm" id="adult_qty" name="adult_qty" value="<?= $adult_count > 0 ? $adult_count : 1 ?>" min="0" oninput="calculateTotals()"></div><div class="col-3"><input type="number" class="form-control form-control-sm" id="adult_rate" name="adult_rate" value="0.00" step="0.01" oninput="calculateTotals()"></div><div class="col-3"><input type="number" class="form-control form-control-sm" id="adult_cost" name="adult_cost" value="0.00" step="0.01" oninput="calculateTotals()"></div></div>
                                                <div class="row g-2 mb-2 align-items-center"><div class="col-3"><label for="child_qty" class="form-label mb-0">Children</label></div><div class="col-3"><input type="number" class="form-control form-control-sm" id="child_qty" name="child_qty" value="<?= $child_count ?>" min="0" oninput="calculateTotals()"></div><div class="col-3"><input type="number" class="form-control form-control-sm" id="child_rate" name="child_rate" value="0.00" step="0.01" oninput="calculateTotals()"></div><div class="col-3"><input type="number" class="form-control form-control-sm" id="child_cost" name="child_cost" value="0.00" step="0.01" oninput="calculateTotals()"></div></div>
                                                <div class="row g-2 align-items-center"><div class="col-3"><label for="infant_qty" class="form-label mb-0">Infants</label></div><div class="col-3"><input type="number" class="form-control form-control-sm" id="infant_qty" name="infant_qty" value="<?= $infant_count ?>" min="0" oninput="calculateTotals()"></div><div class="col-3"><input type="number" class="form-control form-control-sm" id="infant_rate" name="infant_rate" value="0.00" step="0.01" oninput="calculateTotals()"></div><div class="col-3"><input type="number" class="form-control form-control-sm" id="infant_cost" name="infant_cost" value="0.00" step="0.01" oninput="calculateTotals()"></div></div>
                                            </div>
                                            <div class="summary-grid">
                                                <div class="summary-label">Total Fare:</div><div id="summary-total-fare" class="text-end">0.00</div>
                                                <div class="summary-label">Service Fee:</div><div><input type="number" name="service_fee_pkr" id="service-fee-pkr-input" class="form-control form-control-sm text-end" value="0.00" step="0.01" oninput="calculateTotals()"></div>
                                                <div class="summary-label">Discount:</div><div><input type="number" name="discount_pkr" id="discount-pkr-input" class="form-control form-control-sm text-end" value="0.00" step="0.01" oninput="calculateTotals()"></div>
                                                <hr style="grid-column: 1 / -1; margin: 5px 0;"><div class="summary-label grand-total">Grand Total:</div><div class="grand-total text-end" id="summary-grand-total">0.00</div>
                                                <hr style="grid-column: 1 / -1; margin: 10px 0; border-style: dashed;">
                                                <div class="summary-label cost-label">Vendor Commission:</div><div><input type="number" name="vendor_commission_pkr" id="vendor-commission-pkr-input" class="form-control form-control-sm text-end" value="0.00" step="0.01" oninput="calculateTotals()"></div>
                                                <div class="summary-label cost-label">Total Cost:</div><div class="cost-label text-end" id="summary-total-cost">0.00</div>
                                                <div class="summary-label profit-label grand-total">Net Profit:</div><div class="profit-label grand-total text-end" id="summary-net-profit">0.00</div>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </fieldset>
                            
                            <div class="text-center my-4"><button type="submit" class="btn btn-primary btn-lg"><i class="fas fa-check-circle"></i> Create Invoice</button></div>
                        </form>
                    </div>
                </div>
            </main>
        </div>
    </div>
    
    <script src="https://code.jquery.com/jquery-3.6.0.min.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/select2@4.1.0-rc.0/dist/js/select2.min.js"></script>
    <script>
        const addRow = (tbodyId, rowHTML) => {
            const tableBody = document.getElementById(tbodyId);
            const newRow = tableBody.insertRow();
            newRow.innerHTML = rowHTML;
        };
        const removeRow = (btn) => {
            btn.closest('tr').remove();
            calculateTotals();
        };

        function addPassengerRow() {
            const html = `
                <td><input type="text" class="form-control form-control-sm" name="passenger_name[]" required></td>
                <td><select class="form-select form-select-sm" name="passenger_type[]"><option value="Adult" selected>Adult</option><option value="Child">Child</option><option value="Infant">Infant</option></select></td>
                <td><input type="text" class="form-control form-control-sm" name="passenger_passport[]"></td>
                <td><input type="date" class="form-control form-control-sm" name="passenger_dob[]"></td>
                <td><input type="date" class="form-control form-control-sm" name="passenger_issue_date[]"></td>
                <td><input type="date" class="form-control form-control-sm" name="passenger_expiry_date[]"></td>
                <td><input type="text" class="form-control form-control-sm" name="passenger_pnr[]"></td>
                <td><input type="text" class="form-control form-control-sm" name="passenger_ticket_no[]"></td>
                <td class="text-center"><button type="button" class="btn btn-sm btn-outline-danger" onclick="removeRow(this)"><i class="fas fa-trash"></i></button></td>`;
            addRow('passengers-tbody', html);
        }

        function addFlightRow() {
            const html = `
                <td><input type="text" class="form-control form-control-sm" name="flight_airline[]" required></td>
                <td><input type="text" class="form-control form-control-sm" name="flight_no[]"></td>
                <td><input type="text" class="form-control form-control-sm" name="flight_sector[]" placeholder="e.g. KHI-DXB"></td>
                <td><input type="datetime-local" class="form-control form-control-sm" name="flight_departure[]"></td>
                <td><input type="datetime-local" class="form-control form-control-sm" name="flight_arrival[]"></td>
                <td class="text-center"><button type="button" class="btn btn-sm btn-outline-danger" onclick="removeRow(this)"><i class="fas fa-trash"></i></button></td>`;
            addRow('flights-tbody', html);
        }

        function calculateTotals() {
            const format = (num) => isNaN(num) ? '0.00' : num.toLocaleString('en-US', { minimumFractionDigits: 2, maximumFractionDigits: 2 });
            const getVal = (id) => parseFloat(document.getElementById(id).value) || 0;
            const totalFare = (getVal('adult_qty') * getVal('adult_rate')) + (getVal('child_qty') * getVal('child_rate')) + (getVal('infant_qty') * getVal('infant_rate'));
            const totalCost = (getVal('adult_qty') * getVal('adult_cost')) + (getVal('child_qty') * getVal('child_cost')) + (getVal('infant_qty') * getVal('infant_cost'));
            const serviceFee = getVal('service-fee-pkr-input');
            const vendorCommission = getVal('vendor-commission-pkr-input');
            const discount = getVal('discount-pkr-input');
            const grandTotal = (totalFare + serviceFee) - discount;
            const grandTotalCost = totalCost; // Updated calculation
            const netProfit = grandTotal - grandTotalCost - vendorCommission; // Updated calculation
            
            document.getElementById('summary-total-fare').textContent = format(totalFare);
            document.getElementById('summary-grand-total').textContent = format(grandTotal);
            document.getElementById('summary-total-cost').textContent = format(grandTotalCost);
            document.getElementById('summary-net-profit').textContent = format(netProfit);
        }

        document.addEventListener('DOMContentLoaded', () => {
            $('#user_selector, #vendor_selector').select2({ width: '100%' });
            
            $('#user_selector').on('change', function() {
                const selectedOption = $(this).find('option:selected');
                const guestNameInput = $('#guest_name');
                if (guestNameInput.val() === '' || guestNameInput.data('auto-filled')) {
                    guestNameInput.val(selectedOption.data('name') || '');
                    guestNameInput.data('auto-filled', true);
                }
            });
            
            if ($('#user_selector').val() !== '0' && $('#guest_name').val() === '') {
                 $('#user_selector').trigger('change');
            }

            if (document.getElementById('passengers-tbody').rows.length === 0) addPassengerRow();
            if (document.getElementById('flights-tbody').rows.length === 0) addFlightRow();
            calculateTotals();
        });

        document.addEventListener('contextmenu', function(e) { e.preventDefault(); });
    </script>
</body>
</html>