<?php
session_start();
include_once '../db-config.php';

// --- SECURITY CHECK ---
if (!isset($_SESSION['user_id']) || !isset($_SESSION['user_type']) || $_SESSION['user_type'] !== 'admin') {
    header("location: ../login.php");
    exit;
}

function e($string)
{
    return htmlspecialchars($string ?? '', ENT_QUOTES, 'UTF-8');
}

// --- 1. SETUP FILTERS & VENDOR LIST ---
$filter_vendor_id = (int)($_GET['vendor_id'] ?? 0);
$filter_start_date = $_GET['start_date'] ?? '';
$filter_end_date = $_GET['end_date'] ?? '';

$vendors_list = $conn->query("SELECT id, name FROM vendors ORDER BY name ASC")->fetch_all(MYSQLI_ASSOC);
$selected_vendor_details = null;
$transactions = [];
$opening_balance = 0;
$running_balance = 0;

if ($filter_vendor_id > 0) {
    // --- 2. FETCH SELECTED VENDOR DETAILS ---
    $stmt_vendor = $conn->prepare("SELECT * FROM vendors WHERE id = ?");
    $stmt_vendor->bind_param("i", $filter_vendor_id);
    $stmt_vendor->execute();
    $selected_vendor_details = $stmt_vendor->get_result()->fetch_assoc();
    $stmt_vendor->close();

    // --- 3. CALCULATE OPENING BALANCE ---
    // The opening balance is calculated by running the same transaction queries but for dates *before* the start date.
    if (!empty($filter_start_date)) {
        $opening_sql_parts = [];
        $opening_params = [];
        $opening_types = '';

        // Part 1: Costs from Package Invoices (New Detailed Logic)
        $cost_sql_opening = "
            SELECT 
                (
                    CASE 
                        -- Scenario 1: Main vendor is assigned AND NO other vendors are assigned to any sub-item.
                        WHEN 
                            i.vendor_id = ? 
                            AND i.pilgrims_vendor_id IS NULL
                            AND i.transport_vendor_id IS NULL
                            AND i.tickets_vendor_id IS NULL
                            AND NOT EXISTS (SELECT 1 FROM invoice_hotels ih_check WHERE ih_check.invoice_id = i.id AND ih_check.vendor_id IS NOT NULL)
                            AND NOT EXISTS (SELECT 1 FROM invoice_other_services ios_check WHERE ios_check.invoice_id = i.id AND ios_check.vendor_id IS NOT NULL)
                        THEN i.grand_total_pkr_cost
                        
                        -- Scenario 2: Granular calculation for multiple vendors or specific assignments.
                        ELSE 
                            -- Visa cost
                            (SELECT COALESCE(SUM(ip.visa_price_sar_cost), 0) FROM invoice_pilgrims ip WHERE ip.invoice_id = i.id AND i.pilgrims_vendor_id = ?) * i.exchange_rate +
                            -- Transport cost (assuming cost is in SAR)
                            (SELECT COALESCE(SUM(it.total_amount_cost), 0) FROM invoice_transports it WHERE it.invoice_id = i.id AND i.transport_vendor_id = ?) * i.exchange_rate +
                            -- Ticket cost (already PKR)
                            (SELECT COALESCE(SUM(iat.total_amount_cost), 0) FROM invoice_airline_tickets iat WHERE iat.invoice_id = i.id AND i.tickets_vendor_id = ?) +
                            -- Hotel cost (assigned to specific vendor OR default to main vendor)
                            (SELECT COALESCE(SUM(ih.total_sar_cost), 0) FROM invoice_hotels ih WHERE ih.invoice_id = i.id AND (ih.vendor_id = ? OR (i.vendor_id = ? AND ih.vendor_id IS NULL))) * i.exchange_rate +
                            -- Other Services cost (assigned to specific vendor OR default to main vendor)
                            (SELECT COALESCE(SUM(ios.total_amount_cost), 0) FROM invoice_other_services ios WHERE ios.invoice_id = i.id AND (ios.vendor_id = ? OR (i.vendor_id = ? AND ios.vendor_id IS NULL))) * i.exchange_rate
                    END
                ) as credit, 
                0 as debit
            FROM invoices i
            WHERE i.issue_date < ? AND (
                i.vendor_id = ? OR 
                i.pilgrims_vendor_id = ? OR
                i.transport_vendor_id = ? OR
                i.tickets_vendor_id = ? OR
                EXISTS (SELECT 1 FROM invoice_hotels ih_ex WHERE ih_ex.invoice_id = i.id AND ih_ex.vendor_id = ?) OR
                EXISTS (SELECT 1 FROM invoice_other_services ios_ex WHERE ios_ex.invoice_id = i.id AND ios_ex.vendor_id = ?)
            )
        ";
        $opening_sql_parts[] = "($cost_sql_opening)";
        array_push($opening_params, $filter_vendor_id, $filter_vendor_id, $filter_vendor_id, $filter_vendor_id, $filter_vendor_id, $filter_vendor_id, $filter_vendor_id, $filter_vendor_id, $filter_start_date, $filter_vendor_id, $filter_vendor_id, $filter_vendor_id, $filter_vendor_id, $filter_vendor_id, $filter_vendor_id);
        $opening_types .= 'iiiiiiiisiiiiii';

        // Part 2: Costs from Standalone Ticket Invoices
        $opening_sql_parts[] = "(SELECT ti.grand_total_pkr_cost as credit, 0 as debit FROM ticket_invoices ti WHERE ti.vendor_id = ? AND ti.issue_date < ?)";
        array_push($opening_params, $filter_vendor_id, $filter_start_date);
        $opening_types .= 'is';

        // Part 3: Payments
        $opening_sql_parts[] = "(SELECT p.credit_amount as credit, p.debit_amount as debit FROM payments p WHERE p.vendor_id = ? AND p.payment_date < ?)";
        array_push($opening_params, $filter_vendor_id, $filter_start_date);
        $opening_types .= 'is';

        $opening_balance_sql = "SELECT SUM(credit - debit) as opening_balance FROM (" . implode(" UNION ALL ", $opening_sql_parts) . ") AS opening_transactions";

        $stmt_opening = $conn->prepare($opening_balance_sql);
        $stmt_opening->bind_param($opening_types, ...$opening_params);
        $stmt_opening->execute();
        $opening_balance = (float)($stmt_opening->get_result()->fetch_assoc()['opening_balance'] ?? 0);
        $stmt_opening->close();
    }

    // --- 4. FETCH ALL TRANSACTIONS FOR THE PERIOD ---
    $sql_parts = [];
    $params = [];
    $types = '';

    // Query 1: Package Invoice Costs (New Logic)
    $package_cost_sql = "
    SELECT 
        i.id as transaction_id, 
        i.issue_date as date, 
        'Booking Cost' as type, 
        i.invoice_number as number, 
        CONCAT(i.guest_name, ' x ', (SELECT COUNT(*) FROM invoice_pilgrims ip_count WHERE ip_count.invoice_id = i.id), ' Pax') as particulars,
        0 as debit,
        (
            CASE 
                -- Scenario 1: Main vendor is assigned AND NO other vendors are assigned to any sub-item.
                WHEN 
                    i.vendor_id = ? 
                    AND i.pilgrims_vendor_id IS NULL
                    AND i.transport_vendor_id IS NULL
                    AND i.tickets_vendor_id IS NULL
                    AND NOT EXISTS (SELECT 1 FROM invoice_hotels ih_check WHERE ih_check.invoice_id = i.id AND ih_check.vendor_id IS NOT NULL)
                    AND NOT EXISTS (SELECT 1 FROM invoice_other_services ios_check WHERE ios_check.invoice_id = i.id AND ios_check.vendor_id IS NOT NULL)
                THEN i.grand_total_pkr_cost
                
                -- Scenario 2: Granular calculation for multiple vendors or specific assignments.
                ELSE 
                    -- Visa cost
                    (SELECT COALESCE(SUM(ip.visa_price_sar_cost), 0) FROM invoice_pilgrims ip WHERE ip.invoice_id = i.id AND i.pilgrims_vendor_id = ?) * i.exchange_rate +
                    -- Transport cost (NOTE: Assuming total_amount_cost is in SAR, adjust if it's already PKR)
                    (SELECT COALESCE(SUM(it.total_amount_cost), 0) FROM invoice_transports it WHERE it.invoice_id = i.id AND i.transport_vendor_id = ?) * i.exchange_rate +
                    -- Ticket cost (already PKR)
                    (SELECT COALESCE(SUM(iat.total_amount_cost), 0) FROM invoice_airline_tickets iat WHERE iat.invoice_id = i.id AND i.tickets_vendor_id = ?) +
                    -- Hotel cost (assigned to specific vendor OR default to main vendor if unassigned)
                    (SELECT COALESCE(SUM(ih.total_sar_cost), 0) FROM invoice_hotels ih WHERE ih.invoice_id = i.id AND (ih.vendor_id = ? OR (i.vendor_id = ? AND ih.vendor_id IS NULL))) * i.exchange_rate +
                    -- Other Services cost (assigned to specific vendor OR default to main vendor if unassigned)
                    (SELECT COALESCE(SUM(ios.total_amount_cost), 0) FROM invoice_other_services ios WHERE ios.invoice_id = i.id AND (ios.vendor_id = ? OR (i.vendor_id = ? AND ios.vendor_id IS NULL))) * i.exchange_rate
            END
        ) as credit,
        'package' as source
    FROM invoices i
    WHERE (
        i.vendor_id = ? OR 
        i.pilgrims_vendor_id = ? OR
        i.transport_vendor_id = ? OR
        i.tickets_vendor_id = ? OR
        EXISTS (SELECT 1 FROM invoice_hotels ih_ex WHERE ih_ex.invoice_id = i.id AND ih_ex.vendor_id = ?) OR
        EXISTS (SELECT 1 FROM invoice_other_services ios_ex WHERE ios_ex.invoice_id = i.id AND ios_ex.vendor_id = ?)
    ) ";
    $local_params = [$filter_vendor_id, $filter_vendor_id, $filter_vendor_id, $filter_vendor_id, $filter_vendor_id, $filter_vendor_id, $filter_vendor_id, $filter_vendor_id, $filter_vendor_id, $filter_vendor_id, $filter_vendor_id, $filter_vendor_id, $filter_vendor_id, $filter_vendor_id];
    $local_types = 'iiiiiiiiiiiiii';
    if (!empty($filter_start_date)) {
        $package_cost_sql .= " AND i.issue_date >= ?";
        $local_params[] = $filter_start_date;
        $local_types .= 's';
    }
    if (!empty($filter_end_date)) {
        $package_cost_sql .= " AND i.issue_date <= ?";
        $local_params[] = $filter_end_date;
        $local_types .= 's';
    }

    $sql_parts[] = "($package_cost_sql)";
    $params = array_merge($params, $local_params);
    $types .= $local_types;

    // Query 2: Standalone Ticket Invoice Costs
    $ticket_cost_sql = "
    SELECT 
        ti.id as transaction_id, 
        ti.issue_date as date, 
        'Ticket Cost' as type, 
        ti.invoice_number as number, 
        CONCAT('Ticket for ', ti.guest_name) as particulars, 
        0 as debit, 
        ti.grand_total_pkr_cost as credit, 
        'ticket' as source 
    FROM ticket_invoices ti 
    WHERE ti.vendor_id = ? AND ti.grand_total_pkr_cost > 0";
    $local_params = [$filter_vendor_id];
    $local_types = 'i';
    if (!empty($filter_start_date)) {
        $ticket_cost_sql .= " AND ti.issue_date >= ?";
        $local_params[] = $filter_start_date;
        $local_types .= 's';
    }
    if (!empty($filter_end_date)) {
        $ticket_cost_sql .= " AND ti.issue_date <= ?";
        $local_params[] = $filter_end_date;
        $local_types .= 's';
    }
    $sql_parts[] = "($ticket_cost_sql)";
    $params = array_merge($params, $local_params);
    $types .= $local_types;

    // Query 3: Payments
    $payment_sql = "
    SELECT 
        p.id as transaction_id, 
        p.payment_date as date, 
        CASE 
            WHEN p.debit_amount > 0 AND p.payment_method IN ('Bank Transfer', 'Card') THEN 'BP' 
            WHEN p.debit_amount > 0 AND p.payment_method = 'Cash' THEN 'CP' 
            WHEN p.credit_amount > 0 AND p.payment_method IN ('Bank Transfer', 'Card') THEN 'BR' 
            WHEN p.credit_amount > 0 AND p.payment_method = 'Cash' THEN 'CR' 
            ELSE 'Payment' 
        END as type, 
        CONCAT(
            CASE 
                WHEN p.debit_amount > 0 AND p.payment_method IN ('Bank Transfer', 'Card') THEN 'BP-' 
                WHEN p.debit_amount > 0 AND p.payment_method = 'Cash' THEN 'CP-' 
                WHEN p.credit_amount > 0 AND p.payment_method IN ('Bank Transfer', 'Card') THEN 'BR-' 
                WHEN p.credit_amount > 0 AND p.payment_method = 'Cash' THEN 'CR-' 
                ELSE 'PAY-' 
            END, p.id
        ) as number, 
        p.notes as particulars, 
        p.debit_amount as debit, 
        p.credit_amount as credit, 
        'payment' as source 
    FROM payments p 
    WHERE p.vendor_id = ?";
    $local_params = [$filter_vendor_id];
    $local_types = 'i';
    if (!empty($filter_start_date)) {
        $payment_sql .= " AND p.payment_date >= ?";
        $local_params[] = $filter_start_date;
        $local_types .= 's';
    }
    if (!empty($filter_end_date)) {
        $payment_sql .= " AND p.payment_date <= ?";
        $local_params[] = $filter_end_date;
        $local_types .= 's';
    }
    $sql_parts[] = "($payment_sql)";
    $params = array_merge($params, $local_params);
    $types .= $local_types;

    // --- Final Combination and Execution ---
    $final_sql = "SELECT * FROM (
        SELECT transaction_id, date, type, number, particulars, debit, credit, source FROM (" . implode(" UNION ALL ", $sql_parts) . ") AS all_transactions
    ) AS final_result
    WHERE credit > 0 OR debit > 0
    ORDER BY date ASC, transaction_id ASC";

    $stmt_period = $conn->prepare($final_sql);
    if ($stmt_period) {
        if (!empty($params)) {
            $stmt_period->bind_param($types, ...$params);
        }
        $stmt_period->execute();
        $transactions = $stmt_period->get_result()->fetch_all(MYSQLI_ASSOC);
        $stmt_period->close();
    }
}
?>
<!DOCTYPE html>
<html lang="en">

<head>
    <meta charset="UTF-8">
    <title>Vendor Ledger</title>
    <link rel="icon" type="image/png" href="../images/logo-icon.png">

    <link rel="stylesheet" href="admin-style.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.1/css/all.min.css">
    <style>
        .filter-container {
            background-color: #fff;
            padding: 1.5rem;
            border-radius: 8px;
            margin-bottom: 2rem;
        }

        .filter-form {
            display: flex;
            align-items: flex-end;
            gap: 1rem;
            flex-wrap: wrap;
        }

        .summary-container {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
            gap: 1rem;
            margin-bottom: 2rem;
        }

        .summary-item {
            text-align: center;
            padding: 1rem;
            background-color: #fff;
            border-radius: 6px;
            border: 1px solid #e0e0e0;
        }

        .summary-item .label {
            font-size: 0.9em;
            color: #6c757d;
            margin-bottom: 5px;
            text-transform: uppercase;
        }

        .summary-item .value {
            font-size: 1.75em;
            font-weight: 600;
        }

        .summary-item .debit {
            color: #44bd32;
        }

        .summary-item .credit {
            color: #c23616;
        }

        .summary-item .balance {
            color: #0056b3;
        }

        .table-responsive .data-table td:nth-child(4) {
            white-space: normal;
            max-width: 300px;
        }

        .btn.btn-print {
            background-color: #17a2b8;
            color: #fff;
            border: none;
        }

        .btn.btn-print:hover {
            background-color: #138496;
        }
    </style>
</head>

<body>
    <div class="dashboard-wrapper">
        <?php include 'sidebar.php'; ?>
        <div class="main-content">
            <header class="main-header">
                <div class="user-info"><span>Vendor Ledger</span></div>
            </header>
            <main class="content-body">
                <div class="filter-container">
                    <form method="GET" action="vendor-ledger.php" class="filter-form">
                        <div class="form-group">
                            <label>Select Vendor</label>
                            <select name="vendor_id" class="form-control" required onchange="this.form.submit()">
                                <option value="">-- Choose a Vendor --</option>
                                <?php foreach ($vendors_list as $vendor): ?>
                                    <option value="<?= e($vendor['id']) ?>" <?= ($filter_vendor_id == $vendor['id']) ? 'selected' : '' ?>><?= e($vendor['name']) ?></option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                        <div class="form-group"><label>Start Date</label><input type="date" name="start_date" value="<?= e($filter_start_date) ?>" class="form-control"></div>
                        <div class="form-group"><label>End Date</label><input type="date" name="end_date" value="<?= e($filter_end_date) ?>" class="form-control"></div>
                        <button type="submit" class="btn btn-primary"><i class="fas fa-filter"></i> Filter</button>
                        <a href="vendor-ledger.php" class="btn btn-secondary"><i class="fas fa-times"></i> Clear</a>

                        <?php if ($filter_vendor_id > 0): ?>
                            <a href="vendor-ledger-print.php?<?= http_build_query($_GET) ?>" target="_blank" class="btn btn-print">
                                <i class="fas fa-print"></i> Print
                            </a>
                        <?php endif; ?>
                    </form>
                </div>

                <?php if ($selected_vendor_details): ?>
                    <h1 class="page-title">Ledger for: <?= e($selected_vendor_details['name']) ?></h1>

                    <?php
                    $total_debit_period = 0;
                    $total_credit_period = 0;
                    foreach ($transactions as $t) {
                        $total_debit_period += (float)$t['debit'];
                        $total_credit_period += (float)$t['credit'];
                    }
                    $closing_balance = $opening_balance + $total_credit_period - $total_debit_period;
                    ?>
                    <div class="summary-container">
                        <div class="summary-item">
                            <div class="label">Opening Balance</div>
                            <div class="value balance"><?= number_format($opening_balance, 2) ?></div>
                        </div>
                        <div class="summary-item">
                            <div class="label">Total Cost (Credit)</div>
                            <div class="value credit"><?= number_format($total_credit_period, 2) ?></div>
                        </div>
                        <div class="summary-item">
                            <div class="label">Total Paid (Debit)</div>
                            <div class="value debit"><?= number_format($total_debit_period, 2) ?></div>
                        </div>
                        <div class="summary-item">
                            <div class="label">Closing Balance</div>
                            <div class="value balance"><?= number_format($closing_balance, 2) ?></div>
                        </div>
                    </div>

                    <div class="content-card">
                        <div class="table-responsive">
                            <table class="data-table">
                                <thead>
                                    <tr>
                                        <th>Date</th>
                                        <th>Type</th>
                                        <th>Reference</th>
                                        <th>Particulars</th>
                                        <th style="text-align: right;">Paid (Debit)</th>
                                        <th style="text-align: right;">Cost (Credit)</th>
                                        <th style="text-align: right;">Balance</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <tr style="font-weight: bold; background-color: #f8f9fa;">
                                        <td colspan="6">Opening Balance</td>
                                        <td style="text-align: right;"><?= number_format($opening_balance, 2) ?></td>
                                    </tr>
                                    <?php if (empty($transactions)): ?>
                                        <tr>
                                            <td colspan="7" class="empty-state">No transactions found for this vendor in the selected criteria.</td>
                                        </tr>
                                        <?php else:
                                        $running_balance = $opening_balance;
                                        foreach ($transactions as $transaction):
                                            $debit = (float)$transaction['debit'];
                                            $credit = (float)$transaction['credit'];
                                            $running_balance += $credit - $debit;
                                        ?>
                                            <tr>
                                                <td><?= date('d M, Y', strtotime(e($transaction['date']))) ?></td>
                                                <td><?= e($transaction['type']) ?></td>
                                                <td>
                                                    <?php
                                                    $link = '#';
                                                    if ($transaction['source'] === 'package') {
                                                        $link = "view-invoice.php?id=" . e($transaction['transaction_id']);
                                                    } elseif ($transaction['source'] === 'ticket') {
                                                        $link = "ticket-invoice-view.php?id=" . e($transaction['transaction_id']);
                                                    } elseif ($transaction['source'] === 'payment') {
                                                        // Assuming a page exists to view/edit payments
                                                        $link = "edit-payment.php?id=" . e($transaction['transaction_id']);
                                                    }
                                                    ?>
                                                    <a href="<?= $link ?>" target="_blank"><?= e($transaction['number']) ?></a>
                                                </td>
                                                <td><?= e($transaction['particulars']) ?></td>
                                                <td style="text-align: right; color: #44bd32;"><?= $debit > 0 ? number_format($debit, 2) : '' ?></td>
                                                <td style="text-align: right; color: #c23616;"><?= $credit > 0 ? number_format($credit, 2) : '' ?></td>
                                                <td style="text-align: right; font-weight: 600;"><?= number_format($running_balance, 2) ?></td>
                                            </tr>
                                    <?php endforeach;
                                    endif; ?>
                                    <tr style="font-weight: bold; background-color: #f8f9fa; border-top: 2px solid #dee2e6;">
                                        <td colspan="4">Period Totals</td>
                                        <td style="text-align: right;"><?= number_format($total_debit_period, 2) ?></td>
                                        <td style="text-align: right;"><?= number_format($total_credit_period, 2) ?></td>
                                        <td></td>
                                    </tr>
                                    <tr style="font-weight: bold; background-color: #e9ecef; border-top: 2px solid #343a40;">
                                        <td colspan="6">Closing Balance</td>
                                        <td style="text-align: right;"><?= number_format($closing_balance, 2) ?></td>
                                    </tr>
                                </tbody>
                            </table>
                        </div>
                    </div>
                <?php else: ?>
                    <div class="notice info">Please select a vendor to view their financial ledger.</div>
                <?php endif; ?>
            </main>
        </div>
    </div>


    <script>
        // This script disables the right-click context menu on the entire page.
        document.addEventListener('contextmenu', function(e) {
            e.preventDefault();
        });
    </script>
</body>

</html>