<?php
session_start();
include_once 'db-config.php';

// --- SECURITY CHECK 1: Is user logged in? ---
if (!isset($_SESSION['user_id'])) {
    header("location: login.php");
    exit;
}

// --- SECURITY CHECK 2: Is the logged-in user an AGENT? ---
if (!isset($_SESSION['user_type']) || $_SESSION['user_type'] !== 'agent') {
    header("location: my-account.php"); // Redirect non-agents
    exit;
}

// A helper function for safely echoing output
function e($string)
{
    return htmlspecialchars($string ?? '', ENT_QUOTES, 'UTF-8');
}

// Get the logged-in agent's ID and name from the session
$logged_in_agent_id = (int)$_SESSION['user_id'];
$user_name = $_SESSION['user_name'];

// --- CALCULATE SUMMARY STATISTICS FOR THE LOGGED-IN AGENT ---
$summary_stats = [
    'package_invoice_count' => 0,
    'ticket_invoice_count' => 0,
    'pilgrim_count' => 0,
    'hotel_count' => 0
];

$stats_sql = "
    SELECT
        (SELECT COUNT(id) FROM invoices WHERE user_id = ?) as package_invoice_count,
        (SELECT COUNT(id) FROM ticket_invoices WHERE user_id = ?) as ticket_invoice_count,
        (SELECT COUNT(ip.id) FROM invoice_pilgrims ip JOIN invoices i ON ip.invoice_id = i.id WHERE i.user_id = ?) as pilgrim_count,
        (SELECT COUNT(ih.id) FROM invoice_hotels ih JOIN invoices i ON ih.invoice_id = i.id WHERE i.user_id = ?) as hotel_count
";

$stmt_stats = $conn->prepare($stats_sql);
if ($stmt_stats) {
    $stmt_stats->bind_param("iiii", $logged_in_agent_id, $logged_in_agent_id, $logged_in_agent_id, $logged_in_agent_id);
    $stmt_stats->execute();
    $stats_result = $stmt_stats->get_result();
    if ($stats_result) {
        $summary_stats = $stats_result->fetch_assoc();
    }
    $stmt_stats->close();
}

// --- CALCULATE AGENT INCENTIVES ---
$incentive_total = 0;
$incentive_label = "Lifetime Incentive Total";
$start_date = '';
$end_date = '';

// Check if the form has been submitted with a date range
if ($_SERVER["REQUEST_METHOD"] == "GET" && isset($_GET['filter_incentives'])) {
    $start_date = $_GET['start_date'] ?? '';
    $end_date = $_GET['end_date'] ?? '';

    if (!empty($start_date) && !empty($end_date)) {
        $incentive_label = "Incentive from " . e(date('d M Y', strtotime($start_date))) . " to " . e(date('d M Y', strtotime($end_date)));
        $incentive_sql = "SELECT SUM(up.agent_incentive) as total 
                          FROM umrah_inquiries ui 
                          JOIN umrah_packages up ON ui.package_id = up.package_id 
                          WHERE ui.referred_by_agent_id = ? 
                          AND DATE(ui.created_at) BETWEEN ? AND ?";
        $stmt_incentive = $conn->prepare($incentive_sql);
        $stmt_incentive->bind_param("iss", $logged_in_agent_id, $start_date, $end_date);
    } else {
        // If dates are empty, just calculate lifetime total (fall through to the block below)
        $incentive_label = "Lifetime Incentive Total";
        $incentive_sql = "SELECT SUM(up.agent_incentive) as total 
                          FROM umrah_inquiries ui 
                          JOIN umrah_packages up ON ui.package_id = up.package_id 
                          WHERE ui.referred_by_agent_id = ?";
        $stmt_incentive = $conn->prepare($incentive_sql);
        $stmt_incentive->bind_param("i", $logged_in_agent_id);
    }
} else {
    // Default: Calculate lifetime total
    $incentive_sql = "SELECT SUM(up.agent_incentive) as total 
                      FROM umrah_inquiries ui 
                      JOIN umrah_packages up ON ui.package_id = up.package_id 
                      WHERE ui.referred_by_agent_id = ?";
    $stmt_incentive = $conn->prepare($incentive_sql);
    $stmt_incentive->bind_param("i", $logged_in_agent_id);
}

if (isset($stmt_incentive) && $stmt_incentive) {
    $stmt_incentive->execute();
    $incentive_result = $stmt_incentive->get_result()->fetch_assoc();
    $incentive_total = $incentive_result['total'] ?? 0;
    $stmt_incentive->close();
}
?>
<!DOCTYPE html>
<html>

<head>
    <title>Agent Dashboard</title>
    <link rel="icon" type="image/png" href="images/logo-icon.png">
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <link rel="stylesheet" href="css/style.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.2.0/css/all.min.css">
    <link rel="stylesheet" href="css/account-style.css">
    <style>
        /* Dashboard Specific Styles */
        .summary-container {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
            gap: 1rem;
            margin-bottom: 2rem;
            background-color: #f8f9fa;
            border: 1px solid #e9ecef;
            padding: 1rem;
            border-radius: 8px;
        }

        .summary-item {
            text-align: center;
            padding: 1rem;
            background-color: #fff;
            border-radius: 6px;
            border: 1px solid #e9ecef;
        }

        .summary-item .label {
            font-size: 0.9em;
            color: #6c757d;
            margin-bottom: 5px;
            text-transform: uppercase;
        }

        .summary-item .value {
            font-size: 2em;
            font-weight: 600;
            color: #0d2d4c;
        }

        .dashboard-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(280px, 1fr));
            gap: 1.5rem;
        }

        .dashboard-card {
            background-color: #fff;
            border: 1px solid #e9ecef;
            border-radius: 8px;
            padding: 1.5rem;
            box-shadow: 0 4px 6px -1px rgba(0, 0, 0, 0.05);
            transition: transform 0.2s ease-in-out, box-shadow 0.2s ease-in-out;
        }

        .dashboard-card:hover {
            transform: translateY(-5px);
            box-shadow: 0 10px 15px -3px rgba(0, 0, 0, 0.1);
        }

        .dashboard-card-header {
            display: flex;
            align-items: center;
            gap: 1rem;
            border-bottom: 1px solid #f0f0f0;
            padding-bottom: 1rem;
            margin-bottom: 1rem;
        }

        .dashboard-card-header .icon {
            font-size: 1.8rem;
            color: #0d2d4c;
            width: 50px;
            height: 50px;
            display: grid;
            place-items: center;
            background-color: #f0f5fa;
            border-radius: 50%;
        }

        .dashboard-card-header h3 {
            margin: 0;
            font-size: 1.2rem;
            color: #212529;
        }

        .dashboard-card p {
            font-size: 0.95rem;
            color: #6c757d;
            line-height: 1.5;
            margin-top: 0;
        }

        .action-list {
            list-style: none;
            padding: 0;
            margin: 1.5rem 0 0 0;
        }

        .action-list li {
            margin-bottom: 0.5rem;
        }

        .action-list a {
            text-decoration: none;
            color: #0056b3;
            font-weight: 500;
            display: block;
            padding: 0.5rem;
            border-radius: 4px;
            transition: background-color 0.2s ease;
        }

        .action-list a:hover {
            background-color: #e9ecef;
            color: #003d80;
        }

        .action-list a i {
            margin-right: 0.75rem;
            width: 20px;
            text-align: center;
        }
        
        /* Styles for Incentive Card */
        .incentive-card-content .incentive-display {
            text-align: center;
            margin: 1rem 0;
        }
        .incentive-card-content .incentive-label {
            font-size: 0.9em;
            color: #6c757d;
        }
        .incentive-card-content .incentive-value {
            font-size: 2.2em;
            font-weight: 700;
            color: #28a745;
            margin: 0.25rem 0;
        }
        .incentive-filter-form {
            margin-top: 1rem;
            border-top: 1px solid #f0f0f0;
            padding-top: 1rem;
        }
        .incentive-filter-form .form-row {
            display: flex;
            gap: 0.5rem;
            margin-bottom: 0.5rem;
        }
        .incentive-filter-form .form-group { flex: 1; }
        .incentive-filter-form label {
            display: block;
            font-size: 0.8em;
            margin-bottom: 3px;
        }
        .incentive-filter-form input[type="date"] {
            width: 100%;
            padding: 6px;
            border: 1px solid #ced4da;
            border-radius: 4px;
            font-size: 0.9em;
        }
        .incentive-filter-form .btn-filter {
            width: 100%;
            padding: 8px;
            font-size: 0.9em;
            background-color: #007bff;
            color: white;
            border: none;
            border-radius: 4px;
            cursor: pointer;
        }
        .incentive-filter-form .btn-filter:hover { background-color: #0056b3; }
    </style>
</head>

<body>

    <?php include 'header.php'; ?>

    <main class="account-page-wrapper">
        <div class="account-container">

            <?php include 'agent-sidebar.php'; ?>

            <!-- ========== AGENT DASHBOARD CONTENT ========== -->
            <div class="account-content">
                <div class="content-card">
                    <h2><i class="fa-solid fa-gauge-high"></i> Welcome, <?php echo e(explode(' ', $user_name)[0]); ?>!</h2>
                    <p class="content-description">Quickly access your tools and see your account summary below.</p>

                    <div class="summary-container">
                        <div class="summary-item">
                            <div class="label">Package Invoices</div>
                            <div class="value"><?= e($summary_stats['package_invoice_count']) ?></div>
                        </div>
                        <div class="summary-item">
                            <div class="label">Ticket Invoices</div>
                            <div class="value"><?= e($summary_stats['ticket_invoice_count']) ?></div>
                        </div>
                        <div class="summary-item">
                            <div class="label">Total Pilgrims</div>
                            <div class="value"><?= e($summary_stats['pilgrim_count']) ?></div>
                        </div>
                        <div class="summary-item">
                            <div class="label">Hotel Bookings</div>
                            <div class="value"><?= e($summary_stats['hotel_count']) ?></div>
                        </div>
                    </div>

                    <hr style="margin: 2rem 0;">

                    <div class="dashboard-grid">
                        
                        <!-- Card 1: My Incentives -->
                        <div class="dashboard-card">
                            <div class="dashboard-card-header"><span class="icon" style="color:#28a745;"><i class="fa-solid fa-hand-holding-dollar"></i></span>
                                <h3>My Incentives</h3>
                            </div>
                            <div class="incentive-card-content">
                                <div class="incentive-display">
                                    <p class="incentive-label"><?= e($incentive_label) ?></p>
                                    <h4 class="incentive-value"><?= e(number_format($incentive_total, 2)) ?> SAR</h4>
                                </div>
                                <form action="agent-dashboard.php" method="GET" class="incentive-filter-form">
                                    <div class="form-row">
                                        <div class="form-group">
                                            <label for="start_date">From</label>
                                            <input type="date" name="start_date" id="start_date" value="<?= e($start_date) ?>">
                                        </div>
                                        <div class="form-group">
                                            <label for="end_date">To</label>
                                            <input type="date" name="end_date" id="end_date" value="<?= e($end_date) ?>">
                                        </div>
                                    </div>
                                    <button type="submit" name="filter_incentives" value="1" class="btn-filter"><i class="fa-solid fa-filter"></i> Filter</button>
                                </form>
                            </div>
                        </div>
                        
                        <!-- Card 2: Bookings -->
                        <div class="dashboard-card">
                            <div class="dashboard-card-header"><span class="icon"><i class="fa-solid fa-suitcase-rolling"></i></span>
                                <h3>Bookings</h3>
                            </div>
                            <p>Manage existing flight bookings or create new ones for your clients.</p>
                            <ul class="action-list">
                                <li><a href="agent-dashboard.php"><i class="fa-solid fa-eye"></i> View All Bookings</a></li>
                                <li><a href="./index.php"><i class="fa-solid fa-plus-circle"></i> Create New Booking</a></li>
                            </ul>
                        </div>

                        <!-- Card 3: Hotel Vouchers -->
                        <div class="dashboard-card">
                            <div class="dashboard-card-header"><span class="icon"><i class="fa-solid fa-hotel"></i></span>
                                <h3>Hotel Vouchers</h3>
                            </div>
                            <p>Generate and view hotel vouchers for accommodation bookings.</p>
                            <ul class="action-list">
                                <li><a href="agent-vouchers.php"><i class="fa-solid fa-list-ul"></i> View All Vouchers</a></li>
                                <li><a href="manage-requests.php"><i class="fa-solid fa-plus-circle"></i> Create Hotel Voucher</a></li>
                            </ul>
                        </div>

                        <!-- Card 4: Invoices -->
                        <div class="dashboard-card">
                            <div class="dashboard-card-header"><span class="icon"><i class="fa-solid fa-file-invoice"></i></span>
                                <h3>Invoices</h3>
                            </div>
                            <p>Access both package and ticket invoices for your records and for clients.</p>
                            <ul class="action-list">
                                <li><a href="agent-invoices.php"><i class="fa-solid fa-eye"></i> View All Invoices</a></li>
                                <li><a href="agent-invoices.php?print=all"><i class="fa-solid fa-print"></i> Print Invoices</a></li>
                            </ul>
                        </div>

                        <!-- Card 5: My Flights -->
                        <div class="dashboard-card">
                            <div class="dashboard-card-header"><span class="icon"><i class="fa-solid fa-plane-up"></i></span>
                                <h3>My Flights</h3>
                            </div>
                            <p>Manage group seat inventories and individual flight tickets.</p>
                            <ul class="action-list">
                                <li><a href="agent-flights.php"><i class="fa-solid fa-users"></i> Group Seats Bookings</a></li>
                                <li><a href="agent-invoices.php?type=ticket"><i class="fa-solid fa-ticket"></i> Tickets Invoices</a></li>
                            </ul>
                        </div>

                        <!-- Card 6: Account Ledger -->
                        <div class="dashboard-card">
                            <div class="dashboard-card-header"><span class="icon"><i class="fa-solid fa-book-open"></i></span>
                                <h3>Account Ledger</h3>
                            </div>
                            <p>Review your complete financial statement, including all debits and credits.</p>
                            <ul class="action-list">
                                <li><a href="agent-ledger.php"><i class="fa-solid fa-eye"></i> View My Ledger</a></li>
                                <li><a href="agent-ledger-print.php" target="_blank"><i class="fa-solid fa-print"></i> Print Full Statement</a></li>
                            </ul>
                        </div>

                        <!-- Card 7: Profile -->
                        <div class="dashboard-card">
                            <div class="dashboard-card-header"><span class="icon"><i class="fa-solid fa-user-gear"></i></span>
                                <h3>My Profile</h3>
                            </div>
                            <p>Update your personal and company information, including your logo.</p>
                            <ul class="action-list">
                                <li><a href="agent-edit-profile.php"><i class="fa-solid fa-user-pen"></i> Edit Personal Profile</a></li>
                                <li><a href="edit-company-profile.php"><i class="fa-solid fa-building"></i> Edit Company Profile</a></li>
                                <li><a href="logout.php"><i class="fa-solid fa-arrow-right-from-bracket"></i> Logout</a></li>
                            </ul>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </main>

    <?php include 'footer.php'; ?>
</body>

</html>