<?php
// checkout.php
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}
include 'db-config.php';

// SECURITY 1: Check if user is logged in
if (!isset($_SESSION['user_id'])) {
    $redirect_url = 'login.php?redirect_url=' . urlencode($_SERVER['REQUEST_URI']);
    header('Location: ' . $redirect_url);
    exit();
}

// SECURITY 2: Check that the necessary session data exists.
if (!isset($_SESSION['flight_search_results']) || !isset($_SESSION['flight_search_params'])) {
    $_SESSION['checkout_error'] = "Your flight search session has expired. Please search again.";
    header('Location: index.php');
    exit();
}

$flight_id = $_GET['flight_id'] ?? null;
if ($flight_id === null || !isset($_SESSION['flight_search_results'][$flight_id])) {
    $_SESSION['checkout_error'] = "Invalid flight selected. Please search again.";
    header('Location: index.php');
    exit();
}

// Safely get the flight and search parameters
$flight = $_SESSION['flight_search_results'][$flight_id];
$search_params = $_SESSION['flight_search_params'];

$adults_count = intval($search_params['adults'] ?? 0);
$children_count = intval($search_params['children'] ?? 0);
$infants_count = intval($search_params['infants'] ?? 0);
$total_passengers = $adults_count + $children_count + $infants_count;

$first_itinerary = $flight['itineraries'][0] ?? null;
$last_itinerary = end($flight['itineraries']) ?? null;

// NEW: Check if this is part of a package building flow
$is_package_flow = isset($search_params['service_redirect']) && $search_params['service_redirect'] === 'custom_package';


// Fetch logged-in user's data for auto-filling contact details
$user_email = '';
$user_phone = '';
if (isset($_SESSION['user_id'])) {
    try {
        $stmt = $conn->prepare("SELECT email, mobile_number FROM users WHERE id = ?");
        $stmt->bind_param("i", $_SESSION['user_id']);
        $stmt->execute();
        $result = $stmt->get_result();
        if ($user = $result->fetch_assoc()) {
            $user_email = $user['email'] ?? '';
            $user_phone = $user['mobile_number'] ?? '';
        }
        $stmt->close();
    } catch (Exception $e) {
        error_log("Failed to fetch user details for checkout: " . $e->getMessage());
    }
}

$checkout_error = $_SESSION['checkout_error'] ?? null;
unset($_SESSION['checkout_error']);
?>
<!DOCTYPE html>
<html>
<head>
    <title>Flight Checkout - RF Travel & Tours</title>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    
    <link rel="icon" type="image/png" href="images/logo-icon.png">

    <link rel="stylesheet" href="css/style.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.2.0/css/all.min.css">
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/flatpickr/dist/flatpickr.min.css">
    <link href="https://fonts.googleapis.com/css2?family=Poppins:wght@400;500;600;700&display=swap" rel="stylesheet">

    <style>
        body { background-color: #f8f9fa; }
        .checkout-container { max-width: 1200px; margin: 2rem auto; padding: 0 1rem; }
        .checkout-layout { display: grid; grid-template-columns: 2fr 1fr; gap: 2rem; align-items: flex-start; }
        .left-column, .right-column { display: flex; flex-direction: column; gap: 1.5rem; }
        .itinerary-card, .summary-card, .details-card { background: #fff; border-radius: 12px; padding: 1.5rem; box-shadow: 0 4px 12px rgba(0, 0, 0, 0.08); }
        .itinerary-card h2, .summary-card h3, .details-card h3 { margin-top: 0; margin-bottom: 1.5rem; color: #1a2a4d; font-weight: 600; display: flex; align-items: center; gap: 0.75rem; }
        .itinerary-card h2 { font-size: 1.4rem; }
        .details-card h3 { font-size: 1.2rem; }
        .info-text { font-size: 0.9rem; color: #6c757d; background-color: #f8f9fa; border-left: 3px solid #31a7e2; padding: 0.75rem 1rem; margin-bottom: 1.5rem; border-radius: 0 4px 4px 0; }
        .journey-section { border-top: 1px solid #e9ecef; padding-top: 1.5rem; margin-top: 1.5rem; }
        .journey-section:first-child { border-top: none; margin-top: 0; padding-top: 0; }
        .journey-header { display: flex; align-items: center; gap: 0.75rem; color: #343a40; margin-bottom: 1rem; }
        .journey-header i { color: #31a7e2; font-size: 1.2rem; }
        .journey-header h3 { margin: 0; font-size: 1.1rem; font-weight: 600; }
        .flight-segment { display: flex; gap: 1rem; align-items: flex-start; }
        .timeline { flex-shrink: 0; display: flex; flex-direction: column; align-items: center; }
        .timeline-dot { width: 12px; height: 12px; background-color: #fff; border: 2px solid #31a7e2; border-radius: 50%; }
        .timeline-line { flex-grow: 1; width: 2px; background-color: #e9ecef; margin: 4px 0; }
        .flight-segment:last-child .timeline-line { background: transparent; }
        .segment-details { flex-grow: 1; }
        .segment-details p { margin: 0 0 0.5rem 0; }
        .segment-time { display: flex; justify-content: space-between; font-size: 1rem; }
        .segment-time span { font-size: 0.85rem; color: #6c757d; }
        .segment-airports { font-size: 0.9rem; color: #495057; }
        .segment-airline { display: flex; align-items: center; gap: 0.5rem; font-size: 0.85rem; color: #6c757d; margin-top: 0.5rem; }
        .segment-airline img { width: 20px; height: 20px; }
        .baggage-info { margin-left: auto; background: #e9f5fd; color: #31a7e2; padding: 2px 8px; border-radius: 12px; font-size: 0.75rem; font-weight: 500; }
        .layover-info { padding-left: 1.7rem; margin: 1rem 0; font-size: 0.9rem; color: #6c757d; display: flex; align-items: center; gap: 0.5rem; }
        .layover-info i { color: #fd7e14; }
        .price-line { display: flex; justify-content: space-between; margin-bottom: 0.75rem; font-size: 0.95rem; }
        .price-line span:last-child { font-weight: 600; }
        .total-price { font-size: 1.2rem; font-weight: 700; color: #1a2a4d; }
        .passenger-details-section { margin-top: 2rem; }
        .passenger-form-group { border: 1px solid #e9ecef; border-radius: 8px; padding: 1.5rem; margin-bottom: 1.5rem; }
        .passenger-form-group h4 { margin-top: 0; margin-bottom: 1.5rem; color: #31a7e2; font-size: 1.1rem; }
        .passenger-fields-expanded, .passport-fields, .contact-details-fields { display: grid; gap: 1.5rem; }
        .passenger-fields-expanded { grid-template-columns: 1fr 2fr 2fr 1.5fr; margin-bottom: 1.5rem; }
        .passport-fields { grid-template-columns: repeat(3, 1fr); }
        .contact-details-fields { grid-template-columns: 1fr 1fr; }
        .form-field { display: flex; flex-direction: column; position: relative; }
        .form-field label { margin-bottom: 0.5rem; font-size: 0.85rem; font-weight: 500; color: #495057; }
        .form-field input, .form-field select { width: 100%; padding: 0.75rem; border: 1px solid #ced4da; border-radius: 6px; font-size: 0.95rem; font-family: 'Poppins', sans-serif; transition: border-color 0.2s, box-shadow 0.2s; }
        .form-field input:focus, .form-field select:focus { outline: none; border-color: #31a7e2; box-shadow: 0 0 0 2px rgba(49, 167, 226, 0.25); }
        .submit-container { text-align: center; margin-top: 2rem; }
        .checkout-submit-button { background-color: #28a745; color: #fff; border: none; padding: 1rem 3rem; font-size: 1.1rem; font-weight: 600; border-radius: 8px; cursor: pointer; transition: background-color 0.3s; }
        .checkout-submit-button:disabled { background-color: #6c757d; cursor: not-allowed; }
        .error-message { background-color: #f8d7da; color: #721c24; border: 1px solid #f5c6cb; padding: 1rem; border-radius: 8px; margin-bottom: 1.5rem; text-align: center; }
        .passport-error-message { font-size: 0.8rem; color: #721c24; padding-top: 5px; }
        .form-field input.is-invalid { border-color: #dc3545; }
        
        /* --- NEW: Styles for Package Flow --- */
        .package-flow-options { border-top: 2px dashed #31a7e2; margin-top: 1.5rem; padding-top: 1.5rem; }
        .package-flow-options p { text-align: center; font-weight: 500; color: #1a2a4d; margin-bottom: 1rem; }
        .package-flow-buttons { display: flex; flex-direction: column; gap: 1rem; }
        .btn-package-flow { width: 100%; padding: 1rem; font-size: 1rem; font-weight: 600; border-radius: 8px; text-decoration: none; text-align: center; transition: all 0.2s; }
        .btn-flight-only { background-color: #fff; color: #0d6efd; border: 2px solid #0d6efd; }
        .btn-flight-only:hover { background-color: #eef6ff; }
        .btn-build-package { background-color: #0d6efd; color: #fff; border: 2px solid #0d6efd; }
        .btn-build-package:hover { background-color: #0b5ed7; }
        
        @media (max-width: 992px) { .checkout-layout { grid-template-columns: 1fr; } }
        @media (max-width: 768px) { .passenger-fields-expanded, .passport-fields, .contact-details-fields { grid-template-columns: 1fr; } }
    </style>
</head>

<body>
    <?php include 'header.php'; ?>

    <div class="checkout-container">

        <?php if ($checkout_error): ?>
            <div class="error-message"><?php echo htmlspecialchars($checkout_error); ?></div>
        <?php endif; ?>

        <form action="<?php echo $is_package_flow ? '#' : 'process-booking.php'; ?>" method="POST" id="checkout-form">
            <input type="hidden" name="flight_id" value="<?php echo htmlspecialchars($flight_id); ?>">

            <div class="checkout-layout">
                <!-- === LEFT COLUMN === -->
                <div class="left-column">
                    <div class="itinerary-card">
                        <h2>Itinerary:
                            <?php if ($first_itinerary && $last_itinerary): ?>
                                <?php echo htmlspecialchars($first_itinerary['origin']); ?>
                                <i class="fa-solid fa-arrow-right-long"></i>
                                <?php echo htmlspecialchars($last_itinerary['destination']); ?>
                            <?php else: ?>
                                Flight Details
                            <?php endif; ?>
                        </h2>
                        <?php foreach ($flight['itineraries'] as $index => $itinerary): ?>
                            <div class="journey-section">
                                <div class="journey-header">
                                    <i class="fa-solid fa-plane-<?php echo ($index === 0) ? 'departure' : 'circle-right'; ?>"></i>
                                    <h3>
                                        <?php echo ($index === 0) ? 'Outbound' : 'Return/Next Leg'; ?>:
                                        <?php echo htmlspecialchars(date('D, M j, Y', strtotime($itinerary['departureDate']))); ?>
                                    </h3>
                                </div>
                                <?php foreach ($itinerary['segments'] as $segment): ?>
                                    <div class="flight-segment">
                                        <div class="timeline"><div class="timeline-dot"></div><div class="timeline-line"></div></div>
                                        <div class="segment-details">
                                            <p class="segment-time"><strong><?php echo htmlspecialchars($segment['departure']['time']); ?> - <?php echo htmlspecialchars($segment['arrival']['time']); ?></strong><span><?php echo htmlspecialchars($segment['duration']); ?></span></p>
                                            <p class="segment-airports"><?php echo htmlspecialchars($segment['departure']['iata']); ?> to <?php echo htmlspecialchars($segment['arrival']['iata']); ?></p>
                                            <div class="segment-airline">
                                                <img src="<?php echo htmlspecialchars($segment['airlineLogo']); ?>" alt="">
                                                <?php echo htmlspecialchars($segment['airlineName']); ?> - <?php echo htmlspecialchars($segment['flightNumber']); ?>
                                                <span class="baggage-info"><i class="fa-solid fa-suitcase"></i> <?php echo htmlspecialchars($flight['baggage']); ?></span>
                                            </div>
                                        </div>
                                    </div>
                                    <?php if (isset($segment['layover'])): ?>
                                        <div class="layover-info"><i class="fa-solid fa-hourglass-half"></i> Connection of <?php echo htmlspecialchars($segment['layover']['duration']); ?> in <?php echo htmlspecialchars($segment['layover']['airport']); ?></div>
                                    <?php endif; ?>
                                <?php endforeach; ?>
                            </div>
                        <?php endforeach; ?>
                    </div>

                    <div class="details-card">
                        <h3><i class="fa-solid fa-address-book"></i> Contact Details</h3>
                        <p class="info-text" style="margin-top: -1rem;">We will send all booking updates to this email and phone number.</p>
                        <div class="contact-details-fields">
                            <div class="form-field">
                                <label for="contact_email">Email Address*</label>
                                <input id="contact_email" type="email" name="contact_details[email]" required placeholder="e.g., yourname@example.com" value="<?php echo htmlspecialchars($user_email); ?>">
                            </div>
                            <div class="form-field">
                                <label for="contact_phone">Phone Number*</label>
                                <input id="contact_phone" type="tel" name="contact_details[phone]" required placeholder="e.g., 03001234567" value="<?php echo htmlspecialchars($user_phone); ?>">
                            </div>
                        </div>
                    </div>
                </div>

                <!-- === RIGHT COLUMN === -->
                <div class="right-column">
                    <div class="summary-card">
                        <h3>Price Summary</h3>
                        <div class="price-line"><span>Base Fare (<?php echo $total_passengers; ?> Traveller<?php echo $total_passengers > 1 ? 's' : ''; ?>)</span><span><?php echo htmlspecialchars($flight['currency']); ?> <?php echo number_format($flight['price'] / 1.05, 0); ?></span></div>
                        <div class="price-line"><span>Taxes & Service Fee</span><span><?php echo htmlspecialchars($flight['currency']); ?> <?php echo number_format($flight['price'] - ($flight['price'] / 1.05), 0); ?></span></div>
                        <hr>
                        <div class="price-line total-price"><span>Total Price</span><span><?php echo htmlspecialchars($flight['currency']); ?> <?php echo number_format($flight['price'], 0); ?></span></div>
                        
                        <?php if ($is_package_flow): ?>
                            <div class="package-flow-options">
                                <p>Continue with your booking:</p>
                                <div class="package-flow-buttons">
                                    <button type="submit" class="btn-package-flow btn-flight-only" form="checkout-form" formaction="process-booking.php">
                                        <i class="fa-solid fa-plane"></i> Book Flight Only
                                    </button>
                                    <a href="process-package-redirect.php?flight_id=<?php echo urlencode($flight_id); ?>" class="btn-package-flow btn-build-package">
                                        <i class="fa-solid fa-kaaba"></i> Book Flight & Build Package
                                    </a>
                                </div>
                            </div>
                        <?php endif; ?>
                    </div>
                </div>
            </div>

            <!-- PASSENGER DETAILS (FULL WIDTH) -->
            <div class="passenger-details-section">
                <div class="details-card">
                    <h3><i class="fa-solid fa-user-pen"></i> Traveller Details</h3>
                    <p class="info-text">Please use given names and surnames exactly as they appear in the passport to avoid boarding complications.</p>
                    
                    <?php for ($i = 1; $i <= $adults_count; $i++): ?>
                        <div class="passenger-form-group">
                            <h4>Adult <?php echo $i; ?></h4>
                            <div class="passenger-fields-expanded">
                                <div class="form-field"><label>Title*</label><select name="passengers[adults][<?php echo $i; ?>][title]" required><option>Mr</option><option>Mrs</option><option>Ms</option></select></div>
                                <div class="form-field"><label>First/Given Name*</label><input type="text" name="passengers[adults][<?php echo $i; ?>][firstName]" required></div>
                                <div class="form-field"><label>Last/Surname*</label><input type="text" name="passengers[adults][<?php echo $i; ?>][lastName]" required></div>
                                <div class="form-field"><label>Date of Birth*</label><input type="text" class="flatpickr-dob" name="passengers[adults][<?php echo $i; ?>][dob]" placeholder="DD/MM/YYYY" required></div>
                            </div>
                            <div class="passport-fields">
                                <div class="form-field"><label>Passport Number*</label><input type="text" name="passengers[adults][<?php echo $i; ?>][passport]" required></div>
                                <div class="form-field"><label>Issue Date*</label><input type="text" class="flatpickr-issue" name="passengers[adults][<?php echo $i; ?>][doi]" placeholder="DD/MM/YYYY" required></div>
                                <div class="form-field"><label>Expiry Date*</label><input type="text" class="flatpickr-expiry" name="passengers[adults][<?php echo $i; ?>][doe]" placeholder="DD/MM/YYYY" required></div>
                            </div>
                        </div>
                    <?php endfor; ?>
                    
                    <?php for ($i = 1; $i <= $children_count; $i++): ?>
                         <div class="passenger-form-group">
                            <h4>Child <?php echo $i; ?></h4>
                            <div class="passenger-fields-expanded">
                                <div class="form-field"><label>Title*</label><select name="passengers[children][<?php echo $i; ?>][title]" required><option>Master</option><option>Miss</option></select></div>
                                <div class="form-field"><label>First/Given Name*</label><input type="text" name="passengers[children][<?php echo $i; ?>][firstName]" required></div>
                                <div class="form-field"><label>Last/Surname*</label><input type="text" name="passengers[children][<?php echo $i; ?>][lastName]" required></div>
                                <div class="form-field"><label>Date of Birth*</label><input type="text" class="flatpickr-dob" name="passengers[children][<?php echo $i; ?>][dob]" placeholder="DD/MM/YYYY" required></div>
                            </div>
                             <div class="passport-fields">
                                <div class="form-field"><label>Passport Number*</label><input type="text" name="passengers[children][<?php echo $i; ?>][passport]" required></div>
                                <div class="form-field"><label>Issue Date*</label><input type="text" class="flatpickr-issue" name="passengers[children][<?php echo $i; ?>][doi]" placeholder="DD/MM/YYYY" required></div>
                                <div class="form-field"><label>Expiry Date*</label><input type="text" class="flatpickr-expiry" name="passengers[children][<?php echo $i; ?>][doe]" placeholder="DD/MM/YYYY" required></div>
                            </div>
                        </div>
                    <?php endfor; ?>

                    <?php for ($i = 1; $i <= $infants_count; $i++): ?>
                        <div class="passenger-form-group">
                            <h4>Infant <?php echo $i; ?></h4>
                            <div class="passenger-fields-expanded">
                                <div class="form-field"><label>Title*</label><select name="passengers[infants][<?php echo $i; ?>][title]" required><option>Master</option><option>Miss</option></select></div>
                                <div class="form-field"><label>First/Given Name*</label><input type="text" name="passengers[infants][<?php echo $i; ?>][firstName]" required></div>
                                <div class="form-field"><label>Last/Surname*</label><input type="text" name="passengers[infants][<?php echo $i; ?>][lastName]" required></div>
                                <div class="form-field"><label>Date of Birth*</label><input type="text" class="flatpickr-dob" name="passengers[infants][<?php echo $i; ?>][dob]" placeholder="DD/MM/YYYY" required></div>
                            </div>
                             <div class="passport-fields">
                                <div class="form-field"><label>Passport Number*</label><input type="text" name="passengers[infants][<?php echo $i; ?>][passport]" required></div>
                                <div class="form-field"><label>Issue Date*</label><input type="text" class="flatpickr-issue" name="passengers[infants][<?php echo $i; ?>][doi]" placeholder="DD/MM/YYYY" required></div>
                                <div class="form-field"><label>Expiry Date*</label><input type="text" class="flatpickr-expiry" name="passengers[infants][<?php echo $i; ?>][doe]" placeholder="DD/MM/YYYY" required></div>
                            </div>
                        </div>
                    <?php endfor; ?>
                </div>

                <?php if (!$is_package_flow): ?>
                    <div class="submit-container">
                        <button type="submit" id="checkout-submit-button" class="checkout-submit-button">Confirm Booking Request</button>
                    </div>
                <?php endif; ?>
            </div>
        </form>
    </div>
    <?php include 'footer.php'; ?>

    <script src="https://cdn.jsdelivr.net/npm/flatpickr"></script>
    <script>
        document.addEventListener('DOMContentLoaded', function() {
            const submitButton = document.getElementById('checkout-submit-button');
            const allExpiryInputs = document.querySelectorAll(".flatpickr-expiry");
            
            function validateAllPassportExpiries() {
                let isAnyInvalid = false;
                allExpiryInputs.forEach(input => {
                    if (input.classList.contains('is-invalid')) {
                        isAnyInvalid = true;
                    }
                });
                if (submitButton) {
                    submitButton.disabled = isAnyInvalid;
                }
            }

            function handleExpiryValidation(selectedDates, dateStr, instance) {
                const inputElement = instance.element;
                const parentFormField = inputElement.closest('.form-field');
                let errorMessageElement = parentFormField.querySelector('.passport-error-message');

                if (errorMessageElement) errorMessageElement.remove();
                inputElement.classList.remove('is-invalid');

                if (selectedDates.length > 0) {
                    const expiryDate = selectedDates[0];
                    const sixMonthsFromNow = new Date();
                    sixMonthsFromNow.setMonth(sixMonthsFromNow.getMonth() + 6);

                    if (expiryDate < sixMonthsFromNow) {
                        errorMessageElement = document.createElement('div');
                        errorMessageElement.className = 'passport-error-message';
                        errorMessageElement.textContent = 'Passport must be valid for at least 6 months.';
                        parentFormField.appendChild(errorMessageElement);
                        inputElement.classList.add('is-invalid');
                    }
                }
                validateAllPassportExpiries();
            }

            const commonConfig = {
                altInput: true,
                altFormat: "d/m/Y",
                dateFormat: "Y-m-d",
            };

            flatpickr(".flatpickr-dob", { ...commonConfig, maxDate: "today" });
            flatpickr(".flatpickr-issue", { ...commonConfig, maxDate: "today" });
            flatpickr(".flatpickr-expiry", { ...commonConfig, minDate: "today", onChange: handleExpiryValidation });
            
            validateAllPassportExpiries();
        });
    </script>
</body>
</html>