<?php
// --- AJAX INQUIRY SUBMISSION HANDLER ---
if ($_SERVER["REQUEST_METHOD"] == "POST" && isset($_POST['action']) && $_POST['action'] == 'submit_inquiry') {
    header('Content-Type: application/json');
    if (session_status() === PHP_SESSION_NONE) { session_start(); }
    require_once 'db-config.php';

    $package_data_json = $_POST['package_data'] ?? '{}';
    $data = json_decode($package_data_json, true);

    if (json_last_error() !== JSON_ERROR_NONE || empty($data) || !isset($data['flight'])) {
        echo json_encode(['success' => false, 'message' => 'Invalid or incomplete package data received.']);
        exit;
    }

    try {
        $user_id = $_SESSION['user_id'] ?? null;
        $customer_name = htmlspecialchars($data['flight']['customerName'] ?? 'N/A');
        $customer_email = htmlspecialchars($data['flight']['customerEmail'] ?? null);
        $customer_phone = htmlspecialchars($data['flight']['customerPhone'] ?? null);
        
        $booking_ref = 'RF-CP-' . time();
        $total_pax = $data['flight']['pax'] ?? 1;

        $sar_rate = $data['settings']['sar_to_pkr'] ?? 75.0;
        $total_sar = ($data['visa']['costSAR'] ?? 0) + ($data['accommodation']['totalCostSAR'] ?? 0) + ($data['transport']['costSAR'] ?? 0) + ($data['ziyarat']['costSAR'] ?? 0);
        $total_pkr_no_ticket = $total_sar * $sar_rate;
        $total_pkr_with_ticket = $total_pkr_no_ticket + ($data['flight']['price'] ?? 0);
        
        $per_person_ground_pkr = ($total_pax > 0) ? round($total_pkr_no_ticket / $total_pax) : 0;
        $per_person_total_pkr = ($total_pax > 0) ? round($total_pkr_with_ticket / $total_pax) : 0;

        $components = [
            'flight_ticket' => [
                'details' => "Route: {$data['flight']['arrivalAirport']} -> {$data['flight']['departureAirport']}",
                'cost_pkr' => round($data['flight']['price'] ?? 0)
            ],
            'visa' => [
                'details' => "{$data['visa']['pax']} Pax | {$data['visa']['typeName']}",
                'cost_sar' => $data['visa']['costSAR'] ?? 0
            ],
            'accommodation' => [
                'details' => "Total {$data['accommodation']['totalNights']} nights",
                'cost_sar' => $data['accommodation']['totalCostSAR'] ?? 0
            ],
            'transport' => [
                'details' => "Ground Transport",
                'cost_sar' => $data['transport']['costSAR'] ?? 0
            ],
            'ziyarat' => [
                'details' => "Optional Tours",
                'cost_sar' => $data['ziyarat']['costSAR'] ?? 0
            ]
        ];
        
        if (empty($components['flight_ticket']['cost_pkr'])) {
            unset($components['flight_ticket']);
        }

        $summary_details = [
            'grand_total_sar' => 'SAR ' . number_format($total_sar, 2),
            'grand_total_pkr' => 'PKR ' . number_format(round($total_pkr_with_ticket)),
            'per_person_ground' => 'PKR ' . number_format($per_person_ground_pkr),
            'total_per_person' => 'PKR ' . number_format($per_person_total_pkr)
        ];

        $details_to_store = ['pax' => ['adults' => $total_pax], 'components' => $components, 'summary' => $summary_details, 'rate_list_type' => $data['accommodation']['rate_list_type'] ?? 'N/A'];
        $package_details_json_db = json_encode($details_to_store);
        
        $stmt = $conn->prepare("INSERT INTO custom_package_inquiries (user_id, customer_name, customer_email, customer_phone, booking_ref, package_details_json, status) VALUES (?, ?, ?, ?, ?, ?, 'Pending')");
        $stmt->bind_param("isssss", $user_id, $customer_name, $customer_email, $customer_phone, $booking_ref, $package_details_json_db);
        
        if ($stmt->execute()) {
            $_SESSION['submission_success'] = true;
            $_SESSION['booking_type'] = 'custom_package';
            $_SESSION['success_context_name'] = $customer_name;
            $_SESSION['success_booking_ref'] = $booking_ref;
            echo json_encode(['success' => true, 'redirect' => 'thank-you.php']);
        } else {
            throw new Exception('Database insert failed: ' . $stmt->error);
        }
        $stmt->close();
    } catch (Exception $e) {
        error_log("Custom Package Submission Error: " . $e->getMessage());
        echo json_encode(['success' => false, 'message' => 'An internal error occurred. Please try again later.']);
    }
    exit;
}

if (session_status() === PHP_SESSION_NONE) {
    session_start();
}
require_once 'db-config.php';

$ticket_price_from_redirect = $_SESSION['redirect_ticket_price'] ?? null;
$ticket_pax_from_redirect = $_SESSION['redirect_ticket_pax'] ?? null;
unset($_SESSION['redirect_ticket_price'], $_SESSION['redirect_ticket_pax']);

function e($string) {
    return htmlspecialchars($string ?? '', ENT_QUOTES, 'UTF-8');
}

$is_agent_only = (isset($_SESSION['user_type']) && $_SESSION['user_type'] === 'agent');

$all_rates_json = '{}';
try {
    $response_data = [
        'settings' => [],
        'visa_rates' => [],
        'hotel_rates' => ['Rates 1' => ['Makkah' => [], 'Madinah' => []], 'Rates 2' => ['Makkah' => [], 'Madinah' => []]],
        'transport_rates' => [],
        'ziyarat_rates' => [],
        'user_type' => $_SESSION['user_type'] ?? 'customer',
        'post_assets' => []
    ];

    $settings_result = $conn->query("SELECT * FROM custom_package_settings");
    if ($settings_result) {
        while ($row = $settings_result->fetch_assoc()) {
            $response_data['settings'][$row['setting_key']] = $row['setting_value'];
        }
    }

    $visa_result = $conn->query("SELECT * FROM custom_package_visa_rates ORDER BY duration_days ASC");
    if ($visa_result) {
        $response_data['visa_rates'] = $visa_result->fetch_all(MYSQLI_ASSOC);
    }

    $hotel_result = $conn->query("SELECT id, hotel_name, city, star_rating, distance_info, notes, rate_sharing, rate_quad, rate_trpl, rate_dubl, rate_room_basis, rate_list_type FROM custom_package_hotel_rates ORDER BY hotel_name");
    if ($hotel_result) {
        $all_hotels_by_list = ['Rates 1' => [], 'Rates 2' => []];
        while ($row = $hotel_result->fetch_assoc()) {
            foreach(['rate_sharing', 'rate_quad', 'rate_trpl', 'rate_dubl', 'rate_room_basis'] as $rate_key) {
                 $row[$rate_key] = !is_null($row[$rate_key]) ? (float)$row[$rate_key] : null;
            }

            if ($is_agent_only) {
                if ($row['city'] === 'Makkah') {
                    if (!is_null($row['rate_sharing'])) $row['rate_sharing'] = max(0, $row['rate_sharing'] - 2);
                    if (!is_null($row['rate_quad']))    $row['rate_quad']    = max(0, $row['rate_quad'] - 2);
                    if (!is_null($row['rate_trpl']))    $row['rate_trpl']    = max(0, $row['rate_trpl'] - 2);
                    if (!is_null($row['rate_dubl']))    $row['rate_dubl']    = max(0, $row['rate_dubl'] - 2);
                } elseif ($row['city'] === 'Madinah') {
                    if (!is_null($row['rate_sharing'])) $row['rate_sharing'] = max(0, $row['rate_sharing'] - 3);
                    if (!is_null($row['rate_quad']))    $row['rate_quad']    = max(0, $row['rate_quad'] - 3);
                    if (!is_null($row['rate_trpl']))    $row['rate_trpl']    = max(0, $row['rate_trpl'] - 3);
                    if (!is_null($row['rate_dubl']))    $row['rate_dubl']    = max(0, $row['rate_dubl'] - 3);
                }
            }

            $list_type = $row['rate_list_type'];
            if (isset($response_data['hotel_rates'][$list_type])) {
                 if (isset($response_data['hotel_rates'][$list_type][$row['city']])) {
                     $response_data['hotel_rates'][$list_type][$row['city']][] = $row;
                 }
            }
            if (isset($all_hotels_by_list[$list_type])) {
                $all_hotels_by_list[$list_type][] = $row;
            }
        }
        $response_data['all_hotels_by_list'] = $all_hotels_by_list;
    }

    $transport_result = $conn->query("SELECT id, sector_name, trip_details, vehicle_type, rate_sar FROM custom_package_transport_rates ORDER BY display_order, vehicle_type, id");
    if ($transport_result) {
        $response_data['transport_rates'] = $transport_result->fetch_all(MYSQLI_ASSOC);
    }
    
    $ziyarat_result = $conn->query("SELECT id, city, vehicle_type, rate_sar FROM custom_package_ziyarat_rates ORDER BY city, vehicle_type");
    if ($ziyarat_result) {
        $response_data['ziyarat_rates'] = $ziyarat_result->fetch_all(MYSQLI_ASSOC);
    }
    
    $response_data['post_assets'] = [
        'post_bg_main' => 'images/bg-post.jpg'
    ];
    
    $all_rates_json = json_encode($response_data);

} catch (Exception $e) {
    error_log("Failed to fetch package data for builder: " . $e->getMessage());
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Build Your Custom Umrah Package - Amere Taiba</title>
    <link rel="icon" type="image/png" href="images/logo-icon.png">
    <link rel="stylesheet" href="css/style.css">
    <link rel="preconnect" href="https://fonts.googleapis.com">
    <link rel="preconnect" href="https://fonts.gstatic.com" crossorigin>
    <link href="https://fonts.googleapis.com/css2?family=Poppins:wght@400;500;600;700;800&family=Roboto:wght@400;500;700&display=swap" rel="stylesheet">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.1/css/all.min.css">
    <link href="https://cdn.jsdelivr.net/npm/select2@4.1.0-rc.0/dist/css/select2.min.css" rel="stylesheet" />
    <style>
        .hero-section { padding-top: 100px; padding-bottom: 200px; display: flex; align-items: center; justify-content: center; background-image: linear-gradient(rgba(0, 0, 0, 0.5), rgba(0, 0, 0, 0.5)), url('./images/bg.jpg'); background-size: cover; background-position: center; }
        .search-container { position: relative; z-index: 10; max-width: 1200px; padding: 0 15px; margin-top: -450px; margin-bottom: 200px; }
        
 
        .package-builder-content { width: 100%; max-width: 1200px; margin: 0 auto; background-color: #ffffff; padding: 30px 40px; border-radius: 12px; box-shadow: 0 10px 30px rgba(0,0,0,0.07); border: 1px solid #e9ecef; }
        #calculator-wrapper, #package-builder-start { display: none; }
        .calculator-grid { display: grid; grid-template-columns: 2fr 1fr; gap: 2rem; align-items: flex-start; }
        .form-sections { display: flex; flex-direction: column; gap: 1.5rem; }
        .calc-section { background: #fdfdfd; padding: 1.5rem; border-radius: 12px; border: 1px solid #e9ecef; position: relative; }
        
        .section-locked-overlay { position: absolute; top: 0; left: 0; right: 0; bottom: 0; background: rgba(255, 255, 255, 0.85); z-index: 2; display: flex; flex-direction: column; align-items: center; justify-content: center; border-radius: 12px; backdrop-filter: blur(3px); cursor: pointer; transition: opacity 0.3s ease; }
        .section-locked-overlay i { font-size: 2rem; color: #6c757d; margin-bottom: 1rem; }
        .section-locked-overlay p { margin: 0; color: #495057; font-weight: 500; text-align: center; padding: 0 1rem; }

        .calc-section h3, .initial-screen h2 { font-size: 1.2rem; color: #1a2a4d; display: flex; align-items: center; font-weight: 600; }
        .calc-section h3 { margin: -1.5rem -1.5rem 1.5rem -1.5rem; padding: 1rem 1.5rem; border-bottom: 1px solid #e9ecef; background-color: #f8f9fa; border-top-left-radius: 12px; border-top-right-radius: 12px; }
        .calc-section h3 i:first-child, .initial-screen h2 i:first-child { color:#31a7e2; margin-right: 12px; font-size: 1.3rem; }

        .form-row { display: grid; gap: 1.5rem; margin-bottom: 1rem; }
        .col-2 { grid-template-columns: 1fr 1fr; }
        .col-3 { grid-template-columns: 1fr 1fr 1fr; }
        .form-field label { display: block; margin-bottom: 0.5rem; font-size: 0.85rem; font-weight: 500; color: #495057; }
        .form-field input, .form-field select { width: 100%; height: 44px; padding: 0 .75rem; border: 1px solid #ced4da; border-radius: 6px; background-color: #fff; font-size: 0.95rem; transition: border-color 0.2s ease, box-shadow 0.2s ease; }
        .form-field input:focus, .form-field select:focus { outline: none; border-color: #31a7e2; box-shadow: 0 0 0 3px rgba(49, 167, 226, 0.2); }
        .pax-summary-display { padding: 0 .75rem; background: #f8f9fa; border-radius: 6px; border: 1px solid #e9ecef; font-weight: 500; color: #333; height: 44px; display: flex; align-items: center; }
        .select2-container .select2-selection--single { height: 44px; border: 1px solid #ced4da; }
        .select2-container--default .select2-selection--single .select2-selection__rendered { line-height: 42px; }
        .select2-container--default .select2-selection--single .select2-selection__arrow { height: 42px; }

        .btn-primary { display: inline-block; background-color: #0d6efd; color: white; padding: 12px 25px; border: none; border-radius: 6px; font-size: 1rem; font-weight: 600; cursor: pointer; text-align: center; transition: background-color 0.2s ease, transform 0.1s ease; }
        .btn-primary:hover { background-color: #0b5ed7; }
        .btn-primary:active { transform: translateY(1px); }
        .btn-primary:disabled { background-color: #6c757d; cursor: not-allowed; opacity: 0.7; }
        .btn-secondary { background-color: #6c757d; }
        .btn-secondary:hover { background-color: #5a6268; }
        #add-accommodation-btn { width: 100%; font-weight: 600; }
        #proceed-btn { width: 100%; margin-top: 1rem; font-size: 1.1rem; padding: 14px; background-color: #28a745; }
        #proceed-btn:hover { background-color: #218838; }
        #proceed-btn:disabled { background-color: #6c757d; }
        #download-post-btn { width: 100%; margin-top: 0.75rem; }

        #visa-cost-display, .stop-cost-display, .ziyarat-cost-display { margin-top: 1rem; padding: 12px; background-color: #e9f5fd; border: 1px solid #d0e9fc; border-radius: 6px; font-size: 0.9rem; font-weight: 500; text-align: center; }
        #visa-pax-group-container { display: none; margin-top: 1rem; }

        .summary-card-wrapper { position: sticky; top: 2rem; }
        .summary-line { display: flex; justify-content: space-between; padding: 0.85rem 0; border-bottom: 1px solid #f1f3f5; font-size: 0.9rem; }
        .grand-total { margin-top: 1rem; padding-top: 1rem; border-top: 2px solid #31a7e2; }
        .grand-total .summary-line span { font-size: 1.2rem; font-weight: 700; color: #0353a4; }
        .per-person-total { margin-top: 1rem; background-color: #f8f9fa; border: 1px solid #e9ecef; padding: 1rem; border-radius: 8px; }
        .per-person-total .summary-line { padding: 0.5rem 0; border-bottom: none; }
        .per-person-total .summary-line span:first-child { font-weight: 600; color: #495057; }
        .per-person-total .summary-line span:last-child { font-weight: 700; color: #0353a4; font-size: 1.1rem; }
        
        .transport-option { border: 1px solid #e0e6f1; border-radius: 8px; padding: 1rem; margin-bottom: 1rem; transition: all 0.2s ease-in-out; position: relative; cursor: pointer; }
        .transport-option:hover { background-color: #f8f9fa; }
        .transport-option.selected { background-color: #e9f5fd; border-color: #31a7e2; box-shadow: 0 2px 4px rgba(49, 167, 226, 0.1); }
        .transport-option .badge { position: absolute; top: 10px; right: 10px; padding: 3px 8px; font-size: 0.75rem; font-weight: 600; border-radius: 12px; }
        .badge.private { background-color: #0353a4; color: #fff; }
        .badge.free { background-color: #28a745; color: #fff; }
        .transport-option h5 { margin: 0 0 5px 0; color: #0353a4; font-size: 1rem; }
        .vehicle-selector-container, .ziyarat-bus-pax-container { margin-top: 1rem; padding-top: 1rem; border-top: 1px dashed #ced4da; display: none; }
        .vehicle-selector-container label, .ziyarat-bus-pax-container label { font-size: 0.9rem; font-weight: 500; margin-bottom: 0.5rem; display: block; }
        .ziyarat-option-card { border: 1px solid #e0e6f1; border-radius: 8px; padding: 1rem; margin-bottom: 1rem; }
        .ziyarat-option-card h5 { margin: 0 0 5px 0; color: #0353a4; font-size: 1rem; }
        .ziyarat-type-selector { display: flex; gap: 0.5rem; flex-wrap: wrap; margin-top: 0.5rem; }
        .ziyarat-type-selector label { display: inline-block; padding: 8px 12px; border: 1px solid #ced4da; border-radius: 20px; cursor: pointer; font-size: 0.85rem; line-height: 1; transition: all 0.2s ease; }
        .ziyarat-type-selector input[type="radio"] { display: none; }
        .ziyarat-type-selector input[type="radio"]:checked + label { background-color: #0353a4; color: white; border-color: #0353a4; }
        .ziyarat-option-card .vehicle-selector-container, .ziyarat-option-card .ziyarat-bus-pax-container { margin-top: 1rem; padding: 0; border: none; display: none; }
        
        .accommodation-stop { background: #fff; border: 1px solid #e9ecef; padding: 1.5rem; border-radius: 8px; margin-bottom: 1rem; }
        .stop-header { display: flex; justify-content: space-between; align-items: center; margin: -1.5rem -1.5rem 1rem -1.5rem; padding: 0.75rem 1.5rem; border-bottom: 1px solid #e9ecef; background: #f8f9fa; border-top-left-radius: 8px; border-top-right-radius: 8px; }
        .stop-header h4 { margin: 0; font-size: 1rem; font-weight: 600; color: #1a2a4d; }
        .stop-header .drag-handle { cursor: grab; margin-right: 10px; color: #6c757d; }
        .btn-remove-stop { background: none; border: none; color: #dc3545; font-size: 1.2rem; cursor: pointer; transition: color 0.2s ease; }
        .btn-remove-stop:hover { color: #a71d2a; }
        .ui-state-highlight { border: 2px dashed #ced4da; background-color: #f8f9fa; height: 100px; margin-bottom: 1rem; border-radius: 8px; }
        #total-nights-warning { color: #dc3545; font-weight: 500; text-align: center; margin-top: 1rem; display: none; background: #fbebee; padding: 10px; border-radius: 6px; border: 1px solid #f5c6cb; }
        
        #rate-list-selector-container { display: grid; grid-template-columns: 1fr 1fr; gap: 1rem; margin-bottom: 1.5rem; max-width: 800px; margin-left: auto; margin-right: auto; }
        .rate-list-card { border: 2px solid #e9ecef; border-radius: 8px; cursor: pointer; transition: all 0.2s ease; text-align: center; position: relative; overflow: hidden; }
        .rate-list-card:hover { border-color: #31a7e2; box-shadow: 0 4px 12px rgba(49,167,226,0.1); }
        .rate-list-card.selected { border-color: #0d6efd; border-width: 3px; }
        .rate-list-card.preview-active { border-color: #28a745; }
        .rate-list-card-thumb { height: 120px; background-size: cover; background-position: center; display: flex; align-items: center; justify-content: center; background-color: #f8f9fa; }
        .rate-list-card-thumb.no-img { background-image: linear-gradient(45deg, #f1f3f5, #e9ecef); }
        .rate-list-card-thumb.no-img i { font-size: 2.5rem; color: #adb5bd; }
        .rate-list-card-body { padding: 1rem; }
        .rate-list-card-body h5 { margin: 0 0 0.5rem 0; font-size: 1rem; color: #343a40; }
        .btn-preview-list { background: rgba(0,0,0,0.5); color: #fff; border: none; padding: 5px 12px; font-size: 0.8rem; border-radius: 15px; cursor: pointer; position: absolute; top: 10px; right: 10px; z-index: 2; transition: background 0.2s; }
        .btn-preview-list:hover { background: rgba(0,0,0,0.7); }

        #rate-list-preview-container { margin-top: 1.5rem; border: 1px solid #dee2e6; border-radius: 8px; background-color: #fdfdfd; padding: 1rem; position: relative; overflow-x: auto; }
        .rate-list-preview-header { display: flex; justify-content: space-between; align-items: center; border-bottom: 1px solid #ced4da; padding-bottom: 0.75rem; margin-bottom: 1rem; }
        .rate-list-preview-header h3 { margin: 0; font-size: 1.1rem; color: #343a40; }
        .btn-close-preview { background: none; border: none; font-size: 1.5rem; font-weight: bold; color: #6c757d; cursor: pointer; line-height: 1; padding: 0.25rem 0.5rem; }
        .btn-close-preview:hover { color: #343a40; }

        #social-post-preview-area { display: none; margin-top: 1rem; padding: 1rem; background-color: #f8f9fa; border: 1px solid #e9ecef; border-radius: 8px; }
        #social-post-preview-area h4 { margin: 0 0 1rem 0; text-align: center; }
        #social-post-canvas-wrapper { max-width: 540px; margin: 0 auto; box-shadow: 0 4px 15px rgba(0,0,0,0.1); }
        #social-post-canvas-wrapper canvas { width: 100%; height: auto; }

        .toast-notification { position: fixed; top: 20px; left: 50%; transform: translateX(-50%); background-color: #333; color: #fff; padding: 15px 25px; border-radius: 8px; z-index: 9999; opacity: 0; visibility: hidden; transition: opacity 0.3s, visibility 0.3s, top 0.3s; font-size: 0.95rem; box-shadow: 0 5px 15px rgba(0,0,0,0.2); }
        .toast-notification.show { top: 80px; opacity: 1; visibility: visible; }
        .toast-notification.warning { background-color: #f0ad4e; color: #fff; }
        .toast-notification.error { background-color: #d9534f; color: #fff; }
        
        @media (max-width: 992px) { 
            .hero-section { padding-top: 80px; padding-bottom: 150px; background-position: center top; } 
            .search-container { margin-top: -350px; margin-bottom: 150px; padding: 0 20px; }
            .calculator-grid { grid-template-columns: 1fr; } 
            .summary-card-wrapper { position: static; top: auto; margin-top: 1.5rem; } 
            #rate-list-selector-container { max-width: 100%; }
        }
        @media (max-width: 768px) { 
            .hero-section { padding-top: 70px; padding-bottom: 120px; flex-direction: column; text-align: center; } 
            .search-container { margin-top: -200px; margin-bottom: 200px; padding: 0 10px; }
            .package-builder-content { padding: 20px; } 
            .form-row.col-2, .form-row.col-3 { grid-template-columns: 1fr; } 
            .calc-section h3, .initial-screen h2 { font-size: 1.1rem; }
            #rate-list-selector-container { grid-template-columns: 1fr; }
            .rate-list-card-body h5 { font-size: 0.9rem; }
        }
        @media (max-width: 480px) { 
            .hero-section { padding-top: 60px; padding-bottom: 100px; background-position: top center; } 
            .search-container { margin-top: -200px; margin-bottom: 200px; padding: 0 10px; }
            .package-builder-content { padding: 15px; } 
            .calc-section { padding: 1rem; } 
            .calc-section h3 { margin: -1rem -1rem 1rem -1rem; padding: 0.75rem 1rem; } 
        }
    </style>
</head>
<body>
    <?php include 'header.php'; ?>
    
    <main>
        <section class="hero-section">
            <div class="hero-content">
                <h1>Build Your Perfect Umrah Journey</h1>
                <p>Personalize every detail of your sacred trip, from hotels to transport, with our easy-to-use package builder.</p>
            </div>
        </section>

        <div class="search-container">
            <div class="tabs-wrapper">
                <div class="service-tabs">
                    <a href="index.php" class="tab"><i class="fa-solid fa-plane-up"></i> Ticket</a>
                    <a href="group-fares.php" class="tab"><i class="fa-solid fa-users"></i> Groups</a>
                    <a href="umrah-packages.php" class="tab"><i class="fa-solid fa-kaaba"></i> Umrah</a>
                    <a href="hotels.php" class="tab"><i class="fa-solid fa-hotel"></i> Hotels</a>
                    <a href="visa-services.php" class="tab"><i class="fa-solid fa-passport"></i> Visas</a>
                    <a href="custom-package.php" class="tab active"><i class="fa-solid fa-kaaba"></i> Custom Package</a>
                </div>
            </div>
            <div class="search-form-wrapper">
                <div class="umrah-welcome-text">
                    <h2>Design Your <strong>Sacred Journey</strong></h2>
                    <p>Use our interactive tool below to build and price your custom Umrah package.</p>
                    <p class="guarantee-line">Transparent Pricing & Expert Service!</p>
                </div>
            </div>
        </div>

        <section class="package-builder-section" style="padding-bottom: 50px;">
            <div class="package-builder-content">
                
                <div id="service-selection-screen" class="initial-screen">
                    <h2><i class="fas fa-concierge-bell"></i> Choose Your Service</h2>
                    <p style="text-align:center; color: #495057; margin-bottom: 2rem;">What would you like to book today?</p>
                    <div class="form-field">
                        <label for="service-type-select">Select a service</label>
                        <select id="service-type-select" class="select2-init" style="width: 100%;">
                            <option value="">Please select...</option>
                            <option value="pkg_with_ticket">Complete Package (with Ticket)</option>
                            <option value="pkg_without_ticket">Complete Package (without Ticket)</option>
                            <option value="only_hotels">Only Hotels</option>
                            <option value="only_transport">Only Transport</option>
                            <option value="only_ziyarat">Only Ziyarat</option>
                            <option value="transport_ziyarat">Transport + Ziyarat</option>
                        </select>
                    </div>
                    <div style="text-align:center; margin-top:2.5rem;"><button type="button" id="proceed-service-selection" class="btn-primary" disabled>Proceed</button></div>
                </div>


                <div id="package-builder-start" class="initial-screen">
                    <h2><i class="fas fa-plane-departure"></i> Let's Get Started</h2>
                    <?php if ($is_agent_only): ?>
                        <div class="form-row col-3">
                            <div class="form-field"><label for="customer-name">Customer Name</label><input type="text" id="customer-name" placeholder="Full Name" required></div>
                            <div class="form-field"><label for="customer-email">Customer Email (Optional)</label><input type="email" id="customer-email" placeholder="email@example.com"></div>
                            <div class="form-field"><label for="customer-phone">Customer Phone (Optional)</label><input type="tel" id="customer-phone" placeholder="+92 300 1234567"></div>
                        </div>
                    <?php else: ?>
                        <div class="form-field" style="margin-bottom: 2rem;"><label for="customer-name">Your Name</label><input type="text" id="customer-name" placeholder="e.g., John Doe" required></div>
                    <?php endif; ?>

                    <p style="text-align:center; color: #495057; margin: 2rem 0;">Provide travel details to begin.</p>
                    <div class="form-row col-3">
                         <div class="form-field"><label for="manual-ticket-price">Total Ticket Price (PKR) (Optional)</label><input type="number" id="manual-ticket-price" placeholder="e.g., 185000" value="<?= e($ticket_price_from_redirect) ?>"></div>
                         <div class="form-field"><label for="manual-pax">Total Passengers</label><input type="number" id="manual-pax" placeholder="e.g., 2" min="1" required value="<?= e($ticket_pax_from_redirect) ?>"></div>
                    </div>
                    <p style="font-size: 0.9rem; color: #6c757d; text-align: center; margin-top: 1.5rem;">Enter IATA codes for arrival and departure airports in Saudi Arabia.</p>
                    <div class="form-row col-2">
                        <div class="form-field"><label for="arrival-city">Arrival Airport</label><select id="arrival-city" class="select2-init" style="width: 100%;" required><option value="">Select Airport</option><option value="JED">Jeddah (JED)</option><option value="MED">Madinah (MED)</option></select></div>
                        <div class="form-field"><label for="departure-city">Departure Airport</label><select id="departure-city" class="select2-init" style="width: 100%;" required><option value="">Select Airport</option><option value="JED">Jeddah (JED)</option><option value="MED">Madinah (MED)</option></select></div>
                    </div>
                    <div style="text-align:center; margin-top:2.5rem;"><button type="button" id="start-building-btn" class="btn-primary" disabled>Start Building Package</button></div>
                </div>

                <div id="calculator-wrapper">
                    <form id="custom-package-form">
                        <div class="calculator-grid">
                            <div class="form-sections">
                                <!-- Passenger Info -->
                                <div class="calc-section"><h3><i class="fas fa-users"></i> Passenger &amp; Flight Info</h3>
                                    <div id="customer-info-display" style="font-weight: 500; margin-bottom: 1rem; display:none;">Building for: <span id="selected-customer-name"></span></div>
                                    <div class="form-row col-3"><div class="form-field"><label>Customer</label><div class="pax-summary-display" id="customer-name-display"></div></div><div class="form-field"><label>Passengers</label><div class="pax-summary-display" id="pax-summary-display"></div></div><div class="form-field"><label>Flight Route</label><div class="pax-summary-display" id="flight-route-display"></div></div></div>
                                </div>
                                <!-- Visa Section -->
                                <div class="calc-section" id="visa-section">
                                    <h3><i class="fas fa-passport"></i> Visa</h3>
                                    <div class="form-row col-2">
                                        <div class="form-field"><label for="visa-type-select">Visa Type</label><select id="visa-type-select" class="select2-init" style="width: 100%;"><option value="">Select Visa Type...</option></select></div>
                                        <div class="form-field"><label for="visa-pax-select">Number of people needing a visa</label><select id="visa-pax-select" class="select2-init" style="width: 100%;"><option value="0">0</option></select></div>
                                    </div>
                                    <div id="visa-pax-group-container"><div class="form-field"><label for="visa-pax-group-count">Enter number of people (5 to 47)</label><input type="number" id="visa-pax-group-count" min="5" max="47" value="5"></div></div>
                                    <div id="visa-cost-display">Select visa options to see cost.</div>
                                </div>
                                <!-- Transport Section -->
                                <div class="calc-section" id="transport-section"><h3><i class="fas fa-bus"></i> Ground Transport</h3><div id="transport-options-container"></div></div>
                                <!-- Accommodation Section -->
                                <div class="calc-section" id="accommodation-section">
                                    <h3><i class="fas fa-hotel"></i> Accommodation</h3>
                                    <p>First, select a rate list to see available hotels.</p>
                                    <div id="rate-list-selector-container"></div>
                                    <div id="rate-list-preview-container" style="display: none;"></div>
                                    
                                    <div id="accommodation-wrapper" style="margin-top: 1.5rem;">
                                        <div id="accommodation-content" style="display:none; min-height: 100px;">
                                            <div id="accommodation-stops-container"></div>
                                            <button type="button" id="add-accommodation-btn" class="btn-primary"><i class="fas fa-plus"></i> Add Hotel Stay</button>
                                            <div id="total-nights-warning"></div>
                                        </div>
                                        <div id="accommodation-lock-overlay" class="section-locked-overlay" style="position:relative; min-height:150px;"><i class="fas fa-lock"></i><p>Select a Rate List above to begin</p></div>
                                    </div>
                                </div>
                                <!-- Ziyarat Section -->
                                <div class="calc-section" id="ziyarat-section"><h3><i class="fas fa-kaaba"></i> Ziyarat (Optional)</h3><div id="ziyarat-options-container"></div></div>
                            </div>
                            <!-- Summary Card Wrapper -->
                            <div class="summary-card-wrapper">
                                <div class="summary-card calc-section">
                                    <h3><i class="fas fa-file-invoice-dollar"></i> Package Summary</h3>
                                    <div class="summary-line" id="summary-ticket"><span >Flight Ticket</span><span id="ticket-total">PKR 0</span></div>
                                    <div class="summary-line" id="summary-visa"><span>Visa</span><span id="visa-total">SAR 0.00</span></div>
                                    <div class="summary-line" id="summary-accommodation"><span>Accommodation</span><span id="accommodation-total">SAR 0.00</span></div>
                                    <div class="summary-line" id="summary-transport"><span>Transport</span><span id="transport-total">SAR 0.00</span></div>
                                    <div class="summary-line" id="summary-ziyarat"><span>Ziyarat</span><span id="ziyarat-total">SAR 0.00</span></div>
                                    <div class="grand-total"><div class="summary-line"><span>Package Cost (SAR)</span><span id="grand-total-sar">SAR 0.00</span></div><div class="summary-line" style="font-size: 1rem;"><span>Package Cost (PKR)</span><span id="grand-total-pkr-noticket">PKR 0</span></div><hr style="border: 0; border-top: 1px dashed #ccc; margin: 0.5rem 0;"><div class="summary-line" id="summary-total-with-ticket"><span>Total with Ticket (PKR)</span><span id="grand-total-pkr">PKR 0</span></div></div>
                                    <div class="per-person-total">
                                        <div class="summary-line"><span>Per Person (Ground)</span><span id="per-person-ground">PKR 0</span></div>
                                        <div class="summary-line" id="summary-pp-with-ticket"><span>Per Person (w/ Ticket)</span><span id="per-person-total">PKR 0</span></div>
                                    </div>
                                    <button type="submit" id="proceed-btn" class="btn-primary" disabled>Proceed with Inquiry</button>
                                    <button type="button" id="download-post-btn" class="btn-primary btn-secondary" disabled><i class="fas fa-download"></i> Download Package Post</button>
                                </div>
                                <div id="social-post-preview-area">
                                    <h4>Generated Image Preview</h4>
                                    <div id="social-post-canvas-wrapper"></div>
                                </div>
                            </div>
                        </div>
                    </form>
                </div>
            </div>
        </section>
    </main>
    
    <div id="toast-notification" class="toast-notification"></div>
    
    <?php include 'floating-icon.php'; ?>
    <?php include 'footer.php'; ?>
    
    <script src="https://code.jquery.com/jquery-3.6.0.min.js"></script>
    <script src="https://code.jquery.com/ui/1.13.2/jquery-ui.min.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/select2@4.1.0-rc.0/dist/js/select2.min.js"></script>
    <script> 
        const allRates = <?php echo $all_rates_json; ?>; 
        const ticketFromRedirect = <?= json_encode($ticket_price_from_redirect !== null); ?>;
    </script>
    <script>
    $(document).ready(function() {
        // --- TOAST NOTIFICATION ---
        let toastTimeout;
        function showToast(message, type = 'normal', duration = 5000) {
            const toast = $('#toast-notification');
            clearTimeout(toastTimeout);
            toast.removeClass('show warning error').addClass(type).text(message);
            toast.addClass('show');
            toastTimeout = setTimeout(() => {
                toast.removeClass('show');
            }, duration);
        }
        
        // --- STATE & INITIAL SETUP ---
        let state = {
            flight: { price: 0, pax: 1, arrivalAirport: '', departureAirport: '', customerName: '', customerEmail: '', customerPhone: '' },
            visa: { id: null, typeName: '', duration_days: 0, pax: 0, costSAR: 0, perPersonCostSAR: 0, required: true },
            accommodation: { stops: [], totalNights: 0, totalCostSAR: 0, required: true, rate_list_type: '' },
            transport: { costSAR: 0, required: true, type: '', details: '' },
            ziyarat: { costSAR: 0, required: false, selections: [] },
            settings: {},
            user_type: 'customer',
            selected_customer: null,
            service_type: ''
        };

        if (allRates.settings && allRates.settings.sar_to_pkr_rate) {
            state.settings.sar_to_pkr = parseFloat(allRates.settings.sar_to_pkr_rate);
        }
        state.user_type = allRates.user_type || 'customer';
        
        // Initialize all select elements with the 'select2-init' class
        $('.select2-init').select2({ theme: "default", width: '100%' });

        $('#customer-name').on('input', function() {
            $('#start-building-btn').prop('disabled', $(this).val().trim() === '');
        }).trigger('input');

        $('#service-selection-screen').show();

        $('#service-type-select').on('change', function() {
            $('#proceed-service-selection').prop('disabled', !$(this).val());
        });

        $('#proceed-service-selection').on('click', function() {
            const service = $('#service-type-select').val();
            state.service_type = service;
            
            if (service === 'pkg_with_ticket') {
                window.location.href = 'index.php?service_redirect=custom_package';
            } else if (service === 'pkg_without_ticket') {
                configureForService(service);
                $('#service-selection-screen').fadeOut(() => $('#package-builder-start').fadeIn());
            } else if (service === 'only_hotels') {
                window.location.href = 'hotels.php';
            } else {
                configureForService(service);
                $('#service-selection-screen').fadeOut(() => $('#package-builder-start').fadeIn());
            }
        });

        function configureForService(serviceType) {
            state.visa.required = (serviceType === 'pkg_without_ticket');
            state.accommodation.required = (serviceType === 'pkg_without_ticket');
            state.transport.required = (serviceType === 'pkg_without_ticket' || serviceType === 'only_transport' || serviceType === 'transport_ziyarat');
            state.ziyarat.required = (serviceType === 'only_ziyarat');

            $('#visa-section').toggle(state.visa.required);
            $('#accommodation-section').toggle(state.accommodation.required);
            $('#transport-section').toggle(state.transport.required);
            $('#ziyarat-section').toggle(serviceType === 'pkg_without_ticket' || serviceType === 'only_ziyarat' || serviceType === 'transport_ziyarat');
            
            $('#summary-visa').toggle(state.visa.required);
            $('#summary-accommodation').toggle(state.accommodation.required);
            $('#summary-transport').toggle(state.transport.required);
            $('#summary-ziyarat').toggle(serviceType === 'pkg_without_ticket' || serviceType === 'only_ziyarat' || serviceType === 'transport_ziyarat');
        }

        $('#start-building-btn').on('click', function() {
            let customerName = $('#customer-name').val().trim();
            let customerEmail = '', customerPhone = '';

            if (state.user_type === 'agent') {
                customerEmail = $('#customer-email').val().trim();
                customerPhone = $('#customer-phone').val().trim();
            }

            if (!customerName) return alert('Please enter the customer\'s name.');
            
            const price = parseFloat($('#manual-ticket-price').val()) || 0;
            const pax = parseInt($('#manual-pax').val());
            const arrival = $('#arrival-city').val();
            const departure = $('#departure-city').val();
            
            if (isNaN(pax) || pax <= 0) return alert('Please enter a valid number of passengers.');
            if (state.visa.required || state.accommodation.required || state.transport.required) {
                if (!arrival || !departure) return alert('Please select arrival and departure airports for this service.');
            }
            
            state.flight = { customerName, customerEmail, customerPhone, price, pax, arrivalAirport: arrival, departureAirport: departure };
            startCalculator();
        });

        if (ticketFromRedirect) {
            state.service_type = 'pkg_without_ticket';
            configureForService(state.service_type);
            $('#package-builder-start').show();
            $('#service-selection-screen').hide();
            const ticketPrice = $('#manual-ticket-price').val();
            if (ticketPrice > 0) {
                $('#manual-ticket-price').prop('readonly', true);
                $('#manual-pax').prop('readonly', true);
            }
        }

        function startCalculator() {
            $('#package-builder-start').fadeOut(() => {
                populateCalculator();
                $('#calculator-wrapper').fadeIn();
            });
        }

        function populateCalculator() {
            $('#customer-name-display').text(state.flight.customerName);
            $('#pax-summary-display').text(`${state.flight.pax} Person(s)`);
            $('#flight-route-display').text(`${state.flight.arrivalAirport} → ${state.flight.departureAirport}`);
            
            if(state.visa.required) renderVisaOptions();
            if(state.transport.required) renderTransportOptions();
            renderZiyaratOptions();
            if(state.accommodation.required) renderRateListCards();
            $("#accommodation-stops-container").sortable({ handle: ".drag-handle", placeholder: "ui-state-highlight", update: calculateCosts });
            updateSummary();
        }
        
        $('#custom-package-form').on('submit', function(e) {
            e.preventDefault();
            const proceedBtn = $('#proceed-btn');
            proceedBtn.prop('disabled', true).html('<i class="fas fa-spinner fa-spin"></i> Submitting...');

            $.ajax({
                url: 'custom-package.php', type: 'POST', data: { action: 'submit_inquiry', package_data: JSON.stringify(state) }, dataType: 'json',
                success: function(response) {
                    if (response.success && response.redirect) {
                        window.location.href = response.redirect;
                    } else {
                        showToast('Submission failed: ' + (response.message || 'Unknown error.'), 'error');
                        proceedBtn.prop('disabled', false).text('Proceed with Inquiry');
                    }
                },
                error: function() {
                    showToast('An error occurred. Please try again.', 'error');
                    proceedBtn.prop('disabled', false).text('Proceed with Inquiry');
                }
            });
        });

        $('#visa-type-select, #visa-pax-select').on('change', function() {
             // This event handler is specifically for the dropdowns
            if ($(this).is('#visa-pax-select')) {
                $('#visa-pax-group-container').toggle($(this).val() === '5+');
            }
            calculateCosts();
        });

        $('#visa-pax-group-count').on('input', calculateCosts); // Use 'input' for instant updates as user types
        
        $('body').on('click', '.vehicle-selector-container', function(e) { e.stopPropagation(); });

        $('body').on('click', '.transport-option', function(e) {
            if (!$(e.target).closest('.vehicle-selector-container').length) {
                if ($(this).hasClass('selected')) return;
                $('.transport-option').removeClass('selected').find('.vehicle-selector-container').slideUp();
                $(this).addClass('selected');
                if ($(this).data('type') === 'private') {
                    $(this).find('.vehicle-selector-container').slideDown();
                }
                calculateCosts();
            }
        });
        $('body').on('change', '.transport-vehicle-select', calculateCosts);

        $('body').on('change', '.ziyarat-type-selector input, .ziyarat-bus-pax-count, .ziyarat-vehicle-select', function() {
            const card = $(this).closest('.ziyarat-option-card');
            const type = card.find('input[type="radio"]:checked').val();
            card.find('.vehicle-selector-container').toggle(type === 'private');
            card.find('.ziyarat-bus-pax-container').toggle(type === 'bus');
            calculateCosts();
        });
        
        $('body').on('click', '.rate-list-card', function(e) {
            if ($(e.target).hasClass('btn-preview-list')) { return; }
            $('#rate-list-preview-container').slideUp(() => {
                $('#rate-list-preview-container').empty();
                $('.rate-list-card').removeClass('preview-active');
            });
            $('.rate-list-card').removeClass('selected');
            $(this).addClass('selected');
            const selectedList = $(this).data('rate-type');
            state.accommodation.rate_list_type = selectedList;
            $('#accommodation-lock-overlay').hide();
            $('#accommodation-content').show();
            // Clear existing stops and re-render them to ensure correct hotel list
            $('#accommodation-stops-container').empty(); 
            state.accommodation.stops.forEach(addAccommodationStop);
            if (state.accommodation.stops.length === 0) {
                // If there are no stops, just calculate costs to reset accommodation part
                calculateCosts();
            }
        });
        
        $('body').on('click', '.btn-preview-list', function(e) {
            e.stopPropagation();
            const $card = $(this).closest('.rate-list-card');
            const rateType = $card.data('rate-type');
            const $previewContainer = $('#rate-list-preview-container');

            if ($card.hasClass('preview-active')) {
                $previewContainer.slideUp(() => {
                    $previewContainer.empty();
                    $('.rate-list-card').removeClass('preview-active');
                });
                return;
            }

            $('.rate-list-card').removeClass('preview-active');
            $card.addClass('preview-active');
            const previewHtml = generateRateListPreviewHtml(rateType);
            $previewContainer.html(previewHtml);
            if (!$previewContainer.is(':visible')) { $previewContainer.slideDown(); }
        });

        $('body').on('click', '.btn-close-preview', function() {
            $('#rate-list-preview-container').slideUp(() => {
                $('#rate-list-preview-container').empty();
                $('.rate-list-card').removeClass('preview-active');
            });
        });

        $('#add-accommodation-btn').on('click', () => addAccommodationStop());
        $('body').on('click', '.btn-remove-stop', function() { $(this).closest('.accommodation-stop').remove(); calculateCosts(); });
        $('body').on('input change', '.accommodation-stop input, .accommodation-stop select', calculateCosts);

        $('#download-post-btn').on('click', function() {
            const button = $(this);
            button.prop('disabled', true).html('<i class="fas fa-spinner fa-spin"></i> Generating...');
            $('#social-post-preview-area').slideDown();
            
            generateSocialPostCanvas().then(() => {
                const canvas = document.getElementById('social-post-canvas');
                const link = document.createElement('a');
                const customerNameSlug = state.flight.customerName.replace(/[^a-z0-9]/gi, '_').toLowerCase();
                link.download = `umrah-package-${customerNameSlug}.png`;
                link.href = canvas.toDataURL('image/png');
                link.click();
                button.prop('disabled', false).html('<i class="fas fa-download"></i> Download Package Post');
            }).catch(err => {
                console.error("Canvas generation failed:", err);
                showToast("Could not generate the image. Please check console for errors.", 'error');
                button.prop('disabled', false).html('<i class="fas fa-download"></i> Download Package Post');
            });
        });

        function renderVisaOptions() {
            const typeSelect = $('#visa-type-select').empty().append('<option value="">Select Visa Type...</option>');
            allRates.visa_rates.forEach(visa => typeSelect.append(`<option value="${visa.id}">${visa.visa_type}</option>`));
            const paxSelect = $('#visa-pax-select');
            paxSelect.empty().append('<option value="0">0</option>');
            for (let i = 1; i <= 4; i++) paxSelect.append(`<option value="${i}">${i} Person${i > 1 ? 's' : ''}</option>`);
            paxSelect.append('<option value="5+">5 or more</option>');
            
            typeSelect.select2({theme: "default", width: '100%'});
            paxSelect.select2({theme: "default", width: '100%'});
        }

        function getVisaPaxCount() {
            const selection = $('#visa-pax-select').val();
            if (selection === '5+') {
                const paxCount = parseInt($('#visa-pax-group-count').val()) || 5;
                return Math.max(5, Math.min(47, paxCount));
            }
            return parseInt(selection) || 0;
        }

        function renderTransportOptions() {
            const container = $('#transport-options-container').empty();
            const { arrivalAirport, departureAirport } = state.flight;

            container.append(`<p style="font-size: 0.85rem; color: #6c757d; text-align: center; margin-bottom: 1.5rem; padding-bottom: 1rem; border-bottom: 1px solid #e9ecef;">Please note: Complimentary bus transport is typically included with your visa.</p>`);

            const routesToShow = [];
            if (arrivalAirport === 'JED' && departureAirport === 'JED') { routesToShow.push({ name: 'Round Trip B', details: 'JED APT → MAK → MED → JED APT' }, { name: 'Round Trip B Flipped', details: 'JED APT → MED → MAK → JED APT' }, { name: 'Round Trip C', details: 'JED APT → MAK → MED → MAK → JED APT' }); }
            else if (arrivalAirport === 'MED' && departureAirport === 'JED') { routesToShow.push({ name: 'Round Trip A Flipped', details: 'MAD APT → MED → MAK → JED APT' }); }
            else if (arrivalAirport === 'JED' && departureAirport === 'MED') { routesToShow.push({ name: 'Round Trip A', details: 'JED APT → MAK → MED → MED APT' }); }
            
            if (routesToShow.length === 0) {
                container.append(`<p style="text-align:center; color:#6c757d;">No standard round trips available for this route.</p>`);
                return;
            }

            if (arrivalAirport === 'JED' && departureAirport === 'JED') {
                const busRate = allRates.transport_rates.find(r => r.sector_name === 'Round Trip C' && r.vehicle_type.toLowerCase() === 'bus');
                if (busRate) {
                    container.append(`<div class="transport-option" data-route-details="JED APT → MAK → MED → MAK → JED APT" data-type="free"><span class="badge free">INCLUDED IN VISA</span><h5>Complimentary Bus</h5><div class="route-def">JED APT → MAK → MED → MAK → JED APT</div><div class="price-info">SAR 0.00</div></div>`);
                }
            }

            routesToShow.forEach(route => {
                const originalRouteName = route.name.includes('Flipped') ? route.name.replace(' Flipped', '') : route.name;
                const privateRates = allRates.transport_rates.filter(r => r.sector_name === originalRouteName && r.vehicle_type.toLowerCase() !== 'bus');
                if (privateRates.length > 0) {
                    const cheapestRate = Math.min(...privateRates.map(r => parseFloat(r.rate_sar)));
                    const vehicleOptionsHtml = [...new Set(privateRates.map(r => r.vehicle_type))].map(v => `<option value="${v}">${v}</option>`).join('');
                    container.append(`<div class="transport-option" data-route-name="${originalRouteName}" data-type="private"><span class="badge private">PRIVATE</span><h5>Ya Transport (Private)</h5><div class="route-def">${route.details}</div><div class="price-info">From SAR ${cheapestRate.toFixed(2)}</div><div class="vehicle-selector-container"><label>Select Vehicle</label><select class="form-field transport-vehicle-select select2-init" style="width:100%"><option value="">Select Vehicle</option>${vehicleOptionsHtml}</select></div></div>`);
                }
            });
            // Init select2 for newly added transport selectors
             $('.transport-vehicle-select.select2-init').select2({theme: "default", width: '100%'});
        }

        function renderZiyaratOptions() {
            const container = $('#ziyarat-options-container').empty();
            const locations = [...new Set(allRates.ziyarat_rates.map(r => r.city))];
            locations.forEach(city => {
                const rates = allRates.ziyarat_rates.filter(r => r.city === city);
                const hasBus = rates.some(r => r.vehicle_type.toLowerCase() === 'bus');
                const privateVehicles = rates.filter(r => r.vehicle_type.toLowerCase() !== 'bus');
                const privateOptions = [...new Set(privateVehicles.map(v => v.vehicle_type))].map(v => `<option value="${v}">${v}</option>`).join('');
                container.append(`<div class="ziyarat-option-card" data-city="${city}"><h5>${city} Ziyarat</h5><div class="ziyarat-type-selector"><input type="radio" id="ziyarat_${city.replace(/\s+/g,'')}_none" name="ziyarat_type_${city.replace(/\s+/g,'')}" value="none" checked><label for="ziyarat_${city.replace(/\s+/g,'')}_none">None</label>${hasBus?`<input type="radio" id="ziyarat_${city.replace(/\s+/g,'')}_bus" name="ziyarat_type_${city.replace(/\s+/g,'')}" value="bus"><label for="ziyarat_${city.replace(/\s+/g,'')}_bus">Bus</label>`:''}${privateVehicles.length>0?`<input type="radio" id="ziyarat_${city.replace(/\s+/g,'')}_private" name="ziyarat_type_${city.replace(/\s+/g,'')}" value="private"><label for="ziyarat_${city.replace(/\s+/g,'')}_private">Private</label>`:''}</div><div class="ziyarat-bus-pax-container form-field"><label>Number of Passengers for Bus</label><input type="number" class="ziyarat-bus-pax-count" min="1" value="${state.flight.pax}"></div><div class="vehicle-selector-container form-field"><label>Select Private Vehicle</label><select class="ziyarat-vehicle-select select2-init" style="width:100%"><option value="">Select Vehicle</option>${privateOptions}</select></div><div class="ziyarat-cost-display" style="display:none;"></div></div>`);
            });
            // Init select2 for newly added ziyarat selectors
            $('.ziyarat-vehicle-select.select2-init').select2({theme: "default", width: '100%'});
        }
        
        function renderRateListCards() {
            const container = $('#rate-list-selector-container').empty();
            const s = allRates.settings;
            const listTypes = ['Rates 1', 'Rates 2'];
            listTypes.forEach(type => {
                const title = `${s.rate_list_main_title || 'UMRAH 1447-H'} - ${type}`;
                const bgImage = s.rate_list_bg_image ? `../${s.rate_list_bg_image}` : '';
                const thumbClass = bgImage ? '' : 'no-img';
                const thumbStyle = bgImage ? `background-image: url('${bgImage}')` : '';
                const cardHtml = `<div class="rate-list-card" data-rate-type="${type}"><button type="button" class="btn-preview-list">Preview</button><div class="rate-list-card-thumb ${thumbClass}" style="${thumbStyle}">${!bgImage ? '<i class="fas fa-image"></i>' : ''}</div><div class="rate-list-card-body"><h5>${title}</h5></div></div>`;
                container.append(cardHtml);
            });
        }

        function calculateCosts() {
            if(state.visa.required) calculateVisa();
            if(state.transport.required) calculateTransport();
            if(state.accommodation.required) calculateAccommodation();
            calculateZiyarat();
            updateSummary();
            updateAddButtonState();
        }

        function calculateVisa() {
            const visaId = $('#visa-type-select').val();
            const pax = getVisaPaxCount();
            let cost = 0, perPersonCost = 0, duration = 0, typeName = '';
            if (visaId && pax > 0) {
                const data = allRates.visa_rates.find(v => v.id == visaId);
                if (data) {
                    duration = parseInt(data.duration_days);
                    typeName = data.visa_type;
                    let rateKey = (pax >= 5) ? 'rate_per_pax_5_plus' : `rate_per_pax_${pax}`;
                    perPersonCost = parseFloat(data[rateKey] || 0);
                    cost = perPersonCost * pax;
                    $('#visa-cost-display').html(`Per Person: <strong>SAR ${perPersonCost.toFixed(2)}</strong> | Total for ${pax} Person(s): <strong>SAR ${cost.toFixed(2)}</strong>`);
                }
            } else { $('#visa-cost-display').text('Select visa options to see cost.'); }
            state.visa = { ...state.visa, id: visaId, typeName, duration_days: duration, pax, costSAR: cost, perPersonCostSAR: perPersonCost };
        }
        
        function calculateTransport() {
            const selected = $('.transport-option.selected');
            let cost = 0, type = '', details = '';
            if (selected.length > 0) {
                type = selected.data('type');
                details = selected.find('.route-def').text();
                if (type === 'private') {
                    const vehicle = selected.find('.transport-vehicle-select').val();
                    const routeName = selected.data('routeName');
                    if (vehicle && routeName) {
                        const rateData = allRates.transport_rates.find(r => r.sector_name === routeName && r.vehicle_type === vehicle);
                        if (rateData) cost = parseFloat(rateData.rate_sar);
                    }
                }
            }
            state.transport = { ...state.transport, costSAR: cost, type: type, details: details };
        }

        function calculateAccommodation() {
            let totalCost = 0, totalNights = 0;
            let stops = [];
            const rateListType = state.accommodation.rate_list_type;
            if (!rateListType) {
                state.accommodation = { ...state.accommodation, stops: [], totalCostSAR: 0, totalNights: 0 };
                return;
            }
            $('.accommodation-stop').each(function() {
                const $stop = $(this);
                const nights = parseInt($stop.find('.hotel-nights').val()) || 0;
                totalNights += nights;
                const hotelId = $stop.find('.hotel-select').val(), roomType = $stop.find('.room-type').val(), city = $stop.find('.hotel-city').val(), display = $stop.find('.stop-cost-display');
                let stopDetails = { id: $stop.attr('id'), city, hotelId, hotelName: '', nights, roomType, stopCostSAR: 0 };
                if (!hotelId || nights <= 0 || !roomType || !city) { display.text('Fill all details to see cost.'); stops.push(stopDetails); return; }
                const hotelData = allRates.hotel_rates[rateListType]?.[city]?.find(h => h.id == hotelId);
                if (!hotelData) { display.text('Hotel data not found.'); stops.push(stopDetails); return; }
                stopDetails.hotelName = hotelData.hotel_name;
                stopDetails.notes = hotelData.notes;
                let ratePerPerson = hotelData[`rate_${roomType === 'triple' ? 'trpl' : roomType}`] !== null ? parseFloat(hotelData[`rate_${roomType === 'triple' ? 'trpl' : roomType}`]) : null;
                if (ratePerPerson === null) {
                    const roomRate = parseFloat(hotelData.rate_room_basis);
                    const paxMap = { double: 2, triple: 3, quad: 4 };
                    if (!isNaN(roomRate) && paxMap[roomType]) {
                         ratePerPerson = roomRate / paxMap[roomType];
                    } else {
                         ratePerPerson = 0;
                    }
                }
                
                const stopPP = ratePerPerson * nights, stopTotal = stopPP * state.flight.pax;
                totalCost += stopTotal;
                stopDetails.stopCostSAR = stopTotal;
                display.html(`Per Person: <strong>SAR ${stopPP.toFixed(2)}</strong> | Total: <strong>SAR ${stopTotal.toFixed(2)}</strong>`);
                stops.push(stopDetails);
            });
            state.accommodation = { ...state.accommodation, stops, totalCostSAR: totalCost, totalNights };
            const visaDuration = state.visa.duration_days || 0;
            $('#total-nights-warning').html(`<strong>Warning:</strong> Total nights (${totalNights}) exceed the ${visaDuration}-day visa limit.`).toggle(state.visa.required && visaDuration > 0 && totalNights > visaDuration);
        }

        function calculateZiyarat() {
            let totalCost = 0;
            let selections = [];
            $('.ziyarat-option-card').each(function() {
                const card = $(this), city = card.data('city'), type = card.find('input:checked').val(), display = card.find('.ziyarat-cost-display');
                let stopCost = 0;

                if (type !== 'none') {
                    selections.push({ city: city, type: type });
                }

                if (type === 'bus') {
                    const rateData = allRates.ziyarat_rates.find(r => r.city === city && r.vehicle_type.toLowerCase() === 'bus');
                    if (rateData) {
                        let rate = parseFloat(rateData.rate_sar);
                        if (state.user_type === 'agent') rate = Math.max(0, rate - 5);
                        const pax = parseInt(card.find('.ziyarat-bus-pax-count').val()) || 0;
                        stopCost = rate * pax;
                        display.html(`Per Person: <strong>SAR ${rate.toFixed(2)}</strong> | Total for ${pax}: <strong>SAR ${stopCost.toFixed(2)}</strong>`).show();
                    }
                } else if (type === 'private') {
                    const vehicle = card.find('.ziyarat-vehicle-select').val();
                    if (vehicle) {
                        const rateData = allRates.ziyarat_rates.find(r => r.city === city && r.vehicle_type === vehicle);
                        if (rateData) {
                            stopCost = parseFloat(rateData.rate_sar);
                            display.html(`Total Cost: <strong>SAR ${stopCost.toFixed(2)}</strong>`).show();
                        }
                    } else { display.hide(); }
                } else { display.hide(); }
                totalCost += stopCost;
            });
            state.ziyarat = { ...state.ziyarat, costSAR: totalCost, selections: selections };
        }
        
        function updateSummary() {
            const groundTotalSAR = (state.visa.costSAR) + (state.accommodation.totalCostSAR) + (state.transport.costSAR) + (state.ziyarat.costSAR);
            const totalPKR_noTicket = groundTotalSAR * (state.settings.sar_to_pkr || 75.00);
            const totalPKR_withTicket = totalPKR_noTicket + state.flight.price;
            const perPersonGroundPKR = state.flight.pax > 0 ? totalPKR_noTicket / state.flight.pax : 0;
            const perPersonTotalPKR = state.flight.pax > 0 ? totalPKR_withTicket / state.flight.pax : 0;
            
            $('#ticket-total').text(`PKR ${Math.round(state.flight.price).toLocaleString()}`);
            $('#visa-total').text(`SAR ${state.visa.costSAR.toFixed(2)}`);
            $('#accommodation-total').text(`SAR ${state.accommodation.totalCostSAR.toFixed(2)}`);
            $('#transport-total').text(`SAR ${state.transport.costSAR.toFixed(2)}`);
            $('#ziyarat-total').text(`SAR ${state.ziyarat.costSAR.toFixed(2)}`);
            $('#grand-total-sar').text(`SAR ${groundTotalSAR.toFixed(2)}`);
            $('#grand-total-pkr-noticket').text(`PKR ${Math.round(totalPKR_noTicket).toLocaleString()}`);
            $('#grand-total-pkr').text(`PKR ${Math.round(totalPKR_withTicket).toLocaleString()}`);
            $('#per-person-ground').text(`PKR ${Math.round(perPersonGroundPKR).toLocaleString()}`);
            $('#per-person-total').text(`PKR ${Math.round(perPersonTotalPKR).toLocaleString()}`);
            
            const hasTicket = state.flight.price > 0;
            $('#summary-ticket, #summary-total-with-ticket, #summary-pp-with-ticket').toggle(hasTicket);
            
            let isValid = true;
            if (state.visa.required && (!state.visa.id || state.visa.pax <= 0)) isValid = false;
            if (state.accommodation.required && state.accommodation.stops.length === 0) isValid = false;
            if (state.transport.required && !state.transport.type) isValid = false;

            const allStopsValid = state.accommodation.stops.every(s => s.hotelId && s.nights > 0 && s.roomType && s.city);
            if (state.accommodation.required && !allStopsValid) isValid = false;

            $('#proceed-btn, #download-post-btn').prop('disabled', !isValid);
        }
        
        function updateAddButtonState() {
            const totalStops = $('.accommodation-stop').length;
            const maxStaysReached = totalStops >= 4; // Max 2 for Makkah, 2 for Madinah
            $('#add-accommodation-btn').prop('disabled', maxStaysReached)
                .html(maxStaysReached ? '<i class="fas fa-ban"></i> Max Stays Reached' : '<i class="fas fa-plus"></i> Add Hotel Stay');
        }

        function addAccommodationStop(stopData = {}) {
            if ($('#add-accommodation-btn').is(':disabled')) return;
            const id = `stop_${Date.now()}`;
            const html = `<div class="accommodation-stop" id="${id}"><div class="stop-header"><h4><i class="fas fa-arrows-alt drag-handle"></i> Hotel Stay</h4><button type="button" class="btn-remove-stop"><i class="fas fa-times-circle"></i></button></div><div class="form-row col-2"><div class="form-field"><label>City</label><select class="hotel-city select2-init" style="width:100%"><option value="">Select</option><option value="Makkah">Makkah</option><option value="Madinah">Madinah</option></select></div><div class="form-field"><label>Hotel</label><select class="hotel-select select2-init" style="width:100%;" disabled><option value="">Select city first</option></select></div></div><div class="form-row col-2"><div class="form-field"><label>Nights</label><input type="number" class="hotel-nights" min="1" value="1"></div><div class="form-field"><label>Room Type</label><select class="room-type select2-init" style="width:100%"><option value="sharing">Sharing</option><option value="quad">Quad</option><option value="triple">Triple</option><option value="double">Double</option></select></div></div><div class="stop-cost-display">Fill all details to see cost.</div></div>`;
            $('#accommodation-stops-container').append(html);
            // Initialize Select2 for all dropdowns in the new stop
            $(`#${id} .select2-init`).select2({ theme: "default" });
            calculateCosts();
        }

        $('body').on('change', '.hotel-city', function() {
            const $currentSelect = $(this);
            const selectedCity = $currentSelect.val();
            const currentStopId = $currentSelect.closest('.accommodation-stop').attr('id');

            if (selectedCity) {
                let cityCount = 0;
                $('.accommodation-stop').each(function() {
                    const $stop = $(this);
                    if ($stop.attr('id') !== currentStopId && $stop.find('.hotel-city').val() === selectedCity) {
                        cityCount++;
                    }
                });

                if (cityCount >= 2) {
                    showToast('You can add a maximum of two stays per city.', 'warning');
                    $currentSelect.val('').trigger('change'); 
                    return; 
                }
            }

            const city = $(this).val();
            const rateListType = state.accommodation.rate_list_type;
            const hotelSelect = $(this).closest('.accommodation-stop').find('.hotel-select').empty().prop('disabled', true);
            if (city && rateListType && allRates.hotel_rates[rateListType]?.[city]) {
                hotelSelect.append(new Option('Select Hotel', ''));
                allRates.hotel_rates[rateListType][city].forEach(h => { hotelSelect.append(new Option(`${h.hotel_name} ${h.star_rating ? `(${h.star_rating} Star)`:''} - ${h.distance_info}`, h.id)); });
                hotelSelect.prop('disabled', false);
            } else { hotelSelect.append(new Option('Select city first', '')); }
            hotelSelect.trigger('change');
        });

        $('body').on('change', '.hotel-select', function() {
            const $currentSelect = $(this);
            const currentStopId = $currentSelect.closest('.accommodation-stop').attr('id');
            const currentCity = $currentSelect.closest('.accommodation-stop').find('.hotel-city').val();
            const currentHotelId = $currentSelect.val();

            if (!currentCity || !currentHotelId) return;

            let firstStayHotelId = null;
            $('.accommodation-stop').each(function() {
                const $stop = $(this);
                if ($stop.attr('id') !== currentStopId && $stop.find('.hotel-city').val() === currentCity) {
                    firstStayHotelId = $stop.find('.hotel-select').val();
                    return false; 
                }
            });

            if (firstStayHotelId && currentHotelId !== firstStayHotelId) {
                showToast("For a second stay in the same city, you must select the same hotel. For custom arrangements, please contact our team.", 'warning');
                $currentSelect.val('').trigger('change.select2');
            }
        });

        function loadGoogleFont(fontName) {
            const fontId = `google-font-${fontName.replace(' ', '-')}`;
            if (document.getElementById(fontId)) return;
            const link = document.createElement('link');
            link.id = fontId; link.rel = 'stylesheet';
            link.href = `https://fonts.googleapis.com/css2?family=${fontName.replace(' ', '+')}:wght@400;500;600;700;800&display=swap`;
            document.head.appendChild(link);
        }

        function generateRateListPreviewHtml(rateType) {
            const s = allRates.settings;
            const hotelsForList = allRates.all_hotels_by_list[rateType] || [];
            const makkahHotels = hotelsForList.filter(h => h.city === 'Makkah');
            const madinahHotels = hotelsForList.filter(h => h.city === 'Madinah');
            const title = `${s.rate_list_main_title || 'UMRAH 1447-H'} - ${rateType}`;

            loadGoogleFont(s.rate_list_font || 'Poppins');
            
            const dynamicStyles = `
            <style>
                .preview-wrapper, .preview-wrapper *, .preview-wrapper *::before, .preview-wrapper *::after {
                    box-sizing: border-box; margin: 0; padding: 0; border: 0; font: inherit;
                }
                .preview-wrapper { font-family: '${s.rate_list_font || 'Poppins'}', sans-serif; }
                .preview-table { width: 100%; border-collapse: collapse; margin-bottom: 1.5rem; background-color: #fff; }
                .preview-table th, .preview-table td { 
                    padding: 8px 10px; border: 1px solid #dee2e6; text-align: center; white-space: nowrap;
                    font-size: ${s.rate_list_table_body_fs ? parseFloat(s.rate_list_table_body_fs) * 0.1 : 0.9}rem; 
                }
                .preview-table thead tr { 
                    background-color: ${s.rate_list_table_header_bg || '#343A40'}; 
                    color: ${s.rate_list_table_header_text || '#ffffff'}; 
                }
                 .preview-table thead th {
                    font-size: ${s.rate_list_table_head_fs ? parseFloat(s.rate_list_table_head_fs) * 0.1 : 1}rem;
                    font-weight: 600;
                 }
                .preview-table tbody td:first-child { text-align: left; font-weight: 500; }
                .preview-table .notes { font-size: 0.8em; color: #6c757d; display: block; margin-top: 2px; font-weight: 400; white-space: normal; }
                .preview-wrapper h4 { 
                    font-size: 1rem; margin: 1.5rem 0 0.5rem 0; color: #1a2a4d; font-weight: 600;
                }
                 .preview-wrapper h4:first-child { margin-top: 0; }

                @media screen and (max-width: 768px) {
                    .preview-table thead { display: none; }
                    .preview-table, .preview-table tbody, .preview-table tr, .preview-table td { display: block; width: 100%; }
                    .preview-table tr {
                        border: 1px solid #dee2e6;
                        border-radius: 8px;
                        margin-bottom: 1rem;
                        padding: 1rem;
                    }
                    .preview-table td {
                        text-align: right;
                        padding-left: 50%;
                        position: relative;
                        border: none;
                        padding-top: 0.5rem;
                        padding-bottom: 0.5rem;
                        white-space: normal;
                    }
                    .preview-table td:not(:last-child) {
                        border-bottom: 1px solid #f1f3f5;
                    }
                    .preview-table td:before {
                        content: attr(data-label);
                        position: absolute;
                        left: 0;
                        width: 45%;
                        padding-left: 1rem;
                        font-weight: 600;
                        text-align: left;
                    }
                    .preview-table td:first-child { text-align: right; }
                    .preview-table td:first-child:before {
                         padding-left: 1rem;
                    }
                    .preview-table .notes { white-space: normal; text-align: right; }
                }
            </style>`;

            const generateSimpleHotelTableHtml = (title, hotels) => {
                if (!hotels || hotels.length === 0) {
                    return `<h4>${title}</h4><p>No hotels found for this category.</p>`;
                }
                const rows = hotels.map(hotel => `
                    <tr>
                        <td data-label="HOTELS">${e(hotel.hotel_name)}<span class="notes">${e(hotel.notes || '')}</span></td>
                        <td data-label="STAR">${e(hotel.star_rating)}</td>
                        <td data-label="DISTANCE">${e(hotel.distance_info)}</td>
                        <td data-label="SHARING">${hotel.rate_sharing !== null ? parseFloat(hotel.rate_sharing).toFixed(0) : 'N/A'}</td>
                        <td data-label="QUAD">${hotel.rate_quad !== null ? parseFloat(hotel.rate_quad).toFixed(0) : 'N/A'}</td>
                        <td data-label="TRIPLE">${hotel.rate_trpl !== null ? parseFloat(hotel.rate_trpl).toFixed(0) : 'N/A'}</td>
                        <td data-label="DOUBLE">${hotel.rate_dubl !== null ? parseFloat(hotel.rate_dubl).toFixed(0) : 'N/A'}</td>
                    </tr>`).join('');

                return `
                    <h4>${title}</h4>
                    <table class="preview-table">
                        <thead><tr><th>HOTELS</th><th>STAR</th><th>DISTANCE</th><th>SHARING</th><th>QUAD</th><th>TRIPLE</th><th>DOUBLE</th></tr></thead>
                        <tbody>${rows}</tbody>
                    </table>`;
            };

            return `
                <div class="rate-list-preview-header">
                    <h3>${title}</h3>
                    <button type="button" class="btn-close-preview" title="Close Preview">&times;</button>
                </div>
                <div class="preview-wrapper">
                    ${dynamicStyles}
                    ${generateSimpleHotelTableHtml('Makkah Hotels', makkahHotels)}
                    ${generateSimpleHotelTableHtml('Madinah Hotels', madinahHotels)}
                </div>
            `;
        }
        
        function e(str) { 
            return String(str||'').replace(/[&<>"']/g, s => ({'&':'&amp;','<':'&lt;','>':'&gt;','"':'&quot;',"'":'&#39;'}[s])); 
        }

        async function generateSocialPostCanvas() {
            const wrapper = document.getElementById('social-post-canvas-wrapper');
            wrapper.innerHTML = '';
            const canvas = document.createElement('canvas');
            canvas.id = 'social-post-canvas';
            canvas.width = 1080;
            canvas.height = 1080;
            wrapper.appendChild(canvas);
            const ctx = canvas.getContext('2d');

            const loadImage = src => new Promise((resolve, reject) => {
                if (!src) return reject(new Error('Image source is missing.'));
                const img = new Image();
                img.crossOrigin = "Anonymous";
                img.onload = () => resolve(img);
                img.onerror = () => reject(new Error(`Failed to load image: ${src}`));
                img.src = src;
            });
            
            const wrapText = (context, text, x, y, maxWidth, lineHeight) => {
                const words = text.split(' ');
                let line = '';
                for(let n = 0; n < words.length; n++) {
                    let testLine = line + words[n] + ' ';
                    if (context.measureText(testLine).width > maxWidth && n > 0) {
                        context.fillText(line.trim(), x, y);
                        line = words[n] + ' ';
                        y += lineHeight;
                    } else {
                        line = testLine;
                    }
                }
                context.fillText(line.trim(), x, y);
            };

            await Promise.all([
                document.fonts.load('bold 28px Poppins'),
                document.fonts.load('bold 49px Poppins'),
                document.fonts.load('20px Roboto'),
                document.fonts.load('500 23px Poppins'),
                document.fonts.load('500 20px Poppins'),
                document.fonts.load('bold 26px Poppins'),
                document.fonts.load('500 22px Poppins'),
                document.fonts.load('bold 23px Poppins'),
                document.fonts.load('bold 39px Poppins'),
                document.fonts.load('500 13px Poppins')
            ]).catch(err => console.warn("Font loading may be inconsistent:", err));

            try {
                const assets = allRates.post_assets;
                const templateImg = await loadImage(assets.post_bg_main);

                ctx.clearRect(0, 0, 1080, 1080);
                ctx.drawImage(templateImg, 0, 0, 1080, 1080);

                ctx.fillStyle = '#FFFFFF';
                ctx.textBaseline = 'top';
                ctx.textAlign = 'left';

                const totalNights = state.accommodation.totalNights || state.visa.duration_days;
                ctx.font = 'bold 28px Poppins';
                ctx.fillText(`${totalNights} DAYS UMRAH PACKAGE FOR`, 48, 200);

                ctx.font = 'bold 49px Poppins';
                ctx.fillText(state.flight.customerName.toUpperCase(), 48, 232);

                ctx.font = '20px Roboto';
                ctx.fillText(`FROM ${state.flight.arrivalAirport}-${state.flight.departureAirport}`, 48, 298);
                ctx.fillText("TRAVEL DATES: AS PER YOUR CHOICE", 48, 323);
                
                wrapText(ctx, "Blessed Umrah with Spiritual and Rewarding Ziyarats.", 48, 380, 450, 26);
                
                ctx.font = '500 23px Poppins';
                ctx.fillText("", 48, 440);

                const makkahStops = state.accommodation.stops.filter(s => s.city === 'Makkah');
                const madinahStops = state.accommodation.stops.filter(s => s.city === 'Madinah');
                
                ctx.font = '500 20px Poppins';
                let hotelY_Makkah = 500;
                if (makkahStops.length > 0) {
                    makkahStops.forEach(stop => {
                        const hotelText = stop.hotelName.toUpperCase();
                        const notesText = `SHUTTLE SERVICE: | N ${stop.nights}`;
                        ctx.fillText(hotelText, 48, hotelY_Makkah); hotelY_Makkah += 24;
                        ctx.fillText(notesText, 48, hotelY_Makkah); hotelY_Makkah += 36;
                    });
                }
                let hotelY_Madinah = 500;
                 if (madinahStops.length > 0) {
                     madinahStops.forEach(stop => {
                        const hotelText = stop.hotelName.toUpperCase();
                        const notesText = `SHUTTLE SERVICE: | N ${stop.nights}`;
                        ctx.fillText(hotelText, 530, hotelY_Madinah); hotelY_Madinah += 24;
                        ctx.fillText(notesText, 530, hotelY_Madinah); hotelY_Madinah += 36;
                     });
                 }
                
                ctx.font = '500 22px Poppins';
                let inclusionsY = 680;
                const inclusions = [];
                if (state.flight.price > 0) inclusions.push('✓ Return Tickets');
                if (state.visa.costSAR > 0) inclusions.push('✓ Visa');
                if (state.accommodation.totalCostSAR > 0) inclusions.push('✓ Hotels');

                // --- MODIFIED INCLUSION LOGIC FOR TRANSPORT & ZIYARAT ---
                const hasTransport = state.transport.costSAR > 0 || state.transport.type === 'free';
                const isPrivateTransport = state.transport.type === 'private';
                const hasZiyarat = state.ziyarat.costSAR > 0 && state.ziyarat.selections.length > 0;
                const isPrivateZiyarat = state.ziyarat.selections.some(s => s.type === 'private');

                if (hasTransport && hasZiyarat) {
                    const transportText = isPrivateTransport ? "Private Transport" : "Transport";
                    const ziyaratText = isPrivateZiyarat ? "Private Ziyarat" : "Ziyarat";
                    inclusions.push(`✓ ${transportText} + ${ziyaratText}`);
                } else if (hasTransport) {
                    inclusions.push(isPrivateTransport ? "✓ Private Transport" : "✓ Transport");
                } else if (hasZiyarat) {
                    inclusions.push(isPrivateZiyarat ? "✓ Private Ziyarat" : "✓ Ziyarat");
                }
                
                inclusions.slice(0, 4).forEach(item => {
                    ctx.fillText(item, 85, inclusionsY);
                    inclusionsY += 35;
                });
                
                const sarToPkrRate = state.settings.sar_to_pkr || 75.00;
                const groundTotalSAR = state.visa.costSAR + state.accommodation.totalCostSAR + state.transport.costSAR + state.ziyarat.costSAR;
                const groundTotalPKR = groundTotalSAR * sarToPkrRate;
                const totalWithTicketPKR = groundTotalPKR + state.flight.price;
                const perPersonPKR = state.flight.pax > 0 ? Math.round(totalWithTicketPKR / state.flight.pax) : 0;
                
                ctx.textAlign = 'center';
                ctx.fillStyle = '#FFFFFF';
                
                ctx.font = 'bold 23px Poppins';
                ctx.fillText(`PACKAGE FOR ${state.flight.pax} PERSON`, 810, 870); 
                ctx.font = 'bold 39px Poppins';
                ctx.fillText(`${Math.round(totalWithTicketPKR).toLocaleString()}`, 810, 900);
                
                ctx.font = 'bold 23px Poppins';
                ctx.fillText("TOTAL PACKAGE PRICE", 360, 870);
                ctx.font = 'bold 39px Poppins';
                ctx.fillText(`${perPersonPKR.toLocaleString()}`, 360, 900);

                ctx.fillStyle = '#000000';
                ctx.textAlign = 'left';
                ctx.font = '500 13px Poppins';
                wrapText(ctx, "AL QURESH NEAR RAILWAY PAH., INFRONT OF AL QURESH HSG SCHEME SHER SHAH RD MULTAN", 178, 1018, 380, 16);
                ctx.font = 'bold 14px Poppins';
                ctx.fillText("RAEES FALAKSHER: +92 305 2394810", 655, 1015);
                ctx.fillText("SANWAL FALAKSHER: +92 309 9705022", 655, 1033);

            } catch (error) {
                console.error('Error during canvas generation:', error);
                ctx.fillStyle = 'red';
                ctx.font = '30px Arial';
                ctx.textAlign = 'center';
                ctx.textBaseline = 'middle';
                ctx.fillText('Image could not be generated.', canvas.width / 2, canvas.height / 2 - 20);
                ctx.font = '20px Arial';
                ctx.fillText('Please check browser console for errors.', canvas.width / 2, canvas.height / 2 + 20);
            }
        }
    });
    </script>
</body>
</html>