<?php
require_once 'db-config.php';
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

// --- NEW SECURITY & LOGIN CHECK ---
// --- 1. User MUST be logged in ---
if (!isset($_SESSION['user_id'])) {
    // Save the current page URL so we can redirect back after login
    $redirect_url = "group-booking-detail.php?" . $_SERVER['QUERY_STRING'];
    $_SESSION['login_redirect'] = $redirect_url;

    $_SESSION['error_message'] = "Please log in to your account to proceed with the booking.";
    header("Location: login.php");
    exit;
}

// --- 2. User MUST be a customer or an agent (not an admin) ---
if (isset($_SESSION['user_type']) && $_SESSION['user_type'] === 'admin') {
    // Admins should not be making bookings on the frontend.
    header("Location: index.php"); // Or wherever you want to send admins
    exit;
}

// --- INITIALIZATION ---
$fare_details = null;
$error_message = '';

// --- GET FARE DETAILS ---
$group_ref_id = $_GET['ref'] ?? null;

if (!$group_ref_id) {
    $error_message = "No booking reference was provided. Please go back and select a fare.";
} else {
    $sql = "SELECT gf.*, al.airline_name, al.logo_url FROM group_fares gf LEFT JOIN airlines al ON gf.airline_id = al.id WHERE gf.is_active = 1 AND gf.group_ref_id = ?";
    $stmt = $conn->prepare($sql);
    $stmt->bind_param("s", $group_ref_id);
    $stmt->execute();
    $result = $stmt->get_result();
    if ($result->num_rows > 0) {
        $fare_details = $result->fetch_assoc();
        $fare_details['flight_details_json'] = json_decode($fare_details['flight_details_json'], true) ?: [];
    } else {
        $error_message = "The requested booking could not be found or is no longer available.";
    }
}

// --- HELPER FUNCTIONS ---
function format_flight_date_short($date_string)
{
    if (empty($date_string)) return '';
    return date("d M Y", strtotime($date_string));
}

function render_passenger_row($type, $index, $srText)
{
    $namePrefix = "{$type}[{$index}]";
    return "
        <tr>
            <td>{$srText}</td>
            <td>
                <select name='{$namePrefix}[title]' required>
                    <option value=''>Select</option>
                    <option>Mr</option>
                    <option>Mrs</option>
                    <option>Ms</option>
                </select>
            </td>
            <td><input type='text' name='{$namePrefix}[surname]' required></td>
            <td><input type='text' name='{$namePrefix}[given_name]' required></td>
            <td><input type='text' name='{$namePrefix}[passport]' required></td>
            <td><input type='date' name='{$namePrefix}[dob]' required></td>
            <td><input type='date' name='{$namePrefix}[doi]' required></td>
            <td><input type='date' name='{$namePrefix}[doe]' required></td>
        </tr>";
}
?>
<!DOCTYPE html>
<html lang="en">

<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Umrah Booking - RF Travel & Tours</title>
    <link rel="stylesheet" href="css/style.css">
    <link rel="preconnect" href="https://fonts.googleapis.com">
    <link rel="preconnect" href="https://fonts.gstatic.com" crossorigin>
    <link href="https://fonts.googleapis.com/css2?family=Poppins:wght@400;500;600;700&display=swap" rel="stylesheet">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.2.0/css/all.min.css">

    <style>
        :root {
            --primary-blue: #31a7e2;
            --secondary-blue: #31a7e2;
            --light-grey-bg: #f4f7f9;
            --border-color: #e4e7eb;
            --text-dark: #212529;
            --text-light: #6c757d;
            --white: #fff;
            --green: #28a745;
            --box-shadow: 0 5px 15px rgba(0, 0, 0, 0.07);
        }

        body {
            font-family: 'Poppins', sans-serif;
            background-color: var(--light-grey-bg);
            margin: 0;
            color: var(--text-dark);
            font-size: 14px;
        }

        .container {
            max-width: 1700px;
            margin: 0 auto;
            padding: 2rem;
        }

        .card {
            background: var(--white);
            border: 1px solid var(--border-color);
            border-radius: 8px;
            margin-bottom: 2rem;
            box-shadow: var(--box-shadow);
            overflow: hidden;
        }

        .page-header {
            margin-bottom: 2rem;
            display: flex;
            justify-content: space-between;
            align-items: center;
        }

        .page-header-title h1 {
            font-size: 2.25rem;
            color: var(--primary-blue);
            margin: 0 0 0.5rem 0;
        }

        .breadcrumbs a {
            text-decoration: none;
            color: var(--secondary-blue);
            font-weight: 500;
        }

        .breadcrumbs span {
            color: var(--text-light);
        }

        .btn-back {
            display: inline-flex;
            align-items: center;
            gap: 0.5rem;
            text-decoration: none;
            background-color: var(--white);
            color: var(--primary-blue);
            border: 1px solid var(--border-color);
            padding: 0.7rem 1.2rem;
            border-radius: 6px;
            font-weight: 600;
            transition: all 0.2s ease;
        }

        .btn-back:hover {
            background-color: var(--primary-blue);
            color: var(--white);
            border-color: var(--primary-blue);
        }

        .summary-header {
            padding: 1rem 1.5rem;
            border-bottom: 1px solid var(--border-color);
        }

        .summary-header p {
            font-weight: 600;
            color: var(--text-dark);
            margin: 0;
        }

        .summary-header span {
            color: var(--green);
        }

        .summary-body {
            display: flex;
            flex-wrap: wrap;
            justify-content: space-between;
            align-items: flex-start;
            gap: 1.5rem;
            padding: 1.5rem;
        }

        .summary-airline-info .airline-name {
            font-weight: 600;
            font-size: 1.1rem;
            display: block;
        }

        .summary-airline-info img {
            max-height: 25px;
            margin-top: 5px;
        }

        .summary-details {
            flex-grow: 1;
        }

        .summary-details table {
            width: 100%;
            border-collapse: collapse;
        }

        .summary-details th,
        .summary-details td {
            padding: 0.5rem;
            text-align: left;
            vertical-align: top;
        }

        .summary-details th {
            font-weight: 600;
            color: var(--text-light);
        }

        .summary-details td a {
            color: var(--secondary-blue);
            text-decoration: none;
            display: block;
            font-weight: 500;
        }

        .summary-details td strong {
            font-size: 1.2rem;
            font-weight: 600;
        }

        .form-section-header {
            background: var(--primary-blue);
            color: #fff;
            padding: 1rem 1.5rem;
            font-weight: 600;
            font-size: 1.1rem;
        }

        .form-section-header.flex {
            display: flex;
            justify-content: space-between;
        }

        .pax-selection-grid {
            display: grid;
            grid-template-columns: repeat(3, 1fr);
            gap: 1.5rem;
            padding: 1.5rem;
        }

        .pax-selection-grid table {
            width: 100%;
            text-align: center;
        }

        .pax-selection-grid th,
        .pax-selection-grid td {
            padding: 0.5rem;
        }

        .pax-selection-grid th {
            color: var(--text-light);
        }

        .pax-selection-grid td {
            font-weight: 500;
        }

        .pax-selection-grid strong {
            color: var(--primary-blue);
            font-weight: 600;
        }

        .pax-selection-grid input {
            width: 100%;
            box-sizing: border-box;
            text-align: center;
            padding: 0.6rem;
            border: 1px solid var(--border-color);
            border-radius: 4px;
            font-size: 1rem;
        }

        .passenger-details-wrapper {
            overflow-x: auto;
        }

        .passenger-details-table {
            width: 100%;
            border-collapse: collapse;
        }

        .passenger-details-table th,
        .passenger-details-table td {
            border: 1px solid var(--border-color);
            padding: 0.75rem;
            text-align: center;
            white-space: nowrap;
        }

        .passenger-details-table th {
            background-color: var(--light-grey-bg);
            font-weight: 600;
        }

        .passenger-details-table input,
        .passenger-details-table select {
            width: 100%;
            box-sizing: border-box;
            padding: 0.6rem;
            border: 1px solid var(--border-color);
            border-radius: 4px;
        }

        .submit-container {
            padding: 2rem;
            text-align: center;
            background: #fdfdfd;
            border-top: 1px solid var(--border-color);
        }

        .btn-submit {
            background: var(--primary-blue);
            color: #fff;
            border: none;
            padding: 0.8rem 2.5rem;
            font-size: 1.1rem;
            cursor: pointer;
            border-radius: 6px;
            font-weight: 600;
        }

        .error-message {
            color: #dc3545;
            background-color: #f8d7da;
            border: 1px solid #f5c6cb;
            padding: 1rem;
            text-align: center;
            border-radius: 8px;
        }

        @media (max-width: 768px) {
            .container {
                padding: 1rem;
            }

            .page-header {
                flex-direction: column;
                align-items: flex-start;
                gap: 1.5rem;
            }

            .pax-selection-grid {
                grid-template-columns: 1fr;
            }

            .summary-body {
                flex-direction: column;
                align-items: stretch;
            }
        }
    </style>
</head>

<body>
    <?php include 'header.php'; ?>
    <div class="container">
        <div class="page-header">
            <div class="page-header-title">
                <h1>Umrah Booking</h1>
                <div class="breadcrumbs">
                    <a href="index.php">Home</a> / <a href="group-fares.php">Umrah Group</a> / <span>Umrah Group Booking</span>
                </div>
            </div>
            <a href="javascript:history.back()" class="btn-back"><i class="fas fa-arrow-left"></i> Go Back</a>
        </div>

        <?php if (isset($_SESSION['error_message'])): ?>
            <p class="error-message" style="margin-bottom: 2rem;"><?= htmlspecialchars($_SESSION['error_message']);
                                                                    unset($_SESSION['error_message']); ?></p>
        <?php endif; ?>

        <?php if ($error_message): ?>
            <p class="error-message"><?= htmlspecialchars($error_message) ?></p>
        <?php else: ?>
            <main>
                <div class="card summary-card">
                    <div class="summary-header">
                        <p>Umrah | <?= htmlspecialchars($fare_details['airline_name'] . ' ' . $fare_details['route']) ?> | Number Of Days <?= htmlspecialchars($fare_details['duration_days']) ?> | <span><?= htmlspecialchars($fare_details['group_ref_id']) ?></span></p>
                    </div>
                    <div class="summary-body">
                        <div class="summary-airline-info">
                            <span class="airline-name"><?= htmlspecialchars($fare_details['airline_name']) ?></span>
                            <img src="<?= htmlspecialchars($fare_details['logo_url']) ?>" alt="Airline Logo">
                        </div>
                        <div class="summary-details">
                            <table>
                                <thead>
                                    <tr>
                                        <th>Sector Details (Umrah)</th>
                                        <th>Baggage</th>
                                        <th>Seats</th>
                                        <th>Dep Date</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <tr>
                                        <td>
                                            <a href="#"><?= htmlspecialchars($fare_details['flight_details_json']['outbound'] ?? 'N/A') ?></a>
                                            <a href="#"><?= htmlspecialchars($fare_details['flight_details_json']['inbound'] ?? 'N/A') ?></a>
                                        </td>
                                        <td><?= nl2br(htmlspecialchars($fare_details['flight_details_json']['baggage'] ?? 'N/A')) ?></td>
                                        <td>Available Seats: <?= htmlspecialchars($fare_details['remaining_seats']) ?></td>
                                        <td><strong><?= format_flight_date_short($fare_details['departure_date']) ?></strong></td>
                                    </tr>
                                </tbody>
                            </table>
                        </div>
                    </div>
                </div>

                <form id="booking-form" class="card booking-form" action="submit-group-booking.php" method="POST">
                    <input type="hidden" name="group_ref_id" value="<?= htmlspecialchars($fare_details['group_ref_id']) ?>">
                    <input type="hidden" name="full_fare_details" value="<?= htmlspecialchars(json_encode($fare_details)) ?>">

                    <div class="form-section-header flex">
                        <span>Passengers</span><span>Price/Seat</span><span>Total Price</span>
                    </div>
                    <div class="pax-selection-grid">
                        <div class="pax-count-section">
                            <table>
                                <thead>
                                    <tr>
                                        <th>Adults</th>
                                        <th>Child</th>
                                        <th>Infants</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <tr>
                                        <td><input type="number" id="adults-count" name="pax_adults" value="1" min="1" max="<?= htmlspecialchars($fare_details['remaining_seats']) ?>"></td>
                                        <td><input type="number" id="child-count" name="pax_child" value="0" min="0" max="<?= htmlspecialchars($fare_details['remaining_seats']) ?>"></td>
                                        <td><input type="number" id="infants-count" name="pax_infants" value="0" min="0" max="<?= htmlspecialchars($fare_details['remaining_seats']) ?>"></td>
                                    </tr>
                                </tbody>
                            </table>
                        </div>
                        <div class="price-per-seat-section">
                            <table>
                                <thead>
                                    <tr>
                                        <th>Adult</th>
                                        <th>Child</th>
                                        <th>Infant</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <tr>
                                        <td>PKR <?= number_format($fare_details['price_adult']) ?></td>
                                        <td>PKR <?= number_format($fare_details['price_child']) ?></td>
                                        <td>PKR <?= number_format($fare_details['price_infant']) ?></td>
                                    </tr>
                                </tbody>
                            </table>
                        </div>
                        <div class="total-price-section">
                            <table>
                                <thead>
                                    <tr>
                                        <th>Adults</th>
                                        <th>Childs</th>
                                        <th>Infants</th>
                                        <th>Total Price</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <tr>
                                        <td id="total-adults-price">PKR: 0</td>
                                        <td id="total-child-price">PKR: 0</td>
                                        <td id="total-infants-price">PKR: 0</td>
                                        <td id="total-booking-price"><strong>PKR: 0.00</strong></td>
                                    </tr>
                                    <tr>
                                        <td colspan="3"><strong>Total Seats:</strong></td>
                                        <td id="total-seats-display">1</td>
                                    </tr>
                                </tbody>
                            </table>
                        </div>
                    </div>

                    <div class="form-section-header">Passenger Details</div>
                    <div class="passenger-details-wrapper">
                        <table class="passenger-details-table">
                            <thead>
                                <tr>
                                    <th>Sr#</th>
                                    <th>Title</th>
                                    <th>Sur Name</th>
                                    <th>Given Name</th>
                                    <th>Passport#</th>
                                    <th>DOB</th>
                                    <th>DOI</th>
                                    <th>DOE</th>
                                </tr>
                            </thead>
                            <tbody id="passenger-rows-container">
                                <!-- JS will populate this -->
                            </tbody>
                        </table>
                    </div>
                    <div class="submit-container">
                        <button type="submit" class="btn-submit">Submit Booking</button>
                    </div>
                </form>
            </main>
        <?php endif; ?>
    </div>
    <?php include 'footer.php'; ?>
    <script>
        document.addEventListener('DOMContentLoaded', function() {
            const adultsCountInput = document.getElementById('adults-count');
            const childCountInput = document.getElementById('child-count');
            const infantsCountInput = document.getElementById('infants-count');
            const passengerRowsContainer = document.getElementById('passenger-rows-container');
            const priceAdult = <?= $fare_details['price_adult'] ?? 0 ?>;
            const priceChild = <?= $fare_details['price_child'] ?? 0 ?>;
            const priceInfant = <?= $fare_details['price_infant'] ?? 0 ?>;
            const maxSeats = <?= $fare_details['remaining_seats'] ?? 1 ?>;
            const totalAdultsPriceEl = document.getElementById('total-adults-price');
            const totalChildPriceEl = document.getElementById('total-child-price');
            const totalInfantsPriceEl = document.getElementById('total-infants-price');
            const totalBookingPriceEl = document.getElementById('total-booking-price');
            const totalSeatsDisplayEl = document.getElementById('total-seats-display');

            function createRowHtml(type, index, srText) {
                const namePrefix = `${type}[${index}]`;
                return `<tr><td>${srText}</td><td><select name="${namePrefix}[title]" required><option value="">Select</option><option>Mr</option><option>Mrs</option><option>Ms</option></select></td><td><input type="text" name="${namePrefix}[surname]" required></td><td><input type="text" name="${namePrefix}[given_name]" required></td><td><input type="text" name="${namePrefix}[passport]" required></td><td><input type="date" name="${namePrefix}[dob]" required></td><td><input type="date" name="${namePrefix}[doi]" required></td><td><input type="date" name="${namePrefix}[doe]" required></td></tr>`;
            }

            function generatePassengerRows() {
                passengerRowsContainer.innerHTML = '';
                const adults = parseInt(adultsCountInput.value) || 0;
                const children = parseInt(childCountInput.value) || 0;
                const infants = parseInt(infantsCountInput.value) || 0;
                let currentSr = 1;
                for (let i = 0; i < adults; i++) {
                    passengerRowsContainer.innerHTML += createRowHtml('adults', i, `A-${currentSr++}`);
                }
                for (let i = 0; i < children; i++) {
                    passengerRowsContainer.innerHTML += createRowHtml('children', i, `C-${currentSr++}`);
                }
                for (let i = 0; i < infants; i++) {
                    passengerRowsContainer.innerHTML += createRowHtml('infants', i, `I-${currentSr++}`);
                }
            }

            function updateTotals() {
                let adults = parseInt(adultsCountInput.value) || 0;
                let children = parseInt(childCountInput.value) || 0;
                let infants = parseInt(infantsCountInput.value) || 0;
                if (adults + children > maxSeats) {
                    alert(`The total number of adults and children cannot exceed the ${maxSeats} available seats.`);
                    adultsCountInput.value = 1; // Reset to default
                    childCountInput.value = 0; // Reset to default
                    return handleUpdate(); // Re-run the update
                }
                const totalAdultPrice = adults * priceAdult;
                const totalChildPrice = children * priceChild;
                const totalInfantPrice = infants * priceInfant;
                const totalBookingPrice = totalAdultPrice + totalChildPrice + totalInfantPrice;
                totalAdultsPriceEl.textContent = 'PKR ' + totalAdultPrice.toLocaleString();
                totalChildPriceEl.textContent = 'PKR ' + totalChildPrice.toLocaleString();
                totalInfantsPriceEl.textContent = 'PKR ' + totalInfantPrice.toLocaleString();
                totalBookingPriceEl.innerHTML = '<strong>PKR ' + totalBookingPrice.toLocaleString('en-US', {
                    minimumFractionDigits: 2,
                    maximumFractionDigits: 2
                }) + '</strong>';
                totalSeatsDisplayEl.textContent = adults + children;
            }

            function handleUpdate() {
                updateTotals();
                generatePassengerRows();
            }

            [adultsCountInput, childCountInput, infantsCountInput].forEach(input => {
                input.addEventListener('input', handleUpdate);
            });

            // Initial call to set up the form correctly on page load
            handleUpdate();
        });
    </script>
</body>

</html>