<?php
// FILE: featured-hotels.php
// PURPOSE: To be included on a homepage, showing the top 3 hotels with the new design.
if (session_status() === PHP_SESSION_NONE) { session_start(); }

if (!isset($conn)) {
    include_once 'db-config.php';
}

// ===================================================================
//  CONFIGURATION
// ===================================================================
if (!defined('DEFAULT_SAR_TO_PKR_RATE_FEATURED')) {
    define('DEFAULT_SAR_TO_PKR_RATE_FEATURED', 75.50); // Fallback rate
}


// ===================================================================
//  PREPARE AGENT/COMPANY DATA
// ===================================================================
$is_agent_featured = (isset($_SESSION['user_type']) && $_SESSION['user_type'] === 'agent');
$display_logo_featured = 'images/logo.png';
$footer_address_featured = 'Al Quresh, Railway Phatak, Multan';
$footer_phone_1_featured = '923052394810';

if ($is_agent_featured && isset($_SESSION['user_id'])) {
    $agent_id_featured = $_SESSION['user_id'];
    $stmt_featured = $conn->prepare("SELECT logo_path, company_address, company_phone_1 FROM users WHERE id = ?");
    if($stmt_featured) {
        $stmt_featured->bind_param("i", $agent_id_featured);
        $stmt_featured->execute();
        $result_agent_featured = $stmt_featured->get_result();
        if ($agent_data_featured = $result_agent_featured->fetch_assoc()) {
            if (!empty($agent_data_featured['logo_path'])) { $display_logo_featured = 'admin/' . $agent_data_featured['logo_path']; }
            if (!empty($agent_data_featured['company_address'])) { $footer_address_featured = $agent_data_featured['company_address']; }
            if (!empty($agent_data_featured['company_phone_1'])) { $footer_phone_1_featured = preg_replace('/[^0-9]/', '', $agent_data_featured['company_phone_1']); }
        }
        $stmt_featured->close();
    }
}
if (!function_exists('safe_htmlspecialchars_featured')) {
    function safe_htmlspecialchars_featured($string) { return htmlspecialchars($string ?? '', ENT_QUOTES, 'UTF-8'); }
}
if (!function_exists('generate_star_rating_html_featured')) {
    function generate_star_rating_html_featured($rating) {
        $html = ''; $rating = (float) $rating;
        for ($i = 1; $i <= 5; $i++) {
            if ($rating >= $i) { $html .= '<i class="fa-solid fa-star"></i>'; } 
            elseif ($rating > ($i - 1) && $rating < $i) { $html .= '<i class="fa-solid fa-star-half-stroke"></i>'; } 
            else { $html .= '<i class="fa-regular fa-star"></i>'; }
        } return $html;
    }
}
?>

<!-- ======================================================== -->
<!-- === FEATURED HOTEL DEALS (LAYOUT FIX) === -->
<!-- ======================================================== -->
<section class="hotel-featured-section">
    <div class="listings-container">
        <div class="listings-header">
            <div class="listings-header-text">
                <h2 class="listings-title">Top Hotel Deals</h2>
                <p class="listings-subtitle">Discover our curated selection of top-rated hotels at unbeatable prices.</p>
            </div>
            <a href="hotels.php" class="btn-view-more">View All Hotels</a>
        </div>
        <div class="featured-grid-wrapper">
            <div class="featured-grid">
                <?php
                try {
                    $sql_featured = "SELECT * FROM hotels WHERE is_active = 1 ORDER BY display_order ASC, last_updated DESC LIMIT 3";
                    $result_featured = $conn->query($sql_featured);

                    if ($result_featured && $result_featured->num_rows > 0):
                        while ($hotel = $result_featured->fetch_assoc()):
                            $sar_price_featured = ($is_agent_featured && !empty($hotel['price_per_night_agent'])) ? $hotel['price_per_night_agent'] : $hotel['price_per_night'];
                            $exchange_rate_featured = !empty($hotel['exchange_rate_pkr']) ? $hotel['exchange_rate_pkr'] : DEFAULT_SAR_TO_PKR_RATE_FEATURED;
                            $pkr_price_featured = number_format($sar_price_featured * $exchange_rate_featured);
                            $image_url_with_version_featured = safe_htmlspecialchars_featured($hotel['image_url']) . '?v=' . strtotime($hotel['last_updated']);
                            $logo_bg_class_featured = ($hotel['logo_background_style'] === 'transparent') ? 'transparent-logo' : 'white-bg-logo';
                ?>
                            <!-- This HTML structure now matches the working one from all-hotels.php -->
                            <div class="hotel-card-v2">
                                <div class="hotel-card-v2__banner">
                                    <div class="hotel-card-v2__logo-overlay <?= $logo_bg_class_featured ?>">
                                        <img src="<?= safe_htmlspecialchars_featured($display_logo_featured) ?>" alt="Agency Logo" crossorigin="anonymous">
                                    </div>
                                    <img src="<?= $image_url_with_version_featured ?>" alt="Image of <?= safe_htmlspecialchars_featured($hotel['hotel_name']) ?>" class="hotel-card-v2__image" crossorigin="anonymous">
                                </div>
                                <div class="hotel-card-v2__content">
                                    <h3 class="hotel-card-v2__title"><?= safe_htmlspecialchars_featured($hotel['hotel_name']) ?></h3>
                                    <p class="hotel-card-v2__location"><i class="fa-solid fa-location-dot"></i> <?= safe_htmlspecialchars_featured($hotel['location']) ?></p>
                                    <div class="hotel-card-v2__rating"><?= generate_star_rating_html_featured($hotel['rating']); ?></div>
                                    <p class="hotel-card-v2__price">from <span>PKR <?= $pkr_price_featured ?></span>/night</p>
                                </div>
                                <div class="hotel-card-v2__actions">
                                    <?php $whatsapp_message_featured = urlencode("Hi, I'm interested in the hotel: '" . $hotel['hotel_name'] . "' in " . $hotel['location']); ?>
                                    <a href="https://wa.me/<?= $footer_phone_1_featured ?>?text=<?= $whatsapp_message_featured ?>" target="_blank" class="btn-hotel-primary">Enquire</a>
                                    <button class="btn-hotel-secondary btn-download-featured" data-download-id="featured-hotel-<?= $hotel['id'] ?>"><i class="fa-solid fa-download"></i> Download</button>
                                </div>
                            </div>

                             <!-- HIDDEN POST FOR DOWNLOADING -->
                            <div class="downloadable-hotel-post" id="download-post-featured-hotel-<?= $hotel['id'] ?>">
                                <img src="<?= $image_url_with_version_featured ?>" class="hotel-post-bg-image" crossorigin="anonymous" alt="">
                                <div class="hotel-post-overlay">
                                    <header class="hotel-post-header">
                                        <div class="hotel-post-logo <?= $logo_bg_class_featured ?>"><img src="<?= safe_htmlspecialchars_featured($display_logo_featured) ?>" alt="Agency Logo" crossorigin="anonymous"></div>
                                    </header>
                                    <main class="hotel-post-body">
                                        <div class="hotel-post-title"><h1><?= safe_htmlspecialchars_featured($hotel['hotel_name']) ?></h1><h2><?= safe_htmlspecialchars_featured($hotel['location']) ?></h2></div>
                                        <div class="hotel-post-details-box">
                                            <div class="hotel-post-detail-item"><div class="rating-display"><?= generate_star_rating_html_featured($hotel['rating']) ?></div><h3>Star Rating</h3></div>
                                            <div class="hotel-post-detail-item"><p>PKR <?= $pkr_price_featured ?></p><h3>Price Per Night</h3></div>
                                        </div>
                                    </main>
                                    <footer class="hotel-post-footer">
                                        <span><i class="fas fa-map-marker-alt"></i> <?= safe_htmlspecialchars_featured($footer_address_featured) ?></span>
                                        <span><i class="fab fa-whatsapp"></i> <?= safe_htmlspecialchars_featured($footer_phone_1_featured) ?></span>
                                    </footer>
                                </div>
                            </div>
                <?php
                        endwhile;
                    else:
                        echo '<p>No featured hotels are available at the moment.</p>';
                    endif;
                } catch (Exception $e) {
                    error_log("Featured Hotel Listing Error: " . $e->getMessage());
                    echo '<p>Unable to display hotel deals.</p>';
                }
                ?>
                <a href="hotels.php" class="view-more-card">
                    <span>View More Hotels</span>
                    <i class="fa-solid fa-arrow-right"></i>
                </a>
            </div>
        </div>
    </div>
</section>

<!-- STYLES AND SCRIPTS FOR THE FEATURED SECTION (NOW SCOPED TO PREVENT CONFLICTS) -->
<style>
/* Scoping all styles to the unique parent class ".hotel-featured-section" */
.hotel-featured-section { padding: 40px 0; }
.hotel-featured-section .featured-grid-wrapper { min-height: 400px; }
.hotel-featured-section .featured-grid { display: grid; grid-template-columns: repeat(auto-fit, minmax(320px, 1fr)); gap: 2rem; }
@media (min-width: 1200px) {
    .hotel-featured-section .featured-grid { grid-template-columns: repeat(3, 1fr); }
}
.hotel-featured-section .hotel-card-v2 { background: #fff; border: 1px solid #eef2f6; border-radius: 12px; overflow: hidden; display: flex; flex-direction: column; transition: all 0.3s ease; }
.hotel-featured-section .hotel-card-v2:hover { transform: translateY(-5px); box-shadow: 0 8px 25px rgba(0,0,0,0.08); }
.hotel-featured-section .hotel-card-v2__banner { position: relative; }
.hotel-featured-section .hotel-card-v2__image { width: 100%; height: 220px; object-fit: cover; }
.hotel-featured-section .hotel-card-v2__logo-overlay { position: absolute; top: 15px; left: 15px; width: 45px; height: 45px; display: flex; align-items: center; justify-content: center; }
.hotel-featured-section .hotel-card-v2__logo-overlay.white-bg-logo { background: rgba(255,255,255,0.9); backdrop-filter: blur(4px); border-radius: 50%; padding: 5px; box-shadow: 0 1px 5px rgba(0,0,0,0.2); }
.hotel-featured-section .hotel-card-v2__logo-overlay.transparent-logo { background: none; backdrop-filter: none; box-shadow: none; }
.hotel-featured-section .hotel-card-v2__logo-overlay.transparent-logo img { filter: drop-shadow(0 2px 3px rgba(0,0,0,0.5)); }
.hotel-featured-section .hotel-card-v2__logo-overlay img { max-width: 100%; max-height: 100%; object-fit: contain; }
.hotel-featured-section .hotel-card-v2__content { padding: 1.25rem; flex-grow: 1; display: flex; flex-direction: column; }
.hotel-featured-section .hotel-card-v2__actions { display: grid; grid-template-columns: 1fr 1fr; gap: 10px; padding: 1rem; background-color: #f8f9fa; border-top: 1px solid #eef2f6; }
.hotel-featured-section .btn-hotel-primary, .hotel-featured-section .btn-hotel-secondary { padding: 12px; font-size: 0.9rem; font-weight: 600; border-radius: 8px; cursor: pointer; text-align: center; text-decoration: none; border: 1px solid transparent; transition: all 0.3s; }
.hotel-featured-section .btn-hotel-primary { background-color: #212529; color: #fff; }
.hotel-featured-section .btn-hotel-primary:hover { background-color: #343a40; }
.hotel-featured-section .btn-hotel-secondary { background-color: #f8f9fa; color: #555; border-color: #dee2e6; }
.hotel-featured-section .btn-hotel-secondary:hover:not(:disabled) { background-color: #e9ecef; }
.hotel-featured-section .btn-hotel-secondary:disabled { opacity: 0.7; cursor: not-allowed; }
.hotel-featured-section .btn-hotel-secondary i { margin-right: 5px; }
.hotel-featured-section .hotel-card-v2__title { font-size: 1.25rem; font-weight: 600; color: var(--text-dark); margin-bottom: 0.25rem; }
.hotel-featured-section .hotel-card-v2__location { font-size: 0.9rem; color: var(--text-light); margin-bottom: 1rem; }
.hotel-featured-section .hotel-card-v2__rating { color: #f39c12; margin-bottom: 1rem; }
.hotel-featured-section .hotel-card-v2__price { margin-top: auto; font-size: 0.9rem; color: var(--text-light); }
.hotel-featured-section .hotel-card-v2__price span { font-weight: 700; font-size: 1.5rem; color: var(--text-dark); }
@media (max-width: 768px) { .hotel-featured-section .featured-grid { display: flex; overflow-x: auto; gap: 1rem; padding: 0.5rem 1rem 1rem 1rem; scroll-snap-type: x mandatory; -ms-overflow-style: none; scrollbar-width: none; } .hotel-featured-section .featured-grid::-webkit-scrollbar { display: none; } .hotel-featured-section .hotel-card-v2 { flex: 0 0 85%; max-width: 320px; scroll-snap-align: start; } }

/* --- ** CRITICAL FIX: Hide the downloadable post template ** --- */
.hotel-featured-section .downloadable-hotel-post { display: none; width: 1080px; height: 1080px; font-family: 'Poppins', sans-serif; position: relative; color: #fff; }
.hotel-featured-section .hotel-post-bg-image { position: absolute; top: 0; left: 0; width: 100%; height: 100%; object-fit: cover; z-index: 1; }
.hotel-featured-section .hotel-post-overlay { position: relative; z-index: 2; width: 100%; height: 100%; padding: 60px; display: flex; flex-direction: column; justify-content: space-between; background: linear-gradient(to top, rgba(10, 15, 20, 0.98) 0%, rgba(10, 15, 20, 0.9) 50%, rgba(10, 15, 20, 0.4) 100%); }
.hotel-featured-section .hotel-post-header { display: flex; }
.hotel-featured-section .hotel-post-logo { width: 140px; height: 140px; display: flex; align-items: center; justify-content: center; padding: 15px; }
.hotel-featured-section .hotel-post-logo.white-bg-logo { background-color: rgba(255, 255, 255, 0.95); border-radius: 50%; box-shadow: 0 4px 15px rgba(0,0,0,0.3); }
.hotel-featured-section .hotel-post-logo.transparent-logo img { filter: drop-shadow(0px 3px 5px rgba(0,0,0,0.6)); }
.hotel-featured-section .hotel-post-logo img { max-width: 100%; max-height: 100%; }
.hotel-featured-section .hotel-post-body { text-align: center; flex-grow: 1; display: flex; flex-direction: column; justify-content: center; }
.hotel-featured-section .hotel-post-title h1 { font-size: 90px; font-weight: 700; line-height: 1.1; margin: 0; text-shadow: 2px 3px 8px rgba(0,0,0,0.6); }
.hotel-featured-section .hotel-post-title h2 { font-size: 40px; font-weight: 400; opacity: 0.9; margin: 5px 0 50px 0; }
.hotel-featured-section .hotel-post-details-box { background-color: rgba(0, 0, 0, 0.5); backdrop-filter: blur(12px); border-radius: 20px; padding: 30px 50px; display: grid; grid-template-columns: 1fr 1fr; gap: 20px; max-width: 700px; margin: 0 auto; border: 1px solid rgba(255,255,255,0.1); }
.hotel-featured-section .hotel-post-detail-item .rating-display { font-size: 36px; color: #f39c12; margin-bottom: 10px; }
.hotel-featured-section .hotel-post-detail-item p { font-size: 48px; font-weight: 700; margin: 0 0 5px 0; }
.hotel-featured-section .hotel-post-detail-item h3 { font-size: 22px; color: #ccc; font-weight: 500; margin: 0; }
.hotel-featured-section .hotel-post-footer { background-color: #3498db; border-radius: 20px; padding: 25px 40px; display: flex; justify-content: space-between; align-items: center; font-size: 24px; font-weight: 500; }
.hotel-featured-section .hotel-post-footer i { margin-right: 15px; font-size: 28px; }
</style>

<script>
(function() {
    document.addEventListener('DOMContentLoaded', function() {
        // Scoped Javascript to only affect this section
        document.querySelectorAll('.hotel-featured-section .btn-download-featured').forEach(button => {
            button.addEventListener('click', function (e) {
                e.preventDefault();
                const originalButtonHTML = this.innerHTML; this.innerHTML = '<i class="fa-solid fa-spinner fa-spin"></i>'; this.disabled = true;
                if (typeof html2canvas === 'undefined') { console.error("html2canvas not loaded."); this.disabled = false; return; }
                const downloadId = this.dataset.downloadId; const postToDownload = document.getElementById(`download-post-${downloadId}`);
                if (!postToDownload) { console.error('Post element not found for ID:', downloadId); this.disabled = false; return; }
                const hotelName = postToDownload.querySelector('h1').textContent || 'hotel'; const sanitizedFileName = hotelName.replace(/[^a-z0-9]/gi, '_').toLowerCase();
                document.body.appendChild(postToDownload); postToDownload.style.display = 'block';
                html2canvas(postToDownload, { useCORS: true, scale: 1.5 }).then(canvas => {
                    const link = document.createElement('a'); link.href = canvas.toDataURL('image/jpeg', 0.9); link.download = `${sanitizedFileName}.jpg`; link.click(); document.body.removeChild(link);
                }).catch(err => { console.error('Download failed!', err); alert("Download Failed.");
                }).finally(() => { postToDownload.style.display = 'none'; this.innerHTML = originalButtonHTML; this.disabled = false; });
            });
        });
    });
})();
</script>