<?php
session_start();
include 'db-config.php';

$errors = [];

// --- DEFINITIVE REDIRECT LOGIC ---
// 1. Check for a high-priority redirect from a secure flow (like group bookings).
if (isset($_SESSION['login_redirect'])) {
    $redirect_url = $_SESSION['login_redirect'];
    unset($_SESSION['login_redirect']); // Use it once and clear it.
}
// 2. Fallback to the redirect URL passed in the query string (like from flight search).
else {
    $redirect_url = filter_var($_GET['redirect_url'] ?? 'index.php', FILTER_SANITIZE_URL);
}

// Get the 'view' parameter from the URL to show the correct form.
$initial_view = $_GET['view'] ?? 'login';

if ($_SERVER["REQUEST_METHOD"] == "POST") {
    // On POST, the redirect URL comes from the hidden input field.
    $post_redirect_url = $_POST['redirect_url'] ?? 'index.php';

    // --- SIGNUP LOGIC ---
    if (isset($_POST['signup'])) {
        // Get form data
        $user_type = trim($_POST['user_type'] ?? 'customer');
        $name = trim($_POST['name'] ?? '');
        $email = trim($_POST['email'] ?? '');
        $password = $_POST['password'] ?? '';
        $mobile_number = trim($_POST['mobile_number'] ?? '');
        $company_name = ($user_type === 'agent') ? trim($_POST['company_name'] ?? '') : null;

        // --- Validation ---
        if (empty($name) || empty($email) || empty($password)) {
            $errors[] = "Full Name, Email, and Password are required.";
        }
        if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
            $errors[] = "Invalid email format.";
        }
        if (strlen($password) < 6) {
            $errors[] = "Password must be at least 6 characters long.";
        }
        if (!empty($mobile_number) && !preg_match('/^[0-9\s\+]{10,15}$/', $mobile_number)) {
            $errors[] = "Invalid mobile number format.";
        }

        // Check if email already exists
        if (empty($errors)) {
            $stmt_check = $conn->prepare("SELECT id FROM users WHERE email = ?");
            $stmt_check->bind_param("s", $email);
            $stmt_check->execute();
            $stmt_check->store_result();
            if ($stmt_check->num_rows > 0) {
                $errors[] = "An account with this email already exists.";
            }
            $stmt_check->close();
        }

        if (empty($errors)) {
            // Hash the password for security
            $hashed_password = password_hash($password, PASSWORD_DEFAULT);

            $sql_insert = "INSERT INTO users (name, email, password, user_type, company_name, mobile_number) VALUES (?, ?, ?, ?, ?, ?)";
            $stmt_insert = $conn->prepare($sql_insert);
            $stmt_insert->bind_param("ssssss", $name, $email, $hashed_password, $user_type, $company_name, $mobile_number);

            if ($stmt_insert->execute()) {
                $_SESSION['user_id'] = $stmt_insert->insert_id;
                $_SESSION['user_name'] = $name;
                $_SESSION['user_type'] = $user_type;

                if ($user_type === 'agent') {
                    header("Location: agent-dashboard.php");
                } else {
                    header("Location: " . $post_redirect_url);
                }
                exit();
            } else {
                $errors[] = "Error creating account. Please try again.";
            }
            $stmt_insert->close();
        }
    }

    // --- LOGIN LOGIC ---
    if (isset($_POST['login'])) {
        $email = trim($_POST['email'] ?? '');
        $password = $_POST['password'] ?? '';

        if (empty($email) || empty($password)) {
            $errors[] = "Email and password are required.";
        } else {
            $stmt = $conn->prepare("SELECT * FROM users WHERE email = ?");
            $stmt->bind_param("s", $email);
            $stmt->execute();
            $result = $stmt->get_result();

            if ($result->num_rows == 1) {
                $user = $result->fetch_assoc();
                if (password_verify($password, $user['password'])) {
                    $_SESSION['user_id'] = $user['id'];
                    $_SESSION['user_name'] = $user['name'];
                    $_SESSION['user_type'] = $user['user_type'];

                    if ($user['user_type'] === 'agent') {
                        header("Location: agent-dashboard.php");
                    } else {
                        header("Location: " . $post_redirect_url);
                    }
                    exit();
                }
            }
            $errors[] = "Invalid email or password.";
            $stmt->close();
        }
    }
}
?>
<!DOCTYPE html>
<html lang="en">

<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Login / Sign Up</title>
    <link rel="icon" type="image/png" href="images/logo-icon.png">

    <link rel="preconnect" href="https://fonts.googleapis.com">
    <link rel="preconnect" href="https://fonts.gstatic" crossorigin>
    <link href="https://fonts.googleapis.com/css2?family=Poppins:wght@400;500;600;700&display=swap" rel="stylesheet">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.1/css/all.min.css">
    <link rel="stylesheet" href="css/login-style.css">

    <!-- START: ROBUST CSS FIX FOR ALL ICONS AND INPUTS -->
    <style>
        /* This rule targets the icon that is the FIRST element in the group */
        .form-group>i:first-child {
            position: absolute;
            left: 15px;
            /* Adjust if needed */
            top: 50%;
            transform: translateY(-50%);
            color: #aaa;
            /* A common color for placeholder icons */
            z-index: 2;
            /* Ensure icon is on top */
        }

        /* This rule applies left padding to ANY input that is a direct child of form-group */
        .form-group>input {
            padding-left: 45px !important;
            /* !important to override specific external styles */
        }
    </style>
    <!-- END: CSS FIX -->

</head>

<body>
    <div class="auth-page-wrapper">
        <div class="auth-container">
            <div class="auth-image-panel">
                <img src="images/kaaba-gate.jpg" alt="The Holy Kaaba in Makkah">
            </div>
            <div class="auth-form-panel">
                <div class="auth-form" id="login-form">
                    <img src="images/logo.png" alt="Company Logo" class="auth-logo">
                    <h2>Sign in to your account</h2>
                    <p class="auth-subheading">Enter your credentials to access your dashboard</p>
                    <?php
                    if ((!empty($errors) && isset($_POST['login'])) || isset($_SESSION['error_message'])) {
                        echo '<div class="error-container">';
                        if (isset($_SESSION['error_message'])) {
                            echo "<p>" . htmlspecialchars($_SESSION['error_message']) . "</p>";
                            unset($_SESSION['error_message']);
                        }
                        foreach ($errors as $error) echo "<p>" . htmlspecialchars($error) . "</p>";
                        echo '</div>';
                    }
                    ?>
                    <form method="POST" action="login.php">
                        <input type="hidden" name="login" value="1">
                        <input type="hidden" name="redirect_url" value="<?php echo htmlspecialchars($redirect_url); ?>">
                        <div class="form-group"><i class="fa-regular fa-user"></i><input type="email" name="email" placeholder="Email Address" required></div>
                        <div class="form-group"><i class="fa-solid fa-lock"></i><input type="password" name="password" class="password-input" placeholder="Password" required><i class="fa-regular fa-eye toggle-password"></i></div>
                        <button type="submit" class="form-button">Sign In</button>
                        <p class="form-switcher">Don't have an account? <a href="#" class="toggle-link">Sign Up</a></p>
                    </form>
                </div>
                <div class="auth-form is-hidden" id="signup-form">
                    <img src="images/logo.png" alt="Company Logo" class="auth-logo">
                    <h2>Create Your Account</h2>
                    <p class="auth-subheading">Join us to start your journey</p>
                    <?php
                    if (!empty($errors) && isset($_POST['signup'])) {
                        echo '<div class="error-container">';
                        foreach ($errors as $error) echo "<p>" . htmlspecialchars($error) . "</p>";
                        echo '</div>';
                    }
                    ?>
                    <form method="POST" action="login.php">
                        <input type="hidden" name="signup" value="1">
                        <input type="hidden" name="redirect_url" value="<?php echo htmlspecialchars($redirect_url); ?>">
                        <div class="form-group"><i class="fa-solid fa-briefcase"></i><select name="user_type" id="user-type-select">
                                <option value="customer" selected>I am a Customer</option>
                                <option value="agent">I am a Travel Agent</option>
                            </select></div>
                        <div class="form-group"><i class="fa-regular fa-user"></i><input type="text" name="name" placeholder="Full Name" required></div>
                        <div class="agent-fields" style="display:none;">
                            <div class="form-group"><i class="fa-regular fa-building"></i><input type="text" name="company_name" placeholder="Company Name"></div>
                        </div>
                        <div class="form-group"><i class="fa-regular fa-envelope"></i><input type="email" name="email" placeholder="Email Address" required></div>
                        <div class="form-group"><i class="fa-solid fa-phone"></i><input type="tel" name="mobile_number" placeholder="Mobile Number"></div>
                        <div class="form-group"><i class="fa-solid fa-lock"></i><input type="password" name="password" class="password-input" placeholder="Password" required><i class="fa-regular fa-eye toggle-password"></i></div>
                        <button type="submit" class="form-button">Create Account</button>
                        <p class="form-switcher">Already have an account? <a href="#" class="toggle-link">Login</a></p>
                    </form>
                </div>
            </div>
        </div>
    </div>
    <script>
        document.addEventListener('DOMContentLoaded', function() {
            const loginForm = document.getElementById('login-form');
            const signupForm = document.getElementById('signup-form');
            const toggleLinks = document.querySelectorAll('.toggle-link');

            toggleLinks.forEach(link => {
                link.addEventListener('click', (event) => {
                    event.preventDefault();
                    loginForm.classList.toggle('is-hidden');
                    signupForm.classList.toggle('is-hidden');
                });
            });

            const userTypeSelect = document.getElementById('user-type-select');
            const agentFields = document.querySelector('.agent-fields');
            if (userTypeSelect) {
                userTypeSelect.addEventListener('change', function() {
                    agentFields.style.display = (this.value === 'agent') ? 'block' : 'none';
                });
            }

            document.querySelectorAll('.toggle-password').forEach(toggle => {
                toggle.addEventListener('click', function() {
                    const passwordInput = this.previousElementSibling;
                    if (passwordInput.type === 'password') {
                        passwordInput.type = 'text';
                        this.classList.replace('fa-eye', 'fa-eye-slash');
                    } else {
                        passwordInput.type = 'password';
                        this.classList.replace('fa-eye-slash', 'fa-eye');
                    }
                });
            });

            <?php if ($initial_view === 'signup' || (!empty($errors) && isset($_POST['signup']))): ?>
                loginForm.classList.add('is-hidden');
                signupForm.classList.remove('is-hidden');
                // Keep agent fields visible on signup error if agent was selected
                if (document.getElementById('user-type-select').value === 'agent') {
                    document.querySelector('.agent-fields').style.display = 'block';
                }
            <?php endif; ?>
        });
    </script>
</body>

</html>