<?php
require_once 'db-config.php';
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

// --- NEW HELPER FUNCTION TO PARSE FLIGHT STRINGS (Made backward-compatible) ---
function parse_flight_string($flight_string, $year_from_fare)
{
    // Regex to capture: (Flight Number) (Day) (Month) (Sector) (Dep Time) (Arr Time)
    // Example: SV727 31 Jul ISB-JED 0300 0610
    $pattern = '/^([A-Z0-9]{2,}\d+)\s+(\d{1,2})\s+([A-Za-z]{3})\s+([A-Z]{3}-[A-Z]{3})\s+(\d{4})\s+(\d{4})$/';
    if (preg_match($pattern, $flight_string, $matches)) {
        list(, $flight_no, $day, $month_str, $sector, $dep_time, $arr_time) = $matches;

        // COMPATIBILITY FIX: Use strtotime which is more reliable than date_parse across versions.
        $month_num = date('m', strtotime($month_str));

        $dep_datetime_str = "$year_from_fare-$month_num-$day " . substr($dep_time, 0, 2) . ":" . substr($dep_time, 2, 2);
        $arr_datetime_str = "$year_from_fare-$month_num-$day " . substr($arr_time, 0, 2) . ":" . substr($arr_time, 2, 2);

        // Convert to 'Y-m-d H:i:s' format for MySQL
        $dep_mysql_datetime = date('Y-m-d H:i:s', strtotime($dep_datetime_str));
        $arr_mysql_datetime = date('Y-m-d H:i:s', strtotime($arr_datetime_str));

        // Check for overnight flights, add a day to arrival if necessary
        if (strtotime($arr_mysql_datetime) < strtotime($dep_mysql_datetime)) {
            $arr_mysql_datetime = date('Y-m-d H:i:s', strtotime($arr_datetime_str . ' +1 day'));
        }

        // COMPATIBILITY FIX: Use older array() syntax instead of []
        return array(
            'flight_no' => $flight_no,
            'sector' => $sector,
            'departure_datetime' => $dep_mysql_datetime,
            'arrival_datetime' => $arr_mysql_datetime,
        );
    }
    return null; // Return null if the string doesn't match the expected format
}

// --- INITIALIZATION ---
$booking_details_for_display = null;
$error_message = '';
$group_ref_id_for_redirect = '';

// --- SERVER-SIDE PROCESSING ---
if ($_SERVER["REQUEST_METHOD"] === "POST") {

    // --- SECURITY CHECK: User MUST be logged in ---
    if (!isset($_SESSION['user_id'])) {
        $_SESSION['error_message'] = "Your session has expired. Please log in to complete your booking.";
        header("Location: login.php");
        exit;
    }

    // COMPATIBILITY FIX: Replace null coalescing operator (??) with isset() ternary
    $group_ref_id_for_redirect = isset($_POST['group_ref_id']) ? $_POST['group_ref_id'] : null;
    $full_fare_details_json = isset($_POST['full_fare_details']) ? $_POST['full_fare_details'] : null;

    if (!$group_ref_id_for_redirect || !$full_fare_details_json) {
        $error_message = "Invalid submission. Incomplete fare information provided.";
    } else {
        $fare_details = json_decode($full_fare_details_json, true);

        $conn->begin_transaction();
        try {
            // 1. Lock the fare row to prevent race conditions
            $sql_fare_check = "SELECT id, remaining_seats, price_adult, price_child, price_infant, price_currency FROM group_fares WHERE group_ref_id = ? AND is_active = 1 FOR UPDATE";
            $stmt_fare_check = $conn->prepare($sql_fare_check);
            $stmt_fare_check->bind_param("s", $group_ref_id_for_redirect);
            $stmt_fare_check->execute();
            $result_fare = $stmt_fare_check->get_result();
            if ($result_fare->num_rows === 0) {
                throw new Exception("This group fare is no longer available.");
            }
            $current_fare_state = $result_fare->fetch_assoc();
            $stmt_fare_check->close();

            // 2. Validate passenger counts and seat availability
            // COMPATIBILITY FIX: Replace ?? with isset() ternary
            $pax_adults = (int)(isset($_POST['pax_adults']) ? $_POST['pax_adults'] : 0);
            $pax_child = (int)(isset($_POST['pax_child']) ? $_POST['pax_child'] : 0);
            $pax_infants = (int)(isset($_POST['pax_infants']) ? $_POST['pax_infants'] : 0);
            $seats_needed = $pax_adults + $pax_child;

            if ($seats_needed <= 0) {
                throw new Exception("You must book at least one adult or child seat.");
            }
            if ($current_fare_state['remaining_seats'] < $seats_needed) {
                throw new Exception("Sorry, there are not enough seats available for this group fare.");
            }

            // 3. Create the generic booking record FIRST to get a booking_id
            $user_id = $_SESSION['user_id'];
            // COMPATIBILITY FIX: Replace random_bytes() with openssl_random_pseudo_bytes()
            $booking_ref = "GRP-" . strtoupper(bin2hex(openssl_random_pseudo_bytes(4)));
            $status = 'pending';

            // COMPATIBILITY FIX: Replace ?? and use older array() syntax
            $adults_data = isset($_POST['adults']) ? $_POST['adults'] : array();
            $children_data = isset($_POST['children']) ? $_POST['children'] : array();
            $infants_data = isset($_POST['infants']) ? $_POST['infants'] : array();
            $passenger_details_json = json_encode(array('adults' => $adults_data, 'children' => $children_data, 'infants' => $infants_data));

            $grand_total_for_booking = ($pax_adults * $current_fare_state['price_adult']) + ($pax_child * $current_fare_state['price_child']) + ($pax_infants * $current_fare_state['price_infant']);

            $sql_insert_booking = "INSERT INTO bookings (booking_ref, user_id, booking_type, flight_details, passenger_details, total_price, price_currency, status) VALUES (?, ?, ?, ?, ?, ?, ?, ?)";
            $stmt_insert = $conn->prepare($sql_insert_booking);
            $booking_type = 'group';
            $stmt_insert->bind_param("sisssdss", $booking_ref, $user_id, $booking_type, $full_fare_details_json, $passenger_details_json, $grand_total_for_booking, $current_fare_state['price_currency'], $status);
            if (!$stmt_insert->execute()) {
                throw new Exception("Database error: Could not create booking record. " . $stmt_insert->error);
            }
            $booking_id = $conn->insert_id;
            $stmt_insert->close();

            // 4. Now, create the Ticket Invoice and LINK it to the booking
            // COMPATIBILITY FIX: Replace ?? with isset() ternary
            $user_name = isset($_SESSION['user_name']) ? $_SESSION['user_name'] : 'Customer';
            $total_fare = ($pax_adults * $current_fare_state['price_adult']) + ($pax_child * $current_fare_state['price_child']) + ($pax_infants * $current_fare_state['price_infant']);
            $grand_total_invoice = $total_fare;

            $sql_invoice = "INSERT INTO ticket_invoices (booking_id, user_id, guest_name, issue_date, status, adult_qty, adult_rate, child_qty, child_rate, infant_qty, infant_rate, total_fare_pkr, grand_total_pkr) VALUES (?, ?, ?, CURDATE(), 'Tentative', ?, ?, ?, ?, ?, ?, ?, ?)";
            $stmt_invoice = $conn->prepare($sql_invoice);
            $stmt_invoice->bind_param("iisidididdd", $booking_id, $user_id, $user_name, $pax_adults, $current_fare_state['price_adult'], $pax_child, $current_fare_state['price_child'], $pax_infants, $current_fare_state['price_infant'], $total_fare, $grand_total_invoice);
            if (!$stmt_invoice->execute()) {
                throw new Exception("Could not create the invoice record. " . $stmt_invoice->error);
            }
            $ticket_invoice_id = $conn->insert_id;

            // 5. Generate and update invoice number
            $invoice_number = "TKT-" . str_pad($ticket_invoice_id, 4, '0', STR_PAD_LEFT);
            $stmt_update_inv_num = $conn->prepare("UPDATE ticket_invoices SET invoice_number = ? WHERE id = ?");
            $stmt_update_inv_num->bind_param("si", $invoice_number, $ticket_invoice_id);
            $stmt_update_inv_num->execute();

            // 6. Insert Passengers into the ticket_invoice_passengers table
            // COMPATIBILITY FIX: Use temporary variables instead of using ?? directly in function calls
            $adults_to_map = isset($_POST['adults']) ? $_POST['adults'] : array();
            $children_to_map = isset($_POST['children']) ? $_POST['children'] : array();
            $infants_to_map = isset($_POST['infants']) ? $_POST['infants'] : array();

            $passengers_to_insert = array_merge(
                array_map(function ($p) {
                    $p['type'] = 'Adult';
                    return $p;
                }, $adults_to_map),
                array_map(function ($p) {
                    $p['type'] = 'Child';
                    return $p;
                }, $children_to_map),
                array_map(function ($p) {
                    $p['type'] = 'Infant';
                    return $p;
                }, $infants_to_map)
            );

            $sql_pass = "INSERT INTO ticket_invoice_passengers (ticket_invoice_id, full_name, passenger_type, passport_no, dob, passport_issue_date, passport_expiry_date) VALUES (?, ?, ?, ?, ?, ?, ?)";
            $stmt_pass = $conn->prepare($sql_pass);
            foreach ($passengers_to_insert as $pax) {
                // COMPATIBILITY FIX: Replace ?? for constructing full name
                $pax_title = isset($pax['title']) ? $pax['title'] : '';
                $pax_given_name = isset($pax['given_name']) ? $pax['given_name'] : '';
                $pax_surname = isset($pax['surname']) ? $pax['surname'] : '';
                $full_name = trim($pax_title . ' ' . $pax_given_name . ' ' . $pax_surname);

                $stmt_pass->bind_param("issssss", $ticket_invoice_id, $full_name, $pax['type'], $pax['passport'], $pax['dob'], $pax['doi'], $pax['doe']);
                $stmt_pass->execute();
            }

            // 7. Parse flight details and insert into ticket_invoice_flights
            $sql_flight = "INSERT INTO ticket_invoice_flights (ticket_invoice_id, airline, flight_no, sector, departure_datetime, arrival_datetime) VALUES (?, ?, ?, ?, ?, ?)";
            $stmt_flight = $conn->prepare($sql_flight);
            $flight_details_for_db = $fare_details['flight_details_json'];
            $fare_year = date('Y', strtotime($fare_details['departure_date']));

            if (!empty($flight_details_for_db['outbound'])) {
                $parsed_outbound = parse_flight_string($flight_details_for_db['outbound'], $fare_year);
                if ($parsed_outbound) {
                    $stmt_flight->bind_param("isssss", $ticket_invoice_id, $fare_details['airline_name'], $parsed_outbound['flight_no'], $parsed_outbound['sector'], $parsed_outbound['departure_datetime'], $parsed_outbound['arrival_datetime']);
                    $stmt_flight->execute();
                }
            }
            if (!empty($flight_details_for_db['inbound'])) {
                $parsed_inbound = parse_flight_string($flight_details_for_db['inbound'], $fare_year);
                if ($parsed_inbound) {
                    $stmt_flight->bind_param("isssss", $ticket_invoice_id, $fare_details['airline_name'], $parsed_inbound['flight_no'], $parsed_inbound['sector'], $parsed_inbound['departure_datetime'], $parsed_inbound['arrival_datetime']);
                    $stmt_flight->execute();
                }
            }

            // 8. Update seat availability
            $new_remaining_seats = $current_fare_state['remaining_seats'] - $seats_needed;
            $sql_update_seats = "UPDATE group_fares SET remaining_seats = ? WHERE id = ?";
            $stmt_update = $conn->prepare($sql_update_seats);
            $stmt_update->bind_param("ii", $new_remaining_seats, $current_fare_state['id']);
            if (!$stmt_update->execute()) {
                throw new Exception("Could not update seat availability.");
            }

            $conn->commit();

            // Success! Prepare details for display
            // COMPATIBILITY FIX: Use older array() syntax
            $booking_details_for_display = array(
                'booking_ref' => $booking_ref,
                'route' => $fare_details['route'],
                'departure_date' => date("D, d M Y", strtotime($fare_details['departure_date'])),
                'passengers' => $pax_adults + $pax_child + $pax_infants,
                'total_price' => number_format($grand_total_invoice, 2),
                'currency' => $current_fare_state['price_currency'],
                'invoice_id' => $ticket_invoice_id
            );
        } catch (Exception $e) {
            $conn->rollback();
            $error_message = "An error occurred: " . $e->getMessage();
        }
    }
} else {
    header("Location: index.php");
    exit;
}

// Dynamic Account Page URL Logic
$account_page_url = 'my-invoices.php';
if (isset($_SESSION['user_type']) && $_SESSION['user_type'] === 'agent') {
    $account_page_url = 'agent-invoices.php';
}
?>
<!DOCTYPE html>
<html lang="en">

<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Booking Status - RF Travel & Tours</title>
    <link rel="icon" type="image/png" href="images/logo-icon.png">

    <link rel="stylesheet" href="css/style.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.2.0/css/all.min.css">
    <style>
        :root {
            --primary-blue: #31a7e2;
            --secondary-blue: #31a7e2;
            --light-grey-bg: #f4f7f9;
            --border-color: #e4e7eb;
            --text-dark: #212529;
            --text-light: #6c757d;
            --white: #fff;
            --green: #28a745;
            --red: #dc3545;
            --box-shadow: 0 10px 30px rgba(0, 0, 0, 0.08);
        }

        body {
            font-family: 'Poppins', sans-serif;
            background-color: var(--light-grey-bg);
        }

        .booking-status-page {
            padding: 4rem 1rem;
            min-height: 60vh;
            display: flex;
            align-items: center;
            justify-content: center;
        }

        .status-card {
            background: var(--white);
            border-radius: 12px;
            box-shadow: var(--box-shadow);
            max-width: 650px;
            width: 100%;
            margin: 0 auto;
            text-align: center;
            padding: 3rem;
            border-top: 5px solid;
        }

        .status-icon {
            font-size: 5rem;
            line-height: 1;
            margin-bottom: 1.5rem;
        }

        .status-card h1 {
            font-size: 2rem;
            font-weight: 700;
            color: var(--text-dark);
            margin: 0 0 0.75rem 0;
        }

        .status-card .subtitle {
            font-size: 1.1rem;
            color: var(--text-light);
            margin-bottom: 2.5rem;
        }

        .status-card.success {
            border-color: var(--green);
        }

        .status-card.success .status-icon {
            color: var(--green);
        }

        .booking-details-wrapper {
            border: 1px solid var(--border-color);
            border-radius: 8px;
            background-color: #fafafa;
            margin-bottom: 2.5rem;
            text-align: left;
        }

        .booking-ref-panel {
            padding: 1.5rem;
            border-bottom: 1px solid var(--border-color);
        }

        .booking-ref-panel p {
            margin: 0 0 0.5rem 0;
            color: var(--text-light);
            font-size: 0.9rem;
        }

        .booking-ref-panel strong {
            font-size: 1.75rem;
            color: var(--primary-blue);
            font-weight: 700;
            letter-spacing: 1px;
        }

        .booking-summary-panel {
            padding: 1.5rem;
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(150px, 1fr));
            gap: 1.5rem;
        }

        .summary-item strong {
            display: block;
            color: var(--text-light);
            font-size: 0.8rem;
            font-weight: 500;
            margin-bottom: 0.25rem;
            text-transform: uppercase;
        }

        .summary-item span {
            font-weight: 600;
            font-size: 1rem;
            color: var(--text-dark);
        }

        .status-card.failed {
            border-color: var(--red);
        }

        .status-card.failed .status-icon {
            color: var(--red);
        }

        .error-text {
            background-color: #fff5f5;
            color: #c53030;
            border: 1px solid #fed7d7;
            padding: 1rem;
            border-radius: 8px;
            margin-bottom: 2rem;
        }

        .card-actions {
            display: flex;
            flex-wrap: wrap;
            gap: 1rem;
            justify-content: center;
        }

        .btn {
            text-decoration: none;
            padding: 0.8rem 1.8rem;
            border-radius: 6px;
            font-weight: 600;
            transition: all 0.2s ease;
            font-size: 0.95rem;
            border: 1px solid transparent;
        }

        .btn-primary {
            background-color: var(--primary-blue);
            color: var(--white);
        }

        .btn-primary:hover {
            background-color: #31a7e2;
        }

        .btn-secondary {
            background-color: var(--white);
            color: var(--secondary-blue);
            border-color: var(--border-color);
        }

        .btn-secondary:hover {
            border-color: var(--secondary-blue);
        }
    </style>
</head>

<body>
    <?php include 'header.php'; ?>
    <main class="booking-status-page">
        <?php if ($error_message): ?>
            <div class="status-card failed">
                <div class="status-icon"><i class="fas fa-times-circle"></i></div>
                <h1>Booking Failed</h1>
                <p class="subtitle">Unfortunately, we couldn't process your booking at this time.</p>
                <div class="error-text"><strong>Reason:</strong> <?= htmlspecialchars($error_message) ?></div>
                <div class="card-actions">
                    <a href="group-booking-detail.php?ref=<?= urlencode($group_ref_id_for_redirect) ?>" class="btn btn-primary">Go Back & Try Again</a>
                    <a href="contact-us.php" class="btn btn-secondary">Contact Support</a>
                </div>
            </div>
        <?php else: ?>
            <div class="status-card success">
                <div class="status-icon"><i class="fas fa-check-circle"></i></div>
                <h1>Booking Confirmed & Invoice Created!</h1>
                <p class="subtitle">Your reservation and invoice are now in our system. You can view the full details in your account.</p>
                <div class="booking-details-wrapper">
                    <div class="booking-ref-panel">
                        <p>Your Booking Reference</p>
                        <strong><?= htmlspecialchars($booking_details_for_display['booking_ref']) ?></strong>
                    </div>
                    <div class="booking-summary-panel">
                        <div class="summary-item"><strong>Route</strong><span><?= htmlspecialchars($booking_details_for_display['route']) ?></span></div>
                        <div class="summary-item"><strong>Departure</strong><span><?= htmlspecialchars($booking_details_for_display['departure_date']) ?></span></div>
                        <div class="summary-item"><strong>Passengers</strong><span><?= htmlspecialchars($booking_details_for_display['passengers']) ?></span></div>
                        <div class="summary-item"><strong>Total Fare</strong><span><?= htmlspecialchars($booking_details_for_display['currency']) ?> <?= htmlspecialchars($booking_details_for_display['total_price']) ?></span></div>
                    </div>
                </div>
                <div class="card-actions">
                    <a href="<?= htmlspecialchars($account_page_url) ?>" class="btn btn-primary"><i class="fas fa-file-invoice"></i> View My Invoices</a>
                    <a href="index.php" class="btn btn-secondary"><i class="fas fa-home"></i> Return to Homepage</a>
                </div>
            </div>
        <?php endif; ?>
    </main>
    <?php include 'footer.php'; ?>
</body>

</html>